/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.xml.refactoring;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.modules.xml.xam.Component;
import org.netbeans.modules.xml.xam.ComponentEvent;
import org.netbeans.modules.xml.xam.ComponentListener;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.Referenceable;
import org.openide.filesystems.FileObject;
import org.openide.util.WeakListeners;

/**
 * This class represents an atomic workset of usages for a usage search or 
 * a refactoring request.  Any change to sources within the scope of request
 * (currently project) would invalidate the usage set and undo/redo state of
 * the refactoring change..
 *
 * @author Nam Nguyen
 */

public class UsageSet implements PropertyChangeListener, ComponentListener {
    public static final String VALID_PROPERTY = "valid";
    private Model targetModel;
    private Referenceable target;
    private Set<UsageGroup> usageGroups;
    private boolean valid;
    private PropertyChangeSupport pcs;
    
    /** Creates a new instance of RefactoringPreviewSet */
    public UsageSet(Referenceable target) {
        assert target != null;
        this.target = target;
        targetModel = RefactorRequest.getModel(target);
        assert targetModel != null;
        pcs = new PropertyChangeSupport(this);
        valid = true;
        usageGroups = new HashSet<UsageGroup>();
        addModelListeners(targetModel);
    }
    
    private void addModelListeners(Model model) {
        model.addPropertyChangeListener(WeakListeners.propertyChange(this, model));
        model.addComponentListener((ComponentListener) 
            WeakListeners.create(ComponentListener.class, this, model));
    }
    
    public void addPropertyChangeListener(PropertyChangeListener pcl) {
        pcs.addPropertyChangeListener(pcl);
    }

    public void removePropertyChangeListener(PropertyChangeListener pcl) {
        pcs.removePropertyChangeListener(pcl);
    }

    public Collection<UsageGroup> getUsages() {
        return usageGroups;
    }
    
    public Set<Model> getModels() {
        Set<Model> models = new HashSet<Model>();
        for (UsageGroup ug : usageGroups ) {
            if (ug.getModel() != null) {
                models.add(ug.getModel());
            }
        }
        return models;
    }
    
    public Map<SourceGroup, Map<FileObject, Set<UsageGroup>>> getSortedUsages() {
        Map<SourceGroup,Map<FileObject,Set<UsageGroup>>> result = new HashMap<SourceGroup,Map<FileObject,Set<UsageGroup>>>();
        for (UsageGroup u : getUsages()) {
            Map<FileObject,Set<UsageGroup>> usagesInGroup = result.get(u.getSourceGroup());
            if (usagesInGroup == null) {
                usagesInGroup = new HashMap<FileObject,Set<UsageGroup>>();
                result.put(u.getSourceGroup(), usagesInGroup);
            }
            
            Set<UsageGroup> usagesInFolder = usagesInGroup.get(u.getFolder());
            if (usagesInFolder == null) {
                usagesInFolder = new HashSet<UsageGroup>();
                usagesInGroup.put(u.getFolder(), usagesInFolder);
            }
            usagesInFolder.add(u);
        }
        return result;
    }
    
    public void addUsages(List<UsageGroup> usages) {
        if (usages == null) return;
        for (UsageGroup u : usages) {
            addUsage(u);
        }
    }
    
    public void addUsage(UsageGroup p) {
        if (p.getErrors().isEmpty()) {
            assert p.getModel() != null : "Usage group does not have model"; //NOI18N
        }
        if (! p.getItems().isEmpty() || ! p.getErrors().isEmpty()) {
            addModelListeners(p.getModel());
            usageGroups.add(p);
        }
    }
    
    public void removeUsage(UsageGroup p) {
        usageGroups.remove(p);
    }
    
    public boolean isEmpty() {
        return isEmpty(true);
    }
    public boolean isEmpty(boolean countErrors) {
        for (UsageGroup ug : usageGroups) {
            if (! ug.isEmpty(countErrors)) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Any change to components in usages set invalidate the usage set.
     */
    public boolean isValid() {
        return valid;
    }
    
    protected void setValid(boolean valid) {
        boolean old = isValid();
        this.valid = valid;
        pcs.firePropertyChange(VALID_PROPERTY, old, valid);
    }
    
    private boolean isMySource(Object source) {
        if (source == target) {
            return true;
        }
        Model sourceModel = null;
        if (source instanceof Model) {
            sourceModel = (Model) source;
        } else if (source instanceof Component) {
            sourceModel = ((Component)source).getModel();
        }
        if (sourceModel == null) {
            return false;
        }
        if (targetModel == sourceModel) {
            return true;
        }
        for (UsageGroup u : getUsages()) {
            if (u.getModel() == sourceModel) {
                return true;
            }
        }
        return false;
    }
    
    public void propertyChange(PropertyChangeEvent evt) {
        handleChange(evt.getSource());
    }
    
    private void handleChange(Object source) {
        // ignore changes from the refactoring itself
        if (RefactoringManager.getInstance().isRunning(this) || //
            ! isValid() || ! isMySource(source)) {
            return;
        }
        setValid(false);
    }

    public void valueChanged(ComponentEvent evt) {
        handleChange(evt.getSource());
    }

    public void childrenDeleted(ComponentEvent evt) {
        handleChange(evt.getSource());
    }

    public void childrenAdded(ComponentEvent evt) {
        handleChange(evt.getSource());
    }
}
