/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.xml.refactoring.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.api.project.Sources;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.modules.xml.refactoring.DeleteRequest;
import org.netbeans.modules.xml.refactoring.ErrorItem;
import org.netbeans.modules.xml.refactoring.FileRenameRequest;
import org.netbeans.modules.xml.refactoring.RefactorRequest;
import org.netbeans.modules.xml.refactoring.RenameRequest;
import org.netbeans.modules.xml.refactoring.Usage;
import org.netbeans.modules.xml.retriever.catalog.ProjectCatalogSupport;
import org.netbeans.modules.xml.xam.Component;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.ModelSource;
import org.netbeans.modules.xml.xam.Nameable;
import org.netbeans.modules.xml.xam.Referenceable;
import org.netbeans.modules.xml.xam.dom.Utils;
import org.netbeans.spi.project.SubprojectProvider;
import org.openide.cookies.SaveCookie;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.util.NbBundle;

/**
 *
 * @author Nam Nguyen
 */
public class RefactoringUtil {
    
    public static Project findCurrentProject(Referenceable referenced) {
        Model model = referenced instanceof Model ? (Model) referenced : ((Component)referenced).getModel();
        if (model == null) return null;
        return FileOwnerQuery.getOwner((FileObject)
            model.getModelSource().getLookup().lookup(FileObject.class));
    }
    
    public static Set<Project> getReferencingProjects(Project project) {
        Set<Project> result = new HashSet<Project>();
        for (Project p : OpenProjects.getDefault().getOpenProjects()) {
            if (p.getLookup().lookup(ProjectCatalogSupport.class) == null) {
                continue;
            }
        
            SubprojectProvider spp = (SubprojectProvider) p.getLookup().lookup(SubprojectProvider.class);
            if (spp == null) continue;
            for (Object o : spp.getSubprojects()) {
                Project sp = (Project) o;
                if (sp == project) {
                    result.add(p);
                    break;
                }
            }
        }

        return result;
    }
    
    public static List<SourceGroup> findSourceRoots(Project project) {
	// get the generic roots so that all roots will be identified
	SourceGroup[] groups =
	        ProjectUtils.getSources(project).getSourceGroups(Sources.TYPE_GENERIC);
        return Arrays.asList(groups);
    }
    
    public static List<FileObject> findSourceFiles(FileObject folder) {
        Enumeration children = folder.getChildren(true);
        List<FileObject> ret = new ArrayList<FileObject>();
        while (children.hasMoreElements()) {
            FileObject fo = (FileObject) children.nextElement();
            if (fo.isData()) {
                ret.add(fo);
            }
        }
        return ret;
    }
    
    public static void precheckTarget(RefactorRequest request) {
        Model model = request.getTargetModel();
        if (model.getState() != Model.State.VALID) {
            String msg = NbBundle.getMessage(RefactoringUtil.class, "MSG_ModelSourceNotWelformed");
            request.addError(new ErrorItem(model, msg));
        }
        if (request.getAutosave() && ! RefactoringUtil.isWritable(model)) {
            String msg = NbBundle.getMessage(RefactoringUtil.class, "MSG_ModelSourceNotWritable");
            request.addError(new ErrorItem(model, msg));
        }
    }

    public static void precheckUsageModels(RefactorRequest request) {
        Set<Model> models = request.getUsages().getModels();
        for (Model model : models) {
            if (model.getState() != Model.State.VALID) {
                String msg = NbBundle.getMessage(RefactoringUtil.class, "MSG_ModelSourceNotWelformed");
                request.addError(new ErrorItem(model, msg));
            }
            if (request.getAutosave() && ! RefactoringUtil.isWritable(model)) {
                String msg = NbBundle.getMessage(RefactoringUtil.class, "MSG_ModelSourceNotWritable");
                request.addError(new ErrorItem(model, msg));
            }
        }
    }
    
    public static void precheckForUnsafeDelete(DeleteRequest request) {
        if (! request.getUsages().getUsages().isEmpty()) {
            String msg = NbBundle.getMessage(RefactoringUtil.class, "MSG_UnsafeDelete");
            request.addError(new ErrorItem(request.getTarget(), msg));
        }
    }
    
    public static String getDescription(RefactorRequest request) {
        if (request instanceof RenameRequest) {
            return NbBundle.getMessage(RefactoringUtil.class, "LBL_Rename"); //NOI18N
        } else if (request instanceof DeleteRequest) {
            return NbBundle.getMessage(RefactoringUtil.class, "LBL_Safe_Delete"); //NOI18N
        } else if (request instanceof FileRenameRequest) {
            return NbBundle.getMessage(RefactoringUtil.class, "LBL_File_Rename"); //NOI18N
        } else {
            return "";  //NOI18N
        }
    }

    public static boolean isDirty(Model model) {
        DataObject obj = (DataObject) model.getModelSource().getLookup().lookup(DataObject.class);
        if (obj != null) {
            return obj.isModified();
        }
        return false;
    }

    public static DataObject getDataObject(Model model) {
        return (DataObject) model.getModelSource().getLookup().lookup(DataObject.class);
    }
    
    public static void saveTargetFile(FileRenameRequest request) {
        Model target = request.getTarget();
        Set<Model> all = request.getUsages().getModels();
        all.remove(target);
        save(request, all);
    }
    
    public static void save(RefactorRequest request, Set<Model> excludeds) {
        Set<Model> all = request.getUsages().getModels();
        all.add(request.getTargetModel());
        for (Model model : all) {
            if (excludeds.contains(model)) {
                continue;
            }
            DataObject obj = getDataObject(model);
            if (obj == null) {
                String msg = NbBundle.getMessage(RefactoringUtil.class, "MSG_CannotFindDataObject");
                request.addError(new ErrorItem(model, msg));
                continue;
            }
            
            SaveCookie save = (SaveCookie) obj.getCookie(SaveCookie.class);
            FileObject fo = obj.getPrimaryFile();
            if (save != null) {
                try {
                    save.save();
                    obj.setModified(false);
                } catch (IOException ioe) {
                    String msg = NbBundle.getMessage(RefactoringUtil.class, "MSG_ErrorSave", fo.getPath(), ioe.getMessage());
                    request.addError(new ErrorItem(model, msg));
                    continue;
                }
            } else {
                String msg = NbBundle.getMessage(RefactoringUtil.class, "MSG_CannotSave", fo.getPath());
                request.addError(new ErrorItem(model, msg));
                continue;
            }
        }
            
    }
    
    public static String getDescription(Usage.Type type) {
        switch(type) {
            case GENERALIZATION:
                return NbBundle.getMessage(RefactoringUtil.class, "LBL_Generalization");
            case REFERENCE:
                return NbBundle.getMessage(RefactoringUtil.class, "LBL_Reference");
            default:
                assert false : "Invalid type " + type;
                return "";
        }
    }
    
    public static ErrorItem precheck(RenameRequest request) {
        if (request.getNewName() == null || !Utils.isValidNCName(request.getNewName())) {
            return new ErrorItem(request.getTarget(),
                    NbBundle.getMessage(RefactoringUtil.class, "MSG_NewNameNullEmpty"),
                    ErrorItem.Level.FATAL);
        } else if (! checkDuplicateName(request)) {
            return new ErrorItem(request.getTarget(),
                    NbBundle.getMessage(RefactoringUtil.class, "MSG_NewNameDuplicate"),
                    ErrorItem.Level.FATAL);
        }
        return null;
    }

    public static ErrorItem precheck(FileRenameRequest request) {
        FileObject current = request.getFileObject();
        FileObject parent = current.getParent();
        assert (parent != null) : "Source file has no parent folder";
        String newName = request.getNewFileName();
        if (newName == null || newName.trim().length() == 0 ||
            parent.getFileObject(newName, current.getExt()) != null) 
        {
            return new ErrorItem(request.getTarget(),
                    NbBundle.getMessage(RefactoringUtil.class, "MSG_NewNameDuplicate"),
                    ErrorItem.Level.FATAL);
        }
        return null;
    }

    /**
     * Returns true if the check result is OK.
     */
    public static boolean checkDuplicateName(RenameRequest request) {
        Nameable component = request.getNameableTarget();
        Component parent = component.getParent();
        Collection<Component> siblings = parent.getChildren(component.getClass());
        for (Component c : siblings) {
            Nameable nameable = (Nameable)c;
            if (nameable.getName() != null && nameable.getName().equals(request.getNewName())) {
                return false;
            }
        }
        return true;
    }


    public static boolean isWritable(Model model) {
        if (model != null) {
            ModelSource ms = model.getModelSource();
            if (ms.isEditable()) {
                FileObject fo = (FileObject) ms.getLookup().lookup(FileObject.class);
                if (fo != null) {
                    return fo.canWrite();
                }
            }
        }
        return false;
    }
}
