/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

/*
 * RetrieverTask.java
 *
 * Created on January 9, 2006, 6:50 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.netbeans.modules.xml.retriever;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.modules.xml.retriever.catalog.Utilities;
import org.openide.filesystems.FileLock;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.NbBundle;

/**
 *
 * @author girix
 */
public class RetrieverTask {
    
    
    
    private File saveFile;
    
    private String baseAddress;
    
    private String sourceToBeGot;
    
    private RetrieveEntry rent;
    
    private RetrieverEngine retEngine = null;
    
    /** Creates a new instance of RetrieverTask */
    public RetrieverTask(RetrieveEntry rent, RetrieverEngine retEngine){
        this.retEngine = retEngine;
        this.sourceToBeGot = rent.getCurrentAddress();
        this.baseAddress = rent.getBaseAddress();
        this.saveFile = rent.getSaveFile();
        this.rent = rent;
    }
    
    
    public HashMap<String,File> goGetIt() throws IOException, URISyntaxException{
        synchronized(RetrieverTask.class){
            if(( saveFile != null ) && (saveFile.isFile()) && (saveFile.length() != 0)){
                //String newfile = saveFile.getParentFile().toString()+File.pathSeparator+saveFile.getName()+System.currentTimeMillis();
                //saveFile = new File(newfile);
                
                //this will prevent a cycle in recursion.
                throw new IOException("File already exists"); //NOI18N
            }
            ResourceRetriever rr = ResourceRetrieverFactory.getResourceRetriever(baseAddress, sourceToBeGot);
            if(rr == null )
                throw new RuntimeException("No Retriever for this Resource address :"+sourceToBeGot); //NOI18N
            
            if(isAlreadyDownloadedInThisSession(rr.getEffectiveAddress(baseAddress , sourceToBeGot))){
                String fileExists = NbBundle.getMessage(RetrieverTask.class,
                        IConstants.EXCEPTION_CYCLIC_REFERENCE_INDICATOR);
                throw new IOException(fileExists);
            }
            
            HashMap<String, InputStream> srcAddrNContent = rr.retrieveDocument(baseAddress , sourceToBeGot);
            if(srcAddrNContent == null)
                return null;
            String effectiveSrcAddr = srcAddrNContent.keySet().iterator().next();
            InputStream is = srcAddrNContent.get(effectiveSrcAddr);
            rent.setEffectiveAddress(effectiveSrcAddr);
            if(saveFile == null)
                saveFile = guessSaveFile(rent);
            if(saveFile == null)
                throw new IOException("Could not determine the save file."); //NOI18N
            
            checkForCycle(saveFile, rr.getStreamLength(), is);
            
            if(retEngine.isSave2SingleFolder() && !retEngine.getFileOverwrite() ){
                //this stream is diff but same file name so give another name
                int i = 0;
                File curFile = saveFile;
                String fileName = saveFile.getName();
                while(curFile.isFile())
                    curFile = new File(retEngine.getCurrentSaveRootFile()+File.separator+"new"+i+++fileName);
                saveFile = curFile;
            }
            
            BufferedInputStream bis = new BufferedInputStream(is, 1024);
            FileObject saveFileObject = FileUtil.toFileObject(FileUtil.normalizeFile(saveFile));
            //create datafile and also all the parents folders
            saveFile.getParentFile().mkdirs();
            FileObject parent = FileUtil.toFileObject(FileUtil.normalizeFile(saveFile.getParentFile()));
            saveFileObject = FileUtil.createData(parent, saveFile.getName());
            FileLock saveFileLock = saveFileObject.lock();
            try{
                OutputStream saveFileOutputStream = saveFileObject.getOutputStream(saveFileLock);
                BufferedOutputStream bos = new BufferedOutputStream(saveFileOutputStream, 1024);
                byte[] buffer = new byte[1024];
                int len = 0;
                while((len = bis.read(buffer)) != -1){
                    bos.write(buffer, 0, len);
                }
                bos.close();
                bis.close();
            }finally{
                //release the lock at any cost
                saveFileLock.releaseLock();
            }
            HashMap<String, File> result = new HashMap<String, File>();
            String modifiedFileExtn = null;
            try {
                modifiedFileExtn = new DocumentTypeSchemaWsdlParser().getFileExtensionByParsing(saveFile);
                if(modifiedFileExtn != null){
                    if(!saveFileObject.getNameExt().endsWith("."+modifiedFileExtn)){
                        String fileName = saveFileObject.getNameExt();
                        if(saveFileObject.getNameExt().endsWith("_"+modifiedFileExtn)){
                            fileName = fileName.substring(0, fileName.length() -
                                    ("_"+modifiedFileExtn).length());
                        }
                        
                        File newFile = new File(saveFile.getParent()+File.separator+fileName+"."+modifiedFileExtn);
                        if(newFile.isFile() && retEngine.getFileOverwrite())
                            newFile.delete();
                        
                        FileObject newsaveFileObject = FileUtil.copyFile(saveFileObject, saveFileObject.getParent(),
                                fileName, modifiedFileExtn);
                        saveFileObject.delete();
                        saveFileObject = newsaveFileObject;
                    }
                }
            } catch (Exception ex) {
                //this is just a rename. So, just ignore any exceptions.
            }
            
            result.put(effectiveSrcAddr, FileUtil.toFile(saveFileObject));
            //commented out the ref file generation
            //createReferenceFile(effectiveSrcAddr, saveFile);
            return result;
        }
    }
    
    
    private File guessSaveFile(RetrieveEntry rent) throws URISyntaxException, IOException{
        if(rent.getSaveFile() != null)
            return rent.getSaveFile();
        URI curUri = new URI(rent.getEffectiveAddress());
        //get file name
        String curAddr = rent.getEffectiveAddress();
        String curFileName = null;
        int index = curAddr.lastIndexOf("/"); //NOI18N
        if(index != -1){
            curFileName = curAddr.substring(index+1);
        }else{
            curFileName = curAddr;
        }
        
        if(retEngine.isSave2SingleFolder()){
            curFileName = convertAllSpecialChars(curFileName);
            return new File(retEngine.getCurrentSaveRootFile()+File.separator+curFileName);
        }
        
        File result = null;
        
        //get directory to be stored
        if(curUri.isAbsolute()){
            if("http".equalsIgnoreCase(curUri.getScheme()) || "https".equalsIgnoreCase(curUri.getScheme())) { //NOI18N
                //treat URLs differently
                result = getSaveFileForURL(curUri);
            } else{
                result = new File(new URI(retEngine.getFixedSaveRootFolder().toURI().toString()+"/"+curFileName));
            }
        }else{
            File newFile = new File(new URI(rent.getLocalBaseFile().getParentFile().toURI().normalize().toString()+"/"+rent.getCurrentAddress())).getCanonicalFile();
            File newParentFile = getModifiedParentFile(rent.getLocalBaseFile(), newFile);
            if(rent.getLocalBaseFile() != newParentFile)
                result = new File(new URI(newParentFile.getParentFile().toURI().toString()+"/"+rent.getCurrentAddress()));
            else
                result = newFile;
        }
        return result;
    }
    
    private File getModifiedParentFile(File parentFile, File curSaveFile) {
        File result = parentFile;
        
        String curSaveStr = curSaveFile.toURI().toString();
        String saveRootStr = retEngine.getFixedSaveRootFolder().toURI().toString();
        
        if(curSaveStr.startsWith(saveRootStr)){
            return result;
        }
        File newRootFile = null;
        int pushCount = Utilities.countPushdownFolders(curSaveFile.toURI(), retEngine.getFixedSaveRootFolder().toURI());
        retEngine.pushDownRoot(pushCount);
        result = retEngine.getNewFileForOld(parentFile, pushCount);
        return result;
    }
    
    private File getSaveFileForURL(URI absURI) {
        String rootFolderStr = retEngine.getFixedSaveRootFolder().toURI().toString();
        String absURIStr = absURI.toString();
        //replace http:// with the saverootfolder
        String resultStr = absURI.getSchemeSpecificPart().replace(':','_');
        resultStr = resultStr.replace('?', '.');
        if(resultStr.lastIndexOf(".") != -1 ){
            String fileExtension = resultStr.substring(resultStr.lastIndexOf("."), resultStr.length());
            
            if(!fileExtension.equals(fileExtension.toLowerCase())){
                resultStr = resultStr.substring(0, resultStr.lastIndexOf("."))+fileExtension.toLowerCase();
            }
        }
        resultStr = convertAllSpecialChars(resultStr);
        resultStr = rootFolderStr+"/"+resultStr;
        try {
            return new File(new URI(resultStr).normalize());
        } catch (URISyntaxException ex) {
            return null;
        }
    }
    
    
    public String convertAllSpecialChars(String resultStr){
        StringBuffer sb = new StringBuffer(resultStr);
        for(int i = 0; i < sb.length(); i++){
            char c = sb.charAt(i);
            if( Character.isLetterOrDigit(c) ||
                    (c == '/') ||
                    (c == '.') ||
                    (c == '_') ||
                    (c == ' ') ||
                    (c == '-')){
                continue;
            }else{
                sb.setCharAt(i, '_');
            }
        }
        return sb.toString();
    }
    
    
    private void createReferenceFile(String effectiveSrcAddr, File saveFile) {
        Properties prop = new Properties();
        prop.setProperty(IConstants.SOURCE_URL_PROP_KEY, effectiveSrcAddr);
        File propsFile = new File(saveFile.toString()+IConstants.SOURCE_FILE_PROPS_FILE_EXTN);
        try{
            BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(propsFile));
            prop.store(bos," This file is a A pair of:"+saveFile+" . Do not delete."); //NOI18N
        }catch(IOException e){
            Logger.getLogger(getClass().getName()).log(Level.FINE, "createReferenceFile", e);
        }
    }
    
    private void checkForCycle(File saveFile, long l, InputStream is) throws IOException {
        String fileExists = NbBundle.getMessage(RetrieverTask.class,
                IConstants.EXCEPTION_CYCLIC_REFERENCE_INDICATOR);
        if(saveFile.isFile()){
            if( (isAlreadyDownloadedInThisSession(saveFile)) ||
                    ((saveFile.length() == l) && !retEngine.getFileOverwrite()) ) {
                //file is already there...Breaks cyclic link traversals
                is.close();
                throw new IOException(fileExists+" : "+saveFile);
            }
            if(retEngine.getFileOverwrite()){
                //let the retriever overwrite
                return;
            } else{
                //dont overwrite.
                is.close();
                throw new IOException(fileExists+" : "+saveFile);
            }
        }
        if(saveFile.isDirectory()){
            is.close();
            String dirExists = NbBundle.getMessage(RetrieverTask.class,
                    IConstants.EXCEPTION_DIRECTORY_ALREADY_EXISTS);
            throw new IOException(dirExists + " : "+saveFile.getCanonicalPath()); //NOI18N
        }
    }
    
    private boolean isAlreadyDownloadedInThisSession(File thisFile){
        for(RetrieveEntry rent : retEngine.retrievedList){
            if(rent.getSaveFile().equals(thisFile))
                return true;
        }
        return false;
    }
    
    private boolean isAlreadyDownloadedInThisSession(String uri){
        for(RetrieveEntry rent : retEngine.retrievedList){
            if(rent.getEffectiveAddress().equals(uri))
                return true;
        }
        return false;
    }
}
