/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.xml.schema.core;

import java.util.Collection;
import java.util.List;
import javax.swing.undo.UndoManager;
import junit.framework.*;
import org.netbeans.api.project.Project;
import org.netbeans.modules.xml.refactoring.DeleteRequest;
import org.netbeans.modules.xml.refactoring.ErrorItem;
import org.netbeans.modules.xml.refactoring.FileRenameRequest;
import org.netbeans.modules.xml.refactoring.FindUsageResult;
import org.netbeans.modules.xml.refactoring.RefactoringManager;
import org.netbeans.modules.xml.refactoring.RenameRequest;
import org.netbeans.modules.xml.refactoring.Usage;
import org.netbeans.modules.xml.refactoring.UsageGroup;
import org.netbeans.modules.xml.refactoring.UsageSet;
import org.netbeans.modules.xml.refactoring.actions.RefactoringRedoAction;
import org.netbeans.modules.xml.refactoring.actions.RefactoringUndoAction;
import org.netbeans.modules.xml.refactoring.impl.RefactoringUtil;
import org.netbeans.modules.xml.schema.model.GlobalComplexType;
import org.netbeans.modules.xml.schema.model.GlobalElement;
import org.netbeans.modules.xml.schema.model.LocalElement;
import org.netbeans.modules.xml.schema.model.ReferenceableSchemaComponent;
import org.netbeans.modules.xml.schema.model.SchemaModel;
import org.netbeans.modules.xml.schema.model.SchemaModelFactory;
import org.netbeans.modules.xml.schema.model.Sequence;
import org.netbeans.modules.xml.xam.Component;
import org.netbeans.modules.xml.xam.ModelSource;
import org.netbeans.modules.xml.retriever.catalog.Utilities;
import org.openide.cookies.SaveCookie;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.util.actions.SystemAction;

/**
 *
 * @author nn136682
 */
public class SchemaRefactoringTest extends TestCase {
    public static final String XSD_PO = "PurchaseOrder.xsd";
    public static final String XSD_LA = "LoanApplication.xsd";
    
    private Project project;
    private SchemaModel model;
    private FileObject purchaseOrderFO;
    
    public SchemaRefactoringTest(String testName) {
        super(testName);
    }

    protected void setUp() throws Exception {
        project = Util.createJavaTestProject();
        purchaseOrderFO = Util.populateProject(
                project, "resources/"+XSD_PO, "src/com/acme/schemas");
        ModelSource ms = Utilities.getModelSource(purchaseOrderFO, true);
        model = SchemaModelFactory.getDefault().getModel(ms);
    }

    protected void tearDown() throws Exception {
        deleteModelFile(purchaseOrderFO);
    }
    
    private static void deleteModelFile(FileObject fo) throws Exception {
        if (fo == null) return;
        SchemaDataObject sdo = (SchemaDataObject) DataObject.find(fo);
        if (sdo != null) {
            SaveCookie save = (SaveCookie) sdo.getCookie(SaveCookie.class);
            if (save != null) save.save();
        }
        fo.delete();
    }

    public static Test suite() {
        TestSuite suite = new TestSuite(SchemaRefactoringTest.class);
        return suite;
    }

    private static int totalItems(UsageSet set) {
        int count = 0;
        for (UsageGroup u : set.getUsages()) {
            count += u.getItems().size();
        }
        return count;
    }
    
    public void testFindUsages() throws Exception {
        GlobalComplexType poType = model.getSchema().getComplexTypes().iterator().next();
        FindUsageResult result = RefactoringManager.getInstance().findUsages(poType);
        assertEquals(1, result.get().getUsages().iterator().next().getItems().size());
        assertTrue(result.get().getUsages().iterator().next().getErrors().isEmpty());
        UsageSet us = result.get();
        UsageGroup u = us.getUsages().iterator().next();
        assertNotNull(u.getSourceGroup());
        assertNotNull(u.getTarget());
        assertNotNull(u.getTargetComponent());
        assertNotNull(u.getModel());
        assertNotNull(u.getTargetModel());
        assertNotNull(u.getEngine());
        assertNotNull(u.getFileObject());
        Collection<Usage> items = u.getItems();
        Usage item = items.iterator().next();
        assertNotNull(item.getContainer());
        assertEquals(2, item.getPathFromRoot().size());
        assertEquals(Usage.Type.REFERENCE, item.getType());
        Component referencing = item.getComponent();
        assertEquals("purchaseOrder", ((GlobalElement)referencing).getName());
    }

    public void testRefactorUsagesUndoRedo() throws Exception {
        UndoManager myUndoManager = new UndoManager();
        model.addUndoableEditListener(myUndoManager);
        
        ReferenceableSchemaComponent poType = Util.findGlobalComponentByName(model.getSchema(), "PurchaseOrderType");
        RenameRequest request = new RenameRequest(poType, "MyPoType");
        List<ErrorItem> errors = RefactoringManager.getInstance().precheckUsages(request);
        assertEquals(0, errors.size());
        RefactoringManager.getInstance().process(request);
        GlobalElement ge = (GlobalElement)Util.findGlobalComponentByName(model.getSchema(), "purchaseOrder");
        assertEquals("po:MyPoType", ge.getType().getRefString());
        
        assertFalse("other undomanager should be emptied", myUndoManager.canUndo());
        assertFalse("other undomanager should be emptied", myUndoManager.canRedo());
        assertTrue(RefactoringManager.getInstance().canUndo());
        RefactoringManager.getInstance().undo(); 
        poType = Util.findGlobalComponentByName(model.getSchema(), "PurchaseOrderType");
        assertNotNull(poType);
        assertEquals("po:PurchaseOrderType", ge.getType().getRefString());
        assertFalse(RefactoringManager.getInstance().canUndo());
        
        assertTrue(RefactoringManager.getInstance().canRedo());
        RefactoringManager.getInstance().redo(); 
        poType = Util.findGlobalComponentByName(model.getSchema(), "MyPoType");
        assertNotNull(poType);
        assertEquals("po:MyPoType", ge.getType().getRefString());
        assertFalse(RefactoringManager.getInstance().canRedo());

        assertTrue(RefactoringManager.getInstance().canUndo());
        assertTrue(RefactoringManager.getInstance().canUndo());
        RefactoringManager.getInstance().undo(); 
        poType = Util.findGlobalComponentByName(model.getSchema(), "PurchaseOrderType");
        assertNotNull(poType);
        assertEquals("po:PurchaseOrderType", ge.getType().getRefString());
        assertFalse(RefactoringManager.getInstance().canUndo());
        
        assertTrue(RefactoringManager.getInstance().canRedo());
        assertTrue(RefactoringManager.getInstance().canRedo());
        RefactoringManager.getInstance().redo(); 
        poType = Util.findGlobalComponentByName(model.getSchema(), "MyPoType");
        assertNotNull(poType);
        assertEquals("po:MyPoType", ge.getType().getRefString());
        assertFalse(RefactoringManager.getInstance().canRedo());

        model.startTransaction();
        poType.setId("foo");
        model.endTransaction();
        
        assertTrue("other undomanager should be intact", myUndoManager.canUndo());
    }

    public void testMultiFiles() throws Exception {
        FileObject loanApplicationFO = Util.populateProject(
                project, "resources/"+XSD_LA, "src/com/acme/schemas");
        try {
        ModelSource ms = Utilities.getModelSource(loanApplicationFO, true);
        SchemaModel laModel = SchemaModelFactory.getDefault().getModel(ms);
        
        ReferenceableSchemaComponent addressType = Util.findGlobalComponentByName(laModel.getSchema(), "AddressType");
        FindUsageResult result = RefactoringManager.getInstance().findUsages(addressType);
        UsageSet us = result.get();
        assertEquals(2, us.getUsages().size());
        assertEquals(3, totalItems(us));
        
        assertTrue(RefactoringManager.getInstance().canChange(RenameRequest.class, addressType));
        RenameRequest request = new RenameRequest(addressType, "MyAddressType");
        List<ErrorItem> errors = RefactoringManager.getInstance().precheckChange(request);
        assertTrue(errors.toString(), errors.isEmpty());
        errors = RefactoringManager.getInstance().precheckUsages(request);
        assertEquals(0, errors.size());
        RefactoringManager.getInstance().process(request);
        String xpath = "/xs:schema/xs:complexType[@name='ResidenceType']/xs:sequence/xs:element[@name='address']";
        LocalElement le = (LocalElement) Util.findComponent(laModel.getSchema(), xpath);
        assertEquals("MyAddressType", le.getType().getRefString());
        addressType = Util.findGlobalComponentByName(laModel.getSchema(), "MyAddressType");
        assertEquals(3, totalItems(RefactoringManager.getInstance().findUsages(addressType).get()));

        assertTrue(RefactoringManager.getInstance().canUndo());
        RefactoringManager.getInstance().undo(); 
        
        addressType = Util.findGlobalComponentByName(laModel.getSchema(), "AddressType");
        assertNotNull(addressType);
        assertEquals("AddressType", le.getType().getRefString());
        GlobalComplexType poType = (GlobalComplexType) Util.findGlobalComponentByName(model.getSchema(), "PurchaseOrderType");
        Sequence seq = (Sequence) poType.getDefinition();
        assertEquals("la:AddressType", ((LocalElement)seq.getContent().get(1)).getType().getRefString());
        addressType = Util.findGlobalComponentByName(laModel.getSchema(), "AddressType");
        assertEquals(3, totalItems(RefactoringManager.getInstance().findUsages(addressType).get()));
        
        assertFalse(RefactoringManager.getInstance().canUndo());
        assertTrue(RefactoringManager.getInstance().canRedo());
        RefactoringManager.getInstance().redo(); 

        addressType = Util.findGlobalComponentByName(laModel.getSchema(), "MyAddressType");
        assertEquals(3, totalItems(RefactoringManager.getInstance().findUsages(addressType).get()));
        } finally {        
            deleteModelFile(loanApplicationFO);
        }
    }

    public void testInvalidDeleteUndoRedo() throws Exception {
        FileObject fo = Util.populateProject(
                project, "resources/newXmlSchema.xsd", "src/com/acme/schemas");
        try {
            ModelSource ms = Utilities.getModelSource(fo, true);
            SchemaModel model = SchemaModelFactory.getDefault().getModel(ms);
            Collection<GlobalElement> ges = model.getSchema().getElements();
            GlobalElement ge = ges.iterator().next();
            GlobalElement copy = (GlobalElement) ge.copy(model.getSchema());
            copy.setName("newName");
            DeleteRequest delete = new DeleteRequest(ge);
            RefactoringManager.getInstance().execute(delete, false);
            assertTrue(RefactoringManager.getInstance().canUndo());
            model.startTransaction();
            model.getSchema().addElement(copy);
            model.endTransaction();
            assertFalse(RefactoringManager.getInstance().canUndo());
            assertFalse(RefactoringManager.getInstance().canRedo());
        } finally {        
            deleteModelFile(fo);
        }
    }

    public void testRefactorLocalElementUndoRedo() throws Exception {
        GlobalComplexType poType = (GlobalComplexType) Util.findGlobalComponentByName(model.getSchema(), "PurchaseOrderType");
        LocalElement le= poType.getDefinition().getChildren(LocalElement.class).get(0);
        String expected = "shipToAddress";
        RenameRequest request = new RenameRequest(le, expected);
        RefactoringManager.getInstance().execute(request, false);
        assertEquals(expected, le.getName());

        SystemAction undoAction = SystemAction.get(RefactoringUndoAction.class);
        SystemAction redoAction = SystemAction.get(RefactoringRedoAction.class);
        
        assertTrue(RefactoringManager.getInstance().canUndo());
        assertTrue(RefactoringManager.getInstance().canUndo());
        assertFalse(RefactoringManager.getInstance().canRedo());
        assertFalse(RefactoringManager.getInstance().canRedo());
        RefactoringManager.getInstance().undo();
        
        assertFalse(RefactoringManager.getInstance().canUndo());
        assertFalse(RefactoringManager.getInstance().canUndo());
        assertTrue(RefactoringManager.getInstance().canRedo());
        assertTrue(RefactoringManager.getInstance().canRedo());
        RefactoringManager.getInstance().redo();

        assertTrue(RefactoringManager.getInstance().canUndo());
        assertTrue(RefactoringManager.getInstance().canUndo());
        assertFalse(RefactoringManager.getInstance().canRedo());
        assertFalse(RefactoringManager.getInstance().canRedo());
        RefactoringManager.getInstance().undo();
        
        assertFalse(RefactoringManager.getInstance().canUndo());
        assertFalse(RefactoringManager.getInstance().canUndo());
        assertTrue(RefactoringManager.getInstance().canRedo());
        assertTrue(RefactoringManager.getInstance().canRedo());
        RefactoringManager.getInstance().redo();
    }

    public void testOneRefactoringInvalidateOther() throws Exception {
        String expected = "address";
        GlobalComplexType poType = (GlobalComplexType) Util.findGlobalComponentByName(model.getSchema(), "PurchaseOrderType");
        LocalElement element1 = poType.getDefinition().getChildren(LocalElement.class).get(0);
        RenameRequest request1 = new RenameRequest(element1, expected);
        LocalElement element2 = poType.getDefinition().getChildren(LocalElement.class).get(1);
        RenameRequest request2 = new RenameRequest(element2, expected);
        RefactoringManager.getInstance().precheckUsages(request1);
        RefactoringManager.getInstance().precheckUsages(request2);
        RefactoringManager.getInstance().process(request1);
        
        assertEquals(expected, element1.getName());
        assertFalse("preview of second refactoring should be invalidated", request2.getUsages().isValid());
        
        assertTrue(RefactoringManager.getInstance().canUndo());
        request2.setUsages(null);
        RefactoringManager.getInstance().precheckUsages(request2);
        assertTrue(request2.getUsages().isValid());
        assertTrue("precheckUsages should not invalidate undo", RefactoringManager.getInstance().canUndo());
        
        model.startTransaction();
        model.getSchema().setVersion("9999");
        model.endTransaction();
        
        assertFalse(RefactoringManager.getInstance().canUndo());
        assertFalse(RefactoringManager.getInstance().canRedo());
    }

    public void testMultipleRefactoringEffectsOnUndo() throws Exception {
        UndoManager um = new UndoManager();
        model.addUndoableEditListener(um);
        
        String expected1 = "PurchaseOrderType1";
        String expected2 = "PurchaseOrderType2";
        String expected3 = "PurchaseOrderType3";
        
        GlobalComplexType poType = (GlobalComplexType) Util.findGlobalComponentByName(model.getSchema(), "PurchaseOrderType");

        RenameRequest request1 = new RenameRequest(poType, expected1);
        RefactoringManager.getInstance().precheckUsages(request1);
        RefactoringManager.getInstance().process(request1);
        assertEquals(expected1, poType.getName());
        assertTrue(RefactoringManager.getInstance().canUndo());
        
        model.startTransaction();
        model.getSchema().setVersion("8888");
        model.endTransaction();
        
        assertFalse(RefactoringManager.getInstance().canUndo());
        assertFalse(RefactoringManager.getInstance().canRedo());
        assertTrue(um.canUndo());

        RenameRequest request2 = new RenameRequest(poType, expected2);
        RefactoringManager.getInstance().precheckUsages(request2);
        RefactoringManager.getInstance().process(request2);
        assertEquals(expected2, poType.getName());
        assertTrue(RefactoringManager.getInstance().canUndo());

        RenameRequest request3 = new RenameRequest(poType, expected3);
        RefactoringManager.getInstance().precheckUsages(request3);
        RefactoringManager.getInstance().process(request3);
        assertEquals(expected3, poType.getName());
        assertTrue(RefactoringManager.getInstance().canUndo());
        
        model.startTransaction();
        model.getSchema().setVersion("0000");
        model.endTransaction();
        
        assertFalse(RefactoringManager.getInstance().canUndo());
        assertFalse(RefactoringManager.getInstance().canRedo());
        assertTrue(um.canUndo());
    }

    public void testFileRefactoring() throws Exception {
        FileObject loanApplicationFO = Util.populateProject(
                project, "resources/"+XSD_LA, "src/com/acme/schemas");
        try {
        ModelSource ms = Utilities.getModelSource(loanApplicationFO, true);
        SchemaModel laModel = SchemaModelFactory.getDefault().getModel(ms);
        UndoManager um = new UndoManager();
        model.addUndoableEditListener(um);
        
        laModel.startTransaction();
        laModel.getSchema().setVersion("1.0");
        laModel.endTransaction();
        
        FileRenameRequest request = new FileRenameRequest(laModel, "newLoanApp");
        RefactoringManager.getInstance().precheckUsages(request);
        assertEquals(1, request.getUsages().getUsages().iterator().next().getItems().size());
        RefactoringManager.getInstance().process(request);
        
        assertEquals("newLoanApp", loanApplicationFO.getName());
        assertEquals("newLoanApp.xsd", model.getSchema().getImports().iterator().next().getSchemaLocation());
        assertTrue(RefactoringManager.getInstance().canUndo());
        
        RefactoringManager.getInstance().undo();
        
        assertEquals("LoanApplication", loanApplicationFO.getName());
        assertEquals("LoanApplication.xsd", model.getSchema().getImports().iterator().next().getSchemaLocation());
        assertTrue(RefactoringManager.getInstance().canRedo());
        } finally {        
            deleteModelFile(loanApplicationFO);
        }
    }
    
    public void testUndoRedoSaveFile() throws Exception {
        FileObject loanApplicationFO = Util.populateProject(
                project, "resources/"+XSD_LA, "src/com/acme/schemas");
        ModelSource ms = Utilities.getModelSource(loanApplicationFO, true);
        SchemaModel laModel = SchemaModelFactory.getDefault().getModel(ms);
        UndoManager um = new UndoManager();
        laModel.addUndoableEditListener(um);

        ReferenceableSchemaComponent addressType = Util.findGlobalComponentByName(laModel.getSchema(), "AddressType");
        RenameRequest request = new RenameRequest(addressType, "addressType2");
        request.setScopeLocal();
        RefactoringManager.getInstance().execute(request, false);

        assertTrue(RefactoringUtil.isDirty(laModel));
        um.undo();
        assertTrue(RefactoringUtil.isDirty(laModel));
        
        request = new RenameRequest(addressType, "addressType2");
        RefactoringManager.getInstance().precheckUsages(request);
        assertEquals(3, totalItems(request.getUsages()));

        RefactoringManager.getInstance().process(request);

        assertTrue(RefactoringUtil.isDirty(laModel));
        assertFalse(RefactoringUtil.isDirty(model));
        request = new RenameRequest(addressType, "addressType3");
        RefactoringManager.getInstance().precheckUsages(request);
        assertEquals(3, totalItems(request.getUsages()));

        RefactoringManager.getInstance().undo();
        
        assertTrue(RefactoringUtil.isDirty(laModel));
        assertFalse(RefactoringUtil.isDirty(model));
        request = new RenameRequest(addressType, "addressType3");
        RefactoringManager.getInstance().precheckUsages(request);
        assertEquals(3, totalItems(request.getUsages()));

        RefactoringManager.getInstance().redo();
        
        assertTrue(RefactoringUtil.isDirty(laModel));
        assertFalse(RefactoringUtil.isDirty(model));
        request = new RenameRequest(addressType, "addressType3");
        RefactoringManager.getInstance().precheckUsages(request);
        assertEquals(3, totalItems(request.getUsages()));
    }
}
