/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.j2ee.ejbjar.project.ui;

import java.awt.Image;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.openide.actions.FileSystemAction;
import org.openide.filesystems.FileChangeAdapter;
import org.openide.filesystems.Repository;
import org.openide.nodes.FilterNode;
import org.openide.nodes.Node;
import org.openide.util.ImageUtilities;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;

import javax.swing.Action;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;
import org.netbeans.api.project.Project;
import org.netbeans.api.queries.VisibilityQuery;
import org.netbeans.spi.project.ui.support.CommonProjectActions;
import org.openide.actions.FindAction;
import org.openide.actions.PasteAction;
import org.openide.filesystems.FileChangeListener;
import org.openide.filesystems.FileEvent;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileRenameEvent;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.ChangeableDataFilter;
import org.openide.loaders.DataFilter;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.nodes.AbstractNode;
import org.openide.util.actions.SystemAction;
import org.openide.util.lookup.Lookups;
import org.openide.nodes.FilterNode.Children;

/**
 * Node to represent the setup folder described in the blueprints.
 *
 * @author Chris Webster, Andrei Badea
 */
public class ServerResourceNode extends FilterNode {
    
    private static final Logger LOGGER = Logger.getLogger(ServerResourceNode.class.getName());
    private static final boolean LOG = LOGGER.isLoggable(Level.FINE);
    
    private static final Image RESOURCE_FILE_BADGE = ImageUtilities.loadImage( "org/netbeans/modules/j2ee/ejbjar/project/ui/resourcesBadge.gif", true ); // NOI18N
    private static final String SETUP_DIR = "setup"; // NOI18N
    private static final DataFilter VISIBILITY_QUERY_FILTER = new VisibilityQueryDataFilter();
    
    private Project project;
    private FileChangeListener projectDirectoryListener;
    
    /** Creates a new instance of ServerResourceNode */
    public ServerResourceNode(Project project) {
        this(getSetupDataFolder(project), project);
    }
    
    private ServerResourceNode(DataFolder folderDo, Project project) {
        super(getDataFolderNode(folderDo, project), getDataFolderNodeChildren(folderDo));
        projectDirectoryListener = new ProjectDirectoryListener();
        if (LOG) {
            LOGGER.log(Level.FINE, "Adding file listener to " + project.getProjectDirectory()); // NOI18N
        }
        project.getProjectDirectory().addFileChangeListener(FileUtil.weakFileChangeListener(projectDirectoryListener, project.getProjectDirectory()));
        this.project = project;
    }
    
    public Image getIcon(int type) {
        return badgeIcon(super.getIcon(type));
    }
    
    public Image getOpenedIcon( int type ) {
        return badgeIcon(super.getOpenedIcon(type));
    }
    
    private static Image badgeIcon(Image icon) {
        return ImageUtilities.mergeImages(icon, RESOURCE_FILE_BADGE, 7, 7);
    }
    
    public String getDisplayName() {
        return NbBundle.getMessage(ServerResourceNode.class, "LBL_Node_ResourceNode");
    }

    public boolean canCopy() {
        return false;
    }

    public boolean canCut() {
        return false;
    }

    public boolean canRename() {
        return false;
    }

    public boolean canDestroy() {
        return false;
    }

    public Action[] getActions( boolean context ) {
        return new Action[] {
            CommonProjectActions.newFileAction(),
            null,
            SystemAction.get(FileSystemAction.class),
            null,
            SystemAction.get(FindAction.class),
            null,
            SystemAction.get(PasteAction.class),
        };
    }
        
    private void refresh() {
        if (LOG) {
            LOGGER.log(Level.FINE, "Refreshing"); // NOI18N
        }
        final DataFolder folderDo = getSetupDataFolder(project);
        if (LOG) {
            LOGGER.log(Level.FINE, "The DataFolder is: " + folderDo); // NOI18N
        }
        final Node original = getDataFolderNode(folderDo, project);
        final org.openide.nodes.Children children = getDataFolderNodeChildren(folderDo);
        // #106687: should not call Children.getNodes(true) under Children.MUTEX
        if (LOG) {
            LOGGER.log(Level.FINE, "New children count: " + children.getNodes(true).length);
        }
        // #64665: should not call FilterNode.changeOriginal() or Node.setChildren() 
        // under Children.MUTEX read access
        Children.MUTEX.postWriteRequest(new Runnable() {
            public void run() {
                changeOriginal(original, false);
                setChildren(children);
            }
        });

    }
    
    private static DataFolder getSetupDataFolder(Project project) {
        FileObject folderFo = project.getProjectDirectory().getFileObject(SETUP_DIR);
        DataFolder folderDo = null;
        if (folderFo != null && folderFo.isFolder()) {
            try {
                folderDo = DataFolder.findFolder(folderFo);
            }
            catch (IllegalArgumentException e) {}
        }
        return folderDo;
    }
    
    private static Node getDataFolderNode(DataFolder folderDo, Project project) {
        // The project in the placeholder node lookup is needed for the New File action.
        return (folderDo != null) ? folderDo.getNodeDelegate() : new PlaceHolderNode(Lookups.singleton(project));
    }
    
    private static org.openide.nodes.Children getDataFolderNodeChildren(DataFolder folderDo) {
        return (folderDo != null) ? folderDo.createNodeChildren(VISIBILITY_QUERY_FILTER) : Children.LEAF;
    }
    
    final private class ProjectDirectoryListener extends FileChangeAdapter {
        
        public void fileDeleted(FileEvent fe) {
            if (isWatchedFile(getFileName(fe)))
                refresh();
        }

        public void fileFolderCreated(FileEvent fe) {
            if (isWatchedFile(getFileName(fe))) {
                refresh();
            }
        }

        public void fileRenamed(org.openide.filesystems.FileRenameEvent fe) {
            if (isWatchedFile(getFileName(fe)) || isWatchedFile(getOldFileName(fe)))
                refresh();
        }

        private boolean isWatchedFile(String fileName) {
            return fileName.equals(SETUP_DIR);
        }

        private String getFileName(FileEvent fe) {
            return fe.getFile().getNameExt();
        }

        private String getOldFileName(FileRenameEvent fe) {
            String result = fe.getName();
            if (fe.getExt() != "") // NOI18N
                result = result + "." + fe.getExt(); // NOI18N

            return result;
        }
    }
    
    final private static class VisibilityQueryDataFilter implements ChangeListener, ChangeableDataFilter {
        
        private EventListenerList ell = new EventListenerList();
        
        public VisibilityQueryDataFilter() {
            VisibilityQuery.getDefault().addChangeListener(this);
        }
                
        public boolean acceptDataObject(DataObject obj) {
            FileObject fo = obj.getPrimaryFile();
            return VisibilityQuery.getDefault().isVisible(fo);
        }
        
        public void stateChanged( ChangeEvent e) {
            Object[] listeners = ell.getListenerList();
            ChangeEvent event = null;
            for (int i = listeners.length - 2; i >= 0;  i-= 2) {
                if (listeners[i] == ChangeListener.class) {             
                    if (event == null) {
                        event = new ChangeEvent( this );
                    }
                    ((ChangeListener)listeners[i + 1]).stateChanged(event);
                }
            }
        }        
    
        public void addChangeListener( ChangeListener listener ) {
            ell.add(ChangeListener.class, listener);
        }        
                        
        public void removeChangeListener( ChangeListener listener ) {
            ell.remove(ChangeListener.class, listener);
        }
    }    
    
    /**
     * A placeholder node for a folder node.
     */
    private static final class PlaceHolderNode extends AbstractNode {
        
        public PlaceHolderNode(Lookup lookup) {
            super(Children.LEAF, lookup);
        }

        public Image getIcon(int type) {
            Image image = null;
            Node imageDelegate = getImageDelegate();
            if (imageDelegate != null) {
                image = imageDelegate.getIcon(type);
            }
            if (image == null) {
                image = super.getIcon(type);
            }
            return image;
        }
        
        public Image getOpenedIcon(int type) {
            Image image = null;
            Node imageDelegate = getImageDelegate();
            if (imageDelegate != null) {
                image = imageDelegate.getOpenedIcon(type);
            }
            if (image == null) {
                image = super.getOpenedIcon(type);
            }
            return image;
        }
        
        private static Node getImageDelegate() {
            FileObject imageFo = Repository.getDefault().getDefaultFileSystem().getRoot();
            if (imageFo != null) {
                try {
                    DataObject imageDo = DataObject.find(imageFo);
                    return imageDo.getNodeDelegate();
                } catch (DataObjectNotFoundException donfe) {
                    Logger.getLogger("global").log(Level.INFO, null, donfe);
                }
            }
            return null;
        }
    }
}

