
/*
 * NetJack - Packet Handling functions
 * 
 * used by the driver and the jacknet_client
 *    
 * Copyright (C) 2006 Torben Hohn <torbenh@gmx.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * $Id: net_driver.c,v 1.16 2006/03/20 19:41:37 torbenh Exp $
 *
 */


#include <math.h>
#include <stdio.h>
#include <memory.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>

#include <jack/types.h>
#include <jack/engine.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <endian.h>

#include <samplerate.h>

#include "net_driver.h"
#include "netjack_packet.h"

void 
packet_header_hton( jacknet_packet_header *pkthdr )
{
    pkthdr->channels = htonl( pkthdr->channels );
    pkthdr->period_size = htonl( pkthdr->period_size );
    pkthdr->sample_rate = htonl( pkthdr->sample_rate );
    pkthdr->sync_state = htonl( pkthdr->sync_state );
    pkthdr->transport_frame = htonl( pkthdr->transport_frame );
    pkthdr->transport_state = htonl( pkthdr->transport_state );
    pkthdr->framecnt = htonl( pkthdr->framecnt );
    pkthdr->latency = htonl( pkthdr->latency );
    pkthdr->reply_port = htonl( pkthdr->reply_port );
}

void 
packet_header_ntoh( jacknet_packet_header *pkthdr )
{
    pkthdr->channels = ntohl( pkthdr->channels );
    pkthdr->period_size = ntohl( pkthdr->period_size );
    pkthdr->sample_rate = ntohl( pkthdr->sample_rate );
    pkthdr->sync_state = ntohl( pkthdr->sync_state );
    pkthdr->transport_frame = ntohl( pkthdr->transport_frame );
    pkthdr->transport_state = ntohl( pkthdr->transport_state );
    pkthdr->framecnt = ntohl( pkthdr->framecnt );
    pkthdr->latency = ntohl( pkthdr->latency );
    pkthdr->reply_port = ntohl( pkthdr->reply_port );
}

int get_sample_size( int bitdepth ) {
    if( bitdepth == 8 )
	return sizeof( int8_t );
    if( bitdepth == 16 )
	return sizeof( int16_t );

    return sizeof(int32_t);
}


// render functions for float

void render_payload_to_jack_ports_float(  void *packet_payload, jack_nframes_t net_period_down, JSList *capture_ports, JSList *capture_srcs, jack_nframes_t nframes) {

	channel_t chn = 0;
	JSList *node = capture_ports;
	JSList *src_node = capture_srcs;


	uint32_t *packet_bufX = (uint32_t *)packet_payload;
	

	while ( node != NULL)
	{
		int i;
		uint32_t val;
		SRC_DATA src;

		jack_port_t *port = (jack_port_t *) node->data;
		jack_default_audio_sample_t* buf = jack_port_get_buffer (port, nframes);
	
		if( net_period_down != nframes ) {
		    SRC_STATE *src_state = src_node->data;
		    for (i = 0; i < net_period_down; i++)
		    {
			packet_bufX[i] = ntohl( packet_bufX[i] );
		    }

		    src.data_in = (float *)packet_bufX;
		    src.input_frames = net_period_down;

		    src.data_out = buf;
		    src.output_frames = nframes;

		    src.src_ratio = (float) nframes / (float) net_period_down;
		    src.end_of_input = 0;

		    src_set_ratio( src_state, src.src_ratio );
		    src_process( src_state, &src );
		    src_node = jack_slist_next (src_node);
		}
		else
		{
		    for (i = 0; i < net_period_down; i++)
		    {
			uint32_t *ptr = (uint32_t *)&packet_bufX[i];
			val = ntohl( *ptr);
			buf[i] = *(float *)&val;
		    }
		}

		packet_bufX = (packet_bufX + net_period_down);
		node = jack_slist_next (node);
		chn++;
	}
}

void render_jack_ports_to_payload_float ( JSList *playback_ports, JSList *playback_srcs, jack_nframes_t nframes, void *packet_payload, jack_nframes_t net_period_up){

    channel_t chn = 0;
    JSList *node = playback_ports;
    JSList *src_node = playback_srcs;

    uint32_t *packet_bufX = (uint32_t *)packet_payload;

    while ( node != NULL)
    {
	SRC_DATA src;
	int i;
	uint32_t val;
	jack_port_t *port = (jack_port_t *) node->data;
	jack_default_audio_sample_t* buf = jack_port_get_buffer (port, nframes);

	if( net_period_up != nframes ) {
	    SRC_STATE *src_state = src_node->data;
	    src.data_in = buf;
	    src.input_frames = nframes;

	    src.data_out = (float *) packet_bufX;
	    src.output_frames = net_period_up;

	    src.src_ratio = (float) net_period_up / (float) nframes;
	    src.end_of_input = 0;

	    src_set_ratio( src_state, src.src_ratio );
	    src_process( src_state, &src );

	    for (i=0; i < net_period_up; i++) {
		packet_bufX[i] = htonl( packet_bufX[i] );
	    }
	    src_node = jack_slist_next (src_node);
	}
	else
	{
	    for (i=0; i < net_period_up; i++) {
		uint32_t *ptr = (uint32_t *)&buf[i];
		val = htonl( *ptr);
		packet_bufX[i]=val;
	    }
	}

	packet_bufX = (packet_bufX + net_period_up);
	node = jack_slist_next (node);
	chn++;
    }
}

// render functions for 16bit

void render_payload_to_jack_ports_16bit(  void *packet_payload, jack_nframes_t net_period_down, JSList *capture_ports, JSList *capture_srcs, jack_nframes_t nframes) {

	channel_t chn = 0;
	JSList *node = capture_ports;
	JSList *src_node = capture_srcs;


	uint16_t *packet_bufX = (uint16_t *)packet_payload;
	

	while ( node != NULL)
	{
		int i;
		//uint32_t val;
		SRC_DATA src;

		jack_port_t *port = (jack_port_t *) node->data;
		jack_default_audio_sample_t* buf = jack_port_get_buffer (port, nframes);
	
		float *floatbuf = alloca( sizeof(float) * net_period_down );

		if( net_period_down != nframes ) {
		    SRC_STATE *src_state = src_node->data;
		    for (i = 0; i < net_period_down; i++)
		    {
			floatbuf[i] = ((float) ntohs( packet_bufX[i] )) / 32767.0 - 1.0;
		    }

		    src.data_in = floatbuf;
		    src.input_frames = net_period_down;

		    src.data_out = buf;
		    src.output_frames = nframes;

		    src.src_ratio = (float) nframes / (float) net_period_down;
		    src.end_of_input = 0;

		    src_set_ratio( src_state, src.src_ratio );
		    src_process( src_state, &src );
		    src_node = jack_slist_next (src_node);
		}
		else
		{
		    for (i = 0; i < net_period_down; i++)
		    {
			buf[i] = ((float) ntohs( packet_bufX[i] )) / 32768.0 - 1.0;
		    }
		}

		packet_bufX = (packet_bufX + net_period_down);
		node = jack_slist_next (node);
		chn++;
	}
}

void render_jack_ports_to_payload_16bit ( JSList *playback_ports, JSList *playback_srcs, jack_nframes_t nframes, void *packet_payload, jack_nframes_t net_period_up){

    channel_t chn = 0;
    JSList *node = playback_ports;
    JSList *src_node = playback_srcs;

    uint16_t *packet_bufX = (uint16_t *)packet_payload;

    while ( node != NULL)
    {
	SRC_DATA src;
	int i;
	jack_port_t *port = (jack_port_t *) node->data;
	jack_default_audio_sample_t* buf = jack_port_get_buffer (port, nframes);

	if( net_period_up != nframes ) {
	    SRC_STATE *src_state = src_node->data;

	    float *floatbuf = alloca( sizeof(float) * net_period_up );

	    src.data_in = buf;
	    src.input_frames = nframes;

	    src.data_out = floatbuf;
	    src.output_frames = net_period_up;

	    src.src_ratio = (float) net_period_up / (float) nframes;
	    src.end_of_input = 0;

	    src_set_ratio( src_state, src.src_ratio );
	    src_process( src_state, &src );

	    for (i=0; i < net_period_up; i++) {
		packet_bufX[i] = htons( (floatbuf[i]+1.0)*32767.0 );
	    }
	    src_node = jack_slist_next (src_node);
	}
	else
	{
	    for (i=0; i < net_period_up; i++) {
		packet_bufX[i]= htons((buf[i]+1.0) * 32767.0);
	    }
	}

	packet_bufX = (packet_bufX + net_period_up);
	node = jack_slist_next (node);
	chn++;
    }
}

// render functions for 8bit

void render_payload_to_jack_ports_8bit(  void *packet_payload, jack_nframes_t net_period_down, JSList *capture_ports, JSList *capture_srcs, jack_nframes_t nframes) {

	channel_t chn = 0;
	JSList *node = capture_ports;
	JSList *src_node = capture_srcs;


	int8_t *packet_bufX = (int8_t *)packet_payload;
	

	while ( node != NULL)
	{
		int i;
		//uint32_t val;
		SRC_DATA src;

		jack_port_t *port = (jack_port_t *) node->data;
		jack_default_audio_sample_t* buf = jack_port_get_buffer (port, nframes);
	
		float *floatbuf = alloca( sizeof(float) * net_period_down );

		if( net_period_down != nframes ) {
		    SRC_STATE *src_state = src_node->data;
		    for (i = 0; i < net_period_down; i++)
		    {
			floatbuf[i] = ((float) packet_bufX[i]) / 127.0;
		    }

		    src.data_in = floatbuf;
		    src.input_frames = net_period_down;

		    src.data_out = buf;
		    src.output_frames = nframes;

		    src.src_ratio = (float) nframes / (float) net_period_down;
		    src.end_of_input = 0;

		    src_set_ratio( src_state, src.src_ratio );
		    src_process( src_state, &src );
		    src_node = jack_slist_next (src_node);
		}
		else
		{
		    for (i = 0; i < net_period_down; i++)
		    {
			buf[i] = ((float) packet_bufX[i]) / 127.0;
		    }
		}

		packet_bufX = (packet_bufX + net_period_down);
		node = jack_slist_next (node);
		chn++;
	}
}

void render_jack_ports_to_payload_8bit ( JSList *playback_ports, JSList *playback_srcs, jack_nframes_t nframes, void *packet_payload, jack_nframes_t net_period_up){

    channel_t chn = 0;
    JSList *node = playback_ports;
    JSList *src_node = playback_srcs;

    int8_t *packet_bufX = (int8_t *)packet_payload;

    while ( node != NULL)
    {
	SRC_DATA src;
	int i;
	jack_port_t *port = (jack_port_t *) node->data;
	jack_default_audio_sample_t* buf = jack_port_get_buffer (port, nframes);

	if( net_period_up != nframes ) {
	    SRC_STATE *src_state = src_node->data;

	    float *floatbuf = alloca( sizeof(float) * net_period_up );

	    src.data_in = buf;
	    src.input_frames = nframes;

	    src.data_out = floatbuf;
	    src.output_frames = net_period_up;

	    src.src_ratio = (float) net_period_up / (float) nframes;
	    src.end_of_input = 0;

	    src_set_ratio( src_state, src.src_ratio );
	    src_process( src_state, &src );

	    for (i=0; i < net_period_up; i++) {
		packet_bufX[i] = floatbuf[i]*127.0;
	    }
	    src_node = jack_slist_next (src_node);
	}
	else
	{
	    for (i=0; i < net_period_up; i++) {
		packet_bufX[i]= buf[i] * 127.0;
	    }
	}

	packet_bufX = (packet_bufX + net_period_up);
	node = jack_slist_next (node);
	chn++;
    }
}


// wrapper functions with bitdepth argument...

void render_payload_to_jack_ports(  int bitdepth, void *packet_payload, jack_nframes_t net_period_down, JSList *capture_ports, JSList *capture_srcs, jack_nframes_t nframes) {
    if( bitdepth == 8 )
	render_payload_to_jack_ports_8bit( packet_payload, net_period_down, capture_ports, capture_srcs, nframes );
    else if( bitdepth == 16 )
	render_payload_to_jack_ports_16bit( packet_payload, net_period_down, capture_ports, capture_srcs, nframes );
    else
	render_payload_to_jack_ports_float( packet_payload, net_period_down, capture_ports, capture_srcs, nframes );
}

void render_jack_ports_to_payload ( int bitdepth, JSList *playback_ports, JSList *playback_srcs, jack_nframes_t nframes, void *packet_payload, jack_nframes_t net_period_up){
    if( bitdepth == 8 )
	render_jack_ports_to_payload_8bit( playback_ports, playback_srcs, nframes, packet_payload, net_period_up );
    else if( bitdepth == 16 )
	render_jack_ports_to_payload_16bit( playback_ports, playback_srcs, nframes, packet_payload, net_period_up );
    else
	render_jack_ports_to_payload_float( playback_ports, playback_srcs, nframes, packet_payload, net_period_up );
}
