/* colorname.c - colorname routines, not dependent on Netpbm formats
**
** Taken from libppm4.c May 2002.

** Copyright (C) 1989 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

#include "pm.h"
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include "colorname.h"

static int lineNo;

void 
pm_canonstr(char * const str) {

    char * p;
    for (p = str; *p; ) {
        if (isspace(*p)) {
            strcpy(p, &(p[1]));
        } else {
            if (isupper(*p))
                *p = tolower(*p);
            ++p;
        }
    }
}



FILE *
pm_openColornameFile(const int must_open) {
/*----------------------------------------------------------------------------
   Open the colorname database file.  Its pathname is the value of the
   environment variable whose name is RGB_ENV (e.g. "RGBDEF").  Except
   if that environment variable is not set, it is RGB_DB1, RGB_DB2,
   or RGB_DB3 (e.g. "/usr/lib/X11/rgb.txt"), whichever exists.
   
   'must_open' is a logical: we must get the file open or die.  If
   'must_open' is true and we can't open the file (e.g. it doesn't
   exist), exit the program with an error message.  If 'must_open' is
   false and we can't open the file, just return a null pointer.
-----------------------------------------------------------------------------*/
    const char *rgbdef;
    FILE *f;

    if ((rgbdef = getenv(RGBENV))==NULL) {
        /* The environment variable isn't set, so try the hardcoded
           default color names database locations.
        */
        if ((f = fopen(RGB_DB1, "r")) == NULL &&
            (f = fopen(RGB_DB2, "r")) == NULL &&
            (f = fopen(RGB_DB3, "r")) == NULL && must_open) {
            pm_error("can't open color names database file named "
                     "%s, %s, or %s "
                     "and Environment variable %s not set.  Set %s to "
                     "the pathname of your rgb.txt file or don't use "
                     "color names.", 
                     RGB_DB1, RGB_DB2, RGB_DB3, RGBENV, RGBENV);
        }
    } else {            
        /* The environment variable is set */
        if ((f = fopen(rgbdef, "r")) == NULL && must_open)
            pm_error("Can't open the color names database file named %s, "
                     "per the %s environment variable.",
                     rgbdef, RGBENV);
    }
    lineNo = 0;
    return(f);
}



struct colorfile_entry
pm_colorget(FILE * const f) {
/*----------------------------------------------------------------------------
   Get next color entry from the color name database file 'f'.

   If eof or error, return a color entry with NULL for the color name.

   Otherwise, return color name in static storage within.
-----------------------------------------------------------------------------*/
    char buf[200];
    static char colorname[200];
    bool gotOne;
    bool eof;
    struct colorfile_entry retval;
    char * rc;
    
    gotOne = FALSE;  /* initial value */
    eof = FALSE;
    while (!gotOne && !eof) {
        lineNo++;
        rc = fgets(buf, sizeof(buf), f);
        if (rc == NULL)
            eof = TRUE;
        else {
            if (buf[0] != '#' && buf[0] != '\n' && buf[0] != '!' &&
                buf[0] != '\0') {
                if (sscanf(buf, "%ld %ld %ld %[^\n]", 
                           &retval.r, &retval.g, &retval.b, colorname) 
                    == 4 )
                    gotOne = TRUE;
                else {
                    if (buf[strlen(buf)-1] == '\n')
                        buf[strlen(buf)-1] = '\0';
                    pm_message("can't parse color names database Line %d:  "
                               "'%s'", 
                               lineNo, buf);
                }
            }
        }
    }
    if (gotOne)
        retval.colorname = colorname;
    else
        retval.colorname = NULL;
    return retval;
}



long
pm_rgbnorm(const long rgb, const long lmaxval, const int n, 
           const char * const colorname) {
/*----------------------------------------------------------------------------
   Normalize the color (r, g, or b) value 'rgb', which was specified with
   'n' digits, to a maxval of 'lmaxval'.  If the number of digits isn't
   valid, issue an error message and identify the complete color 
   color specification in error as 'colorname'.

   For example, if the user says 0ff/000/000 and the maxval is 100,
   then rgb is 0xff, n is 3, and our result is 
   0xff / (16**3-1) * 100 = 6.

-----------------------------------------------------------------------------*/
    long retval;

    switch (n) {
    case 1:
        retval = (long)((double) rgb * lmaxval / 15 + 0.5);
        break;
    case 2:
        retval = (long) ((double) rgb * lmaxval / 255 + 0.5);
        break;
    case 3:
        retval = (long) ((double) rgb * lmaxval / 4095 + 0.5);
        break;
    case 4:
        retval = (long) ((double) rgb * lmaxval / 65535L + 0.5);
        break;
    default:
        pm_error( "invalid color specifier - \"%s\"", colorname );
    }
    return retval;
}
