/* 
** Version 1.0  September 28, 1996
**
** Copyright (C) 1996 by Mike Burns <burns@cac.psu.edu>
**
** Adapted to Netpbm 2005.08.10 by Bryan Henderson
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

/* References
** ----------
** The select k'th value implementation is based on Algorithm 489 by 
** Robert W. Floyd from the "Collected Algorithms from ACM" Volume II.
**
** The histogram sort is based is described in the paper "A Fast Two-
** Dimensional Median Filtering Algorithm" in "IEEE Transactions on 
** Acoustics, Speech, and Signal Processing" Vol. ASSP-27, No. 1, February
** 1979.  The algorithm I more closely followed is found in "Digital
** Image Processing Algorithms" by Ioannis Pitas.
*/


#include "pgm.h"

#define MAX_MEDIAN_TYPES      2
#define SELECT_MEDIAN         0
#define HISTOGRAM_SORT_MEDIAN 1

/* Global variables common to each median sort routine. */
static int const forceplain = 0;
static int format;
static gray maxval;
static gray **grays;
static gray *grayrow;
static gray **rowptr;
static int ccolso2, crowso2;
static int row;



static void
select_489(gray * const a,
           int *  const parray,
           int    const n,
           int    const k) {

    gray t;
    int i, j, l, r;
    int ptmp, ttmp;

    l = 0;
    r = n - 1;
    while ( r > l ) {
        t = a[parray[k]];
        ttmp = parray[k];
        i = l;
        j = r;
        ptmp = parray[l];
        parray[l] = parray[k];
        parray[k] = ptmp;
        if ( a[parray[r]] > t ) {
            ptmp = parray[r];
            parray[r] = parray[l];
            parray[l] = ptmp;
        }
        while ( i < j ) {
            ptmp = parray[i];
            parray[i] = parray[j];
            parray[j] = ptmp;
            ++i;
            --j;
            while ( a[parray[i]] < t )
                ++i;
            while ( a[parray[j]] > t )
                --j;
        }
        if ( a[parray[l]] == t ) {
            ptmp = parray[l];
            parray[l] = parray[j];
            parray[j] = ptmp;
        } else {
            ++j;
            ptmp = parray[j];
            parray[j] = parray[r];
            parray[r] = ptmp;
        }
        if ( j <= k )
            l = j + 1;
        if ( k <= j )
            r = j - 1;
    }
}



static void
select_median(FILE * const ifp,
              int    const ccols,
              int    const crows,
              int    const cols,
              int    const rows,
              int    const median) {

    int ccol, col;
    int crow;
    int rownum, irow, temprow;
    gray *temprptr;
    int i, leftcol;
    int num_values;
    gray *garray;

    int *parray;
    int addcol;
    int *subcol;
    int tsum;

    /* Allocate storage for array of the current gray values. */
    garray = pgm_allocrow( crows * ccols );

    num_values = crows * ccols;

    parray = (int *) pm_allocrow( crows * ccols, sizeof(int) );
    subcol = (int *) pm_allocrow( cols, sizeof(int) );

    for ( i = 0; i < cols; ++i )
        subcol[i] = ( i - (ccolso2 + 1) ) % ccols;

    /* Apply median to main part of image. */
    for ( ; row < rows; ++row ) {
        temprow = row % crows;
        pgm_readpgmrow( ifp, grays[temprow], cols, maxval, format );

        /* Rotate pointers to rows, so rows can be accessed in order. */
        temprow = ( row + 1 ) % crows;
        rownum = 0;
        for ( irow = temprow; irow < crows; ++rownum, ++irow )
            rowptr[rownum] = grays[irow];
        for ( irow = 0; irow < temprow; ++rownum, ++irow )
            rowptr[rownum] = grays[irow];

        for ( col = 0; col < cols; ++col ) {
            if ( col < ccolso2 || col >= cols - ccolso2 ) {
                grayrow[col] = rowptr[crowso2][col];
            } else if ( col == ccolso2 ) {
                leftcol = col - ccolso2;
                i = 0;
                for ( crow = 0; crow < crows; ++crow ) {
                    temprptr = rowptr[crow] + leftcol;
                    for ( ccol = 0; ccol < ccols; ++ccol ) {
                        garray[i] = *( temprptr + ccol );
                        parray[i] = i;
                        ++i;
                    }
                }
                select_489( garray, parray, num_values, median );
                grayrow[col] = garray[parray[median]];
            } else {
                addcol = col + ccolso2;
                for (crow = 0, tsum = 0; crow < crows; ++crow, tsum += ccols)
                    garray[tsum + subcol[col]] = *(rowptr[crow] + addcol );
                select_489( garray, parray, num_values, median );
                grayrow[col] = garray[parray[median]];
            }
        }
        pgm_writepgmrow( stdout, grayrow, cols, maxval, forceplain );
    }

    /* Write out remaining unchanged rows. */
    for ( irow = crowso2 + 1; irow < crows; ++irow )
        pgm_writepgmrow( stdout, rowptr[irow], cols, maxval, forceplain );

    pgm_freerow( garray );
    pm_freerow( (char *) parray );
    pm_freerow( (char *) subcol );
}



static void
histogram_sort_median(FILE * const ifp,
                      int    const ccols,
                      int    const crows,
                      int    const cols,
                      int    const rows,
                      int    const median) {

    int const histmax = maxval + 1;

    int *hist;
    int mdn, ltmdn;
    gray *left_col, *right_col;

    hist = (int *) pm_allocrow( histmax, sizeof( int ) );
    left_col = pgm_allocrow( crows );
    right_col = pgm_allocrow( crows );

    /* Apply median to main part of image. */
    for ( ; row < rows; ++row ) {
        int col;
        int temprow;
        int rownum;
        int irow;
        int i;
        /* initialize hist[] */
        for ( i = 0; i < histmax; ++i )
            hist[i] = 0;

        temprow = row % crows;
        pgm_readpgmrow( ifp, grays[temprow], cols, maxval, format );

        /* Rotate pointers to rows, so rows can be accessed in order. */
        temprow = ( row + 1 ) % crows;
        rownum = 0;
        for ( irow = temprow; irow < crows; ++rownum, ++irow )
            rowptr[rownum] = grays[irow];
        for ( irow = 0; irow < temprow; ++rownum, ++irow )
            rowptr[rownum] = grays[irow];

        for ( col = 0; col < cols; ++col ) {
            if ( col < ccolso2 || col >= cols - ccolso2 )
                grayrow[col] = rowptr[crowso2][col];
            else if ( col == ccolso2 ) {
                int crow;
                int const leftcol = col - ccolso2;
                i = 0;
                for ( crow = 0; crow < crows; ++crow ) {
                    int ccol;
                    gray * const temprptr = rowptr[crow] + leftcol;
                    for ( ccol = 0; ccol < ccols; ++ccol ) {
                        gray const g = *( temprptr + ccol );
                        ++hist[g];
                        ++i;
                    }
                }
                ltmdn = 0;
                for ( mdn = 0; ltmdn <= median; ++mdn )
                    ltmdn += hist[mdn];
                mdn--;
                if ( ltmdn > median ) 
                    ltmdn -= hist[mdn];

                grayrow[col] = mdn;
            } else {
                int crow;
                int const subcol = col - ( ccolso2 + 1 );
                int const addcol = col + ccolso2;
                for ( crow = 0; crow < crows; ++crow ) {
                    left_col[crow] = *( rowptr[crow] + subcol );
                    right_col[crow] = *( rowptr[crow] + addcol );
                }
                for ( crow = 0; crow < crows; ++crow ) {
                    {
                        gray const g = left_col[crow];
                        hist[(int) g]--;
                        if ( (int) g < mdn )
                            ltmdn--;
                    }
                    {
                        gray const g = right_col[crow];
                        hist[(int) g]++;
                        if ( (int) g < mdn )
                            ltmdn++;
                    }
                }
                if ( ltmdn > median )
                    do {
                        mdn--;
                        ltmdn -= hist[mdn];
                    } while ( ltmdn > median );
                else {
                    /* This one change from Pitas algorithm can reduce run
                    ** time by up to 10%.
                    */
                    while ( ltmdn <= median ) {
                        ltmdn += hist[mdn];
                        mdn++;
                    }
                    mdn--;
                    if ( ltmdn > median ) 
                        ltmdn -= hist[mdn];
                }
                grayrow[col] = mdn;
            }
        }
        pgm_writepgmrow( stdout, grayrow, cols, maxval, forceplain );
    }

    {
        /* Write out remaining unchanged rows. */
        int irow;
        for ( irow = crowso2 + 1; irow < crows; ++irow )
            pgm_writepgmrow( stdout, rowptr[irow], cols, maxval, forceplain );
    }
    pm_freerow( (char *) hist );
    pgm_freerow( left_col );
    pgm_freerow( right_col );
}



int
main(int    argc,
     char * argv[]) {

    FILE *ifp;
    int argn;
    int ccols, crows;
    int median_type;
    int i;
    int median_cutoff;  
    int cols, rows;
    int median;
    const char * const usage =
        "[-size width height] [-type median_type] [-cutoff value] [pgmfile]";

    int median_type_selected;
    const char * const median_name[] = {
        "select",
        "histogram_sort",
    };

    int const median_id[] = {
        SELECT_MEDIAN,
        HISTOGRAM_SORT_MEDIAN
    };

    int const median_compare[] = {
        1,
        2
    };

    pgm_init(&argc, argv);

    ccols = 3;
    crows = 3;

    /* Cutoff to choose between select and histogram median. */
    median_cutoff = 250;
    median_type = SELECT_MEDIAN;
    median_type_selected = 0;

    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
    {
        if ( pm_keymatch( argv[argn], "-cutoff", 2 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message("incorrect number of arguments for -cutoff option");
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -cutoff option: %s",
                            argv[argn] );
                pm_usage( usage );
            }
            if ( (median_cutoff = atoi(argv[argn])) == 0 )
                pm_error( "invalid cutoff specification: %s", argv[argn] );
        }
        else if ( pm_keymatch( argv[argn], "-size", 2 ) )
        {
            ++argn;
            if ( argn+1 >= argc )
            {
                pm_message( "incorrect number of arguments for -size option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' || argv[argn+1][0] == '-' )
            {
                pm_message( "invalid arguments to -size option: %s %s",
                            argv[argn], argv[argn+1] );
                pm_usage( usage );
            }
            if ( (ccols = atoi(argv[argn])) == 0 )
                pm_error( "invalid width size specification: %s", argv[argn] );
            ++argn;
            if ( (crows = atoi(argv[argn])) == 0 )
                pm_error( "invalid height size specification: %s",argv[argn] );
            if ( ccols % 2 != 1 || crows % 2 != 1 )
                pm_error( "the convolution matrix must have "
                          "an odd number of rows and columns" );
        }
        else if ( pm_keymatch( argv[argn], "-type", 2 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( "incorrect number of arguments for -type option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -type option: %s",
                            argv[argn] );
                pm_usage( usage );
            }
            i = 0;
            while ( ( i < MAX_MEDIAN_TYPES ) && 
                    !pm_keymatch( argv[argn],
                                  median_name[i], median_compare[i] ) )
                ++i;
            if ( i == MAX_MEDIAN_TYPES )
            {
                pm_message( "invalid argument to -type option: %s",
                            argv[argn] );
                pm_usage( usage );
            }
            median_type = median_id[i];
            median_type_selected = 1;
        }
        else
            pm_usage( usage );
        ++argn;
    }

    if ( argn < argc )
    {
        ifp = pm_openr( argv[argn] );
        ++argn;
    }
    else
        ifp = stdin;

    if ( argn != argc )
        pm_usage( usage );

    ccolso2 = ccols / 2;
    crowso2 = crows / 2;

    pgm_readpgminit(ifp, &cols, &rows, &maxval, &format);
    pgm_writepgminit(stdout, cols, rows, maxval, forceplain);

    /* Allocate space for number of rows in mask size. */
    grays = pgm_allocarray(cols, crows);
    grayrow = pgm_allocrow(cols);

    /* Allocate pointers to mask row buffer. */
    rowptr = pgm_allocarray(1, crows);

    /* Read in and write out initial rows that won't get changed. */
    for (row = 0; row < crows - 1; ++row) {
        pgm_readpgmrow(ifp, grays[row], cols, maxval, format);
        /* Write out the unchanged row. */
        if (row < crowso2)
            pgm_writepgmrow(stdout, grays[row], cols, maxval, forceplain);
    }

    median = (crows * ccols) / 2;

    /* Choose which sort to run. */
    if (!median_type_selected) {
        if ((maxval / ((ccols * crows) - 1)) < median_cutoff)
            median_type = HISTOGRAM_SORT_MEDIAN;
        else
            median_type = SELECT_MEDIAN;
    }

    switch (median_type) {
    case SELECT_MEDIAN:
        select_median(ifp, ccols, crows, cols, rows, median);
        break;

    case HISTOGRAM_SORT_MEDIAN:
        histogram_sort_median(ifp, ccols, crows, cols, rows, median);
        break;

    default:
        pm_error("no median type");
        break;
    }
    
    pm_close(ifp);
    pm_close(stdout);

    pgm_freearray(grays, crows);
    pgm_freerow(grayrow);
    pgm_freearray(rowptr, crows);

    return 0;
}
