/*
** 
** Add gaussian, multiplicative gaussian, impulse, laplacian or 
** poisson noise to a portable anymap.
** 
** Version 1.0  November 1995
**
** Copyright (C) 1995 by Mike Burns (burns@cac.psu.edu)
**
** Adapted to Netpbm 2005.08.09, by Bryan Henderson
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

/* References
** ----------
** "Adaptive Image Restoration in Signal-Dependent Noise" by R. Kasturi
** Institute for Electronic Science, Texas Tech University  1982
**
** "Digital Image Processing Algorithms" by Ioannis Pitas
** Prentice Hall, 1993  ISBN 0-13-145814-0
*/

#define _XOPEN_SOURCE   /* get M_PI in math.h */

#include <math.h>

#include "pnm.h"

#define RANDOM_MASK 0x7FFF  /* only compare lower 15 bits.  Stupid PCs. */

#define GAUSSIAN        0
#define IMPULSE         1   /* aka salt and pepper noise */
#define LAPLACIAN       2
#define MULTIPLICATIVE_GAUSSIAN 3
#define POISSON         4
#define MAX_NOISE_TYPES     5

#define CHECK_GRAY \
    if ( gtemp < 0.0 ) g = 0; \
    else if ( gtemp > maxval ) g = maxval; \
    else g = gtemp + 0.5;

#define CHECK_RED \
    if ( rtemp < 0.0 ) r = 0; \
    else if ( rtemp > maxval ) r = maxval; \
    else r = rtemp + 0.5;

#define CHECK_GREEN \
    if ( gtemp < 0.0 ) g = 0; \
    else if ( gtemp > maxval ) g = maxval; \
    else g = gtemp + 0.5;

#define CHECK_BLUE \
    if ( btemp < 0.0 ) b = 0; \
    else if ( btemp > maxval ) b = maxval; \
    else b = btemp + 0.5;

static double const arand = 32767.0;      /* 2^15-1 in case stoopid computer */



/* Gaussian Noise
**
** Based on Kasturi/Algorithms of the ACM
*/

static void
gaussian_noise(FILE * const ifp,
               float  const sigma1,
               float  const sigma2) {

    xel *xP;
    xel *xelrow;
    xelval r, g, b;
    xelval maxval;
    int rows, cols, format, row;
    double x1, x2, xn, yn;
    double rtemp, gtemp, btemp;

    int const forceplain = 0;

    pnm_readpnminit( ifp, &cols, &rows, &maxval, &format );
    xelrow = pnm_allocrow( cols );

    pnm_writepnminit( stdout, cols, rows, maxval, format, forceplain );

    for ( row = 0; row < rows; ++row ) {
        int col;
        pnm_readpnmrow( ifp, xelrow, cols, maxval, format );
        for ( col = 0, xP = xelrow; col < cols; ++col, ++xP ) {
            switch ( PNM_FORMAT_TYPE(format) ) {
            case PPM_TYPE:
                x1 = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( x1 == 0.0 )
                    x1 = 1.0;
                x2 = ( rand( ) & RANDOM_MASK ) / arand;
                xn = sqrt( -2.0 * log( x1 ) ) * cos( 2.0 * M_PI * x2 );
                yn = sqrt( -2.0 * log( x1 ) ) * sin( 2.0 * M_PI * x2 );
                r = PPM_GETR( *xP );
                rtemp = r + ( sqrt( (double) r ) * sigma1 * xn ) + 
                    ( sigma2 * yn );

                x1 = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( x1 == 0.0 )
                    x1 = 1.0;
                x2 = ( rand( ) & RANDOM_MASK ) / arand;
                xn = sqrt( -2.0 * log( x1 ) ) * cos( 2.0 * M_PI * x2 );
                yn = sqrt( -2.0 * log( x1 ) ) * sin( 2.0 * M_PI * x2 );
                g = PPM_GETG( *xP );
                gtemp = g + ( sqrt( (double) g ) * sigma1 * xn ) + 
                    ( sigma2 * yn );

                x1 = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( x1 == 0.0 )
                    x1 = 1.0;
                x2 = ( rand( ) & RANDOM_MASK ) / arand;
                xn = sqrt( -2.0 * log( x1 ) ) * cos( 2.0 * M_PI * x2 );
                yn = sqrt( -2.0 * log( x1 ) ) * sin( 2.0 * M_PI * x2 );
                b = PPM_GETB( *xP );
                btemp = b + ( sqrt( (double) b ) * sigma1 * xn ) +
                    ( sigma2 * yn );

                CHECK_GREEN;
                CHECK_BLUE;
                PPM_ASSIGN( *xP, r, g, b );
                break;
                
            default: /* PGM */
                /* Need to have a non-zero x1, so if rand() returns 0, set
                   it to 1.0 (the maximum value x1 can take).
                */
                x1 = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( x1 == 0.0 )
                    x1 = 1.0;
                x2 = ( rand( ) & RANDOM_MASK ) / arand;
                xn = sqrt( -2.0 * log( x1 ) ) * cos( 2.0 * M_PI * x2 );
                yn = sqrt( -2.0 * log( x1 ) ) * sin( 2.0 * M_PI * x2 );
                g = PNM_GET1( *xP );
                gtemp = g + ( sqrt( (double) g ) * sigma1 * xn ) +
                    ( sigma2 * yn );

                CHECK_GRAY;
                PNM_ASSIGN1( *xP, g );
                break;
            }
        }
        pnm_writepnmrow(stdout, xelrow, cols, maxval, format, forceplain);
    }
    pnm_freerow(xelrow);
}



/* Impulse (Salt and Pepper) Noise
**
*/

static void
impulse_noise(FILE * const ifp,
              float  const tolerance) {

    xel *xP;
    xel *xelrow;
    xelval r, g, b;
    xelval maxval;
    int rows, cols, format, row;
    double sap;
    float low_tol, high_tol;

    int const forceplain = 0;

    pnm_readpnminit( ifp, &cols, &rows, &maxval, &format );
    xelrow = pnm_allocrow( cols );

    pnm_writepnminit( stdout, cols, rows, maxval, format, forceplain );

    low_tol = tolerance / 2.0;
    high_tol = 1.0 - ( tolerance / 2.0 );

    for ( row = 0; row < rows; ++row ) {
        int col;
        pnm_readpnmrow( ifp, xelrow, cols, maxval, format );
        for ( col = 0, xP = xelrow; col < cols; ++col, ++xP ) {
            switch ( PNM_FORMAT_TYPE( format ) ) {
            case PPM_TYPE:
                sap = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( sap < low_tol ) 
                    r = 0;
                else if ( sap >= high_tol )
                    r = maxval;
                else
                    r = PPM_GETR( *xP );

                sap = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( sap < low_tol ) 
                    g = 0;
                else if ( sap >= high_tol )
                    g = maxval;
                else
                    g = PPM_GETG( *xP );

                sap = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( sap < low_tol ) 
                    b = 0;
                else if ( sap >= high_tol )
                    b = maxval;
                else
                    b = PPM_GETB( *xP );

                PPM_ASSIGN( *xP, r, g, b );
                break;

            default:
                sap = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( sap < low_tol ) 
                    g = 0;
                else if ( sap >= high_tol )
                    g = maxval;
                else
                    g = PNM_GET1( *xP );

                PNM_ASSIGN1( *xP, g );
                break;
            }
        }
        pnm_writepnmrow(stdout, xelrow, cols, maxval, format, forceplain);
    }
    pnm_freerow(xelrow);
}



/* Laplacian Noise 
**
** From Pitas' book.
*/

static void
laplacian_noise(FILE * const ifp,
                float  const lsigma) {

    xel *xP;
    xel *xelrow;
    xelval r, g, b;
    xelval maxval;
    int rows, cols, format, row;
    double rtemp, gtemp, btemp;
    double u, u1;
    double INFINITY;

    double const EPSILON = 1.0e-5;
    int const forceplain = 0;

    pnm_readpnminit( ifp, &cols, &rows, &maxval, &format );
    xelrow = pnm_allocrow( cols );
    INFINITY = (double) maxval;

    pnm_writepnminit( stdout, cols, rows, maxval, format, forceplain );

    for ( row = 0; row < rows; ++row ) {
        int col;
        pnm_readpnmrow( ifp, xelrow, cols, maxval, format );
        for ( col = 0, xP = xelrow; col < cols; ++col, ++xP ) {
            switch ( PNM_FORMAT_TYPE( format ) ) {
            case PPM_TYPE:
                u = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( u <= 0.5 ) {
                    if ( u <= EPSILON )
                        rtemp = PPM_GETR( *xP ) - INFINITY;
                    else
                        rtemp = PPM_GETR( *xP ) + lsigma * log( 2.0 * u );
                } else {
                    if ( ( u1 = 1.0 - u ) <= 0.5 * EPSILON )
                        rtemp = PPM_GETR( *xP ) + INFINITY;
                    else
                        rtemp = PPM_GETR( *xP ) - lsigma * log( 2.0 * u1 );
                }

                u = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( u <= 0.5 ) {
                    if ( u <= EPSILON )
                        gtemp = PPM_GETG( *xP ) - INFINITY;
                    else
                        gtemp = PPM_GETG( *xP ) + lsigma * log( 2.0 * u );
                } else {
                    if ( ( u1 = 1.0 - u ) <= 0.5 * EPSILON )
                        gtemp = PPM_GETG( *xP ) + INFINITY;
                    else
                        gtemp = PPM_GETG( *xP ) - lsigma * log( 2.0 * u1 );
                }

                u = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( u <= 0.5 ) {
                    if ( u <= EPSILON )
                        btemp = PPM_GETB( *xP ) - INFINITY;
                    else
                        btemp = PPM_GETB( *xP ) + lsigma * log( 2.0 * u );
                } else {
                    if ( ( u1 = 1.0 - u ) <= 0.5 * EPSILON )
                        btemp = PPM_GETB( *xP ) + INFINITY;
                    else
                        btemp = PPM_GETB( *xP ) - lsigma * log( 2.0 * u1 );
                }

                CHECK_RED;
                CHECK_GREEN;
                CHECK_BLUE;
                PPM_ASSIGN( *xP, r, g, b );
                break;

            default:
                u = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( u <= 0.5 ) {
                    if ( u <= EPSILON )
                        gtemp = PNM_GET1( *xP ) - INFINITY;
                    else
                        gtemp = PNM_GET1( *xP ) + lsigma * log( 2.0 * u );
                } else {
                    if ( ( u1 = 1.0 - u ) <= 0.5 * EPSILON )
                        gtemp = PNM_GET1( *xP ) + INFINITY;
                    else
                        gtemp = PNM_GET1( *xP ) - lsigma * log( 2.0 * u1 );
                }

                CHECK_GRAY;
                PNM_ASSIGN1( *xP, g );
                break;
            }
        }
        pnm_writepnmrow(stdout, xelrow, cols, maxval, format, forceplain);
    }
    pnm_freerow(xelrow);
}



/* Multiplicative Gaussian Noise 
**
** From Pitas' book. 
*/

static void
multiplicative_gaussian_noise(FILE * const ifp,
                              float  const mgsigma) {

    xel *xP;
    xel *xelrow;
    xelval r, g, b;
    xelval maxval;
    int rows, cols, format, row;
    double rtemp, gtemp, btemp;
    double uniform, rayleigh, gauss;
    double INFINITY;

    double const EPSILON = 1.0e-5;
    int const forceplain = 0;

    pnm_readpnminit( ifp, &cols, &rows, &maxval, &format );
    xelrow = pnm_allocrow( cols );
    INFINITY = (double) maxval;

    pnm_writepnminit( stdout, cols, rows, maxval, format, forceplain );

    for ( row = 0; row < rows; ++row ) {
        int col;
        pnm_readpnmrow( ifp, xelrow, cols, maxval, format );
        for ( col = 0, xP = xelrow; col < cols; ++col, ++xP ) {
            switch ( PNM_FORMAT_TYPE( format ) ) {
            case PPM_TYPE:
                uniform = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( uniform <= EPSILON )
                    rayleigh = INFINITY;
                else
                    rayleigh = sqrt( -2.0 * log( uniform ) );
                uniform = ( rand( ) & RANDOM_MASK ) / arand; 
                gauss = rayleigh * cos( 2.0 * M_PI * uniform );
                rtemp = PPM_GETR(*xP) + (PPM_GETR(*xP) * mgsigma * gauss);

                uniform = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( uniform <= EPSILON )
                    rayleigh = INFINITY;
                else
                    rayleigh = sqrt( -2.0 * log( uniform ) );
                uniform = ( rand( ) & RANDOM_MASK ) / arand; 
                gauss = rayleigh * cos( 2.0 * M_PI * uniform );
                gtemp = PPM_GETG(*xP) + (PPM_GETG(*xP) * mgsigma * gauss);

                uniform = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( uniform <= EPSILON )
                    rayleigh = INFINITY;
                else
                    rayleigh = sqrt( -2.0 * log( uniform ) );
                uniform = ( rand( ) & RANDOM_MASK ) / arand; 
                gauss = rayleigh * cos( 2.0 * M_PI * uniform );
                btemp = PPM_GETB(*xP) + (PPM_GETB(*xP) * mgsigma * gauss);

                CHECK_RED;
                CHECK_GREEN;
                CHECK_BLUE;
                PPM_ASSIGN( *xP, r, g, b );
                break;

            default:
                uniform = ( rand( ) & RANDOM_MASK ) / arand; 
                if ( uniform <= EPSILON )
                    rayleigh = INFINITY;
                else
                    rayleigh = sqrt( -2.0 * log( uniform ) );
                uniform = ( rand( ) & RANDOM_MASK ) / arand; 
                gauss = rayleigh * cos( 2.0 * M_PI * uniform );
                gtemp = PNM_GET1(*xP) + (PNM_GET1(*xP) * mgsigma * gauss);
                
                CHECK_GRAY;
                PNM_ASSIGN1( *xP, g );
                break;
            }
        }
        pnm_writepnmrow( stdout, xelrow, cols, maxval, format, forceplain );
    }
    pnm_freerow( xelrow );
}



/* Poisson Noise
**
*/

static void
poisson_noise(FILE * const ifp,
              float  const lambda) {

    xel *xP;
    xel *xelrow;
    xelval r, g, b;
    xelval maxval;
    int rows, cols, format, row;
    double rtemp, gtemp, btemp;
    double x, x1;
    float rr;
    int k;

    int const forceplain = 0;

    pnm_readpnminit( ifp, &cols, &rows, &maxval, &format );
    xelrow = pnm_allocrow( cols );

    pnm_writepnminit( stdout, cols, rows, maxval, format, forceplain );

    for ( row = 0; row < rows; ++row ) {
        int col;
        pnm_readpnmrow( ifp, xelrow, cols, maxval, format );
        for ( col = 0, xP = xelrow; col < cols; ++col, ++xP ) {
            switch ( PNM_FORMAT_TYPE( format ) ) {
            case PPM_TYPE:
                x = lambda * PPM_GETR( *xP );
                x1 = exp( -x );
                rr = 1.0;
                k = 0;
                rr = rr * ( ( rand( ) & RANDOM_MASK ) / arand );
                while ( rr > x1 ) {
                    ++k;
                    rr = rr * ( ( rand( ) & RANDOM_MASK ) / arand );
                }
                rtemp = k / lambda;

                x = lambda * PPM_GETG( *xP );
                x1 = exp( -x );
                rr = 1.0;
                k = 0;
                rr = rr * ( ( rand( ) & RANDOM_MASK ) / arand );
                while ( rr > x1 ) {
                    ++k;
                    rr = rr * ( ( rand( ) & RANDOM_MASK ) / arand );
                }    
                gtemp = k / lambda;

                x = lambda * PPM_GETB( *xP );
                x1 = exp( -x );
                rr = 1.0;
                k = 0;
                rr = rr * ( ( rand( ) & RANDOM_MASK ) / arand );
                while ( rr > x1 ) {
                    ++k;
                    rr = rr * ( ( rand( ) & RANDOM_MASK ) / arand );
                }
                btemp = k / lambda;

                CHECK_RED;
                CHECK_GREEN;
                CHECK_BLUE;
                PPM_ASSIGN( *xP, r, g, b );
                break;

            default:
                x = lambda * PNM_GET1( *xP );
                x1 = exp( -x );
                rr = 1.0;
                k = 0;
                rr = rr * ( ( rand( ) & RANDOM_MASK ) / arand );
                while ( rr > x1 ) {
                    ++k;
                    rr = rr * ( ( rand( ) & RANDOM_MASK ) / arand );
                }
                gtemp = k / lambda;

                CHECK_GRAY;
                PNM_ASSIGN1( *xP, g );
                break;
            }
        }
        pnm_writepnmrow( stdout, xelrow, cols, maxval, format, forceplain );
    }
    pnm_freerow( xelrow );
}



int 
main(int argc, char * argv[]) {
    FILE *ifp;
    int argn;
    int noise_type;
    int seed;
    int i;
    const char * const usage = "[-type noise_type] [-lsigma x] [-mgsigma x] "
        "[-sigma1 x] [-sigma2 x] [-lambda x] [-seed n] "
        "[-tolerance ratio] [pgmfile]";

    const char * const noise_name[] = { 
        "gaussian",
        "impulse",
        "laplacian",
        "multiplicative_gaussian",
        "poisson"
    };
    int const noise_id[] = { 
        GAUSSIAN,
        IMPULSE,
        LAPLACIAN,
        MULTIPLICATIVE_GAUSSIAN,
        POISSON
    };
    /* minimum number of characters to match noise name for pm_keymatch() */
    int const noise_compare[] = {
        1,
        1,
        1,
        1,
        1
    };

    /* define default values for configurable options */
    float lambda = 0.05;        
    float lsigma = 10.0;
    float mgsigma = 0.5;
    float sigma1 = 4.0;
    float sigma2 = 20.0;
    float tolerance = 0.10;

    pnm_init(&argc, argv);

    seed = time(NULL) ^ getpid();
    noise_type = GAUSSIAN;

    argn = 1;
    while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
    {
        if ( pm_keymatch( argv[argn], "-lambda", 3 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( 
                    "incorrect number of arguments for -lambda option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -lambda option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            lambda = atof( argv[argn] );
        }
        else if ( pm_keymatch( argv[argn], "-lsigma", 3 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( 
                    "incorrect number of arguments for -lsigma option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -lsigma option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            lsigma = atof( argv[argn] );
        }
        else if ( pm_keymatch( argv[argn], "-mgsigma", 2 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( 
                    "incorrect number of arguments for -mgsigma option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -mgsigma option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            mgsigma = atof( argv[argn] );
        }
        else if ( pm_keymatch( argv[argn], "-seed", 3 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( "incorrect number of arguments for -seed option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -seed option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            seed = atoi(argv[argn]);
        }
        else if ( pm_keymatch( argv[argn], "-sigma1", 7 ) ||
                  pm_keymatch( argv[argn], "-s1", 3 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( 
                    "incorrect number of arguments for -sigma1 option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -sigma1 option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            sigma1 = atof( argv[argn] );
        }
        else if ( pm_keymatch( argv[argn], "-sigma2", 7 ) ||
                  pm_keymatch( argv[argn], "-s2", 3 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( 
                    "incorrect number of arguments for -sigma2 option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -sigma2 option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            sigma2 = atof( argv[argn] );
        }
        else if ( pm_keymatch( argv[argn], "-tolerance", 3 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( 
                    "incorrect number of arguments for -tolerance option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -tolerance option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            tolerance = atof( argv[argn] );
        }
        else if ( pm_keymatch( argv[argn], "-type", 3 ) )
        {
            ++argn;
            if ( argn >= argc )
            {
                pm_message( "incorrect number of arguments for -type option" );
                pm_usage( usage );
            }
            else if ( argv[argn][0] == '-' )
            {
                pm_message( "invalid argument to -type option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            /* search through list of valid noise types and compare */
            i = 0;
            while ( ( i < MAX_NOISE_TYPES ) && 
                    !pm_keymatch( argv[argn], 
                                  noise_name[i], noise_compare[i] ) )
                ++i;
            if ( i >= MAX_NOISE_TYPES )
            {
                pm_message( "invalid argument to -type option: %s", 
                            argv[argn] );
                pm_usage( usage );
            }
            noise_type = noise_id[i];
        }
        else
            pm_usage( usage );
        ++argn;
    }

    if ( argn < argc )
    {
        ifp = pm_openr( argv[argn] );
        argn++;
    }
    else
        ifp = stdin;

    if ( argn != argc )
        pm_usage( usage );

    srand(seed);

    switch (noise_type) {
    case GAUSSIAN:
        gaussian_noise(ifp, sigma1, sigma2);
        break;
    
    case IMPULSE:
        impulse_noise(ifp, tolerance);
        break;

    case LAPLACIAN:
        laplacian_noise(ifp, lsigma);
        break;

    case MULTIPLICATIVE_GAUSSIAN:
        multiplicative_gaussian_noise(ifp, mgsigma);
        break;
    
    case POISSON:
        poisson_noise(ifp, lambda);
        break;

    default:  
        pm_error( 
            "This should not be happening to you... no noise of that type");
        break;
    }

    return 0;
}
