/******************************************************************************
                                  Pnmnorm
*******************************************************************************

  This program normalizes the contrast in a Netpbm image.

  by Bryan Henderson bryanh@giraffe-data.com San Jose CA March 2002.
  Adapted from Ppmnorm.

  Ppmnorm is by Wilson H. Bent, Jr. (whb@usc.edu)
  Extensively hacked from pgmnorm.c, which carries the following note:
  
  Copyright (C) 1989, 1991 by Jef Poskanzer.
  
  Permission to use, copy, modify, and distribute this software and its
  documentation for any purpose and without fee is hereby granted, provided
  that the above copyright notice appear in all copies and that both that
  copyright notice and this permission notice appear in supporting
  documentation.  This software is provided "as is" without express or
  implied warranty.
  
  (End of note from pgmnorm.c)

  Pgmnorm's man page also said:
  
  Partially based on the fbnorm filter in Michael Mauldin's "Fuzzy Pixmap"
  package.
*****************************************************************************/

#include "pnm.h"
#include "shhopt.h"
#include "mallocvar.h"

enum brightMethod {BRIGHT_LUMINOSITY, BRIGHT_COLORVALUE, BRIGHT_SATURATION};

struct cmdlineInfo {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    const char *inputFilespec;  /* Filespec of input file */
    unsigned int bvalueSpec;
    xelval bvalue;
    unsigned int bpercentSpec;
    float bpercent;
    unsigned int wvalueSpec;
    xelval wvalue;
    unsigned int wpercentSpec;
    float wpercent;
    enum brightMethod brightMethod;
    unsigned int keephues;
};



static void
parseCommandLine (int argc, char ** argv,
                  struct cmdlineInfo *cmdlineP) {
/*----------------------------------------------------------------------------
   parse program command line described in Unix standard form by argc
   and argv.  Return the information in the options as *cmdlineP.  

   If command line is internally inconsistent (invalid options, etc.),
   issue error message to stderr and abort program.

   Note that the strings we return are stored in the storage that
   was passed to us as the argv array.  We also trash *argv.
-----------------------------------------------------------------------------*/
    optEntry *option_def;
        /* Instructions to optParseOptions3 on how to parse our options.
         */
    optStruct3 opt;

    unsigned int luminosity, colorvalue, saturation;

    unsigned int option_def_index;

    MALLOCARRAY_NOFAIL(option_def, 100);

    option_def_index = 0;   /* incremented by OPTENT3 */
    OPTENT3(0,   "bpercent",      OPT_FLOAT,   
            &cmdlineP->bpercent,   &cmdlineP->bpercentSpec, 0);
    OPTENT3(0,   "wpercent",      OPT_FLOAT,   
            &cmdlineP->wpercent,   &cmdlineP->wpercentSpec, 0);
    OPTENT3(0,   "bvalue",        OPT_UINT,   
            &cmdlineP->bvalue,     &cmdlineP->bvalueSpec, 0);
    OPTENT3(0,   "wvalue",        OPT_UINT,   
            &cmdlineP->wvalue,     &cmdlineP->wvalueSpec, 0);
    OPTENT3(0,   "keephues",      OPT_FLAG,   
            NULL,                  &cmdlineP->keephues, 0);
    OPTENT3(0,   "luminosity",    OPT_FLAG,   
            NULL,                  &luminosity, 0);
    OPTENT3(0,   "colorvalue",    OPT_FLAG,   
            NULL,                  &colorvalue, 0);
    OPTENT3(0,   "saturation",    OPT_FLAG,   
            NULL,                  &saturation, 0);
    OPTENT3(0,   "brightmax",     OPT_FLAG,   
            NULL,                  &colorvalue, 0);

    /* Note: -brightmax was documented and accepted long before it was
       actually implemented.  By the time we implemented it, we
       decided -colorvalue was a better name for it.
    */

    opt.opt_table = option_def;
    opt.short_allowed = FALSE;  /* We have no short (old-fashioned) options */
    opt.allowNegNum = FALSE;  /* We have no parms that are negative numbers */

    optParseOptions3( &argc, argv, opt, sizeof(opt), 0 );
        /* Uses and sets argc, argv, and some of *cmdline_p and others. */

    if (!cmdlineP->wpercentSpec)
        cmdlineP->wpercent = 1.0;
    if (!cmdlineP->bpercentSpec)
        cmdlineP->bpercent = 2.0;

    if (cmdlineP->wpercent < 0.0)
        pm_error("You specified a negative value for wpercent: %f",
                 cmdlineP->wpercent);
    if (cmdlineP->bpercent < 0.0)
        pm_error("You specified a negative value for bpercent: %f",
                 cmdlineP->bpercent);
    if (cmdlineP->wpercent > 100.0)
        pm_error("You specified a per centage > 100 for wpercent: %f",
                 cmdlineP->wpercent);
    if (cmdlineP->bpercent > 100.0)
        pm_error("You specified a per centage > 100 for bpercent: %f",
                 cmdlineP->bpercent);

    if (luminosity + colorvalue + saturation > 0)
        pm_error("You can specify only one of "
                 "-luminosity, -colorvalue, and -saturation");
    else {
        if (colorvalue)
            cmdlineP->brightMethod = BRIGHT_COLORVALUE;
        else if (saturation)
            cmdlineP->brightMethod = BRIGHT_SATURATION;
        else
            cmdlineP->brightMethod = BRIGHT_LUMINOSITY;
    }

    if (argc-1 > 1)
        pm_error("Program takes at most one argument: the input file "
                 "specification.  "
                 "You specified %d arguments.", argc-1);
    if (argc-1 < 1)
        cmdlineP->inputFilespec = "-";
    else
        cmdlineP->inputFilespec = argv[1];
}



static void
buildHistogram(FILE *   const ifp, 
               int      const cols,
               int      const rows,
               xelval   const maxval,
               int      const format,
               unsigned int   hist[],
               enum brightMethod const brightMethod) {
/*----------------------------------------------------------------------------
   Build the histogram of brightness values for the image that is in file
   'ifp', which is positioned just after the header (at the raster).

   The histogram is the array hist[] such that hist[x] is the number
   of xels in the image that have brightness x.  That brightness is
   either the color value (intensity of most intense component) of the
   xel or it is the luminosity of the xel, depending on
   'brightMethod'.  In either case, it is based on the same maxval as
   the image, which is 'maxval'.  The image is 'cols' columns wide by
   'rows' rows high.

   Leave the file positioned arbitrarily.
-----------------------------------------------------------------------------*/
    int row;
    xel * xelrow;
    
    xelrow = pnm_allocrow(cols);

    {
        unsigned int i;
        for (i = 0; i <= maxval; ++i)
            hist[i] = 0;
    }
    for (row = 0; row < rows; ++row) {
        int col;
        pnm_readpnmrow(ifp, xelrow, cols, maxval, format);
        for (col = 0; col < cols; ++col) {
            xelval brightness;
            xel const p = xelrow[col];
            if (PNM_FORMAT_TYPE(format) == PPM_TYPE) {
                switch(brightMethod) {
                case BRIGHT_LUMINOSITY:
                    brightness = PPM_LUMIN(p);
                    break;
                case BRIGHT_COLORVALUE:
                    brightness = ppm_colorvalue(p);
                    break;
                case BRIGHT_SATURATION:
                    brightness = ppm_saturation(p, maxval);
                    break;
                }
            } else
                brightness = PNM_GET1(p);
            ++hist[brightness];
        }
    }
    pnm_freerow(xelrow);
}



static void
computeBottomPercentile(unsigned int         hist[], 
                        unsigned int   const highest,
                        unsigned int   const total,
                        float          const percent, 
                        unsigned int * const percentileP) {
/*----------------------------------------------------------------------------
   Compute the lowest index of hist[] such that the sum of the hist[]
   values with that index and lower represent at least 'percent' per cent of
   'n' (which is assumed to be the sum of all the values in hist[],
   given to us to save us the time of computing it).
-----------------------------------------------------------------------------*/
    unsigned int cutoff = total * percent / 100.0;
    unsigned int count;
    unsigned int percentile;

    percentile = 0; /* initial value */
    count = hist[0];  /* initial value */

    while (count < cutoff) {
        if (percentile == highest)
            pm_error("Internal error: computeBottomPercentile() received"
                     "a 'total' value greater than the sum of the hist[]"
                     "values");
        ++percentile;
        count += hist[percentile];
    }        
    *percentileP = percentile;
}



static void
computeTopPercentile(unsigned int         hist[], 
                     unsigned int   const highest, 
                     unsigned int   const total,
                     float          const percent, 
                     unsigned int * const percentileP) {
/*----------------------------------------------------------------------------
   Compute the highest index of hist[] such that the sum of the hist[]
   values with that index and higher represent 'percent' per cent of
   'n' (which is assumed to be the sum of all the values in hist[],
   given to us to save us the time of computing it).
-----------------------------------------------------------------------------*/
    unsigned int cutoff = total * percent / 100.0;
    unsigned int count;
    unsigned int percentile;

    percentile = highest; /* initial value */
    count = hist[highest];

    while (count < cutoff) {
        --percentile;
        count += hist[percentile];
    }
    *percentileP = percentile;
}



static void
computeEndValues(FILE *   const ifp,
                 int      const cols,
                 int      const rows,
                 xelval   const maxval,
                 int      const format,
                 struct cmdlineInfo const cmdline,
                 xelval * const bvalueP,
                 xelval * const wvalueP) {
/*----------------------------------------------------------------------------
   Figure out what original values will be translated to full white and
   full black -- thus defining to what all the other values get translated.

   This may involve looking at the image.  The image is in the file
   'ifp', which is positioned just past the header (at the raster).
   Leave it positioned arbitrarily.
-----------------------------------------------------------------------------*/
    unsigned int * hist;  /* malloc'ed */

    MALLOCARRAY(hist, PNM_OVERALLMAXVAL+1);

    if (hist == NULL)
        pm_error("Unable to allocate storage for intensity histogram.");
    else {
        buildHistogram(ifp, cols, rows, maxval, format, hist,
                       cmdline.brightMethod);

        if (cmdline.bvalueSpec && !cmdline.bpercentSpec) {
            *bvalueP = cmdline.bvalue;
        } else {
            computeBottomPercentile(hist, maxval, cols*rows, cmdline.bpercent, 
                                bvalueP);
            if (cmdline.bvalueSpec && cmdline.bvalue < *bvalueP) {
                *bvalueP = cmdline.bvalue;
            }
        }

        if (cmdline.wvalueSpec && !cmdline.wpercentSpec) {
            *wvalueP = cmdline.wvalue;
        } else {
            computeTopPercentile(hist, maxval, cols*rows, cmdline.wpercent, 
                                 wvalueP);
            if (cmdline.wvalueSpec && cmdline.wvalue > *wvalueP) {
                *wvalueP = cmdline.wvalue;
            }
        }

        free(hist);
    }
}



static void
computeTransferFunction(xelval            const bvalue, 
                        xelval            const wvalue,
                        xelval            const maxval,
                        xelval **         const newBrightnessP) {
/*----------------------------------------------------------------------------
   Compute the transfer function, i.e. the array *newBrightnessP such that
   (*newBrightnessP)[x] is the brightness of the xel that should replace a
   xel with brightness x.  Brightness in this case means either luminosity
   or color value (and it doesn't matter to us which).

   'bvalue' is the lowest brightness that should map to zero brightness;
   'wvalue' is the highest brightness that should map to full brightness.
   brightnesses in between should be stretched linearly.

   Define function only for values 0..maxval.
-----------------------------------------------------------------------------*/
    xelval * newBrightness;
    xelval i;

    MALLOCARRAY(newBrightness, maxval+1);

    if (newBrightness == NULL)
        pm_error("Unable to allocate memory for transfer function.");

    /* Map the lowest brightnesses to zero */
    if (bvalue > 0) 
        for (i = 0; i <= bvalue-1; ++i)
            newBrightness[i] = 0;

    /* Map the middle brightnesses to 0..maxval */
    {
        unsigned int const range = wvalue - bvalue;
        unsigned int val;
        /* The following for loop is a hand optimization of this one:
           for (i = bvalue; i <= wvalue; ++i)
             newBrightness[i] = (i-bvalue)*maxval/range);
           (with proper rounding)
        */
        for (i = bvalue, val = maxval/2; 
             i <= wvalue; 
             ++i, val += maxval)
            newBrightness[i] = MIN(val / range, maxval);
        }

    /* Map the highest brightnesses to maxval */
    for (i = wvalue+1; i <= maxval; ++i)
        newBrightness[i] = maxval;

    *newBrightnessP = newBrightness;
}
            


static float
brightScaler(xel               const p,
             pixval            const maxval,
             xelval            const newBrightness[],
             enum brightMethod const brightMethod) {

    xelval oldBrightness;
             
    switch (brightMethod) {
    case BRIGHT_LUMINOSITY:
        oldBrightness = PPM_LUMIN(p);
        break;
    case BRIGHT_COLORVALUE:
        oldBrightness = ppm_colorvalue(p);
        break;
    case BRIGHT_SATURATION:
        oldBrightness = ppm_saturation(p, maxval);
        break;
    }
    return (float)newBrightness[oldBrightness]/oldBrightness;
}
            


static void
writeRowNormalized(xel *             const xelrow,
                   int               const cols,
                   xelval            const maxval,
                   int               const format,
                   enum brightMethod const brightMethod,
                   bool              const keephues,
                   xelval            const newBrightness[],
                   xel *             const rowbuf) {
/*----------------------------------------------------------------------------
   Write to Standard Output a normalized version of the xel row 
   'xelrow'.  Normalize it via the transfer function newBrightness[].

   Use 'rowbuf' as a work buffer.  It is at least 'cols' columns wide.
-----------------------------------------------------------------------------*/
    xel * const outrow = rowbuf;
                
    unsigned int col;
    for (col = 0; col < cols; ++col) {
        xel const p = xelrow[col];

        if (PPM_FORMAT_TYPE(format) == PPM_TYPE) {
            if (keephues) {
                float const scaler =
                    brightScaler(p, maxval, newBrightness, brightMethod);

                xelval const r = MIN((int)(PPM_GETR(p)*scaler+0.5), maxval);
                xelval const g = MIN((int)(PPM_GETG(p)*scaler+0.5), maxval);
                xelval const b = MIN((int)(PPM_GETB(p)*scaler+0.5), maxval);
                PNM_ASSIGN(outrow[col], r, g, b);
            } else 
                PNM_ASSIGN(outrow[col], 
                           newBrightness[PPM_GETR(p)], 
                           newBrightness[PPM_GETG(p)], 
                           newBrightness[PPM_GETB(p)]);
        } else 
            PNM_ASSIGN1(outrow[col], newBrightness[PNM_GET1(p)]);
    }
    pnm_writepnmrow(stdout, outrow, cols, maxval, format, 0);
}



int
main(int argc, char *argv[]) {

    struct cmdlineInfo cmdline;
    FILE *ifP;
    pm_filepos imagePos;
    xelval maxval;
    int rows, cols, format;
    xelval bvalue, wvalue;
    
    pnm_init(&argc, argv);

    parseCommandLine(argc, argv, &cmdline);

    ifP = pm_openr_seekable(cmdline.inputFilespec);

    /* Rescale so that bvalue maps to 0, wvalue maps to maxval. */
    pnm_readpnminit(ifP, &cols, &rows, &maxval, &format);
    pm_tell2(ifP, &imagePos, sizeof(imagePos));

    computeEndValues(ifP, cols, rows, maxval, format, cmdline, 
                     &bvalue, &wvalue);
        
    if (wvalue < bvalue)
        pm_error("The colors which become black would overlap the "
                 "colors which become white.");
    else {
        xelval * newBrightness;
        int row;
        xel * xelrow;
        xel * rowbuf;
        
        xelrow = pnm_allocrow(cols);

        pm_message("remapping %d..%d to %d..%d", bvalue, wvalue, 0, maxval);

        computeTransferFunction(bvalue, wvalue, maxval, &newBrightness);

        pm_seek2(ifP, &imagePos, sizeof(imagePos));
        pnm_writepnminit(stdout, cols, rows, maxval, format, 0);

        rowbuf = pnm_allocrow(cols);

        for (row = 0; row < rows; ++row) {
            pnm_readpnmrow(ifP, xelrow, cols, maxval, format);
            writeRowNormalized(xelrow, cols, maxval, format,
                               cmdline.keephues, cmdline.brightMethod,
                               newBrightness, rowbuf);
        }
        free(newBrightness);
        pnm_freerow(rowbuf);
        pnm_freerow(xelrow);
    }
    pm_close(ifP);
    return 0;
}

