/* libpnm1.c - pnm utility library part 1
**
** Copyright (C) 1989 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

#include <string.h>
#include <errno.h>

#include "pnm.h"

#include "ppm.h"
#include "libppm.h"

#include "pgm.h"
#include "libpgm.h"

#include "pbm.h"
#include "libpbm.h"

#include "mallocvar.h"



xel *
pnm_allocrow(unsigned int const cols) {

    xel * xelrow;

    MALLOCARRAY(xelrow, cols);

    if (xelrow == 0)
        pm_error("Unable to allocate space for a %u-column xel row", cols);

    return xelrow;
}



void
pnm_init( argcP, argv )
    int* argcP;
    char* argv[];
    {
    ppm_init( argcP, argv );
    }

void
pnm_nextimage(FILE *file, int * const eofP) {
    pm_nextimage(file, eofP);
}



static void
validateComputableSize(unsigned int const cols,
                       unsigned int const rows) {
/*----------------------------------------------------------------------------
   Validate that the dimensions of the image are such that it can be
   processed in typical ways on this machine without worrying about
   overflows.  Note that in C, arithmetic is always modulus
   arithmetic, so if your values are too big, the result is not what
   you expect.  That failed expectation can be disastrous if you use
   it to allocate memory.

   A common operation is adding 1 or 2 to the highest row or
   column number in the image, so we make sure that's possible.
-----------------------------------------------------------------------------*/
    if (cols > INT_MAX - 2)
        pm_error("image width (%u) too large to be processed", cols);
    if (rows > INT_MAX - 2)
        pm_error("image height (%u) too large to be processed", rows);
}



void
pnm_readpnminit(FILE *   const fileP,
                int *    const colsP,
                int *    const rowsP,
                xelval * const maxvalP,
                int *    const formatP) {

    /* Check magic number. */
    *formatP = pm_readmagicnumber(fileP);
    switch (PNM_FORMAT_TYPE(*formatP)) {
	case PPM_TYPE: {
        pixval maxval;
        ppm_readppminitrest(fileP, colsP, rowsP, &maxval);
        *maxvalP = maxval;
    }
    break;

	case PGM_TYPE: {
        gray maxval;

        pgm_readpgminitrest(fileP, colsP, rowsP, &maxval);
        *maxvalP = maxval;
    }
    break;

	case PBM_TYPE:
        pbm_readpbminitrest(fileP, colsP, rowsP);
        *maxvalP = 1;
	break;

	default:
        pm_error("bad magic number - not a ppm, pgm, or pbm file");
	}
    validateComputableSize(*colsP, *rowsP);
}



#if __STDC__
void
pnm_readpnmrow( FILE* file, xel* xelrow, int cols, xelval maxval, int format )
#else /*__STDC__*/
void
pnm_readpnmrow( file, xelrow, cols, maxval, format )
    FILE* file;
    xel* xelrow;
    xelval maxval;
    int cols, format;
#endif /*__STDC__*/
    {
    register int col;
    register xel* xP;
    gray* grayrow;
    register gray* gP;
    bit* bitrow;
    register bit* bP;

    switch ( PNM_FORMAT_TYPE(format) )
	{
	case PPM_TYPE:
	ppm_readppmrow( file, (pixel*) xelrow, cols, (pixval) maxval, format );
	break;

	case PGM_TYPE:
	grayrow = pgm_allocrow( cols );
	pgm_readpgmrow( file, grayrow, cols, (gray) maxval, format );
	for ( col = 0, xP = xelrow, gP = grayrow; col < cols; ++col, ++xP, ++gP )
	    PNM_ASSIGN1( *xP, *gP );
	pgm_freerow( grayrow );
	break;

	case PBM_TYPE:
	bitrow = pbm_allocrow( cols );
	pbm_readpbmrow( file, bitrow, cols, format );
	for ( col = 0, xP = xelrow, bP = bitrow; col < cols; ++col, ++xP, ++bP )
	    PNM_ASSIGN1( *xP, *bP == PBM_BLACK ? 0: maxval );
	pbm_freerow( bitrow );
	break;

	default:
	pm_error( "can't happen" );
	}
    }

xel**
pnm_readpnm( file, colsP, rowsP, maxvalP, formatP )
    FILE* file;
    int* colsP;
    int* rowsP;
    int* formatP;
    xelval* maxvalP;
    {
    xel** xels;
    int row;

    pnm_readpnminit( file, colsP, rowsP, maxvalP, formatP );

    xels = pnm_allocarray( *colsP, *rowsP );

    for ( row = 0; row < *rowsP; ++row )
	pnm_readpnmrow( file, xels[row], *colsP, *maxvalP, *formatP );

    return xels;
    }


void
pnm_check(FILE * file, const enum pm_check_type check_type, 
          const int format, const int cols, const int rows, const int maxval,
          enum pm_check_code * const retvalP) {

    switch (PNM_FORMAT_TYPE(format)) {
    case PBM_TYPE:
        pbm_check(file, check_type, format, cols, rows, retvalP);
        break;
    case PGM_TYPE: 
        pgm_check(file, check_type, format, cols, rows, maxval, retvalP);
        break;
    case PPM_TYPE:
        ppm_check(file, check_type, format, cols, rows, maxval, retvalP);
        break;
    default:
        pm_error("pnm_check() called with invalid format parameter");
    }
}
