/* pnmtofits.c - read a PNM image and produce a FITS file
**
** Copyright (C) 1989 by Wilson H. Bent (whb@hoh-2.att.com).
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** Modified by Alberto Accomazzi (alberto@cfa.harvard.edu), Dec 1, 1992.
**
** Added PPM input capability; the program is renamed pnmtofits.
** This program produces files with NAXIS = 2 if input file is in PBM
** or PGM format, and NAXIS = 3, NAXIS3 = 3 if input file is a PPM file.
** Data is written out as either 8 bits/pixel or 16 bits/pixel integers,
** depending on the value of maxval in the input file.
** Flags -max, -min can be used to set DATAMAX, DATAMIN, BSCALE and BZERO
** in the FITS header, but do not cause the data to be rescaled.
*/

#include "pam.h"

static void
writeCard(const char * const s) {
    fwrite(s, sizeof(s[0]), 80, stdout);
}



static void
writeFitsHeader(int    const bitpix,
                int    const planes,
                int    const cols,
                int    const rows,
                double const bscale,
                double const fitsBzero,
                double const datamax,
                double const datamin) {

    char card[80+1];
    unsigned int cardsWritten;
                
    cardsWritten = 0;  /* initial value */
    sprintf(card, "%-20.20s%10.10s%-50.50s", "SIMPLE  =", "T", "");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-20.20s%10d%-50.50s", "BITPIX  =", bitpix, "");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-20.20s%10d%-50.50s",
            "NAXIS   =", (planes == 3) ? 3 : 2, "");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-20.20s%10d%-50.50s", "NAXIS1  =", cols, "");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-20.20s%10d%-50.50s", "NAXIS2  =", rows, "");
    writeCard(card); ++cardsWritten;
    if (planes == 3) {
        sprintf(card, "%-20.20s%10d%-50.50s", "NAXIS3  =", 3, "");
        writeCard(card); ++cardsWritten;
    }
    sprintf(card, "%-18.18s%E%-51.51s", "BSCALE  =", bscale, "");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-18.18s%E%-51.51s", "BZERO   =", fitsBzero, "");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-18.18s%E%-51.51s", "DATAMAX =", datamax, "");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-18.18s%E%-51.51s", "DATAMIN =", datamin, "");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-80.80s", "HISTORY Created by pnmtofits.");
    writeCard(card); ++cardsWritten;
    sprintf(card, "%-80.80s", "END");
    writeCard(card); ++cardsWritten;

    {
        /* pad with blanks to multiple of 36 cards */
        unsigned int const npad = 80 * (36 - (cardsWritten % 36));
        unsigned int i;
        
        for (i = 0; i < npad; ++i)
            putchar(' ');
    }
}



static void
writeRaster(struct pam * const pamP,
            tuple **     const tuples,
            unsigned int const bitpix,
            int          const offset) {

    unsigned int plane;

    for (plane = 0; plane < pamP->depth; ++plane) {
        unsigned int row;
        for (row = 0; row < pamP->height; ++row) {
            unsigned int col;
            for (col = 0; col < pamP->width; ++col) {
                if (bitpix == 16) {
                    /* 16 bit FITS samples are signed integers */
                    int const fitsSample =
                        (int)tuples[row][col][plane] - offset;
                    pm_writebigshort(stdout, (short)fitsSample);
                } else
                    /* 8 bit FITS samples are unsigned integers */
                    putchar(tuples[row][col][plane] - offset);
            }
        }
    }
    {
        /* pad raster to 36 cards with nulls */
        unsigned int const bytesWritten =
            pamP->height * pamP->width * pamP->depth * bitpix/8;
        unsigned int const npad = (36*80) - (bytesWritten % (36*80));
        unsigned int i;

        for (i = 0; i < npad; ++i)
            putchar('\0');
    }
}



int
main(int argc, char * argv[]) {

    FILE * ifP;
    tuple ** tuples;
    struct pam pam;
    int bitpix, forcemin, forcemax;
    double datamin, datamax, bscale, fitsBzero, frmax, frmin;
    int pnmSampleOffsetFromFits;
        /* This is what you add to a FITS sample (raster) value in order
           to get the PNM sample value which it represents.  Note that in
           the default case, that PNM sample value is also the FITS "physical"
           value, but user options can change that.
        */
    int argn;
    const char* const usage = "[-max f] [-min f] [pnmfile]";
    
    pnm_init(&argc, argv);

    forcemin = 0;
    forcemax = 0;
    argn = 1;

    while (argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0') {
        if ( pm_keymatch( argv[argn], "-max", 3 ) )
        {
            ++argn;
            forcemax = 1;
            if ( argn == argc || sscanf( argv[argn], "%lf", &frmax ) != 1 )
                pm_usage( usage );
        }
        else if ( pm_keymatch( argv[argn], "-min", 3 ) )
        {
            ++argn;
            forcemin = 1;
            if ( argn == argc || sscanf( argv[argn], "%lf", &frmin ) != 1 )
                pm_usage( usage );
        }
        else
            pm_usage( usage );
        ++argn;
    }

    if ( argn != argc )
    {
        ifP = pm_openr( argv[argn] );
        ++argn;
    }
    else
        ifP = stdin;

    if ( argn != argc )
        pm_usage( usage );

    tuples = pnm_readpam(ifP, &pam, PAM_STRUCT_SIZE(tuple_type));

    if (forcemin)
        datamin = frmin;
    else
        datamin = 0.0;

    if (forcemax)
        datamax = frmax;
    else
        datamax = (double)pam.maxval;

    bscale = (datamax - datamin) / (double) pam.maxval;
    
    if (pam.maxval > 255) {
        bitpix = 16;
        /* Because 16 bit FITS samples are signed, we have to do a 2**15
           offset to get any possible unsigned 16 bit PNM sample into a FITS
           sample.
        */
        fitsBzero = 1 << 15;
        pnmSampleOffsetFromFits = 1 << 15;
    }
    else {
        bitpix = 8;
        fitsBzero = datamin;
        /* Both 8 bit FITS samples and PNM samples are unsigned 8 bits, so
           we make them identical.
        */
        pnmSampleOffsetFromFits = 0;
    }

    fitsBzero = datamin + pnmSampleOffsetFromFits;
    pm_close(ifP);

    writeFitsHeader(bitpix, pam.depth, pam.width, pam.height,
                    bscale, fitsBzero, datamax, datamin);

    writeRaster(&pam, tuples, bitpix, pnmSampleOffsetFromFits);

    return 0;
}
