/* pnmtopalm.c - read a portable pixmap and write a Palm Tbmp file
 *
 * Program originally ppmtoTbmp.c by Ian Goldberg <iang@cs.berkeley.edu>
 *
 * Mods for multiple bits per pixel by George Caswell <tetsujin@sourceforge.net>
 *  and Bill Janssen <bill@janssen.org>
 *
 * Based on ppmtopuzz.c by Jef Paskanzer, from the netpbm-1mar1994 package.
 */

#include <stdio.h>
#include "pnm.h"
#include "palm.h"

int main( int argc, char **argv ) {
    FILE* ifp;
    pixel** pixels;
    pixel transcolor;
    register pixel* pP;
    register unsigned char *outptr;
    unsigned char *rowdata = 0, *lastrow = 0;
    int rows, cols, row, rowbytes, transindex = 0;
    register int col;
    xelval maxval;
    char *transparentValue = 0;
    unsigned char outbyte, outbit;
    unsigned char outbytes[8];
    int format = 0;
    int setOffset = 0;
    int cc, max_bpp = 0, custom_colormap = 0, flags = 0, specified_bpp = 0, bpp;
    int scanline_compression = 0, rle_compression = 0;
    int argsGood = 1, verbose = 0, version, maxmaxval;
    unsigned int newMaxVal, color, last_color, repeatcount, limit;
    Color_s temp_color;
    register Color found;
    Colormap colormap = 0;

    /* Parse default params */
    ppm_init( &argc, argv );

    for (cc = 1; (cc < argc) && (argv[cc][0] == '-'); cc++) {
       if ((strcmp(argv[cc], "-depth") == 0) && ((cc + 1) < argc)) {
	 cc++;
	 specified_bpp = atoi(argv[cc]);
	 if ((specified_bpp != 1) && (specified_bpp != 2) && (specified_bpp != 4) && (specified_bpp != 8))
	   pm_error("bad value for -depth flag; valid values are 1, 2, 4, or 8");
       } else if ((strcmp(argv[cc], "-maxdepth") == 0) && ((cc + 1) < argc)) {
	 cc++;
	 max_bpp = atoi(argv[cc]);
	 if ((max_bpp != 1) && (max_bpp != 2) && (max_bpp != 4) && (max_bpp != 8))
	   pm_error("bad value for -maxdepth flag; valid values are 1, 2, 4, or 8");
       } else if ((strcmp(argv[cc], "-transparent") == 0) && ((cc + 1) < argc)) {
	 cc++;
	 transparentValue = argv[cc];
       } else if (strcmp(argv[cc], "-rle-compression") == 0) {
	 rle_compression = 1;
       } else if (strcmp(argv[cc], "-scanline-compression") == 0) {
	 scanline_compression = 1;
       } else if (strcmp(argv[cc], "-verbose") == 0) {
	 verbose = 1;
       } else if (strcmp(argv[cc], "-colormap") == 0) {
	 custom_colormap = 1;
       } else if (strcmp(argv[cc], "-offset") == 0) {
          setOffset = 1;
       } else {
          argsGood = 0;
       }
    }
    if (!argsGood) {
	pm_usage( "[-depth BITS-PER-PIXEL] [-maxdepth BITS-PER-PIXEL] [-transparent COLOR] [-colormap] [-verbose] [-offset] [ppmfile]" );
    }

    if (max_bpp && specified_bpp && (specified_bpp > max_bpp))
      pm_error ("argument to -depth flag must be less than or equal to argument to -maxdepth flag.");

    if ( cc < argc )
	ifp = pm_openr( argv[cc] );
    else
	ifp = stdin;

    pixels = pnm_readpnm (ifp, &cols, &rows, &maxval, &format);
    pm_close(ifp);

    if ((format == PBM_FORMAT) || (format == RPBM_FORMAT)) {
      /* we can always handle this one */
      if (verbose)
	pm_message("input is binary %dx%d", cols, rows);
      if (specified_bpp)
	bpp = specified_bpp;
      else
	bpp = 1;	/* no point in wasting bits */
      maxmaxval = 1;

    } else if ((format == PGM_FORMAT) || (format == RPGM_FORMAT)) {
      /* we can usually handle this one, but may not have enough pixels.  So check... */
      if (specified_bpp)
	bpp = specified_bpp;
      else if (max_bpp && (maxval >= (1 << max_bpp)))
	bpp = max_bpp;
      else if (maxval > 16)
	bpp = 4;
      else {
	/* scale to minimum number of bpp needed */
	for (bpp = 1;  (1 << bpp) < maxval;  bpp *= 2)
	  ;
      }
      if (bpp > 4)
	bpp = 4;
      if (verbose)
	pm_message("input is greyscale %dx%d, maxval %d -- using %d bits-per-pixel", cols, rows, maxval, bpp);
      maxmaxval = PGM_OVERALLMAXVAL;

    } else if ((format == PPM_FORMAT) || (format == RPPM_FORMAT)) {
      /* We assume that we only get a PPM if the image cannot be represented
	 as PBM or PGM.  So we're going to have to support 8-bit colors.
	 if "custom_colormap" is specified (not recommended by Palm) we will put
	 in our own colormap; otherwise we will assume that the colors have been
	 mapped to the default Palm colormap by appropriate use of ppmquant. */
      if (!custom_colormap) {
	colormap = palmcolor_build_default_8bit_colormap();
	bpp = 8;
	if (((specified_bpp != 0) && (specified_bpp != 8)) ||
	    ((max_bpp != 0) && (max_bpp < 8)))
	  pm_error("Must use depth of 8 for color pixmap without custom color table.");
	if (verbose)
	  pm_message("input is color %dx%d, using default colormap at 8 bpp", cols, rows);
      } else {
	colormap = palmcolor_build_custom_8bit_colormap (rows, cols, pixels);
	for (bpp = 1;  (1 << bpp) < colormap->ncolors;  bpp *= 2)
	  ;
	if (specified_bpp != 0) {
	  if (specified_bpp >= bpp)
	    bpp = specified_bpp;
	  else
	    pm_error("Too many colors for specified depth.  Use ppmquant to reduce.");
	} else if ((max_bpp != 0) && (max_bpp < bpp)) {
	  pm_error("Too many colors for specified max depth.  Use ppmquant to reduce.");
	}
	if (verbose)
	  pm_message("input is color %dx%d, using custom colormap with %d colors at %d bpp", cols, rows, colormap->ncolors, bpp);
      }
      maxmaxval = PPM_OVERALLMAXVAL;
    } else {
      pm_error("unknown format 0x%x on input file\n", (unsigned) format);
    }

    newMaxVal = (1 << bpp) - 1;

    if (transparentValue != 0) {
      transcolor = ppm_parsecolor (transparentValue, maxmaxval);
      temp_color = ((((PPM_GETR(transcolor) * newMaxVal) / maxval) << 16) |
		    (((PPM_GETG(transcolor) * newMaxVal) / maxval) << 8) |
		    ((PPM_GETB(transcolor) * newMaxVal) / maxval));
      found = (bsearch (&temp_color,
			colormap->color_entries, colormap->ncolors,
			sizeof(Color_s), palmcolor_compare_colors));
      if (!found) {
	pm_error("Specified transparent color %s not found in colormap.", transparentValue);
      } else
	transindex = (*found >> 24) & 0xFF;
    }

    flags = 0;
    if (transparentValue != 0)
      flags |= PALM_HAS_TRANSPARENCY_FLAG;
    if (custom_colormap)
      flags |= PALM_HAS_COLORMAP_FLAG;
    if (rle_compression || scanline_compression)
      flags |= PALM_IS_COMPRESSED_FLAG;

    /* Write Tbmp header. */
    (void) pm_writebigshort( stdout, cols );	/* width */
    (void) pm_writebigshort( stdout, rows );	/* height */
    rowbytes = ((cols + (16 / bpp -1)) / (16 / bpp)) * 2;    /* bytes per row - always a word boundary */
    pm_writebigshort(stdout, rowbytes);
    (void) pm_writebigshort( stdout, flags );
    fputc(bpp, stdout);

    /* we need to mark this as version 1 if we use more than 1 bpp,
       version 2 if we use compression or transparency */

    version = ((transparentValue != 0)||rle_compression||scanline_compression) ? 2 : (((bpp > 1)||(custom_colormap)) ? 1 : 0);
    fputc(version, stdout);

    if (setOffset) {
       /* Offset is measured in double-words. Those are 4-byte, right? */
       /* Account for bitmap data size, plus header size, and round up */
       pm_writebigshort(stdout, (rowbytes * rows + 16 + 3)/4);
    } else {
       pm_writebigshort(stdout, 0);
    }

    fputc(transindex, stdout);	/* transparent index */
    if (rle_compression)
      fputc(PALM_COMPRESSION_RLE, stdout);
    else if (scanline_compression)
      fputc(PALM_COMPRESSION_SCANLINE, stdout);
    else
      fputc(PALM_COMPRESSION_NONE, stdout);

    (void) pm_writebigshort( stdout, 0 );	/* reserved by Palm */

    /* if there's a colormap, write it out */
    if (custom_colormap) {
      qsort (colormap->color_entries, colormap->ncolors,
	     sizeof(Color_s), palmcolor_compare_indices);
      pm_writebigshort( stdout, colormap->ncolors );
      for (row = 0;  row < colormap->ncolors;  row++)
	pm_writebiglong (stdout, colormap->color_entries[row]);
      qsort (colormap->color_entries, colormap->ncolors,
	     sizeof(Color_s), palmcolor_compare_colors);
    }

    last_color = 0xFFFFFFFF;
    rowdata = malloc(rowbytes);
    if (scanline_compression)
      lastrow = malloc(rowbytes);
    /* And write out the data. */
    for ( row = 0; row < rows; ++row ) {
        memset(rowdata, 0, rowbytes);
	outbyte = 0x00;
	/* outbit is the lowest bit number we want to access for this pixel */
	outbit = 8 - bpp;
	for ( col = 0, pP = pixels[row], outptr = rowdata; col < cols; ++col, ++pP ) {

	    if (colormap == 0) {
	      /* we assume grayscale, and use simple scaling */
	      color = (PNM_GET1(*pP) * newMaxVal)/maxval;
	      if (color > newMaxVal)
		pm_error("oops.  Bug in color re-calculation code.  color of %u.", color);
	      color = (newMaxVal - color);	/* note greyscale maps are inverted */
	    } else {
	      temp_color = ((((PPM_GETR(*pP) * newMaxVal)/maxval) << 16) |
			    (((PPM_GETG(*pP) * newMaxVal)/maxval) << 8) |
			    (((PPM_GETB(*pP) * newMaxVal)/maxval)));
	      found = (bsearch (&temp_color,
				colormap->color_entries, colormap->ncolors,
				sizeof(Color_s), palmcolor_compare_colors));
	      if (!found) {
		pm_error("Color %d:%d:%d not found in colormap.  "
			 "Try using ppmquant to reduce the number of colors.",
			 PPM_GETR(*pP), PPM_GETG(*pP), PPM_GETB(*pP));
	      }
	      color = (*found >> 24) & 0xFF;
#if 0
	      if (verbose) {
		if (color != last_color) {
		  fprintf (stderr, "Color %d:%d:%d => %d:%d:%d -- %d (0x%x)\n",
			   PPM_GETR(*pP), PPM_GETG(*pP), PPM_GETB(*pP),
			   (*found & 0xFF0000) >> 16,
			   (*found & 0xFF00) >> 8,
			   (*found & 0xFF),
			   color, color);
		  last_color = color;
		};
	      };
#endif
	    }

	    if (color > newMaxVal)
	      pm_error("oops.  Bug in color re-calculation code.  color of %u.", color);
	    outbyte |= (color << outbit);
	    if (outbit == 0) {
	      *outptr++ = outbyte;
	      outbyte = 0x00;
	      outbit = 8 - bpp;
	    } else {
	      outbit -= bpp;
	    }
	}
	if ((cols % (8 / bpp)) != 0) {
	  /* Handle case of partial byte */
	  *outptr++ = outbyte;
	}

	/* now output the row of pixels, compressed appropriately */

	if (scanline_compression) {
	  for (col = 0;  col < rowbytes;  col += 8) {
	    limit = ((rowbytes - col) < 8) ? (rowbytes - col) : 8;
	    for (outbit = 0, outbyte = 0, outptr = outbytes;  outbit < limit;  outbit++) {
	      if ((row == 0) || (lastrow[col + outbit] != rowdata[col + outbit])) {
		outbyte |= (1 << (7 - outbit));
		*outptr++ = rowdata[col + outbit];
	      }
	    }
	    putc(outbyte, stdout);
	    fwrite(outbytes, (outptr - outbytes), 1, stdout);
	  }
	  memcpy (lastrow, rowdata, rowbytes);
	  
	} else if (rle_compression) {
	  /* we output a count of the number of bytes a value is repeated, followed by that byte value */
	  col = 0;
	  while (col < rowbytes) {
	    for (repeatcount = 1, outptr = rowdata + col;  (repeatcount < (rowbytes - col)) && (repeatcount < 256);  repeatcount += 1)
	      if (*outptr != *(outptr + repeatcount))
		break;
	    putc(repeatcount, stdout);
	    putc(*outptr, stdout);
	    col += repeatcount;
	  }

	} else /* no compression */ {
	  fwrite(rowdata, rowbytes, 1, stdout);
	}
    }
    /* If we've set the offset, pad to the next doubleword */
    if (setOffset) {
        int cc;

        for(cc = (rowbytes * rows + 16)%4; cc > 0; cc--) {
           fputc(0, stdout);
        }
    }

    exit( 0 );
}
