/* NVTV null backend -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Null backend for testing/debugging/calculating without hardware present.
 *
 */

#include "local.h" /* before everything else */

#include "backend.h"
#include "back_null.h"
#include "data.h"

static ChipInfo null_chip_ph2 = {
  name: "<Null Philips-2 chip> (5:8a)", chip: TV_PHILIPS_MODEL2, 
  bus:4, addr:0x8a, next: NULL
};
  
static ChipInfo null_chip_ph1 = {
  name: "<Null Philips-1 chip> (4:8a)", chip: TV_PHILIPS_MODEL1, 
  bus:3, addr:0x8a, next: &null_chip_ph2
};
  
static ChipInfo null_chip_ch2 = {
  name: "<Null Chrontel-2 chip> (3:ec)", chip: TV_CHRONTEL_MODEL2, 
  bus:2, addr:0xec, next: &null_chip_ph1
};
  
static ChipInfo null_chip_ch1 = {
  name: "<Null Chrontel-1 chip> (2:ea)", chip: TV_CHRONTEL_MODEL1, 
  bus:2, addr:0xea, next: &null_chip_ch2
};
  
static ChipInfo null_chip_cx = {
  name: "<Null Conexant chip> (1:8a)", chip: TV_CONEXANT, 
  bus:1, addr:0x8a, next: &null_chip_ch1
};
  
static ChipInfo null_chip_bt = {
  name: "<Null Brooktree chip> (0:88)", chip: TV_BROOKTREE, 
  bus:0, addr:0x88, next: &null_chip_cx
};

#define null_chips null_chip_bt
  
static CardInfo null_i810_card = {
  name: "<Null Intel card>", type: CARD_I810, 
  dev:NULL, reg_base:0x0, pci_id:0x7121,
  arch: "No Architecture", chips: &null_chips, next: NULL
};

static CardInfo null_tdfx_card = {
  name: "<Null Voodoo card>", type: CARD_TDFX, 
  dev:NULL, reg_base:0x0, pci_id:0x0005,
  arch: "No Architecture", chips: &null_chips, next: &null_i810_card
};

static CardInfo null_xbox_card = {
  name: "<Null XBox card>", type: CARD_XBOX, 
  dev:NULL, reg_base:0x0, pci_id:0x02A0,
  arch: "No Architecture", chips: &null_chips, next: &null_tdfx_card
};

static CardInfo null_nv_card = {
  name: "<Null NVidia card>", type: CARD_NVIDIA, 
  dev:NULL, reg_base:0x0, pci_id:0x0110,
  arch: "No Architecture", chips: &null_chips, next: &null_xbox_card
};

#define null_cards null_nv_card

static DataFunc *bnull_data = NULL;
static CardPtr bnull_card = NULL;

void bnull_openCard (CardPtr card)
{
  DPRINTF ("bnull_open\n");
  bnull_card = card;
  bnull_data = data_func (card->type, card->chips->chip);
}

void bnull_closeCard (void)
{
  DPRINTF ("bnull_close\n");
}

#ifdef DEBUG_PROBE
void bnull_probeSystem (CardPtr card_list)
{
  DPRINTF ("bnull_probeSystem\n");
}

void bnull_probeCard (void)
{
  DPRINTF ("bnull_probeCard\n");
}

I2CChainPtr bnull_probeBus (void)
{
  DPRINTF ("bnull_probeBus\n");
  return NULL;
}
#endif

void bnull_setHeads (int main, int tv, int video)
{
  DPRINTF ("bnull_setHeads\n");
}

void bnull_getHeads (int *main, int *tv, int *video, int *max) 
{
  DPRINTF ("bnull_getHeads\n");
  if (max)   *max = 2;
}

void bnull_getHeadDev (int head, int *devFlags) 
{
  DPRINTF ("bnull_getHeadDev\n");
  if (devFlags) *devFlags = DEV_MONITOR;
}

void bnull_probeChips (void)
{
  DPRINTF ("bnull_probe\n");
  if (bnull_card) {
    bnull_data = data_func (bnull_card->type, bnull_card->chips->chip);
  } else {
    bnull_data = &null_func;
  }
}

void bnull_setChip (ChipPtr chip, Bool init)
{
  DPRINTF ("bnull_setChip %s %i\n", chip->name, init);
  bnull_data = data_func (bnull_card->type, chip->chip);
}

void bnull_setSettings (TVSettings *set)
{
  DPRINTF ("bnull_setSettings\n");
}

void bnull_getSettings (TVSettings *set)
{
  DPRINTF ("bnull_getSettings\n");
}

void bnull_setMode (TVRegs *r)
{
  DPRINTF ("bnull_setMode\n");
}

void bnull_getMode (TVRegs *r)
{
  DPRINTF ("bnull_getMode\n");
}

void bnull_setModeSettings (TVRegs *r, TVSettings *set)
{
  DPRINTF ("bnull_setModeSettings\n");
}

void bnull_setTestImage (TVEncoderRegs *tv, TVSettings *set)
{
  DPRINTF ("bnull_setTestImage\n");
}

long bnull_getStatus (int index)
{
  DPRINTF ("bnull_getStatus\n");
  return 0;
}

TVConnect bnull_getConnection (void)
{
  DPRINTF ("bnull_getConnection\n");
  return CONNECT_NONE;
}

int bnull_listModes (TVSystem system, TVMode *(list[]))
{
  return data_listModes (bnull_data->modes (), system, list);
}

Bool bnull_findBySize (TVSystem system, int xres, int yres, char *size, 
    TVMode *mode)
{
  TVMode *tvm;

  DPRINTF ("bnull_findBySize %i %i,%i %s\n", system, xres, yres, size);
  /* FIXME: All cards */
  tvm = data_find (bnull_data->modes (), system, xres, yres, size);
  if (!tvm) return FALSE;
  *mode = *tvm;
  return TRUE;
}

Bool bnull_findByOverscan (TVSystem system, int xres, int yres, 
    double hoc, double voc, TVMode *mode)
{
  DPRINTF ("bnull_findByOC %i %i,%i\n", system, xres, yres);
  return FALSE;
}

void bnull_initSharedView (int *view_x, int *view_y)
{
  *view_x = *view_y = 0;
}

Bool bnull_getTwinView (int *view_x, int *view_y)
{
  *view_x = *view_y = 0;
  return FALSE;
}

Bool bnull_adjustViewport (int flags, int *view_x, int *view_y)
{
  *view_x = *view_y = 0;
  return FALSE;
}

Bool bnull_serviceViewportCursor (int flags, int cursor_x, int cursor_y, 
  int *view_x, int *view_y)
{
  *view_x = *view_y = 0;
  return FALSE;
}

BackAccessRec bnull_access_func = {
  openCard:    bnull_openCard,
  closeCard:   bnull_closeCard,
#ifdef DEBUG_PROBE
  probeSystem: bnull_probeSystem,
#endif
};

BackCardRec bnull_card_func = {
  openCard:              bnull_openCard,
  closeCard:             bnull_closeCard,
#ifdef DEBUG_PROBE
  probeCard:             bnull_probeCard,
  probeBus:              bnull_probeBus,
#endif
  setHeads:              bnull_setHeads,
  getHeads:              bnull_getHeads,
  getHeadDev:            bnull_getHeadDev,
  probeChips:            bnull_probeChips,
  setChip:               bnull_setChip,
  setSettings:           bnull_setSettings,
  getSettings:           bnull_getSettings,
  setMode:               bnull_setMode,
  getMode:               bnull_getMode,
  setModeSettings:       bnull_setModeSettings,
  setTestImage:          bnull_setTestImage, 
  getStatus:             bnull_getStatus,    
  getConnection:         bnull_getConnection,
  listModes:		 bnull_listModes,
  findBySize:            bnull_findBySize, 
  findByOverscan:        bnull_findByOverscan,
  initSharedView:        bnull_initSharedView,
  getTwinView:           bnull_getTwinView,
  adjustViewport:        bnull_adjustViewport,
  serviceViewportCursor: bnull_serviceViewportCursor,
};

CardPtr back_null_init (void)
{
  back_access = &bnull_access_func;
  back_card   = &bnull_card_func;
  return &null_cards;
}

