/* NVTV nvidia backend -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Backend for nvidia cards, direct access.
 *
 */

#include "local.h" /* before everything else */

#include <stdio.h>
#include <string.h>

#include "xfree.h" /* via back_direct.h */
#include "mmio.h"
#include "bitmask.h"
#include "backend.h"
#include "card_direct.h"
#include "back_direct.h"
#include "back_nvidia.h"
#include "nv_type.h"
#include "tv_nv.h"
#include "tv_i2c.h"
#include "data.h"
#include "xf86PciInfo.h"

/* -------- State of common direct backend driver -------- */

/* The state of the backend consists of:
 *
 * - an NVRec (to be compatible to X)
 * - the current card
 * - the current main, tv and video head
 * - the current chip (in bnv_nv.TvFunc)
 * - the current chip function table (in bnv_data) 
 * - the settings (in bnv_set, bnv_set_avail)
 * - the crt/cip registers before settings (in bnv_regs)
 * - the current and old viewport position (for dualhead)
 * - the current and old videoport position (for dualhead)
 *
 */

static CardPtr bnv_card;

static int bnv_main_head = 1;
static int bnv_tv_head = 1;
static int bnv_video_head = 1;

static NVRec bnv_nv = 
{
  TvChain: NULL,
  TvBusses: {NULL, NULL, NULL},
};

static DataFunc *bnv_data = NULL;

static TVSettings bnv_set;
static TVRegs bnv_regs;

Bool bnv_set_avail = FALSE;
Bool bnv_enc_avail = FALSE;

static int bnv_view_main_x = 0;
static int bnv_view_main_y = 0;
static int bnv_view_tv_x = 0;
static int bnv_view_tv_y = 0;

/* -------- Common backend driver routines -------- */

/*
 *  Update crt regs from hardware
 */

void bnv_updateCrt (int head)
{
  NVGetAllRegs (&bnv_nv, 0, &bnv_regs);
}

/*
 *  Init heads: find current active heads. 
 */

void bnv_initHeads (void)
{
  int devFlags[2];

  DPRINTF ("bnv init heads %i\n", bnv_nv.arch.heads);
  if (bnv_nv.arch.heads == 1) {
    bnv_main_head = bnv_tv_head = bnv_video_head = 1;
    bnv_nv.TvHead = 0;
  } else {
    bnv_main_head = bnv_tv_head = bnv_video_head = 0;
    devFlags[0] = NVGetDevFlags (&bnv_nv, 0);
    devFlags[1] = NVGetDevFlags (&bnv_nv, 1);
    if (devFlags[0] & DEV_TELEVISION) {
      bnv_tv_head = 1; 
    } else
    if (devFlags[1] & DEV_TELEVISION) {
      bnv_tv_head = 2; 
    }
    /* FIXME This will probably fail on a GF2Go if both monitor and FP 
       are enabled */
    if (devFlags[0] & DEV_FLATPANEL) {
      bnv_main_head = 1; 
      if (bnv_tv_head == 0) bnv_tv_head = 2;
    } else
    if (devFlags[1] & DEV_FLATPANEL) {
      bnv_main_head = 2; 
      if (bnv_tv_head == 0) bnv_tv_head = 1;
    }
    if (bnv_main_head == 0) bnv_main_head = 1;
    if (bnv_tv_head == 0) bnv_tv_head = bnv_main_head;
    bnv_video_head = bnv_tv_head;
    NVSetTvHead (&bnv_nv, bnv_tv_head - 1);
    NVSetVideoHead (&bnv_nv, bnv_video_head - 1);
  }
}

/*
 *  Init card: read crt regs, probe chips 
 */

/* FIXME: Maybe call backend funcs with 'this' argument, and have
   a 'virtual' apply func? */

void bnv_initCard (void)
{
  bnv_updateCrt (0);
  bdir_fakeScreen.driverPrivate = &bnv_nv;
  if (bnv_nv.arch.exact == 0x2A) {
    if (!XBoxBusInit (&bdir_fakeScreen)) return;
  } else {
    if (!NVTvBusInit (&bdir_fakeScreen)) return;
  }
  bnv_initHeads ();
  bnv_updateCrt (bnv_tv_head);
  bnv_probeChips ();
}

/*
 *  Map NVidia card memory 
 */

void bnv_mapNvMem (CardPtr card, NVPtr pNv, int heads)
{
  int fd;

  fd = openDevMem (card);
  pNv->riva.PMC     = mapDevMem(card, fd, pNv->IOAddress+0x000000, 0x1000);
  pNv->riva.PFB     = mapDevMem(card, fd, pNv->IOAddress+0x100000, 0x1000);
  pNv->riva.PEXTDEV = mapDevMem(card, fd, pNv->IOAddress+0x101000, 0x1000);
  pNv->riva.PCRTC   = mapDevMem(card, fd, pNv->IOAddress+0x600000, 
				0x2000 * heads);
  pNv->riva.PCIO    = ((U008 *) pNv->riva.PCRTC) + 0x1000;     
  pNv->riva.PVIO    = mapDevMem(card, fd, pNv->IOAddress+0x0C0000, 0x1000);
  pNv->riva.PRAMDAC = mapDevMem(card, fd, pNv->IOAddress+0x680000, 
				0x2000 * heads);
  pNv->riva.PDIO    = ((U008 *) pNv->riva.PRAMDAC) + 0x1000;     
  pNv->riva.PVIDEO  = mapDevMem(card, fd, pNv->IOAddress+0x008000, 0x1000);
  pNv->riva.PTIMER  = mapDevMem(card, fd, pNv->IOAddress+0x009000, 0x1000);
  closeDevMem (card, fd);
}

void bnv_unmapNvMem (CardPtr card, NVPtr pNv, int heads)
{
  unmapDevMem(card, pNv->IOAddress+0x000000, 0x1000);
  unmapDevMem(card, pNv->IOAddress+0x100000, 0x1000);
  unmapDevMem(card, pNv->IOAddress+0x101000, 0x1000);
  unmapDevMem(card, pNv->IOAddress+0x600000, 0x2000 * heads);
  unmapDevMem(card, pNv->IOAddress+0x680000, 0x2000 * heads);
  unmapDevMem(card, pNv->IOAddress+0x008000, 0x1000);
}

void bnv_setNvArch (CardPtr card, NVPtr pNv);

void bnv_initArch (NVPtr pNv)
{
  pNv->arch.boot = MMIO_IN32 (pNv->riva.PMC, 0x000);
  switch (pNv->arch.exact) {
    case 0x03:
      pNv->arch.maxP = 3;
      break;
    case 0x10:
    case 0x1a:
      pNv->arch.maxVclk[0] = 300000; 
      break;
    case 0x11:
      pNv->arch.maxVclk[0] = 300000; 
      pNv->arch.maxVclk[1] = 150000; 
      break;
    case 0x17:
    case 0x20:
    case 0x25:
      pNv->arch.maxVclk[0] = 350000; 
      pNv->arch.maxVclk[1] = 350000; 
      break;
  }
  switch (pNv->arch.exact) {
    case 0x11:
    case 0x17:
    case 0x18:
    case 0x20:
    case 0x25:
      pNv->arch.maxM [0] = 13;
      pNv->arch.freqM[0] = 150000;
      pNv->arch.maxM [1] = 6;
      pNv->arch.freqM[1] = 200000;
      pNv->arch.maxM [2] = 4;
      pNv->arch.freqM[2] = 340000;
      pNv->arch.maxM [3] = 2;
      pNv->arch.freqM[4] = 0;
      break;
    default:
      pNv->arch.maxM [0] = 13;
      pNv->arch.freqM[0] = 250000;
      pNv->arch.maxM [1] = 6;
      pNv->arch.freqM[1] = 340000;
      pNv->arch.maxM [2] = 2;
      pNv->arch.freqM[2] = 0;
      break;
  }
  switch (pNv->arch.exact) {
    case 0x15:
      pNv->arch.minVco = 250000;
      pNv->arch.maxVco = 500000;
      if ((pNv->arch.boot & 0xff) != 0xa1) break;
      /* NV15 Rev A1 falls through */
    case 0x11:
      pNv->arch.minVco = 200000;
      pNv->arch.maxVco = 400000;
      break;
    case 0x20:
      pNv->arch.minVco = 250000;
      pNv->arch.maxVco = 500000;
      break;
    case 0x17:
    case 0x18:
    case 0x25:
      pNv->arch.minVco = 300000;
      pNv->arch.maxVco = 600000;
      break;
  }
  /* FIXME: Should get minVco/maxVco from BIOS */
  pNv->arch.crystalFreq = 
    (GetBit (MMIO_IN32 (pNv->riva.PEXTDEV, 0x000),6)) ? 14318 : 13500;
  /* In Xfree riva_hw.c */
  if (pNv->arch.exact == 0x17 || pNv->arch.exact == 0x18 ||
      pNv->arch.exact == 0x1F || pNv->arch.exact == 0x25 ||
      pNv->arch.exact == 0x28)
  {
    if (GetBit (MMIO_IN32 (pNv->riva.PEXTDEV, 0x000),22)) {
      pNv->arch.crystalFreq = 27000;
    }
  }
  if (pNv->arch.crystalFreq >= 14000) {
    pNv->arch.maxM [0] = 14;
  }
  if (pNv->arch.exact == 0x03) {
    pNv->arch.maxM[0] --;
    pNv->arch.minM = pNv->arch.maxM[0] - 5;
  }
}

/* 
 *  Open card, setup pNv, map mem, call probe
 */

void bnv_openCard (CardPtr card)
{
  NVPtr pNv;
  int i;

  DPRINTF ("bnv open\n");
  bnv_card = card;
  pNv = &bnv_nv;
  pNv->IOAddress = card->reg_base;
  pNv->Chipset = card->pci_id;
  for (i = 0; i < NV_MAXBUS; i++) pNv->TvBusses[i] = NULL;
  pNv->arch.major = 0x00;
  pNv->arch.exact = 0x00;
  pNv->arch.heads = 1;
  /* initialize values "on the safe side" */
  pNv->arch.minVco = 128000;
  pNv->arch.maxVco = 256000; 
  pNv->arch.maxVclk[0] = 256000;
  pNv->arch.maxVclk[1] = 0;
  pNv->arch.maxP = 4;
  pNv->arch.minM = 1;
  pNv->arch.maxM[0] = 12;
  pNv->arch.freqM[0] = 0;
  switch (card->pci_id) {
    case PCI_CHIP_RIVA128:
      pNv->arch.major = 0x03; /* NV3 */
      break;
    case PCI_CHIP_TNT:
    case PCI_CHIP_TNT2:
    case PCI_CHIP_TNT2_A:
    case PCI_CHIP_TNT2_B:
    case PCI_CHIP_UTNT2:
    case PCI_CHIP_VTNT2:
    case PCI_CHIP_UVTNT2:
    case PCI_CHIP_ITNT2:
      pNv->arch.major = 0x04; /* NV4 */
      break;
    default:
      pNv->arch.major = (card->pci_id & 0x0f00) >> 4;
  }
  pNv->arch.exact = pNv->arch.major; /* To be safe */
  switch (card->pci_id) {
      case PCI_CHIP_RIVA128:
	pNv->arch.exact = 0x03; /* NV3 */
	break;
      case PCI_CHIP_TNT:
	pNv->arch.exact = 0x04; /* NV4 */
	break;
      case PCI_CHIP_TNT2:
      case PCI_CHIP_TNT2_A:
      case PCI_CHIP_TNT2_B:
      case PCI_CHIP_UTNT2:
      case PCI_CHIP_VTNT2:
      case PCI_CHIP_UVTNT2:
	pNv->arch.exact = 0x05; /* NV5 */
	break;
      case PCI_CHIP_ITNT2:
	pNv->arch.exact = 0x0A; /* NV0A */
	break;
      case PCI_CHIP_GEFORCE256:
      case PCI_CHIP_GEFORCEDDR:
      case PCI_CHIP_QUADRO:
	pNv->arch.exact = 0x10; /* NV10 */
	break;
      case PCI_CHIP_GEFORCE2MX:
      case PCI_CHIP_GEFORCE2MXDDR:
      case PCI_CHIP_QUADRO2MXR:
      case PCI_CHIP_GEFORCE2GO:
	pNv->arch.exact = 0x11; /* NV11 */
	pNv->arch.heads = 2;
	break;
      case PCI_CHIP_GEFORCE2GTS:
      case PCI_CHIP_GEFORCE2TI:
      case PCI_CHIP_GEFORCE2ULTRA:
      case PCI_CHIP_QUADRO2PRO:
	pNv->arch.exact = 0x15; /* NV15 */
	break;
      case PCI_CHIP_GEFORCE4MX_460:
      case PCI_CHIP_GEFORCE4MX_440:
      case PCI_CHIP_GEFORCE4MX_420:
      case PCI_CHIP_GEFORCE4MX_440SE:
      case PCI_CHIP_GEFORCE4GO_440:
      case PCI_CHIP_GEFORCE4GO_420:
      case PCI_CHIP_GEFORCE4GO_420_32M:
      case PCI_CHIP_GEFORCE4GO_460:
      case PCI_CHIP_QUADRO4XGL_500:
      case PCI_CHIP_GEFORCE4GO_440_64M:
      case PCI_CHIP_QUADRO4NVS:
      case PCI_CHIP_QUADRO4XGL_550:
      case PCI_CHIP_QUADRO4GOGL_500:
      case PCI_CHIP_GEFORCE4GO_410_16M:
	pNv->arch.exact = 0x17; /* NV17 */
	pNv->arch.heads = 2; /* from X */
	break;
      case PCI_CHIP_GEFORCE4MX_440_AGP8X:  
      case PCI_CHIP_GEFORCE4MX_440SE_AGP8X:
      case PCI_CHIP_GEFORCE4MX_420_AGP8X:  
      case PCI_CHIP_QUADRO4_XGL_580: 
      case PCI_CHIP_QUADRO4_NVS_280: 
      case PCI_CHIP_QUADRO4_XGL_380: 
	pNv->arch.exact = 0x18; /* NV18 */
	pNv->arch.heads = 2; /* from X */
	break;
      case PCI_CHIP_GEFORCE4GPU:
	pNv->arch.exact = 0x1f; /* NV1F */
	pNv->arch.heads = 2; /* from X */
	break;
      case PCI_CHIP_GEFORCE2GPU:
	pNv->arch.exact = 0x1A; /* NV1A */
	break;
      case PCI_CHIP_GEFORCE3:
      case PCI_CHIP_GEFORCE3TI_200:
      case PCI_CHIP_GEFORCE3TI_500:
      case PCI_CHIP_QUADRO_DDC:
	pNv->arch.exact = 0x20; /* NV20 */
	break;
      case PCI_CHIP_GEFORCE3_MCPX:
	pNv->arch.exact = 0x2a; /* NV2A */
	pNv->arch.heads = 1; 
	break;
      case PCI_CHIP_GEFORCE4TI_4600:
      case PCI_CHIP_GEFORCE4TI_4400:
      case PCI_CHIP_GEFORCE4TI_4200:
      case PCI_CHIP_QUADRO4XGL_900:
      case PCI_CHIP_QUADRO4XGL_750:
      case PCI_CHIP_QUADRO4XGL_700:
	pNv->arch.exact = 0x25; /* NV25 */
	pNv->arch.heads = 2; /* from X */
	break;
      case PCI_CHIP_GEFORCE4TI_4800:
      case PCI_CHIP_GEFORCE4TI_4200_AGP8X:
      case PCI_CHIP_GEFORCE4TI_4800SE: 
      case PCI_CHIP_GEFORCE4GO_4200:
      case PCI_CHIP_QUADRO4XGL_980: 
      case PCI_CHIP_QUADRO4XGL_780: 
      case PCI_CHIP_QUADRO4GOGL_700:
	pNv->arch.exact = 0x28; /* NV28 */
	pNv->arch.heads = 2; /* from X */
	break;
      default:
        fprintf (stderr, "Unknown pci card %04x\n", pNv->Chipset);
  }
  bnv_mapNvMem (card, &bnv_nv, bnv_nv.arch.heads);
  bnv_setNvArch (card, &bnv_nv);
  bnv_initArch (&bnv_nv);
  bnv_initCard (); /* does probe */
}

/*
 * Close card, unmap memory, free pNv
 */

void bnv_closeCard (void)
{
  NVDestroyBusses (&bnv_nv);
  bnv_unmapNvMem (bnv_card, &bnv_nv, bnv_nv.arch.heads);
}

/* 
 *  Set heads used for tv, monitor, video scaler.
 */


void bnv_setHeads (int main, int tv, int video)
{
  DPRINTF ("bnv_setHeads %i %i %i\n", main, tv, video);
  if (main  > 0) bnv_main_head = main;
  if (tv    > 0) bnv_tv_head = tv;
  if (video > 0) bnv_video_head = video;
  if (bnv_main_head == bnv_tv_head) bnv_video_head = bnv_main_head;
  /* might refuse to set TV head ... */
  bnv_tv_head = NVSetTvHead (&bnv_nv, bnv_tv_head - 1) + 1;
  NVSetVideoHead (&bnv_nv, bnv_video_head - 1);
}

/* 
 *  Get heads used for tv, monitor, video scaler.
 */

void bnv_getHeads (int *main, int *tv, int *video, int *max) 
{
  DPRINTF ("bnv_getHeads\n");
  if (main ) *main  = bnv_main_head;
  if (tv   ) *tv    = bnv_tv_head;
  if (video) *video = bnv_video_head;
  if (max  ) *max   = bnv_nv.arch.heads;
}

void bnv_getHeadDev (int head, int *devFlags) 
{
  DPRINTF ("bnv_getHeadDev\n");
  if (devFlags) *devFlags = NVGetDevFlags (&bnv_nv, head-1);
}

/* 
 *  Probe all chips on card.
 */

void bnv_probeChips (void)
{
  DPRINTF ("bnv_probeChips\n");
  bdir_freeChips (bnv_card);
  NVDestroyDevices (&bnv_nv); 
  NVProbeTvDevices (&bnv_nv);
  bnv_card->chips = bdir_copyChips (bnv_nv.TvChain);
  bnv_setChip (bnv_card->chips, FALSE); /* don't init */
}

/*
 *  Set active chip
 */

void bnv_setChip (ChipPtr chip, Bool init)
{
  DPRINTF ("bnv_setChip %s %i\n", chip ? chip->name : "NULL", init);
  if (!chip) {
    bnv_enc_avail = FALSE;
    return;
  }
  bnv_enc_avail = TRUE;
  NVSetTvDevice (&bnv_nv, (I2CChainPtr) chip->private, init);
  bnv_data = data_func (CARD_NVIDIA, chip->chip);
  bnv_data->defaults (&bnv_set);
  bnv_set_avail = TRUE;
}

void bxbox_setChip (ChipPtr chip, Bool init)
{
  DPRINTF ("bnv_setChip %s %i\n", chip ? chip->name : "NULL", init);
  if (!chip) {
    bnv_enc_avail = FALSE;
    return;
  }
  bnv_enc_avail = TRUE;
  NVSetTvDevice (&bnv_nv, (I2CChainPtr) chip->private, init);
  bnv_data = data_func (CARD_XBOX, chip->chip);
  bnv_data->defaults (&bnv_set);
  bnv_set_avail = TRUE;
}

/*
 *  Apply any changes: process settings, call lower level part
 */

/* FIXME: Maybe call backend funcs with 'this' argument, and have
   a 'virtual' apply func? */

void bnv_apply (void)
{
  TVRegs temp;

  bnv_data->clamp (&bnv_set, &bnv_regs);
  temp = bnv_regs;
  if (bnv_set_avail) {
    bnv_data->setup (&bnv_set, &temp);
  }
  if (bnv_nv.arch.exact == 0x2A) {
    XBoxSetTvMode (&bnv_nv, &temp);
  } else {
    NVSetTvMode (&bnv_nv, &temp);
  }
}

/*
 * change settings, update tv chip registers 
 */

void bnv_setSettings (TVSettings *set)
{
  if (set) {
    bnv_set = *set;
    bnv_set_avail = TRUE;
  } else {
    bnv_set_avail = FALSE;
  }
  bnv_apply ();
}

/*
 *  Get current settings
 */

void bnv_getSettings (TVSettings *set)
{
  if (set && bnv_set_avail) *set = bnv_set;
}

/*
 *  Set mode by crt and tv regs
 */

void bnv_setMode (TVRegs *r)
{
  if (r) bnv_regs = *r;
  if (!bnv_enc_avail) return;
  bnv_apply ();
}
  
/* 
 *  Get current mode. Update crt from hardware.
 */

void bnv_getMode (TVRegs *r)

{
  bnv_updateCrt (bnv_tv_head);
  /* FIXME: Make sure chip is set here */
  if (bnv_regs.devFlags & DEV_TELEVISION) {
    NVGetEncRegsPort (&bnv_nv, &bnv_regs.enc, &bnv_regs.portEnc);
    NVGetPort (&bnv_nv, &bnv_regs.portHost);
  }
  if (r) *r = bnv_regs;
}

/*
 *  Set both mode and settings
 */

void bnv_setModeSettings (TVRegs *r, TVSettings *set)
{
  if (set) {
    bnv_set = *set;
    bnv_set_avail = TRUE;
  } else {
    bnv_set_avail = FALSE;
  }
  if (r) bnv_regs = *r;
  if (!bnv_enc_avail) return;
  bnv_apply ();
}

/*
 *  Enable test image (color bars)
 */

void bnv_setTestImage (TVEncoderRegs *tv, TVSettings *set)
{
  if (set) bnv_set = *set;
  if (tv)  bnv_regs.enc  = *tv;
  if (!bnv_enc_avail) return;
#if 0 /* FIXME */
  if (set) {
    bnv_data->setup (&bnv_set, ...);
  }
  /* FIXME: Settings?? */
#endif
  NVSetTestImage (&bnv_nv, &bnv_regs.enc);
}

long bnv_getStatus (int index)
{
  if (!bnv_enc_avail) return 0;
  return NVGetTvStatus (&bnv_nv, index);
}

TVConnect bnv_getConnection (void)
{
  if (!bnv_enc_avail) return CONNECT_NONE;
  return NVGetTvConnect (&bnv_nv);
}

/*
 *  List all modes (by system)
 */

int bnv_listModes (TVSystem system, TVMode *(list[]))
{
  if (!bnv_enc_avail) return FALSE;
  return bdir_listModes (CARD_NVIDIA, bnv_data, system, list);
}

/*
 *  Find mode by size (and resolution). 
 */

Bool bnv_findBySize (TVSystem system, int xres, int yres, char *size, 
    TVMode *mode)
{
  if (!bnv_enc_avail) return FALSE;
  return bdir_findBySize (CARD_NVIDIA, bnv_data, system, xres, yres, 
			  size, mode);
}

Bool bxbox_findBySize (TVSystem system, int xres, int yres, char *size, 
    TVMode *mode)
{
  if (!bnv_enc_avail) return FALSE;
  return bdir_findBySize (CARD_XBOX, bnv_data, system, xres, yres, 
			  size, mode);
}

/*
 *  Find mode by overscan (and resolution)
 */

Bool bnv_findByOverscan (TVSystem system, int xres, int yres, 
    double hoc, double voc, TVMode *mode)
{
  if (!bnv_enc_avail) return FALSE;
  return bdir_findByOverscan (CARD_NVIDIA, bnv_data, system, xres, yres, 
			      hoc, voc, mode);
}

Bool bxbox_findByOverscan (TVSystem system, int xres, int yres, 
    double hoc, double voc, TVMode *mode)
{
  if (!bnv_enc_avail) return FALSE;
  return bdir_findByOverscan (CARD_XBOX, bnv_data, system, xres, yres, 
			      hoc, voc, mode);
}

/*
 *  Initialize shared view for both heads, and return position of viewport
 *  of second head.
 */

void bnv_initSharedView (int *view_x, int *view_y)
{
  int addr, ofs;

  NVCopyHead (&bnv_nv, bnv_main_head - 1, bnv_tv_head - 1);
  NVGetCrtLayout (&bnv_nv, bnv_main_head - 1, &addr, &ofs);
  bnv_view_main_x = addr % (ofs * 2);
  bnv_view_main_y = addr / (ofs * 2);
  if (bnv_main_head != bnv_tv_head) {
    NVSetCrtLayout (&bnv_nv, bnv_tv_head - 1, addr, ofs);
  }
  /* The assumption here is that the first screen always starts 
     at address 0, and we are in byte mode (K=2 for offset). FIXME? */
  bnv_view_tv_x = addr % (ofs * 2);
  bnv_view_tv_y = addr / (ofs * 2);
  if (view_x) *view_x = bnv_view_tv_x;
  if (view_y) *view_y = bnv_view_tv_y;
}

/*
 *  Check whether TwinView is active by inspecting the vertical interrupt
 *  of the TV head.
 */

Bool bnv_getTwinView (int *view_x, int *view_y)
{
  int addr, ofs;

  if (bnv_tv_head != bnv_main_head &&
      NVVertIntrEnabled (&bnv_nv, bnv_tv_head - 1)) 
  {
    NVGetCrtLayout (&bnv_nv, bnv_tv_head - 1, &addr, &ofs);
    if (view_x) *view_x = addr % (ofs * 2);
    if (view_y) *view_y = addr / (ofs * 2);
    return TRUE;
  } else {
    if (view_x) *view_x = 0;
    if (view_y) *view_y = 0;
    return FALSE;
  }
}

/*
 *  The 'adjust' and 'service' actions are complex, and called as one 
 *  routine to reduce client/server interaction. They adjust the viewport
 *  and video overlay position, while at the same time watching for
 *  external changes.
 *
 *  The 'adjust' action sets new viewport and video overlay position.
 *
 *  The 'service' action sets the cursor position and modifies the
 *  viewport position if necessary.
 *  
 *  Here are the rules:
 *
 *  cursor service         -> change hw cursor position, copy image etc.
 *  cursor out of viewport -> change viewport position
 *  main viewport moves    -> change viewport position
 *  tv viewport moves      -> change viewport position
 *  any viewport moves     -> change video position
 *  video hw moves         -> change video position
 */

Bool bnv_checkViewport (int flags, int *pofs)
{
  int addr, ofs;
  int x, y;
  Bool changed;

  changed = FALSE;
  if (bnv_tv_head != bnv_main_head && (flags & BACK_SERVICE_VIEW_MAIN)) 
  {
    NVGetCrtLayout (&bnv_nv, bnv_main_head - 1, &addr, &ofs);
    x = addr % (ofs * 2);
    y = addr / (ofs * 2);
    DPRINTF ("bnv check main %i,%i\n", x, y);
    if (x != bnv_view_main_x || y != bnv_view_main_y) {
      changed = TRUE;
      bnv_view_tv_x = bnv_view_main_x = x;
      bnv_view_tv_y = bnv_view_main_y = y;
    }
  }
  if (!changed) {
    NVGetCrtLayout (&bnv_nv, bnv_tv_head - 1, &addr, &ofs);
    x = addr % (ofs * 2);
    y = addr / (ofs * 2);
    if (x != bnv_view_tv_x || y != bnv_view_tv_y) {
      changed = TRUE;
      bnv_view_tv_x = x;
      bnv_view_tv_y = y;
    }
  }
  if (pofs) *pofs = ofs;
  return changed;
}

Bool bnv_adjustViewport (int flags, int *view_x, int *view_y)
{
  int addr, ofs;
  Bool changed;

  DPRINTF ("bnv adj view %i,%i\n", *view_x, *view_y);
  changed = bnv_checkViewport (flags, &ofs);
  if (changed) {
    *view_x = bnv_view_tv_x;
    *view_y = bnv_view_tv_y;
  } else {
    bnv_view_tv_x = *view_x;
    bnv_view_tv_y = *view_y;
  }
  addr = bnv_view_tv_x + bnv_view_tv_y * ofs * 2;
  NVSetCrtLayout (&bnv_nv, bnv_tv_head - 1, addr, ofs);
  return changed;
}

Bool bnv_serviceViewportCursor (int flags, int cursor_x, int cursor_y, 
  int *view_x, int *view_y)
{
  int x, y;
  int addr, ofs;
  Bool changed;

  DPRINTF ("bnv service %i,%i\n", cursor_x, cursor_y);
  changed = bnv_checkViewport (flags, &ofs);
  if (flags & BACK_SERVICE_VIEW_CURSOR) {
    if (cursor_x < bnv_view_tv_x) {
      bnv_view_tv_x = cursor_x; changed = TRUE;
    }
    if (cursor_y < bnv_view_tv_y) {
      bnv_view_tv_y = cursor_y; changed = TRUE;
    }
    if (cursor_x > bnv_view_tv_x + bnv_regs.crtc.nv.HDisplay) 
    {
      bnv_view_tv_x = cursor_x - bnv_regs.crtc.nv.HDisplay; changed = TRUE;
    }
    if (cursor_y > bnv_view_tv_y + bnv_regs.crtc.nv.VDisplay) 
    {
      bnv_view_tv_y = cursor_y - bnv_regs.crtc.nv.VDisplay; changed = TRUE;
    }
  }
  if (changed) {
    *view_x = bnv_view_tv_x;
    *view_y = bnv_view_tv_y;
    addr = bnv_view_tv_x + bnv_view_tv_y * ofs * 2;
    NVSetCrtLayout (&bnv_nv, bnv_tv_head - 1, addr, ofs);
  }
  if ((flags & BACK_SERVICE_CURSOR) && bnv_main_head != bnv_tv_head) {
    NVCopyCursor (&bnv_nv, bnv_main_head - 1, bnv_tv_head - 1);
    NVGetCursorPos (&bnv_nv, bnv_tv_head - 1, &x, &y);
    NVSetCursorPos (&bnv_nv, bnv_tv_head - 1, 
		    cursor_x - bnv_view_tv_x, cursor_y - bnv_view_tv_y);
    if (x != cursor_x - bnv_view_tv_x || y != cursor_y - bnv_view_tv_y)
    {
      changed = TRUE;
    }
  }
  return changed;
}

#ifdef DEBUG_PROBE
void bnv_probe_dump (int regs [], int max)
{
  int i;

  for (i = 0x00; i < max; i++) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    printf (" %02X", regs[i]);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
}


void bnv_probe_crt (NVPtr pNv)
{
  int regs [0xa0];
  int last = 0xa0;
  int i;
  int head;
 
  for (head = 0; head < bnv_nv.arch.heads; head++) {
    printf ("  CRT registers %i/%i:\n", head+1, bnv_nv.arch.heads);
    for (i = 0x00; i < last; i++) regs[i] = 0x00;
    writeCrtNv (pNv, head, 0x1f, 0x57); /* unlock extended registers */
    for (i = 0x00; i < last; i++)
      regs [i] = readCrtNv (pNv, head, i);
    bnv_probe_dump (regs, last);
  }
}

void bnv_probe_attr (NVPtr pNv)
{
  int regs [0x60];
  int i;
  int head;
 
  for (head = 0; head < bnv_nv.arch.heads; head++) {
    printf ("  ATTR registers %i/%i:\n", head+1, bnv_nv.arch.heads);
    for (i = 0x00; i < 0x20; i++) regs[i] = 0x00;
    for (i = 0x00; i < 0x20; i++)
      regs [i] = NVReadAttr (pNv, head, i);
    bnv_probe_dump (regs, 0x20);
  }
}

void bnv_probe_seq (NVPtr pNv, int head)
{
  int regs [0x60];
  int i;
 
  printf ("  SEQ registers %i/%i:\n", head+1, bnv_nv.arch.heads);
  for (i = 0x00; i < 0x10; i++) regs[i] = 0x00;
  for (i = 0x00; i < 0x10; i++)
    regs [i] = NVReadSeq (pNv, i);
  bnv_probe_dump (regs, 0x10);
}

void bnv_probe_gr (NVPtr pNv, int head)
{
  int regs [0x60];
  int i;
 
  printf ("  GR registers %i/%i:\n", head+1, bnv_nv.arch.heads);
  for (i = 0x00; i < 0x10; i++) regs[i] = 0x00;
  for (i = 0x00; i < 0x10; i++)
    regs [i] = NVReadGr (pNv, i);
  bnv_probe_dump (regs, 0x10);
}

void bnv_probe_misc (NVPtr pNv, int head)
{
  printf ("  MISC registers %i/%i:\n", head+1, bnv_nv.arch.heads);
  printf ("    input0=%02X vse=%02X feat=%02X misc=%02X\n",
    MMIO_IN8(bnv_nv.riva.PCIO, 0x3c2),
    MMIO_IN8(bnv_nv.riva.PVIO, 0x3c3),
    MMIO_IN8(bnv_nv.riva.PCIO, 0x3ca),
    MMIO_IN8(bnv_nv.riva.PVIO, 0x3cc));
}

#define pdac(x) ((int) MMIO_H_IN32 (bnv_nv.riva.PRAMDAC, head, x))
#define pcrt(x) ((int) MMIO_H_IN32 (bnv_nv.riva.PCRTC, head, x))

void bnv_probeCard (void)
{
  int head;

  printf ("  MC_boot=%08lX FB_boot=%08lX FB_conf=%08lX EXT_boot=%08lX\n", 
    MMIO_IN32 (bnv_nv.riva.PMC, 0x000), 
    MMIO_IN32 (bnv_nv.riva.PFB, 0x000), 
    MMIO_IN32 (bnv_nv.riva.PFB, 0x200), 
    MMIO_IN32 (bnv_nv.riva.PEXTDEV, 0x000));
  for (head = 0; head < bnv_nv.arch.heads; head++) {
    printf ("  PCRTC %i/%i:\n", head+1, bnv_nv.arch.heads);
    printf ("    Assoc     = %08X Config    = %08X\n",
      pcrt(0x860), pcrt(0x804));
    printf ("  PRAMDAC %i/%i:\n", head+1, bnv_nv.arch.heads);
    printf ("    PLL coeff = %08X 0x524     = %08X VPLL      = %08X / %08X\n",
      pdac(0x50c), pdac(0x524), pdac(0x508), pdac(0x520));
    printf ("    Gen Ctrl  = %08X Test Ctrl = %08X TV Setup  = %08X\n",
      pdac(0x600), pdac(0x608), pdac(0x700));
    printf ("    FP Test   = %08X TG Ctrl   = %08X FP Debug  = %08X\n",
      pdac(0x844), pdac(0x848), pdac(0x880));
    printf ("    TV blank  = v= %08X-%08X h= %08X-%08X  %08X \n", 
      pdac(0x704), pdac(0x708), pdac(0x70c), pdac(0x710), pdac(0x714));
    printf ("    TV slave  = v= %08X-%08X h= %08X-%08X  %08X,%08X\n",
      pdac(0x724), pdac(0x728), pdac(0x730), pdac(0x734), 
      pdac(0x720), pdac(0x72C));
    printf ("    TV slave' = %08X\n",
      pdac(0x738));
  }
  writeCrtNv (&bnv_nv, 0, 0x1f, 0x57); /* unlock extended registers */
  bnv_probe_crt (&bnv_nv); /* unlocks crt regs */
  bnv_probe_attr (&bnv_nv); 
  for (head = 0; head < bnv_nv.arch.heads; head++) {
    if (bnv_nv.arch.heads > 1) {
      switch (head) {
	case 0: writeCrtNv (&bnv_nv, 0, 0x44, 0x00); break;
	case 1: writeCrtNv (&bnv_nv, 0, 0x44, 0x03); break;
      }
    }
    bnv_probe_seq (&bnv_nv, head); 
    bnv_probe_gr (&bnv_nv, head); 
    bnv_probe_misc (&bnv_nv, head);
  }
  if (bnv_nv.arch.heads > 1) {
    writeCrtNv (&bnv_nv, 0, 0x44, 0x00); /* FIXME may not be default!! */
  }
  printf ("\n");
}

I2CChainPtr bnv_probeBus (void)
{
  NVDestroyDevices (&bnv_nv); 
  /* probe needs unlocked crt */
  NVUpdateTvState (&bnv_nv);
  TVProbeCreateAll (bnv_nv.TvBusses, bnv_nv.TvMaxBus, &bnv_nv.TvChain); 
  return bnv_nv.TvChain;
}

#endif /* DEBUG_PROBE */

BackCardRec bnv_func = {
  openCard:              bnv_openCard,
  closeCard:             bnv_closeCard,
#ifdef DEBUG_PROBE
  probeCard:             bnv_probeCard,
  probeBus:              bnv_probeBus,
#endif
  setHeads:              bnv_setHeads,
  getHeads:              bnv_getHeads,
  getHeadDev:            bnv_getHeadDev,
  probeChips:            bnv_probeChips,
  setChip:               bnv_setChip,
  setSettings:           bnv_setSettings,
  getSettings:           bnv_getSettings,
  setMode:               bnv_setMode,
  getMode:               bnv_getMode,
  setModeSettings:       bnv_setModeSettings,
  setTestImage:          bnv_setTestImage, 
  getStatus:             bnv_getStatus,    
  getConnection:         bnv_getConnection,
  listModes:		 bnv_listModes,
  findBySize:            bnv_findBySize, 
  findByOverscan:        bnv_findByOverscan,
  initSharedView:        bnv_initSharedView,
  getTwinView:           bnv_getTwinView,
  adjustViewport:        bnv_adjustViewport,
  serviceViewportCursor: bnv_serviceViewportCursor,
};

BackCardRec bxbox_func = {
  openCard:              bnv_openCard,
  closeCard:             bnv_closeCard,
#ifdef DEBUG_PROBE
  probeCard:             bnv_probeCard,
  probeBus:              bnv_probeBus,
#endif
  setHeads:              bnv_setHeads,
  getHeads:              bnv_getHeads,
  getHeadDev:            bnv_getHeadDev,
  probeChips:            bnv_probeChips,
  setChip:               bxbox_setChip,
  setSettings:           bnv_setSettings,
  getSettings:           bnv_getSettings,
  setMode:               bnv_setMode,
  getMode:               bnv_getMode,
  setModeSettings:       bnv_setModeSettings,
  setTestImage:          bnv_setTestImage, 
  getStatus:             bnv_getStatus,    
  getConnection:         bnv_getConnection,
  listModes:		 bnv_listModes,
  findBySize:            bxbox_findBySize, 
  findByOverscan:        bxbox_findByOverscan,
  initSharedView:        bnv_initSharedView,
  getTwinView:           bnv_getTwinView,
  adjustViewport:        bnv_adjustViewport,
  serviceViewportCursor: bnv_serviceViewportCursor,
};

/* -------- Architecture -------- */

char *bnv_architecture (int boot_mask)
{
  /* FIXME -- not for multiple cards! */
  static char s[14];

  if ((boot_mask & 0xf0000000) == 0 && (boot_mask & 0x0f000000) != 0) {
    snprintf (s, sizeof(s), "NV%X Rev %X.%i", GetBF(boot_mask,27:20), 
	      GetBF(boot_mask,7:0), GetBF(boot_mask,19:16));
    return s;
  }
  switch (boot_mask) 
  {
    case 0x00010100: return "NV1 Rev A";     /* NVX */
    case 0x00010101: return "NV1 Rev B";     /* NVX */
    case 0x00010102: return "NV1 Rev B2";    /* NVX */
    case 0x00010103: return "NV1 Rev B3";    /* NVX */
    case 0x00010104: return "NV1 Rev C1";    /* NVX */
    case 0x10020400: return "NV2 Rev A1";    /* NVX */
    case 0x00030100: return "NV3 Rev A1";    /* NVX */
    case 0x00030110: return "NV3 Rev B1";    /* NVX */
    case 0x20030120: return "NV3T Rev A1";   /* NVX */
    case 0x20030121: return "NV3T Rev A2";   /* NVX */
    case 0x20030122: return "NV3T Rev A3-4"; /* NVX */
    case 0x20004000: return "NV4 Rev A1-3";  /* NVX */
    case 0x20034001: return "NV4 Rev A4";    /* NVX */
    case 0x20044001: return "NV4 Rev A5";    /* NVX */
    case 0x20104000: return "NV5/6 Rev A1";  /* NVX */
    case 0x20114000: return "NV5/6 Rev A2";  /* NVX */
    case 0x20124000: return "NV5/6 Rev A3";  /* NVX */
    case 0x20204000: return "NV5/6 Rev B1";  /* NVX */
    case 0x20214000: return "NV5/6 Rev B2";  /* NVX */
    case 0x20224000: return "NV5/6 Rev B3";  /* NVX */
#if 0 /* FIXME duplicate! */
    case 0x20204000: return "NV0A Rev A1";   /* NVX */
    case 0x20214000: return "NV0A Rev A2";   /* NVX */
    case 0x20224000: return "NV0A Rev B1";   /* NVX */
#endif
    default: break;
  }
  snprintf (s, sizeof(s), "NV Id %08X", boot_mask); 
  return s;
}

void bnv_setNvArch (CardPtr card, NVPtr pNv)
{
  card->arch = bnv_architecture (MMIO_IN32 (pNv->riva.PMC, 0x000));
}

/*

Scenarios:

NVIDIA driver
  
NV driver

If there is a TV head already active, choose that.
If there is a FP head active, choose that as main, and the other as TV.
Else choose the first as main.
If no TV head, choose same as first.

Keep all that out of tv_nv.

Q: Only change basic crtc and encoder registers if
* kernel already had TV enabled
* other cases?

Q: What's the difference between an kernel TV enabled head and an nvtv
   enabled one?
A: The vertical interrupt.

 */
