/* NVTV 3dfx backend -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: back_tdfx.c,v 1.5 2002/03/24 17:42:48 dthierbach Exp $
 *
 * Contents:
 *
 * Backend for 3dfx (Voodoo 3) cards, direct access.
 *
 */

#include "local.h" /* before everything else */

#include <stdio.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <fcntl.h>

#include "xfree.h" /* via back_direct.h */

#ifdef DEBUG_PROBE
#include "bitmask.h"
#include "mmio.h"
#endif

#include "backend.h"
#include "card_direct.h"
#include "back_direct.h"
#include "back_null.h"
#include "back_tdfx.h"
#include "tv_tdfx.h"
#include "data.h"

/* -------- State of common direct backend driver -------- */

static CardPtr btdfx_card;

static TDFXRec btdfx_tdfx = 
{
  TvChain: NULL,
  TvBus: NULL, 
};

static DataFunc *btdfx_data = NULL;

static TVSettings btdfx_set;
static TVRegs btdfx_regs;

Bool btdfx_set_avail = FALSE;

/* -------- Common backend driver routines -------- */

/*
 *  Update crt regs from hardware
 */

void btdfx_updateCrt (void)
{
  TDFXGetAllRegs (&btdfx_tdfx, &btdfx_regs);
}

/*
 *  Init card: read crt regs, probe chips 
 */

void btdfx_initCard (void)
{
  btdfx_updateCrt ();
  bdir_fakeScreen.driverPrivate = &btdfx_tdfx;
  if (TDFXTvBusInit (&bdir_fakeScreen)) {
    TDFXTvBusConfig (&btdfx_tdfx);
    btdfx_probeChips ();
  }
}

/*
 *  Map Tdfx card memory 
 */

void btdfx_mapMem (CardPtr card, TDFXPtr pTdfx)
{
  int fd;

  fd = openDevMem (card);
  pTdfx->MMIOBase[0] = mapDevMem(card, fd, card->reg_base, 0x1000);
  pTdfx->PIOBase[0] = card->pio_base;
  pTdfx->IOBase = 0xd0; 
  closeDevMem (card, fd);
}

void btdfx_unmapMem (CardPtr card, TDFXPtr pTdfx)
{
  unmapDevMem(card, (unsigned long) pTdfx->MMIOBase[0], 0x1000);
}

/* 
 *  Open card, map mem, call probe
 */

void btdfx_openCard (CardPtr card)
{
  DPRINTF ("btdfx open\n");
  btdfx_card = card;
  btdfx_mapMem (card, &btdfx_tdfx);
  btdfx_initCard (); /* does probe */
  btdfx_card->arch = "3DFX"; /* FIXME: Use sub vendor? */
}

/*
 * Close card, unmap memory.
 */

void btdfx_closeCard (void)
{
  TDFXDestroyBusses (&btdfx_tdfx);
  btdfx_unmapMem (btdfx_card, &btdfx_tdfx);
}

void btdfx_setHeads (int main, int tv, int video)
{
  DPRINTF ("btdfx_setHeads\n");
}

void btdfx_getHeads (int *main, int *tv, int *video, int *max) 
{
  DPRINTF ("btdfx_getHeads\n");
  if (tv)    *tv = 1;
  if (video) *video = 1;
  if (max)   *max = 1;
}

void btdfx_getHeadDev (int head, int *devFlags) 
{
  DPRINTF ("btdfx_getHeadDev\n");
  if (devFlags) *devFlags = TDFXGetDevFlags (&btdfx_tdfx);
}

void btdfx_probeChips (void)
{
  DPRINTF ("btdfx_probe\n");
  bdir_freeChips (btdfx_card);
  TDFXDestroyDevices (&btdfx_tdfx);
  TDFXProbeTvDevices (&btdfx_tdfx);
  btdfx_card->chips = bdir_copyChips (btdfx_tdfx.TvChain);
  btdfx_setChip (btdfx_card->chips, FALSE); /* don't init */
}

void btdfx_setChip (ChipPtr chip, Bool init)
{
  DPRINTF ("btdfx_setChip %s %i\n", chip ? chip->name : "NULL", init);
  if (!chip) return;
  TDFXSetTvDevice (&btdfx_tdfx, (I2CChainPtr) chip->private, init);
  btdfx_data = data_func (CARD_TDFX, chip->chip);
  btdfx_data->defaults (&btdfx_set);
  btdfx_set_avail = TRUE;
}

/*
 *  Apply any changes: process settings, call lower level part
 */

void btdfx_apply (void)
{
  TVRegs temp;

  btdfx_data->clamp (&btdfx_set, &btdfx_regs);
  temp = btdfx_regs;
  if (btdfx_set_avail) {
    btdfx_data->setup (&btdfx_set, &temp);
  }
#ifdef CONFIG_TDFX_SETUP_BLANK
  {
    register TVTdfxRegs *r = &temp.crtc.tdfx;

    r->tvHBlankEnd = r->tvHBlankStart + r->HScreenSize;
    r->tvVBlankEnd = r->tvVBlankStart + r->VScreenSize;
  }
#endif
  TDFXSetTvMode (&btdfx_tdfx, &temp);
}

/*
 * change settings, update tv chip registers 
 */

void btdfx_setSettings (TVSettings *set)
{
  DPRINTF ("btdfx_setSettings\n");
  if (set) {
    btdfx_set = *set;
    btdfx_set_avail = TRUE;
  } else {
    btdfx_set_avail = FALSE;
  }
  btdfx_apply ();
}

/*
 *  Get current settings
 */

void btdfx_getSettings (TVSettings *set)
{
  DPRINTF ("btdfx_getSettings\n");
  if (set && btdfx_set_avail) *set = btdfx_set;
}

/*
 *  Set mode by crt and tv regs
 */

void btdfx_setMode (TVRegs *r)
{
  DPRINTF ("btdfx_setMode\n");
  if (r) btdfx_regs = *r;
  btdfx_apply ();
}

/* 
 *  Get current mode. Update crt from hardware.
 */

void btdfx_getMode (TVRegs *r)
{
  DPRINTF ("btdfx_getMode\n");
  btdfx_updateCrt ();
  if (r) *r = btdfx_regs;
}

/*
 *  Set both mode and settings
 */

void btdfx_setModeSettings (TVRegs *r, TVSettings *set)
{
  DPRINTF ("btdfx_setModeSettings\n");
  if (set) {
    btdfx_set = *set;
    btdfx_set_avail = TRUE;
  } else {
    btdfx_set_avail = FALSE;
  }
  if (r) btdfx_regs = *r;
  btdfx_apply ();
}

void btdfx_setTestImage (TVEncoderRegs *tv, TVSettings *set)
{
  DPRINTF ("btdfx_setTestImage\n");
  if (set) btdfx_set = *set;
  if (tv)  btdfx_regs.enc  = *tv;
#if 0 /* FIXME */
  if (set) {
    btdfx_data->setup (&btdfx_set, ...);
  }
  /* FIXME: Settings?? */
#endif
  TDFXSetTestImage (&btdfx_tdfx, &btdfx_regs.enc);
}

long btdfx_getStatus (int index)
{
  DPRINTF ("btdfx_getStatus\n");
  return TDFXGetTvStatus (&btdfx_tdfx, index);
}

TVConnect btdfx_getConnection (void)
{
  DPRINTF ("btdfx_getConnection\n");
  return TDFXGetTvConnect (&btdfx_tdfx);
}

/*
 *  List all modes (by system)
 */

int btdfx_listModes (TVSystem system, TVMode *(list[]))
{
  return bdir_listModes (CARD_TDFX, btdfx_data, system, list);
}

/*
 *  Find mode by size (and resolution). 
 */

Bool btdfx_findBySize (TVSystem system, int xres, int yres, char *size, 
    TVMode *mode)
{
  return bdir_findBySize (CARD_TDFX, btdfx_data, system, xres, yres, 
			  size, mode);
}

/*
 *  Find mode by overscan (and resolution)
 */

Bool btdfx_findByOverscan (TVSystem system, int xres, int yres, 
    double hoc, double voc, TVMode *mode)
{
  return bdir_findByOverscan (CARD_TDFX, btdfx_data, system, xres, yres, 
			      hoc, voc, mode);
}

#ifdef DEBUG_PROBE
void btdfx_probe_dump (int regs [], int max)
{
  int i;

  for (i = 0x00; i < max; i++) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    printf (" %02X", regs[i]);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
}


void btdfx_probe_crt (TDFXPtr pTdfx)
{
  int regs [0x60];
  int i;
 
  printf ("  CRT registers\n");
  for (i = 0x00; i < 0x30; i++) regs[i] = 0x00;
  for (i = 0x00; i < 0x30; i++)
    regs [i] = readCrtTdfx (pTdfx, i);
  btdfx_probe_dump (regs, 0x30);
}

#define TDFX_RD32(p,i)    MMIO_IN32(((p)->MMIOBase[0]), (i))

void btdfx_probeCard (void)
{
  register unsigned long val;

  val = TDFX_RD32(&btdfx_tdfx, 0x10);
  printf ("  miscInit0     (0x10)=%08lX tv_blank=%i tv_sync=%i\n", val, 
	  GetBF(val,10:8), GetBF(val,13:11));
  val = TDFX_RD32(&btdfx_tdfx, 0x14);
  printf ("  miscInit1     (0x14)=%08lX rom=%i\n", val, 
	  GetBF(val,4:4));
  val = TDFX_RD32(&btdfx_tdfx, 0x24);
  printf ("  tmuGbeInit    (0x24)=%08lX inv=%i dly=%i\n", val, 
	  GetBF(val,15:15), GetBF(val,19:16));
  val = TDFX_RD32(&btdfx_tdfx, 0x28);
  printf ("  vgaInit0      (0x28)=%08lX ext=%i\n", val, GetBF(val,1:1));
  val = TDFX_RD32(&btdfx_tdfx, 0x2c);
  printf ("  vgaInit1      (0x2c)=%08lX lock=%i%i%i\n", val, 
	  GetBF(val,21:21), GetBF(val,22:22), GetBF(val,23:23));
  val = TDFX_RD32(&btdfx_tdfx, 0x5c);
  printf ("  vidProcCfg    (0x5c)=%08lX en=%i half=%i 2x=%i\n", val, 
	   GetBF(val,0:0), GetBF(val,4:4), GetBF(val,26:26));
  val = TDFX_RD32(&btdfx_tdfx, 0x70);
  printf ("  vidInFormat   (0x70)=%08lX\n", val);
  val = TDFX_RD32(&btdfx_tdfx, 0x78);
  printf ("  vidSerParPort (0x78)=%08lX\n", val);
  val = TDFX_RD32(&btdfx_tdfx, 0x98);
  printf ("  vidScreenSize (0x98)=%08lX w=%i h=%i\n", val, 
	   GetBF(val,11:0), GetBF(val,23:12));
  btdfx_probe_crt (&btdfx_tdfx); /* unlocks crt regs */
  printf ("\n");
}

I2CChainPtr btdfx_probeBus (void)
{
  /* needs unlocked crt */
  TDFXUpdateTvState (&btdfx_tdfx);
  TVProbeCreateAll (&btdfx_tdfx.TvBus, 1, &btdfx_tdfx.TvChain); 
  return btdfx_tdfx.TvChain;
}
#endif

BackCardRec btdfx_func = {
  openCard:              btdfx_openCard,
  closeCard:             btdfx_closeCard,
#ifdef DEBUG_PROBE
  probeCard:             btdfx_probeCard,
  probeBus:              btdfx_probeBus,
#endif
  setHeads:              btdfx_setHeads,
  getHeads:              btdfx_getHeads,
  getHeadDev:            btdfx_getHeadDev,
  probeChips:            btdfx_probeChips,
  setChip:               btdfx_setChip,
  setSettings:           btdfx_setSettings,
  getSettings:           btdfx_getSettings,
  setMode:               btdfx_setMode,
  getMode:               btdfx_getMode,
  setModeSettings:       btdfx_setModeSettings,
  setTestImage:          btdfx_setTestImage, 
  getStatus:             btdfx_getStatus,    
  getConnection:         btdfx_getConnection,
  listModes:		 btdfx_listModes,
  findBySize:            btdfx_findBySize, 
  findByOverscan:        btdfx_findByOverscan,
  initSharedView:        bnull_initSharedView,
  getTwinView:           bnull_getTwinView,
  adjustViewport:        bnull_adjustViewport,
  serviceViewportCursor: bnull_serviceViewportCursor,
};
