/* NVTV direct card access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Access the graphics card directly (via mmaps)
 *
 */

#include "local.h" /* before everything else */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#include <errno.h>
#include <fcntl.h>

#ifdef HAVE_PCI
#include <pci/pci.h>
#endif
#include <sys/mman.h>
#include <sys/ioctl.h>
#ifdef HAVE_SYS_IO_H
#include <sys/io.h>
#endif

#include "backend.h"
#include "card_direct.h"

#include "xfree.h" /* for xf86PciInfo.h */
#define INIT_PCI_VENDOR_INFO
#undef VENDOR_INCLUDE_NONVIDEO
#include "xf86PciInfo.h"

#ifdef HAVE_WINIO
#include "winio.h"
#endif

/* -------- mmap on devices -------- */

/* originally from os-support/linux/lnx_video.c */

int openDevMem (CardPtr card)
{
  int fd;

  if (!card->dev) return -1;
#ifdef HAVE_WINIO
  if (strcmp (card->dev, "winio") == 0) {
    if (winio_init ()) {
      XERROR (winio_error ());
      return -1;
    }
    return 1;
  } else
#endif /* HAVE_WINIO */
  {
#ifndef FAKE_CARD_MMAP
    if ((fd = open(card->dev, O_RDWR)) < 0) {
      fprintf(stderr, "mapDevMem: failed to open %s (%s)\n", card->dev,
		     strerror(errno));
      exit(1);
    }
#else
    fd = -1;
#endif /* FAKE_CARD_MMAP */
    return fd;
  }
}

void closeDevMem (CardPtr card, int fd)
{
  if (!card->dev) return;
#ifdef HAVE_WINIO
  if (strcmp (card->dev, "winio") == 0) {
    if (fd < 0) return;
    winio_exit ();
  } else
#endif /* HAVE_WINIO */
  {
    close (fd);
  }
}

void *mapDevMem (CardPtr card, int fd, unsigned long phys, unsigned long Size)
{
  void* virt;
  int mapflags = MAP_SHARED;

  if (!card->dev) return (void *) phys;
#ifdef HAVE_WINIO
  if (strcmp (card->dev, "winio") == 0) {
    if (fd < 0) return NULL;
    virt = winio_mmap ((caddr_t) phys, Size);
    return virt;
  } else
#endif /* HAVE_WINIO */
  {
#ifdef FAKE_MMIO
    return (void *) (0xff000000 | (phys & 0xffffff));
#else /*  FAKE_MMIO */
#ifndef FAKE_CARD_MMAP

    virt = mmap((caddr_t)0, Size, PROT_READ|PROT_WRITE,
		mapflags, fd, (off_t)phys);
    return virt;
#else /* FAKE_CARD_MMAP */
    return xcalloc (1, Size);
#endif /* FAKE_CARD_MMAP */
#endif /* FAKE_MMIO */
  }
}

void unmapDevMem (CardPtr card, unsigned long phys, unsigned long Size)
{
  unsigned long alignOff;
#ifdef HAVE_WINIO
  int i;
#endif /* HAVE_WINIO */

  if (!card->dev) return;
#ifdef HAVE_WINIO
  if (strcmp (card->dev, "winio") == 0) {
    winio_munmap ((caddr_t) phys, Size);
  } else
#endif /* HAVE_WINIO */
  {
#ifndef FAKE_MMIO
#ifndef FAKE_CARD_MMAP
    alignOff = phys - (phys & ~(getpagesize() - 1));
    munmap((caddr_t)(phys - alignOff), (Size + alignOff));
#endif /* FAKE_CARD_MMAP */
#endif /* FAKE_MMIO */
  }
}

/* -------- Common -------- */

/*
 * Test if card is supported, and create entry for it if it is.
 */
 
void test_card (CardPtr *card_list, char *dev_name, int vendor_id, 
  int device_id, int base0, int base1, int base2, 
  int addr_bus, int addr_slot, int addr_func, char *format, ...)
{
  pciVendorDeviceInfo *vinfo;
  struct pciDevice *dinfo;
  CardPtr card;
  CardType type;
  va_list ap;
  int reg_base;
  int pio_base;
  char s[40];
  
  pio_base = 0;
  switch (vendor_id) 
  {
    case PCI_VENDOR_NVIDIA:
    case PCI_VENDOR_NVIDIA_SGS:
      type = CARD_NVIDIA; 
      if (device_id == PCI_CHIP_GEFORCE3_MCPX) 
      {
	type = CARD_XBOX;
	/* FIXME get PIO base from device w/ id 01B4 (SMBus controller) */
	pio_base = 0xC000;
#ifndef HAVE_WINIO
	iopl (3);
#endif
      }
      reg_base = base0;
      break;
    case PCI_VENDOR_3DFX:
      type = CARD_TDFX; 
      reg_base = base0;
      pio_base = 0x300; /* maybe also base2 */
#ifndef HAVE_WINIO
      ioperm (pio_base + 0xb0, 0x30, 1);
#endif
      break;
    case PCI_VENDOR_INTEL:
      type = CARD_I810; 
      reg_base = base1;
      break;
    default:
      type = CARD_NONE;
      reg_base = 0;
      break;
  }
  if (type != CARD_NONE) {
    va_start (ap, format);
    for (vinfo = xf86PCIVendorInfoData; vinfo->VendorID; vinfo++) 
    {
      if (vinfo->VendorID == vendor_id) 
      {
	for (dinfo = vinfo->Device; dinfo->DeviceID; dinfo++) 
	  if (dinfo->DeviceID == device_id)
	  {
	    card = (CardPtr) malloc (sizeof (CardInfo));
	    card->next = *card_list; *card_list = card;
	    card->name = (char *) malloc (40 * sizeof (char));
	    vsnprintf (s, 38, format, ap); 
	    snprintf (card->name, 38, "%s (%s)", dinfo->DeviceName, s); 
	    card->type = type;
	    card->dev = dev_name;
	    card->reg_base = reg_base;
	    card->pio_base = pio_base;
	    card->pci_id = device_id;
	    card->chips = NULL;
	    card->arch = "";
	    card->addr_bus  = addr_bus;
	    card->addr_slot = addr_slot;
	    card->addr_func = addr_func;
	    break;
	  }
	break;
      }
    }
    va_end (ap);
  }
}

/* -------- Root backend -------- */

#if HAVE_PCI
void scan_cards_pci (CardPtr *card_root_list, char *devname)
{
  struct pci_access *acc;
  struct pci_dev *dev;

  acc = pci_alloc ();
  acc->method = PCI_ACCESS_AUTO;
  pci_init(acc);
  pci_scan_bus (acc);

#ifdef FAKE_CARD
  test_card (card_root_list, "", FAKE_CARD_VENDOR, FAKE_CARD_DEVICE,
	     0, 0, 0, 0, 0, 0, "Fake");
#else /* FAKE_CARD */
  for (dev = acc->devices; dev; dev = dev->next) 
  {
    pci_fill_info (dev, PCI_FILL_IDENT);
    pci_fill_info (dev, PCI_FILL_BASES);
    test_card (card_root_list, devname, dev->vendor_id, dev->device_id, 
	       dev->base_addr[0] & PCI_ADDR_IO_MASK, 
	       dev->base_addr[1] & PCI_ADDR_IO_MASK, 
	       dev->base_addr[2] & PCI_ADDR_IO_MASK,
	       dev->bus, dev->dev, dev->func,
	       "%02x:%02x.%02x", dev->bus, dev->dev, dev->func);
  }
  /* FIXME: pci_cleanup (acc); segfaults ... but avoid memory leak here. */
#endif /* FAKE_CARD */
}
#endif /* HAVE_PCI */

/* -------- nvdev backend -------- */

/* FIXME: Sort out unix/nvdev, unix/root, windows/nvdll, windows/winio
   backends */

#ifdef USE_UNIX_BACKEND

typedef unsigned int NvU32; /* 0 to 4294967295 */
typedef unsigned char NvU8; 

#define NV_MAX_DEVICES 8 /* was 4 */
#define NV_IOCTL_CARD_INFO_FLAG_PRESENT       0x0001
#define NV_IOCTL_CARD_INFO_FLAG_NEED_MSYNC    0x0002

#define NV_MMAP_REG_OFFSET              (0)
#define NV_MMAP_FB_OFFSET                (256 * 1024 * 1024)
#define NV_MMAP_ALLOCATION_OFFSET (1 * 1024 * 1024 * 1024)
#define NV_MMAP_AGP_OFFSET ((unsigned long)(2)*(1024 * 1024 * 1024))

#define NVIDIA_IOCTL_MAGIC      'F'
#if 0
#define NVIDIA_IOCTL_CARD_INFO _IOWR(NVIDIA_IOCTL_MAGIC, 2, sizeof(void *))
#endif

#define NVIDIA_IOCTL_CARD_INFO     \
  _IOWR(NVIDIA_IOCTL_MAGIC, 2, NvU8[NV_MAX_DEVICES*sizeof(nv_new_ci)])

static char *dev_nvcard [NV_MAX_DEVICES] = {
  "/dev/nvidia0", "/dev/nvidia1", "/dev/nvidia2", "/dev/nvidia3"
  "/dev/nvidia4", "/dev/nvidia5", "/dev/nvidia6", "/dev/nvidia7"
};

/* 1.0-2314 and older */
typedef struct 
{
  int flags;               /* see below                   */
  int bus;		   /* bus number (PCI, AGP, etc)  */
  int slot;                /* card slot                   */
  int vendor_id;           /* PCI vendor id               */
  int device_id;
  int interrupt_line;
} nv_old_ci;

/* 1.0-2802 */
typedef struct 
{
    int    flags;               /* see below                   */
    int    bus;			/* bus number (PCI, AGP, etc)  */
    int    slot;                /* card slot                   */
    int    vendor_id;           /* PCI vendor id               */
    int    device_id;
    int    interrupt_line;
    unsigned int reg_address;   /* register aperture           */
    unsigned int reg_size;
    unsigned int fb_address;    /* framebuffer aperture        */
    unsigned int fb_size;
} nv_2802_ci;

/* 1.0-2880, 1.0-3123, 1.0-4191, 1.0-4349 */
typedef struct
{
    NvU32    flags;               /* see below                   */
    NvU32    bus;		  /* bus number (PCI, AGP, etc)  */
    NvU32    slot;                /* card slot                   */
    NvU32    vendor_id;           /* PCI vendor id               */
    NvU32    device_id;
    NvU32    interrupt_line;
    NvU32    reg_address;         /* register aperture           */
    NvU32    reg_size;
    NvU32    fb_address;          /* framebuffer aperture        */
    NvU32    fb_size;
} nv_new_ci;

/* 1.0-2960 */
typedef struct {
    int    flags;               /* see below                   */
    int    bus;			/* bus number (PCI, AGP, etc)  */
    int    slot;                /* card slot                   */
    int    vendor_id;           /* PCI vendor id               */
    int    device_id;
    int    interrupt_line;
    unsigned int reg_address;   /* register aperture           */
    unsigned int reg_size;
    unsigned int fb_address;    /* framebuffer aperture        */
    unsigned int fb_size;
} nv_2960_ci;

typedef struct 
{
    NvU32 magic;
    NvU32 version;
    NvU32 major;
    NvU32 minor;
    NvU32 patch;
} nv_api_version;

#define NVIDIA_IOCTL_VERSION _IOWR(NVIDIA_IOCTL_MAGIC, 6, nv_api_version)

#define NVIDIA_VERSION_MAGIC_REQ  0x9797fade
#define NVIDIA_VERSION_MAGIC_REP  0xbead2929
#define NVIDIA_VERSION 1

#define MAX_VER 4349

void scan_cards_nvdev (CardPtr *card_nvdev_list)
{
  int fd, error;
  nv_old_ci ci_old [NV_MAX_DEVICES];
  nv_2802_ci ci_2802 [NV_MAX_DEVICES];
  nv_2960_ci ci_2960 [NV_MAX_DEVICES];
  nv_new_ci ci_new [NV_MAX_DEVICES];
  nv_api_version ver_api;
  int ver;
  int ver_major, ver_minor, ver_patch;
  FILE *f;
  int i;

  /* FIXME: Must check /proc/driver/nvidia/version
     which version, use correct info struct */
  if ((f = fopen ("/proc/driver/nvidia/version", "r"))) {
    ver = -1;
    /* poor man's parsing ... */
    while (!feof (f)) {
      if (fgetc (f) != 'K') continue;
      if (fgetc (f) != 'e') continue;
      if (fgetc (f) != 'r') continue;
      if (fgetc (f) != 'n') continue;
      if (fgetc (f) != 'e') continue;
      if (fgetc (f) != 'l') continue;
      if (fgetc (f) != ' ') continue;
      if (fgetc (f) != 'M') continue;
      if (fgetc (f) != 'o') continue;
      if (fgetc (f) != 'd') continue;
      if (fgetc (f) != 'u') continue;
      if (fgetc (f) != 'l') continue;
      if (fgetc (f) != 'e') continue;
      if (fscanf (f, " %i.%i-%i", &ver_major, &ver_minor, &ver_patch) == 3) {
	fprintf (stderr, "Found NVidia Kernel module version %i.%i-%i\n",
		 ver_major, ver_minor, ver_patch);
	ver = ver_patch;
	break;
      }
    }
    if (ver == -1) {
      fprintf (stderr, "Warning: Cannot determine NVidia Kernel module version.\n");
      fprintf (stderr, "  If NVidia has changed the data structures in the kernel module, nvtv might\n");
      fprintf (stderr, "  fail or even core dump. In this case, run nvtv as root.\n");
      ver = MAX_VER;
    }
    if (ver > MAX_VER) {
      fprintf (stderr, "Warning: The NVidia kernel module is newer than the versions nvtv have been\n");
      fprintf (stderr, "  tested with. If NVidia has changed the data structures in the kernel\n");
      fprintf (stderr, "  module, nvtv might fail, or even core dump. In this case, run nvtv as root.\n");
    }
    fclose (f);
  } else {
    ver = 0;
  }
  fd = open(DEV_NVCTL, O_RDWR, 0);
  if (fd < 0) { *card_nvdev_list = NULL; return; }
  switch (ver) 
  {
    case 0: /* 1.0-2314 and older */
      error = ioctl (fd, NVIDIA_IOCTL_CARD_INFO, ci_old);
      if (error < 0) { close(fd); *card_nvdev_list = NULL; return; }
      for (i = 0; i < NV_MAX_DEVICES; i++) {
	if ((ci_old[i].flags & NV_IOCTL_CARD_INFO_FLAG_PRESENT) == 0) {
	  continue;
	}
	test_card (card_nvdev_list, dev_nvcard[i], ci_old[i].vendor_id, 
		   ci_old[i].device_id, NV_MMAP_REG_OFFSET, 0, 0,
		   ci_old[i].bus, ci_old[i].slot, 0,
		   "%02x:%02x", ci_old[i].bus, ci_old[i].slot); 
      }
      break;
    case 2802:
      error = ioctl (fd, NVIDIA_IOCTL_CARD_INFO, ci_2802);
      if (error < 0) { close(fd); *card_nvdev_list = NULL; return; }
      for (i = 0; i < NV_MAX_DEVICES; i++) {
	if ((ci_2802[i].flags & NV_IOCTL_CARD_INFO_FLAG_PRESENT) == 0) {
	  continue;
	}
	test_card (card_nvdev_list, dev_nvcard[i], ci_2802[i].vendor_id, 
		   ci_2802[i].device_id, NV_MMAP_REG_OFFSET, 0, 0,
		   ci_2802[i].bus, ci_2802[i].slot, 0,
		   "%02x:%02x", ci_2802[i].bus, ci_2802[i].slot); 
      }
      break;
    case 2960:
      error = ioctl (fd, NVIDIA_IOCTL_CARD_INFO, ci_2960);
      if (error < 0) { close(fd); *card_nvdev_list = NULL; return; }
      for (i = 0; i < NV_MAX_DEVICES; i++) {
	if ((ci_2960[i].flags & NV_IOCTL_CARD_INFO_FLAG_PRESENT) == 0) {
	  continue;
	}
	test_card (card_nvdev_list, dev_nvcard[i], ci_2960[i].vendor_id, 
		   ci_2960[i].device_id, NV_MMAP_REG_OFFSET, 0, 0,
		   ci_2960[i].bus, ci_2960[i].slot, 0,
		   "%02x:%02x", ci_2960[i].bus, ci_2960[i].slot); 
      }
      break;
    case 2880:
    case 3123:
      error = ioctl (fd, NVIDIA_IOCTL_CARD_INFO, ci_new);
      if (error < 0) { close(fd); *card_nvdev_list = NULL; return; }
      for (i = 0; i < NV_MAX_DEVICES; i++) {
	if ((ci_new[i].flags & NV_IOCTL_CARD_INFO_FLAG_PRESENT) == 0) {
	  continue;
	}
	test_card (card_nvdev_list, dev_nvcard[i], ci_new[i].vendor_id, 
		   ci_new[i].device_id, NV_MMAP_REG_OFFSET, 0, 0,
		   ci_new[i].bus, ci_new[i].slot, 0,
		   "%02x:%02x", ci_new[i].bus, ci_new[i].slot); 
      }
      break;
    case 4191: 
    case 4349:
    default:
      error = ioctl (fd, NVIDIA_IOCTL_VERSION, &ver_api);
      if (error < 0) { close(fd); *card_nvdev_list = NULL; return; }
      if (ver_api.version != 1 || ver_api.major != 1 || ver_api.minor != 0 ||
	  ver_api.patch != ver) 
      {
	fprintf (stderr, "Version control mismatch (found %i / %i.%i-%i)\n", 
		ver_api.version, ver_api.major, ver_api.minor, ver_api.patch);
	close(fd); *card_nvdev_list = NULL; return;
      }
      ver_api.magic = NVIDIA_VERSION_MAGIC_REQ;
      *((nv_api_version *) &ci_new) = ver_api;
      error = ioctl (fd, NVIDIA_IOCTL_CARD_INFO, ci_new);
      if (error < 0) { close(fd); *card_nvdev_list = NULL; return; }
      for (i = 0; i < NV_MAX_DEVICES; i++) {
	if ((ci_new[i].flags & NV_IOCTL_CARD_INFO_FLAG_PRESENT) == 0) {
	  continue;
	}
	test_card (card_nvdev_list, dev_nvcard[i], ci_new[i].vendor_id, 
		   ci_new[i].device_id, ci_new[i].reg_address, 
		   ci_new[i].fb_address, 0, ci_new[i].bus, ci_new[i].slot, 0,
		   "%02x:%02x", ci_new[i].bus, ci_new[i].slot); 
      }
      break;
  }
}

#endif /* USE_UNIX_BACKEND */

