/* NVTV Chrontel chip data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Data tables and setup routines for the Chrontel chip.
 */ 

#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include "nv_type.h"
#include "data_ch.h"

/* -------- CH -------- NTSC -------- */

/* FSCI: NTSC, NTSC-NDC, PAL-M, PAL-60 */

/* PAL-60 is just scaled up PAL-M */

/* -------- 640x480 -------- */

TVChRegs ch_ntsc_large_a = { /* Mode 16: 640x480 1:1  */
  dmr_ir  : 3,
  dmr_vs  : 1,
  dmr_sr  : 1,
  sav     : 154,  /* tvcc: 92  */
  hpr     : 26,   /* tvcc: 31  */
  vpr     : 0,    /* tvcc: 260 */
  pll_m   : 63,
  pll_n   : 110,
  pllcap  : 0,
  dacg    : 0,
  mode    : 16,
};

unsigned long ch_fsci_ntsc_large_a [] = 
  {623153737, 623156346, 622468953, 770448543};

TVChRegs ch_ntsc_small_a = { /* Mode 17: 640x480 7:8  */
  dmr_ir  : 3,
  dmr_vs  : 1,
  dmr_sr  : 2,
  sav     : 160,  /* tvcc: 92  */
  hpr     : 46,   /* tvcc: 47  */
  vpr     : 0,    /* tvcc: 245 */
  pll_m   : 63,
  pll_n   : 126,
  pllcap  : 0,
  dacg    : 0,
  mode    : 17,
};

unsigned long ch_fsci_ntsc_small_a [] = 
  {545259520, 545261803, 544660334, 674142475};

TVChRegs ch_ntsc_tiny_a = { /* Mode 18: 640x480 5:6  */
  dmr_ir  : 3,
  dmr_vs  : 1,
  dmr_sr  : 3,
  sav     : 170,
  hpr     : 54,
  vpr     : 0, 
  pll_m   : 89,
  pll_n   : 190,
  pllcap  : 0,
  dacg    : 0,
  mode    : 18,
};

unsigned long ch_fsci_ntsc_tiny_a [] = 
  {508908885, 508911016, 508349645, 629199644};

/* -------- 800x600 -------- */

TVChRegs ch_ntsc_huge_b = { /* Mode 22: 800x600 5:6  */
  dmr_ir  : 4,
  dmr_vs  : 1,
  dmr_sr  : 3,
  sav     : 234,
  hpr     : 32,
  vpr     : 258, 
  pll_m   : 33,
  pll_n   : 94,
  pllcap  : 1,
  dacg    : 0,
  mode    : 22,
};

unsigned long ch_fsci_ntsc_huge_b [] = 
  {521957831, 521960016, 521384251, 645332967};

TVChRegs ch_ntsc_large_b = { /* Mode 23: 800x600 3:4  */
  dmr_ir  : 4,
  dmr_vs  : 1,
  dmr_sr  : 4,
  sav     : 208,  /* tvcc: 140 */
  hpr     : 47,   /* tvcc: 47  */
  vpr     : 0,    /* tvcc: 266 */
  pll_m   : 19,
  pll_n   : 62,
  pllcap  : 1,
  dacg    : 0,
  mode    : 23,
};

unsigned long ch_fsci_ntsc_large_b [] = 
  {469762048, 469764015, 469245826, 580799671};

TVChRegs ch_ntsc_small_b = { /* Mode 24: 800x600 7:10  */
  dmr_ir  : 4,
  dmr_vs  : 1,
  dmr_sr  : 5,
  sav     : 216,  /* tvcc: 140 */
  hpr     : 60,   /* tvcc: 61  */
  vpr     : 0,    /* tvcc: 257 */
  pll_m   : 89,
  pll_n   : 302,
  pllcap  : 0,
  dacg    : 0,
  mode    : 24,
};

unsigned long ch_fsci_ntsc_small_b [] = 
  {428554851, 428556645, 428083911, 529852331};

/* -------- 640x400 -------- */

TVChRegs ch_ntsc_large_c = { /* Mode 10: 640x400 5:4  */
  dmr_ir  : 2,
  dmr_vs  : 1,
  dmr_sr  : 0,
  sav     : 150,
  hpr     : 90,
  vpr     : 0, 
  pll_m   : 63,
  pll_n   : 94,
  pllcap  : 1,
  dacg    : 0,
  mode    : 10,
};

unsigned long ch_fsci_ntsc_large_c [] = 
  {646233505, 646236211, 645523358, 798983674};

TVChRegs ch_ntsc_small_c = { /* Mode 11: 640x400 1:1  */
  dmr_ir  : 2,
  dmr_vs  : 1,
  dmr_sr  : 1,
  sav     : 194,
  hpr     : 53,
  vpr     : 0, 
  pll_m   : 11,
  pll_n   : 22,
  pllcap  : 1,
  dacg    : 0,
  mode    : 11,
};

unsigned long ch_fsci_ntsc_small_c [] = 
  {516986804, 516988968, 516418687, 639186940};

TVChRegs ch_ntsc_tiny_c = { /* Mode 12: 640x400 7:8  */
  dmr_ir  : 2,
  dmr_vs  : 1,
  dmr_sr  : 3,
  sav     : 194,
  hpr     : 68,
  vpr     : 0, 
  pll_m   : 89,
  pll_n   : 190,
  pllcap  : 0,
  dacg    : 0,
  mode    : 12,
};

unsigned long ch_fsci_ntsc_tiny_c [] = 
  {452363454, 452365347, 451866351, 559288572};

/* -------- 720x400 -------- */

TVChRegs ch_ntsc_large_d = { /* Mode 6: 720x400 5:4  */
  dmr_ir  : 1,
  dmr_vs  : 1,
  dmr_sr  : 0,
  sav     : 210,
  hpr     : 26,
  vpr     : 256, 
  pll_m   : 63,
  pll_n   : 106,
  pllcap  : 1,
  dacg    : 0,
  mode    : 6,
};

unsigned long ch_fsci_ntsc_large_d [] = 
  {574429782, 574432187, 573798541, 710207711};

TVChRegs ch_ntsc_small_d = { /* Mode 7: 720x400 1:1  */
  dmr_ir  : 1,
  dmr_vs  : 1,
  dmr_sr  : 1,
  sav     : 208,
  hpr     : 59,
  vpr     : 260, 
  pll_m   : 33,
  pll_n   : 70,
  pllcap  : 1,
  dacg    : 0,
  mode    : 7,
};

unsigned long ch_fsci_ntsc_small_d [] = 
  {463962517, 463964459, 463452668, 573629305};

/* -------- 512x384 -------- */

TVChRegs ch_ntsc_large_e = { /* Mode 2: 512x384 5:4  */
  dmr_ir  : 0,
  dmr_vs  : 1,
  dmr_sr  : 0,
  sav     : 240,
  hpr     : 27,
  vpr     : 256, 
  pll_m   : 89,
  pll_n   : 126,
  pllcap  : 1,
  dacg    : 0,
  mode    : 2,
};

unsigned long ch_fsci_ntsc_large_e [] = 
  {763363328, 763366524, 762524467, 943799465};

TVChRegs ch_ntsc_small_e = { /* Mode 3: 512x384 1:1  */
  dmr_ir  : 0,
  dmr_vs  : 1,
  dmr_sr  : 1,
  sav     : 226,
  hpr     : 47,
  vpr     : 0, 
  pll_m   : 63,
  pll_n   : 110,
  pllcap  : 0,
  dacg    : 0,
  mode    : 3,
};

unsigned long ch_fsci_ntsc_small_e [] = 
  {623153737, 623156346, 622468953, 770448543};

/* -------- DVD convenience modes -------- */

/* For use with video scaler, 768x576 for 4:3, 800x450 for 16:9,
   if DVD is in 720x576. For 720x480, use 640x480 modes.

   Based on 800x600 NTSC Large (Mode 23)
*/

TVChRegs ch_ntsc_large_f = { /* Mode 23: 800x600 3:4  */
  dmr_ir  : 4,
  dmr_vs  : 1,
  dmr_sr  : 4,
  sav     : 208,
  hpr     : 43,
  vpr     : 260,
  pll_m   : 19,
  pll_n   : 62,
  pllcap  : 1,
  dacg    : 0,
  mode    : 23,
};

TVChRegs ch_ntsc_large_g = { /* Mode 23: 800x600 3:4  */
  dmr_ir  : 4,
  dmr_vs  : 1,
  dmr_sr  : 4,
  sav     : 208,
  hpr     : 45,
  vpr     : 262,
  pll_m   : 19,
  pll_n   : 62,
  pllcap  : 1,
  dacg    : 0,
  mode    : 23,
};

/* Use ch_fsci_ntsc_large_b */

/* -------- 720x{576,480} DVD -------- */

/* Note: Not used yet, crt values not tested */

TVChRegs ch_ntsc_interl_dvd = { /* Mode 26: 720x480 IL  */
  dmr_ir  : 5,
  dmr_vs  : 1,
  dmr_sr  : 1,
  sav     : 142,
  hpr     : 60,
  vpr     : 0,
  pll_m   : 33,
  pll_n   : 31,
  pllcap  : 1,
  dacg    : 0,
  fsci    : 569408543, /* NDC: 569410927 */
  mode    : 26,
};

/* Note: Mode 26 must use 2X XClk. So need to program that. For
   CRT, need to use double hscan, and interlace. */

/* -------- CH -------- PAL -------- */

/* FSCI: PAL, PAL-N, PAL-X */

/* -------- 640x480 -------- */

TVChRegs ch_pal_large_a = { /* Mode 13: 640x480 5:4  */
  dmr_ir  : 3,
  dmr_vs  : 0,
  dmr_sr  : 0,
  sav     : 142, 
  hpr     : 37, 
  vpr     : 308, 
  pll_m   : 13,
  pll_n   : 20,
  pllcap  : 1,
  dacg    : 1,
  mode    : 13,
};

unsigned long ch_fsci_pal_large_a [] = {806021060, 651209077, 806120400};

TVChRegs ch_pal_small_a = { /* Mode 14: 640x480 1:1  */
  dmr_ir  : 3,
  dmr_vs  : 0,
  dmr_sr  : 1,
  sav     : 110, /* tvcc: 100 */
  hpr     : 52,  /* tvcc: 53  */
  vpr     : 289, /* tvcc: 291 */
  pll_m   : 4,
  pll_n   : 9,
  pllcap  : 1,
  dacg    : 1,
  mode    : 14,
};

unsigned long ch_fsci_pal_small_a [] = {644816848, 520967262, 644862000};

TVChRegs ch_pal_tiny_a = { /* Mode 15: 640x480 5:6  */
  dmr_ir  : 3,
  dmr_vs  : 0,
  dmr_sr  : 3,
  sav     : 110, 
  hpr     : 78,  
  vpr     : 4, 
  pll_m   : 3,
  pll_n   : 9,
  pllcap  : 1,
  dacg    : 1,
  mode    : 15,
};

unsigned long ch_fsci_pal_tiny_a [] = {537347373, 434139385, 537430027};

/* -------- 800x600 -------- */

TVChRegs ch_pal_large_b = { /* Mode 19: 800x600 1:1  */
  dmr_ir  : 4,
  dmr_vs  : 0,
  dmr_sr  : 1,
  sav     : 176, /* tvcc: 122 */
  hpr     : 31,  /* tvcc: 33 */
  vpr     : 0,   /* tvcc: 308 */
  pll_m   : 313,
  pll_n   : 647,
  pllcap  : 0,
  dacg    : 1,
  mode    : 19,
};

unsigned long ch_fsci_pal_large_b [] = {645499916, 521519134, 645527100};

TVChRegs ch_pal_small_b = { /* Mode 20: 800x600 5:6  */
  dmr_ir  : 4,
  dmr_vs  : 0,
  dmr_sr  : 3,
  sav     : 166, /* tvcc: 124 */
  hpr     : 60,  /* tvcc: 73 */
  vpr     : 0,   /* tvcc: 296 */
  pll_m   : 33,
  pll_n   : 86,
  pllcap  : 1,
  dacg    : 1,
  mode    : 20,
};

unsigned long ch_fsci_pal_small_b [] = {528951320, 427355957, 529002000};

TVChRegs ch_pal_tiny_b = { /* Mode 21: 800x600 3:4  */
  dmr_ir  : 4,
  dmr_vs  : 0,
  dmr_sr  : 4,
  sav     : 90,
  hpr     : 70,
  vpr     : 0, 
  pll_m   : 103,
  pll_n   : 284,
  pllcap  : 0,
  dacg    : 1,
  mode    : 21,
};

unsigned long ch_fsci_pal_tiny_b [] = {488262757, 394482422, 488330000};

/* Note: Mode 21 has 627 PAL lines instead of 625 */

/* -------- 640x400 -------- */

TVChRegs ch_pal_small_c = { /* Mode 8: 640x400 5:4  */
  dmr_ir  : 2,
  dmr_vs  : 0,
  dmr_sr  : 0,
  sav     : 226, 
  hpr     : 47,  
  vpr     : 0,   
  pll_m   : 61,
  pll_n   : 108,
  pllcap  : 0,
  dacg    : 1,
  mode    : 8,
};

unsigned long ch_fsci_pal_small_c [] = {677057690, 547015625, 677157690};

TVChRegs ch_pal_tiny_c = { /* Mode 9: 640x400 1:1  */
  dmr_ir  : 2,
  dmr_vs  : 0,
  dmr_sr  : 1,
  sav     : 230,
  hpr     : 80,
  vpr     : 0, 
  pll_m   : 3,
  pll_n   : 9,
  pllcap  : 1,
  dacg    : 1,
  mode    : 9,
};

unsigned long ch_fsci_pal_tiny_c [] = {537347373, 434139385, 537447373};

/* -------- 720x400 -------- */

TVChRegs ch_pal_small_d = { /* Mode 4: 720x400 5:4  */
  dmr_ir  : 1,
  dmr_vs  : 0,
  dmr_sr  : 0,
  sav     : 346, 
  hpr     : 53,  
  vpr     : 0,   
  pll_m   : 26,
  pll_n   : 53,
  pllcap  : 1,
  dacg    : 1,
  mode    : 4,
};

unsigned long ch_fsci_pal_small_d [] = {601829058, 486236111, 601879058};

TVChRegs ch_pal_tiny_d = { /* Mode 5: 720x400 1:1  */
  dmr_ir  : 1,
  dmr_vs  : 0,
  dmr_sr  : 1,
  sav     : 322,
  hpr     : 86,
  vpr     : 0, 
  pll_m   : 138,
  pll_n   : 339,
  pllcap  : 0,
  dacg    : 1,
  mode    : 5,
};

unsigned long ch_fsci_pal_tiny_d [] = {485346014, 392125896, 485446014};

/* -------- 512x384 -------- */

TVChRegs ch_pal_small_e = { /* Mode 0: 512x384 5:4  */
  dmr_ir  : 0,
  dmr_vs  : 0,
  dmr_sr  : 0,
  sav     : 256,
  hpr     : 43,
  vpr     : 0,
  pll_m   : 13,
  pll_n   : 20,
  pllcap  : 1,
  dacg    : 1, 
  mode    : 0,
};

unsigned long ch_fsci_pal_small_e [] = {806021060, 651209077, 806121060};

TVChRegs ch_pal_tiny_e = { /* Mode 1: 512x384 1:1  */
  dmr_ir  : 0,
  dmr_vs  : 0,
  dmr_sr  : 1,
  sav     : 266,
  hpr     : 69,
  vpr     : 0,
  pll_m   : 4,
  pll_n   : 9,
  pllcap  : 1,
  dacg    : 1,
  mode    : 1,
};

unsigned long ch_fsci_pal_tiny_e [] = {644816848, 520967262, 644916848};

/* -------- DVD convenience modes -------- */

/* For use with video scaler, 768x576 for 4:3, 800x450 for 16:9,
   if DVD is in 720x576. For 720x480, use 640x480 modes.

   Based on 800x600 PAL Large (Mode 19) and Small (Mode 20), as
   the large mode is buggy (color flickers).
*/

TVChRegs ch_pal_large_f = { /* Mode 19: 800x600 1:1  */
  dmr_ir  : 4,
  dmr_vs  : 0,
  dmr_sr  : 1,
  sav     : 176,
  hpr     : 34,
  vpr     : 311,
  pll_m   : 313,
  pll_n   : 647,
  pllcap  : 0,
  dacg    : 1,
  mode    : 19,
};

TVChRegs ch_pal_large_g = { /* Mode 19: 800x600 1:1  */
  dmr_ir  : 4,
  dmr_vs  : 0,
  dmr_sr  : 1,
  sav     : 176,
  hpr     : 31,
  vpr     : 312,
  pll_m   : 313,
  pll_n   : 647,
  pllcap  : 0,
  dacg    : 1,
  mode    : 19,
};

/* Use ch_fsci_pal_large_b */

TVChRegs ch_pal_small_f = { /* Mode 20: 800x600 5:6  */
  dmr_ir  : 4,
  dmr_vs  : 0,
  dmr_sr  : 3,
  sav     : 210, 
  hpr     : 68,  
  vpr     : 312,
  pll_m   : 33,
  pll_n   : 86,
  pllcap  : 1,
  dacg    : 1,
  mode    : 20,
};

TVChRegs ch_pal_small_g = { /* Mode 20: 800x600 5:6  */
  dmr_ir  : 4,
  dmr_vs  : 0,
  dmr_sr  : 3,
  sav     : 188, 
  hpr     : 64,  
  vpr     : 312,
  pll_m   : 33,
  pll_n   : 86,
  pllcap  : 1,
  dacg    : 1,
  mode    : 20,
};

/* Use ch_fsci_pal_small_b */

/* -------- 720x{576,480} DVD -------- */

TVChRegs ch_pal_interl_dvd = { /* Mode 25: 720x576 IL  */
  dmr_ir  : 5,
  dmr_vs  : 0,
  dmr_sr  : 1,
  sav     : 142,
  hpr     : 60,
  vpr     : 0,
  pll_m   : 33,
  pll_n   : 31,
  pllcap  : 1,
  dacg    : 1,
  fsci    : 705268427,
  mode    : 25,
};

/* Note: Mode 25 must use 2X XClk. So need to program that. For
   CRT, need to use double hscan, and interlace. */

/* -------- -------- */

static int clamp_contrast[8]  = {-75, -50, -25, 0, 25, 50, 75, 100};
static int clamp_flicker[5]   = {0, 25, 50, 75, 100};
static int clamp_sharpness[3] = {0, 50, 100};
static int clamp_luma_bandwidth[5]   = {0, 25, 50, 75, 100};
static int clamp_chroma_bandwidth[4] = {0, 50, 75, 100};

void data_init_ch (TVSystem system, TVChRegs *r) 
{
  if (!r) return;
  r->ffr_fc = 2;
  r->ffr_fy = 1; 
  r->ffr_ft = 2; 
  r->vbw_flff = 0;
  r->vbw_cbw = 0;   /* low bandwidth */
  r->vbw_ypeak = 0; /* disabled */
  r->vbw_ysv = 0;   /* low bandwidth */
  r->vbw_ycv = 0;   /* low bandwidth */
  r->civh = 0;      /* civ hysteresis mode = 0 */
  r->ce = 3;        /* normal contrast */
  switch (system) {
    case TV_SYSTEM_NTSC: /* additional fsci values for NDC (no dot crawl) */
      r->dmr_vs = 1;
      r->dacg = 0;
      r->blr = 127;
      break;
    case TV_SYSTEM_NTSC_J:
      r->dmr_vs = 3;
      r->dacg = 1;
      r->blr = 100;
      break;
    case TV_SYSTEM_NONE:  /* to have a default */
    case TV_SYSTEM_SECAM: /* makes most sense */
    case TV_SYSTEM_PAL:
    case TV_SYSTEM_PAL_X:
    case TV_SYSTEM_PAL_N:  /* PAL-N has different fsci values */
    case TV_SYSTEM_PAL_NC: /* guess */
      r->dmr_vs = 0;
      r->dacg = 1;
      r->blr = 105;
      break;
    case TV_SYSTEM_PAL_M:   /* needs different fsci values */
    case TV_SYSTEM_PAL_M60: /* guess */
      r->dmr_vs = 2;
      r->dacg = 0;
      r->blr = 127;
      break;
    case TV_SYSTEM_PAL_60: /* modify PAL_M */
      r->dmr_vs = 2;
      r->dacg = 1;
      r->blr = 105;
      break;
  }
  r->flags = CH_FLAG_BOTH | CH_FLAG_CFRB; /* no ACIV, no CVBW */
}

void data_default_ch (TVSettings *s)
{
  s->tv_hoffset = s->mon_hoffset = 0;
  s->tv_voffset = s->mon_voffset = 0;
  s->brightness_sig = 0;
  s->contrast_sig = 0;
  s->contrast = 0;
  s->saturation_sig = 0;
  s->saturation = 0;
  s->flicker = 75;
  s->flicker_adapt = 0;
  s->luma_bandwidth = 100;
  s->chroma_bandwidth = 75;
  s->sharpness = 50;
  s->cross_color = 0;
  s->phase = 0;
  s->hue = 0;
  /* FIXME flags */
}

void data_clamp_ch (TVSettings *s, TVRegs *r)
{
  if (s->tv_hoffset <= -100) s->tv_hoffset = -100;
  if (s->tv_hoffset >=  100) s->tv_hoffset =  100;
  if (s->tv_voffset <= -100) s->tv_voffset = -100;
  if (s->tv_voffset >=  100) s->tv_voffset =  100;
  if (s->mon_hoffset <= -100) s->mon_hoffset = -100;
  if (s->mon_hoffset >=  100) s->mon_hoffset =  100;
  if (s->mon_voffset <= -100) s->mon_voffset = -100;
  if (s->mon_voffset >=  100) s->mon_voffset =  100;
  if (s->brightness_sig < -50) s->brightness_sig = -50;
  if (s->brightness_sig >  50) s->brightness_sig = 50;
  s->contrast   = data_clamp (s->contrast, 8, clamp_contrast); 
  s->saturation = 0;
  s->contrast_sig = 0;
  s->saturation_sig = 0;
  s->phase = 0;
  s->hue = 0;
  s->cross_color = 0;
  s->sharpness = data_clamp (s->sharpness, 3, clamp_sharpness); 
  s->flicker = data_clamp (s->flicker, 5, clamp_flicker); 
  s->flicker_adapt = 0;
  s->luma_bandwidth = data_clamp (s->luma_bandwidth, 5, 
				  clamp_luma_bandwidth); 
  s->chroma_bandwidth = data_clamp (s->chroma_bandwidth, 4, 
				    clamp_chroma_bandwidth); 
  /* FIXME: Operate on mode flags */
}

void data_setup_ch (TVSettings *s, TVRegs *r)
{
  register int i;
  int vmax;

  if (r->enc.ch.dmr_vs & 0x01) vmax = 262; else vmax = 312;
  r->enc.ch.hpr += s->tv_hoffset;
  if (r->enc.ch.hpr <   0) r->enc.ch.hpr = 0;
  if (r->enc.ch.hpr > 511) r->enc.ch.hpr = 511;
  r->enc.ch.vpr -= s->tv_voffset;
  while (r->enc.ch.vpr <    0) r->enc.ch.vpr += vmax;
  while (r->enc.ch.vpr > vmax) r->enc.ch.vpr -= vmax;
  i = s->brightness_sig + r->enc.ch.blr;
  if (i < 0) i = 0;
  if (i > 255) i = 255;
  r->enc.ch.blr = i;
  r->enc.ch.ce = data_pick (s->contrast, 8, clamp_contrast); 
  switch (data_pick (s->flicker, 5, clamp_flicker)) {
    case 0: r->enc.ch.ffr_fy = 0; r->enc.ch.ffr_fc = 0; break;
    case 1: r->enc.ch.ffr_fy = 1; r->enc.ch.ffr_fc = 0; break;
    case 2: r->enc.ch.ffr_fy = 1; r->enc.ch.ffr_fc = 1; break;
    case 3: r->enc.ch.ffr_fy = 2; r->enc.ch.ffr_fc = 1; break;
    case 4: r->enc.ch.ffr_fy = 2; r->enc.ch.ffr_fc = 2; break;
  }
  switch (data_pick (s->luma_bandwidth, 5, clamp_luma_bandwidth)) {
    case 0: 
      r->enc.ch.vbw_ycv = 0; r->enc.ch.vbw_ysv = 0; r->enc.ch.vbw_ypeak = 0; 
      break;
    case 1: 
      r->enc.ch.vbw_ycv = 0; r->enc.ch.vbw_ysv = 0; r->enc.ch.vbw_ypeak = 1; 
      break;
    case 2: 
      r->enc.ch.vbw_ycv = 1; r->enc.ch.vbw_ysv = 1; r->enc.ch.vbw_ypeak = 0; 
      break;
    case 3: 
      r->enc.ch.vbw_ycv = 1; r->enc.ch.vbw_ysv = 1; r->enc.ch.vbw_ypeak = 1; 
      break;
    case 4: 
      r->enc.ch.vbw_ycv = 1; r->enc.ch.vbw_ysv = 2; r->enc.ch.vbw_ypeak = 1; 
      break;
  }
  r->enc.ch.vbw_cbw = data_pick (s->chroma_bandwidth, 4, clamp_chroma_bandwidth);
  /* FIXME Chroma dot crawl in fc */
  /* FIXME vbw_flff=1 for 5 line in 7/10 */
  /* FIXME handle CH7003 differently */
  r->enc.ch.ffr_ft = 2 - data_pick (s->sharpness, 3, clamp_sharpness); 
  r->enc.ch.macro = (s->flags & TV_DESC_MACROVISION) ? 1 : 0;
  if (s->flags & TV_DESC_CARRIER_LOCK) {
    r->enc.ch.flags &= ~CH_FLAG_CFRB;
  } else {
    r->enc.ch.flags |= CH_FLAG_CFRB;
  }
  if (s->flags & TV_DESC_MONOCHROME) {
    r->enc.ch.flags |= CH_FLAG_CVBW;
  } else {
    r->enc.ch.flags &= ~CH_FLAG_CVBW;
  }
  if (s->flags & TV_DESC_COLORFIX) {
    r->enc.ch.flags |= CH_FLAG_POUTP;
  } else {
    r->enc.ch.flags &= ~CH_FLAG_POUTP;
  }
  r->enc.ch.flags &= ~CH_FLAG_DAC_MASK; 
  switch (s->connector) 
  {
    case CONNECT_COMPOSITE: 
      r->enc.ch.flags |= CH_FLAG_COMPOSITE; 
      break;
    case CONNECT_CONVERT:
    case CONNECT_SVIDEO: 
      r->enc.ch.flags |= CH_FLAG_SVIDEO;
      break;
    case CONNECT_BOTH: 
    default:
      r->enc.ch.flags |= CH_FLAG_BOTH;
      break;
  }
}

/* 

4:3 * 4:3 -> 16:9

 4/3 = 1.333
16/9 = 1.777

720 / 576 = 1.25 = 5/4    * 4/3 = 5/3 = 1.66
720 / 480 = 1.5  = 3/2    * 4/3 = 2   

 768 / 576 = 4:3 (800/600, use scaler)
1024 / 576 = 16:9 (only on monitor)
 720 / 540 = 4:3
 720 / 405 = 16:9 
 800 / 450 = 16:9 (800/600, use scaler)


800 / 600 = 1.33 = 4/3 
640 / 480 = 1.33 = 4/3 
640 / 400 = 1.6  = 8/5
720 / 400 = 1.8  = 9/5
512 / 384 = 1.33 = 4/3

Can display more lines by changing CRT, but the ___x400 modes have
wrong aspect.

So we have to change the aspect for those modes. Changing the PLL
coeff's doesn't work.

*/

/* Special FCSI values (different crystal?)

Resolution    Tiny;           Small;          Large;
	      		      	 	 
512x384  (1)  644916848  (0)  806121060	 
640x400  (9)  537447373  (8)  677157690	 
640x480  (15) 537430027  (14) 644862000  (13) 806120400
720x400  (5)  485446014  (4)  601879058	 	 
768x576                  (20) 529051320  (19) 645599916
800x450                  (20) 529051320  (19) 645599916
800x600  (21) 488330000  (20) 529002000  (19) 645527100

Differences:

PAL-X                                   n   m               MHz   kHz

13: 806120400/806021060: 1.000123247   20/ 13 = 1.466666  21.00 25.00
14: 644862000/644816848: 1.000070022    9/  4 = 1.833333  26.25 31.25
19: 645527100/645499916: 1.000042113  647/313 = 2.060317  29.50 31.25
20: 529002000/528951320: 1.000095812   86/ 33 = 2.514285  36.00 37.50

Is not linear...

*/

/*

Think arrangement over.

Mode -> dmr_ir, dmr_sr, dmr_vs (lsb), pll_m, pll_n, pllcap

System -> dmr_vs (msb), dacg, (either of two modesets)

NTSC,NTSC-NDC,PAL-M  (NTSC-J ?)
PAL,PAL-N

crt values -> sav. hpr, vpr (are those different for nv/i810 ??)

*/



