/* NVTV Conexant chip data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Data tables and routines for the Conexant chip.
 *
 */

#include "local.h" /* before everything else */

#include <stddef.h>
#include <math.h>

#include "data_bt.h"
#include "data_cx.h"
#include "data_nv.h"

/* -------- CX -------- NTSC -------- */

/* FIXME I do not know if all XBoxes have the CX encoder. For now, we
   keep the one known BIOS mode here. */

/* -------- 640x480 XBox -------- */

TVCxRegs cx_ntsc_xbox = { /* XBox BIOS */
  bt : {
  h_active     : 648,
  h_blanki     : 115,
  h_clki       : 776,
  h_fract      : 0,
  h_blanko     : 237,
  h_clko       : 1552,
  v_activei    : 480,
  v_blanki     : 36,
  v_linesi     : 525,
  v_activeo    : 242,
  v_blanko     : 19,
  v_scale      : 4096,
  pll_fract    : 55912,
  pll_int      : 10,
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 15,
  vsynwidth    : 1,
  hsync_width  : 114,
  hburst_begin : 133,
  hburst_end   : 68,
  sync_amp     : 228,
  bst_amp      : 124,
  mcr          : 203,
  mcb          : 143,
  my           : 154,
  msc          : 629578002,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_NTSC,
  },
  flags4       : 0,
  flags5       : 0,
  msc_db       : 676059667,
  dr_limitp    : 1443,
  dr_limitn    : 1183,
  db_limitp    : 1443,
  db_limitn    : 1183,
  filfsconv    : 0,
  filincr      : 0,
  wsdat        : 196608,
  wssinc       : 19221,
};

/* -------- 1024x768 3:2 -------- */

TVCxRegs cx_ntsc_small_a = { /* Mode 10, hoc=15.11 voc=14.81 */
  bt : {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2912,
  h_active     : 1024,
  hsync_width  : 216,
  hburst_begin : 242,
  hburst_end   : 238,
  h_blanko     : 625,
  v_blanko     : 36,
  v_activeo    : 208,
  h_fract      : 0,
  h_clki       : 1176,
  h_blanki     : 133,
  v_linesi     : 975,
  v_blanki     : 130,
  v_activei    : 768,
  v_scale      : 11118,
  pll_fract    : 35747,
  pll_int      : 30,
  sync_amp     : 229,
  bst_amp      : 116,
  mcr          : 119,
  mcb          : 67,
  my           : 133,
  msc          : 335544320,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_NTSC 
  },
  flags5       : CX_FLAG5_PLL_32CLK
};

TVCxRegs cx_ntsc_normal_a = { /* Mode 26, hoc=11.97 voc=11.93 */
  bt : {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2808,
  h_active     : 1024,
  hsync_width  : 208,
  hburst_begin : 234,
  hburst_end   : 224,
  h_blanko     : 567,
  v_blanko     : 33,
  v_activeo    : 215,
  h_fract      : 0,
  h_clki       : 1170,
  h_blanki     : 127,
  v_linesi     : 945,
  v_blanki     : 115,
  v_activei    : 768,
  v_scale      : 10650,
  pll_fract    : 29789,
  pll_int      : 29,
  sync_amp     : 229,
  bst_amp      : 116,
  mcr          : 119,
  mcb          : 67,
  my           : 133,
  msc          : 347971887,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_NTSC 
  },
  flags5       : CX_FLAG5_PLL_32CLK
};

TVCxRegs cx_ntsc_tiny_a = { /* Mode 42, hoc=18.04 voc=18.11 */
  bt : {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 3016,
  h_active     : 1024,
  hsync_width  : 224,
  hburst_begin : 252,
  hburst_end   : 250,
  h_blanko     : 683,
  v_blanko     : 40,
  v_activeo    : 200,
  h_fract      : 0,
  h_clki       : 1170,
  h_blanki     : 127,
  v_linesi     : 1015,
  v_blanki     : 150,
  v_activei    : 768,
  v_scale      : 11742,
  pll_fract    : 41704,
  pll_int      : 31,
  sync_amp     : 229,
  bst_amp      : 116,
  mcr          : 119,
  mcb          : 67,
  my           : 133,
  msc          : 323973826,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_NTSC 
  },
  flags5       : CX_FLAG5_PLL_32CLK
};

/* -------- 800x600 3:2 -------- */

TVCxRegs cx_ntsc_small_b = { /* Mode 18, hoc=13.79 voc=13.58 */
  bt : {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2240,
  h_active     : 800,
  hsync_width  : 166,
  hburst_begin : 186,
  hburst_end   : 152,
  h_blanko     : 473,
  v_blanko     : 34,
  v_activeo    : 212,
  h_fract      : 0,
  h_clki       : 1176,
  h_blanki     : 329,
  v_linesi     : 750,
  v_blanki     : 94,
  v_activei    : 600,
  v_scale      : 7607,
  pll_fract    : 32539,
  pll_int      : 23,
  sync_amp     : 229,
  bst_amp      : 116,
  mcr          : 120,
  mcb          : 67,
  my           : 133,
  msc          : 436207616,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_NTSC 
  },
  flags5       : CX_FLAG5_PLL_32CLK 
};

TVCxRegs cx_ntsc_tiny_b = { /* Mode 34, hoc=19.26 voc=19.34 */
  bt : {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2392,
  h_active     : 800,
  hsync_width  : 176,
  hburst_begin : 200,
  hburst_end   : 172,
  h_blanko     : 557,
  v_blanko     : 42,
  v_activeo    : 197,
  h_fract      : 0,
  h_clki       : 1170,
  h_blanki     : 323,
  v_linesi     : 805,
  v_blanki     : 125,
  v_activei    : 600,
  v_scale      : 8465,
  pll_fract    : 5958,
  pll_int      : 25,
  sync_amp     : 229,
  bst_amp      : 116,
  mcr          : 119,
  mcb          : 67,
  my           : 133,
  msc          : 408488737,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_NTSC 
  },
  flags5       : CX_FLAG5_PLL_32CLK 
};

TVCxRegs cx_ntsc_mini_b = { /* Mode 40, hoc=15.59 voc=15.64 */
  bt : {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2288,
  h_active     : 800,
  hsync_width  : 170,
  hburst_begin : 190,
  hburst_end   : 158,
  h_blanko     : 499,
  v_blanko     : 37,
  v_activeo    : 206,
  h_fract      : 0,
  h_clki       : 1170,
  h_blanki     : 323,
  v_linesi     : 770,
  v_blanki     : 105,
  v_activei    : 600,
  v_scale      : 7919,
  pll_fract    : 0,
  pll_int      : 24,
  sync_amp     : 229,
  bst_amp      : 116,
  mcr          : 120,
  mcb          : 67,
  my           : 133,
  msc          : 427056407,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_NTSC 
  },
  flags5       : CX_FLAG5_PLL_32CLK 
};

/* -------- CX -------- PAL -------- */

/* -------- 1024x768 3:2 -------- */

TVCxRegs cx_pal_small_a = { /* Mode 11, hoc=13.44 voc=14.24 */
  bt : {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2912,
  h_active     : 1024,
  hsync_width  : 214,
  hburst_begin : 254,
  hburst_end   : 230,
  h_blanko     : 647,
  v_blanko     : 43,
  v_activeo    : 248,
  h_fract      : 0,
  h_clki       : 1400,
  h_blanki     : 329,
  v_linesi     : 975,
  v_blanki     : 131,
  v_activei    : 768,
  v_scale      : 8684,
  pll_fract    : 21845,
  pll_int      : 30,
  sync_amp     : 240,
  bst_amp      : 86,
  mcr          : 127,
  mcb          : 71,
  my           : 140,
  msc          : 418510935,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_PAL_BDGHI,
  },
  flags5       : CX_FLAG5_PLL_32CLK 
};

TVCxRegs cx_pal_mini_a = { /* Mode 43, hoc=16.20 voc=16.67 */
  bt : {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 3008,
  h_active     : 1024,
  hsync_width  : 220,
  hburst_begin : 264,
  hburst_end   : 240,
  h_blanko     : 703,
  v_blanko     : 47,
  v_activeo    : 241,
  h_fract      : 0,
  h_clki       : 1410,
  h_blanki     : 337,
  v_linesi     : 1000,
  v_blanki     : 147,
  v_activei    : 768,
  v_scale      : 9011,
  pll_fract    : 21845,
  pll_int      : 31,
  sync_amp     : 240,
  bst_amp      : 86,
  mcr          : 126,
  mcb          : 71,
  my           : 140,
  msc          : 405154203,
  flags1       : CX_FLAG1_EXT | BT_FLAG1_PAL_BDGHI
  },
  flags5       : CX_FLAG5_PLL_32CLK 
};

/* -------- -------- */

#define two_to_32 4294967296.0
#define two_to_26 67108864.0
#define two_to_20 1048576.0
#define two_to_13 8192.0
#define PI 3.14159265358979323846

static int clamp_attenuate[8] = {-100, -90, -75, -50, -30, -20, -10, 0};
static int clamp_flicker[5]       = {0, 25, 50, 75, 100};
static int clamp_flicker_adapt[6] = {0, 20, 40, 60, 80, 100};
static int clamp_sharpness[4] = {0, 30, 60, 100};
static int clamp_luma_bandwidth[4] = {0, 30, 60, 100};
static int clamp_chroma_bandwidth[3] = {0, 50, 100};
static int clamp_cross_color[2] = {0, 100};

/* Convert PAL to SECAM */

void data_secam_cx (TVCxRegs *r)
{
  double Fxtal = 13500000; 
  double Fclk;
  double Fsc_r = 4406250; /* Hz, 'for' in Table A-1 */
  double Fsc_b = 4250000; /* Hz, 'fob' in Table A-1 */
  register double f;
  register int i;

  Fclk = Fxtal * (r->bt.pll_int + r->bt.pll_fract / 65536.0) / 6.0;
  if ((r->bt.flags1 & CX_FLAG1_EXT) &&
      (r->flags5 & CX_FLAG5_PLL_32CLK)) 
  {
    Fclk *= 2.0/3.0;
  }
  f = PI * Fsc_r / Fclk;
  f = 288036.0 * r->bt.h_clko * sin(f) / f;
  r->bt.mcr = (int) ((920.26 / f) * two_to_26 + 0.5);
  f = PI * Fsc_b / Fclk;
  f = 288036.0 * r->bt.h_clko * sin(f) / f;
  r->bt.mcb = (int) ((598.15 / f) * two_to_26 + 0.5);
  r->bt.msc = (unsigned long) ((282.0 / r->bt.h_clko) * two_to_32 + 0.5);
  r->msc_db = (unsigned long) ((272.0 / r->bt.h_clko) * two_to_32 + 0.5);
  r->dr_limitp = r->db_limitp = (int) (4.756e6 / Fclk * two_to_13 + 0.5);
  r->dr_limitn = r->db_limitn = (int) (3.900e6 / Fclk * two_to_13 + 0.5);
  i = (int) (8192 * 4.286 * 1728 / (27 * r->bt.h_clko) + 0.5);
  r->filincr = ((i >> 1) & 0xff) ^ 0xff;
  r->filfsconv = (int) ((27 * r->bt.h_clko * 1.087) / 1728.0 + 0.5);
  r->bt.flags1 &= ~(BT_FLAG1_PAL_MD | BT_FLAG1_VSYNC_DUR);
  r->bt.flags1 |= CX_FLAG1_FM;
  /* FIXME use dis_scrst? */
}

void data_init_cx (TVSystem system, TVCxRegs *r, int btmode)
{
  double Fxtal = 13500000; 
  double Fclk; 

  if (!r) return;
  data_mux_nv_bt (&r->bt);
  data_init_nomux_bt (system, &r->bt);
  /* Zero extra stuff if it comes from a BT template */
  if (btmode) {
    r->flags4 = 0;
    r->flags5 = 0;
    r->wsdat = 0;
  }
  Fclk = Fxtal * (r->bt.pll_int + r->bt.pll_fract / 65536.0) / 6.0;
  if (r->flags5 & CX_FLAG5_PLL_32CLK) {
    Fclk *= 2.0/3.0;
  }
  r->mcompu = 128; /* default, scale 1.0 */
  r->mcompv = 128; /* default, scale 1.0 */
  r->mcompy = 128; /* default, scale 1.0 */
  if (system != TV_SYSTEM_SECAM) {
    r->msc_db = 0;
    r->dr_limitp = 0;
    r->dr_limitn = 0;
    r->db_limitp = 0;
    r->db_limitn = 0;
    r->filfsconv = 0;
    r->filincr = 0;
  }
  r->y_off = 0; /* default */
  r->y_thresh = 0; /* default */
  r->hue_adj = 0;
  if (r->bt.flags1 & BT_FLAG1_625LINE) {
    /* FIXME: doc says 200E-9 = 0.2E-6, which overflows */
    r->wssinc = (int) two_to_20 / (2.000E-6 * Fclk);
  } else {
    r->wssinc = (int) two_to_20 / (2.234E-6 * Fclk);
  }
  r->c_altff = 0;
  r->y_altff = 0;
  r->c_thresh = 0;
  r->y_thresh = 0;
}

void data_default_cx (TVSettings *s)
{
  s->tv_hoffset = s->mon_hoffset = 0;
  s->tv_voffset = s->mon_voffset = 0;
  s->brightness_sig = 0;
  s->contrast_sig = 0;
  s->saturation_sig = 0;
  s->contrast = 0;
  s->saturation = 0;
  s->phase = 0;
  s->hue = 0;
  s->flicker = 75;      /* FIXME: default depends on mode */
  s->flicker_adapt = 0; /* FIXME: default depends on mode */
  s->sharpness = 0;
  s->cross_color = 100;
  s->luma_bandwidth = 30;
  s->chroma_bandwidth = 100;
  /* FIXME s->flags; */
}

void data_clamp_cx (TVSettings *s, TVRegs *r)
{
  if (s->tv_hoffset <= -100) s->tv_hoffset = -100;
  if (s->tv_hoffset >=  100) s->tv_hoffset =  100;
  if (s->tv_voffset <= -100) s->tv_voffset = -100;
  if (s->tv_voffset >=  100) s->tv_voffset =  100;
  if (s->mon_hoffset <= -100) s->mon_hoffset = -100;
  if (s->mon_hoffset >=  100) s->mon_hoffset =  100;
  if (s->mon_voffset <= -100) s->mon_voffset = -100;
  if (s->mon_voffset >=  100) s->mon_voffset =  100;
  s->tv_voffset = 0; 
  /* FIXME. Clamp to 0 for now, until some 'VTotal magic' */
  if (s->brightness_sig < -50) s->brightness_sig = -50;
  if (s->brightness_sig >  50) s->brightness_sig = 50;
  if (s->contrast_sig < -50) s->contrast_sig = -50;
  if (s->contrast_sig >  50) s->contrast_sig = 50;
  if (s->saturation_sig < -50) s->saturation_sig = -50;
  if (s->saturation_sig >  50) s->saturation_sig = 50;
  if (s->phase < -180) s->phase = -180;
  if (s->phase >  180) s->phase = 180;
  if (s->hue < -180) s->hue = -180;
  if (s->hue >  180) s->hue = 180;
  s->contrast   = data_clamp (s->contrast, 8, clamp_attenuate); 
  s->saturation = data_clamp (s->saturation, 8, clamp_attenuate); 
  s->sharpness = data_clamp (s->sharpness, 4, clamp_sharpness);
  s->flicker = data_clamp (s->flicker, 5, clamp_flicker); 
  s->flicker_adapt = data_clamp (s->flicker_adapt, 6, clamp_flicker_adapt); 
  s->cross_color = data_clamp (s->cross_color, 2, clamp_cross_color);
  s->chroma_bandwidth = data_clamp (s->chroma_bandwidth, 3, 
				    clamp_chroma_bandwidth); 
  s->luma_bandwidth = data_clamp (s->luma_bandwidth, 4, 
				  clamp_luma_bandwidth); 
  /* FIXME: Operate on mode flags */
}

/* Only for nv, does wrong setup for tdfx */

void data_setup_cx (TVSettings *s, TVRegs *r)
{
  float f;

  data_mux_nv_bt (&r->enc.bt);
  data_setup_nomux_bt (s, r);
  r->enc.cx.y_off = s->brightness_sig & 0xff;
  f = s->hue * 256.0 / 360.0;
  if (f < 0.0) f += 256.0;
  if (f < 0.0) f = 0.0;
  if (f > 255.0) f = 255.0;
  r->enc.cx.hue_adj = (int) f;
  r->enc.cx.flags5 &= ~(CX_FLAG5_ADPT_FF | CX_FLAG5_FFRTN | CX_FLAG5_YSELECT);
  switch (data_pick (s->flicker_adapt, 6, clamp_flicker_adapt)) {
    case 0: 
      break;
    case 1: 
      r->enc.cx.y_altff = 3; /* 4 line */
      r->enc.cx.c_altff = 3; /* 4 line */
      r->enc.cx.y_thresh = 0;
      r->enc.cx.c_thresh = 0;
      r->enc.cx.flags5 |= CX_FLAG5_ADPT_FF | CX_FLAG5_FFRTN | CX_FLAG5_YSELECT;
      break;
    case 2: 
      r->enc.cx.y_altff = 3; /* 4 line */
      r->enc.cx.c_altff = 3; /* 4 line */
      r->enc.cx.y_thresh = 4;
      r->enc.cx.c_thresh = 4;
      r->enc.cx.flags5 |= CX_FLAG5_ADPT_FF;
      break;
    case 3: 
      r->enc.cx.y_altff = 0; /* 5 line */
      r->enc.cx.c_altff = 0; /* 5 line */
      r->enc.cx.y_thresh = 2;
      r->enc.cx.c_thresh = 2;
      r->enc.cx.flags5 |= CX_FLAG5_ADPT_FF | CX_FLAG5_FFRTN;
      break;
    case 4: 
      r->enc.cx.y_altff = 0; /* 5 line */
      r->enc.cx.c_altff = 0; /* 5 line */
      r->enc.cx.y_thresh = 4;
      r->enc.cx.c_thresh = 4;
      r->enc.cx.flags5 |= CX_FLAG5_ADPT_FF | CX_FLAG5_YSELECT;
      break;
    case 5: 
      r->enc.cx.y_altff = 0; /* 5 line */
      r->enc.cx.c_altff = 0; /* 5 line */
      r->enc.cx.y_thresh = 6;
      r->enc.cx.c_thresh = 6;
      r->enc.cx.flags5 |= CX_FLAG5_ADPT_FF | CX_FLAG5_FFRTN | CX_FLAG5_YSELECT;
      break;
  }
  r->enc.cx.pkfil_sel = data_pick (s->sharpness, 4, clamp_sharpness);
  if (s->cross_color) {
    r->enc.cx.flags4 |= CX_FLAG4_BY_YCCR;
  } else {
    r->enc.cx.flags4 &= ~CX_FLAG4_BY_YCCR;
  }
  /* FIXME chroma_bandwidth: CX_FLAG4_CHROMA_BW */
}

/*
Setup:

             BT             CX add.    CH             PH
             ------------   ---------  -------------  ---------
Brightness   -              y_off      blr            blckl
Contrast     yattenuate                ce             gy
  (Signal)   my                        -              -
Saturation   cattenuate                -              gcd
  (Signal)   mcr,mcb                   -              gainu,gainv
Phase        phase_off                 -              chps
Hue          -              hue_adj    -              -
Flicker      f_sely/c,                 ffr_{fc,fy},   (-)
             dis_ffilt                 vbw_flff
Adaptive     -              adpt_ff    -              -
C-Bandwidth  clpf           chroma_bw  cbw            scbw
Y-Bandwidth  ylpf                      ycv,ysv,ypeak  -
Sharpness    -              pkfil_sel  ffr_ft         yfil
CrossColRed  -              by_yccr    -              ccrs
Noninterlace ni_out                    -              flc
Monochrome   dis_chroma                vbw_cvbw       -
CarrierReset                                          phres
Coring       {y,c}_coring              -              -

Ticks:

                       BT   CX   CH   PH
Brightness  -50-50      -   oo   oo   oo  lock to contrast?
Contrast   -100-100     8    8  4+4   32
  (Signal)  -50-50     oo   oo    -    -
Saturation -100-100     8    8    -   32
  (Signal)  -50-50     oo   oo    -   oo
Phase                 360  360    -  360  spin? -180 - 180
Hue                     -  360    -       spin? -180 - 180
Flicker       0-100     5    5    5  (oo)
Flicker2      0-100     -    6    -    -
Chroma-Bandw  0-100     3  3*2    4    2
Luma-Bandw    0-100   4*2  4*2  3*2    -
Sharpness     0-100     -    4    3
Cross Color             -    2         4
Carrier Reset           2    2         4
Non-Interlace           x    x    -    -
Monochrome              x    x    x    ?
Macrovision           2/4  2/4  2/4  2/4
Signals                 x    x    -    x
Coring                7+1  7+1    -    -
         
Buttons:   Dualview, Macrovision, Non-Interlace, 
          Monochrome, Carrier Reset, Distort (Signals)

Not used: 
* Coring, Clipping (eclip, ycoring, ccoring)
* Complete disable: dis_yflpf 
* Phase: phase_off (0-360), dis_scrst, hue_adj (0-360), 
* Adaptive flicker
* y_off (CX only)

Philips:
DACF/R,G,BDACC (maybe shouldn't use these)

Alternate FF:

adpt_ff
y_altff >= f_sely
c_altff >= f_selc
y_thresh
c_thresh
yselect

*/

/*
Init:

          SECAM:     

vsync_dur   0  
625line     1  
setup       0
pal_md      0
dis_scrst   1
fm          1

verify against modes on p. 1-70

*/

