/* NVTV Intel 810 CRTC data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Data routines for the Intel 810 CRTC data.
 *
 */

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include "data_i810.h"
#include "data_ch.h"

/* -------- CH -------------------------------- */

#define TV_DEF_CH (TV_CAP_MONOCHROME)

/* -------- I810 -------- CRT -------- */

TVI810Regs i810_ch_pal_large_a = { /* Mode 13 */
  tvHDisplay    : 640,
  tvHSyncStart  : 740,
  tvHSyncEnd    : 804,
  tvHTotal      : 840,
  tvVDisplay    : 480,
  tvVSyncStart  : 490,
  tvVSyncEnd    : 492,
  tvVTotal      : 500,
};

TVI810Regs i810_ch_pal_small_a = { /* Mode 14 */
  tvHDisplay    : 640,
  tvHSyncStart  : 740,
  tvHSyncEnd    : 804,
  tvHTotal      : 840,
  tvVDisplay    : 480,
  tvVSyncStart  : 552,
  tvVSyncEnd    : 554,
  tvVTotal      : 625,
};

TVI810Regs i810_ch_pal_tiny_a = { /* Mode 15 */
  tvHDisplay    : 640,
  tvHSyncStart  : 740,
  tvHSyncEnd    : 804,
  tvHTotal      : 840,
  tvVDisplay    : 480,
  tvVSyncStart  : 615,
  tvVSyncEnd    : 617,
  tvVTotal      : 750,
};

TVI810Regs i810_ch_ntsc_large_a = { /* Mode 16 */
  tvHDisplay    : 640,
  tvHSyncStart  : 712,
  tvHSyncEnd    : 776,
  tvHTotal      : 784,
  tvVDisplay    : 480,
  tvVSyncStart  : 502,
  tvVSyncEnd    : 504,
  tvVTotal      : 525,
};

TVI810Regs i810_ch_ntsc_small_a = { /* Mode 17 */
  tvHDisplay:   640,
  tvHSyncStart: 712,
  tvHSyncEnd:   776,
  tvHTotal:     784,
  tvVDisplay:   480,
  tvVSyncStart: 540,
  tvVSyncEnd:   542,
  tvVTotal:     600,
};

TVI810Regs i810_ch_ntsc_tiny_a = { /* Mode 18 */
  tvHDisplay    : 640,
  tvHSyncStart  : 720,
  tvHSyncEnd    : 784,
  tvHTotal      : 800,
  tvVDisplay    : 480,
  tvVSyncStart  : 555,
  tvVSyncEnd    : 557,
  tvVTotal      : 630,
};

TVI810Regs i810_ch_pal_large_b = { /* Mode 19 */
  tvHDisplay    : 800,
  tvHSyncStart  : 872,
  tvHSyncEnd    : 936,
  tvHTotal      : 944,
  tvVDisplay    : 600,
  tvVSyncStart  : 612,
  tvVSyncEnd    : 614,
  tvVTotal      : 625,
};

TVI810Regs i810_ch_pal_small_b = { /* Mode 20 */
  tvHDisplay    : 800,
  tvHSyncStart  : 880,
  tvHSyncEnd    : 944,
  tvHTotal      : 960,
  tvVDisplay    : 600,
  tvVSyncStart  : 675,
  tvVSyncEnd    : 677,
  tvVTotal      : 750,
};

TVI810Regs i810_ch_pal_tiny_b = { /* Mode 21 */
  tvHDisplay    : 800,
  tvHSyncStart  : 868,
  tvHSyncEnd    : 932,
  tvHTotal      : 936,
  tvVDisplay    : 600,
  tvVSyncStart  : 718,
  tvVSyncEnd    : 720,
  tvVTotal      : 836,
};

TVI810Regs i810_ch_ntsc_huge_b = { /* Mode 22 */
  tvHDisplay    : 800,
  tvHSyncStart  : 920,
  tvHSyncEnd    : 984,
  tvHTotal      : 1040,
  tvVDisplay    : 600,
  tvVSyncStart  : 615,
  tvVSyncEnd    : 617,
  tvVTotal      : 630,
};

TVI810Regs i810_ch_ntsc_large_b = { /* Mode 23 */
  tvHDisplay    : 800,
  tvHSyncStart  : 920,
  tvHSyncEnd    : 984,
  tvHTotal      : 1040,
  tvVDisplay    : 600,
  tvVSyncStart  : 650,
  tvVSyncEnd    : 652,
  tvVTotal      : 700,
};

TVI810Regs i810_ch_ntsc_small_b = { /* Mode 24 */
  tvHDisplay    : 800,
  tvHSyncStart  : 932,
  tvHSyncEnd    : 996,
  tvHTotal      : 1064,
  tvVDisplay    : 600,
  tvVSyncStart  : 675,
  tvVSyncEnd    : 677,
  tvVTotal      : 750,
};

/* -------- I810 -------- modelist -------- */

typedef struct {
  int sav;
  int hpr;
  int vpr;
} TVAdjustI810Ch;

typedef struct {
  TVModeSpec spec;
  TVI810Regs *i810;
  TVChRegs *ch;
  unsigned long *fsci;
  TVAdjustI810Ch adj;
  int portFlags;
  int descFlags;
} TVTemplateI810Ch;

/* NTSC modes with "no dot crawl" get # appended to size */

TVTemplateI810Ch templ_i810_ch_50hz [] = { /* 50 Hz: PAL, PAL-N, PAL-X */

  {{TV_SYSTEM_PAL, 800, 600, "Tiny",   "4:3", 21.875, 21.875}, /* Mode 21 */
   &i810_ch_pal_tiny_b,  &ch_pal_tiny_b,  
   ch_fsci_pal_tiny_b,  {sav:  68, hpr:  76, vpr: 306}, 
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW},
  {{TV_SYSTEM_PAL, 800, 600, "Small",  "4:3", 13.194, 13.194}, /* Mode 20 */
   &i810_ch_pal_small_b, &ch_pal_small_b, 
   ch_fsci_pal_small_b, {sav:  80, hpr:  63, vpr: 306},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_PAL, 800, 600, "Large",  "4:3", -4.167, -4.167}, /* Mode 19 */
   &i810_ch_pal_large_b, &ch_pal_large_b, 
   ch_fsci_pal_large_b, {sav:  72, hpr:  43, vpr: 298},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW},  

  {{TV_SYSTEM_PAL, 640, 480, "Tiny",   "4:3", 30.556, 30.556}, /* Mode 15 */
   &i810_ch_pal_tiny_a,  &ch_pal_tiny_a,  
   ch_fsci_pal_tiny_a,  {sav: 100, hpr:  80, vpr: 306},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW},    
  {{TV_SYSTEM_PAL, 640, 480, "Small",  "4:3", 16.667, 16.667}, /* Mode 14 */
   &i810_ch_pal_small_a, &ch_pal_small_a, 
   ch_fsci_pal_small_a, {sav: 100, hpr:  53, vpr: 305},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},  
  {{TV_SYSTEM_PAL, 640, 480, "Large",  "4:3", -4.167, -4.167}, /* Mode 13 */
   &i810_ch_pal_large_a, &ch_pal_large_a, 
   ch_fsci_pal_large_a, {sav: 100, hpr:  25, vpr: 304},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW},  

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, NULL, {}, 0},
};

/* NTSC modes with "no dot crawl" get # appended to size */

TVTemplateI810Ch templ_i810_ch_60hz [] = { /* 60 Hz: NTSC, NTSC-NDC, PAL-M */

  {{TV_SYSTEM_NTSC, 800, 600, "Small",  "4:3", 12.500, 12.500}, /* Mode 24 */
   &i810_ch_ntsc_small_b, &ch_ntsc_small_b, 
   ch_fsci_ntsc_small_b, {sav: 132, hpr:  59, vpr: 255},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_NTSC, 800, 600, "Large",  "4:3",  6.250,  6.250}, /* Mode 23 */
   &i810_ch_ntsc_large_b, &ch_ntsc_large_b, 
   ch_fsci_ntsc_large_b, {sav: 120, hpr:  46, vpr: 255},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW},
  {{TV_SYSTEM_NTSC, 800, 600, "Huge",   "4:3", -4.167, -4.167}, /* Mode 22 */
   &i810_ch_ntsc_huge_b,  &ch_ntsc_huge_b,  
   ch_fsci_ntsc_huge_b,  {sav: 120, hpr:  29, vpr: 251},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW},

  {{TV_SYSTEM_NTSC, 640, 480, "Tiny",   "4:3", 16.667, 16.667}, /* Mode 18 */
   &i810_ch_ntsc_tiny_a,  &ch_ntsc_tiny_a,  
   ch_fsci_ntsc_tiny_a,  {sav:  80, hpr:  55, vpr: 255},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW},
  {{TV_SYSTEM_NTSC, 640, 480, "Small",  "4:3", 12.500, 12.500}, /* Mode 17 */
   &i810_ch_ntsc_small_a, &ch_ntsc_small_a, 
   ch_fsci_ntsc_small_a, {sav:  72, hpr:  46, vpr: 255},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW | TV_CAP_MACROVISION},
  {{TV_SYSTEM_NTSC, 640, 480, "Large",  "4:3",  0.000,  0.000}, /* Mode 16 */
   &i810_ch_ntsc_large_a, &ch_ntsc_large_a, 
   ch_fsci_ntsc_large_a, {sav:  72, hpr:  30, vpr: 255},
   PORT_I810, TV_DEF_CH | TV_DEF_DUALVIEW}, 

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, NULL, {}, 0},
};

/* -------- -------- */

static TVMode *modes_i810_ch = NULL;

static TVMode *loop_create_i810_ch (TVMode *m, TVTemplateI810Ch *t, 
  TVSystem system, int f, char *suffix)
{
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    m->spec = t->spec;
    m->spec.system = system;
    m->descFlags = t->descFlags;
    if (suffix) {
      register char *s;

      s = (char *) malloc ((strlen(t->spec.size) + strlen (suffix) + 1) *
			   sizeof (char));
      m->spec.size = strcat (strcpy (s, t->spec.size), suffix);
    }
    m->regs.devFlags = (t->descFlags & TV_CAP_DUALVIEW) ? 
      (DEV_MONITOR | DEV_TELEVISION) : (DEV_TELEVISION);
    m->regs.portHost = m->regs.portEnc = t->portFlags;
    m->regs.crtc.i810 = *(t->i810);
    m->regs.enc.ch = *(t->ch);
    m->regs.enc.ch.sav = t->adj.sav;
    m->regs.enc.ch.hpr = t->adj.hpr;
    m->regs.enc.ch.vpr = t->adj.vpr;
    if (t->fsci) m->regs.enc.ch.fsci = t->fsci[f];
    data_init_ch (system, &m->regs.enc.ch);
    data_init_i810 (&m->regs.crtc.i810, m->regs.portHost);
    m++;
  }
  return m;
}

TVMode *data_modes_i810_ch (void)
{
  int c;
  TVMode *m;

  if (modes_i810_ch) return modes_i810_ch;
  c = sizeof (templ_i810_ch_60hz) / sizeof(TVTemplateI810Ch) * 5
    + sizeof (templ_i810_ch_50hz) / sizeof(TVTemplateI810Ch) * 2
    + 1;
  modes_i810_ch = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_i810_ch;
  m = loop_create_i810_ch (m, templ_i810_ch_60hz, TV_SYSTEM_NTSC,   0, NULL);
  m = loop_create_i810_ch (m, templ_i810_ch_60hz, TV_SYSTEM_NTSC_J, 0, NULL);
  m = loop_create_i810_ch (m, templ_i810_ch_60hz, TV_SYSTEM_NTSC,   1, "#");
  m = loop_create_i810_ch (m, templ_i810_ch_60hz, TV_SYSTEM_PAL_M,  2, NULL);
  m = loop_create_i810_ch (m, templ_i810_ch_60hz, TV_SYSTEM_PAL_60, 3, NULL);
  m = loop_create_i810_ch (m, templ_i810_ch_50hz, TV_SYSTEM_PAL,    0, NULL);
  m = loop_create_i810_ch (m, templ_i810_ch_50hz, TV_SYSTEM_PAL_N,  1, NULL);
  m->spec.system = TV_SYSTEM_NONE;
  return modes_i810_ch;
}

/* -------- -------- */

void data_init_i810 (TVI810Regs *r, int portHost)
{
  r->tvHBlankStart = r->tvHDisplay;
  r->tvHBlankEnd   = r->tvHTotal;
  r->tvVBlankStart = r->tvVDisplay;
  r->tvVBlankEnd   = r->tvVTotal;
  r->borderRed   = 0;
  r->borderGreen = 0;
  r->borderBlue  = 0;
}

void data_make_i810 (int hdisplay, int hsyncstart, int hsyncend, 
  int htotal, int vdisplay, int vsyncstart, int vsyncend, int vtotal, 
  int dotclock, TVCrtcRegs *crt)
{
}

/* -------- -------- */

DataCardFunc data_i810_func = {
  make: data_make_i810,
};

DataFunc data_i810_ch1_func = {
  modes: data_modes_i810_ch,
  defaults: data_default_ch,
  setup: data_setup_ch, 
  clamp: data_clamp_ch,
  calc:  data_calc_null,
};

