/* NVTV Philips chip data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: data_ph.c,v 1.18 2002/04/15 21:29:37 hsalokor Exp $
 *
 * Contents:
 *
 * Data tables and setup routines for the Philips chip.
 *
 */

#include "local.h" /* before everything else */

#include <stddef.h>
#include <stdlib.h>

#include "data_ph.h"
#include "data_nv.h"

/*
  Variables (see datasheet for more information):
  ---
  adwhs:  Active display window horizontal start
  adwhe:  Active display window horizontal end
  xofs:   Horizontal offset
  xpix:   Horizontal pixel count (pixels / 2 )
  xinc:   Horizontal scaling engine incremental fraction (x 4096)
  hlen:   Horizontal length (PIXCLKs / line - 1 )
  fal:    First active line
  lal:    Last active line
  yinc:   Vertical scaling engine incremental fraction
  yskip:  Vertical line skip
  yofso:  Vectical offset, odd lines
  yofse:  Vertical offset, even lines  
  ypix:   Vertical pixel count
  yiwgto: Weighting factor, odd lines  
  yiwgte: Weighting factor, even lines
  pcl:    Pixel clock value
  idel:   Input delay (don't change, use xofs to adjust hpos)
*/

#define PH_FLAG3_DEF (PH_FLAG3_EIDIV | PH_FLAG3_IFRA)

/* -------- PH -------- NTSC -------- */

/* -------- 640x480 -------- */

TVPh1Regs ph_ntsc_small_a = { /* PH 620x204 Small, Full-Fl, OC 12.67x16.05 */
  adwhs  : 342,  /* hsize: ntsc 620 +0 */
  adwhe  : 1582, /* hsize: ntsc 620 +0 */
  xofs   : 159,  /* verified, largest possible */
  xpix   : 320,
  xinc   : 3968, /* 640->620 */
  hlen   : 799,
  fal    : 37,
  lal    : 241,
  yinc   : 1734,
  yskip  : 0,
  yofso  : 84,
  yofse  : 84,
  ypix   : 480,
  yiwgto : 2941, 
  yiwgte : 866,
  pcl    : 2309218,
  idel   : 0,
};

TVPh1Regs ph_ntsc_medium_a = { /* PH 640x216 +0 Reg, Full-Fl, OC 09.86x11.11 */
  adwhs  : 326,  /* hsize: ntsc 640 +0 */
  adwhe  : 1606, /* hsize: ntsc 640 +0 */
  xofs   : 159,  /* verified, largest possible */
  xpix   : 320,
  xinc   : 0,    /* 640->640 */
  hlen   : 799,
  fal    : 31,
  lal    : 247,
  yinc   : 1836,
  yskip  : 0,
  yofso  : 66,
  yofse  : 66,
  ypix   : 480,
  yiwgto : 2965,
  yiwgte : 917,
  pcl    : 2178859,
  idel   : 0,
};

/* -------- 800x600 -------- */

TVPh1Regs ph_ntsc_small_b = { /* PH 620x204 Small, Full-Fl, OC 12.67x16.05 */
  adwhs  : 346,   /* hsize: ntsc 620 +0 */
  adwhe  : 1586,  /* hsize: ntsc 620 +0 */
  xofs   : 200,   /* test! */
  xpix   : 400,
  xinc   : 3175,  /* 800->620 */
  hlen   : 1023,
  fal    : 37,
  lal    : 241,
  yinc   : 1389,
  yskip  : 0,
  yofso  : 106,
  yofse  : 106,
  ypix   : 600,
  yiwgto : 2742,
  yiwgte : 694,
  pcl    : 3689981,
  idel   : 0,
};

TVPh1Regs ph_ntsc_medium_b = { /* PH 640x216 +0 Reg, Full-Fl, OC 09.86x11.11 */
  adwhs  : 326,  /* hsize: ntsc 640 +0 */
  adwhe  : 1606, /* hsize: ntsc 640 +0 */
  xofs   : 200,  /* test! */
  xpix   : 400,
  xinc   : 3277, /* 800->640 */
  hlen   : 1023,
  fal    : 31,
  lal    : 247,
  yinc   : 1470,
  yskip  : 0,
  yofso  : 82,
  yofse  : 82,
  ypix   : 600,
  yiwgto : 2782,
  yiwgte : 734,
  pcl    : 3484982,
  idel   : 0,
};

TVPh1Regs ph_ntsc_huge_b = { /* PH 710x241 Full, Full-Fl, OC 0.00x00.82 */
  adwhs  : 256,  /* hsize: ntsc 710 +0 */
  adwhe  : 1676, /* hsize: ntsc 710 +0 */
  xofs   : 200,  /* test! */
  xpix   : 400,
  xinc   : 3636, /* 800 -> 710 */
  hlen   : 1023,
  fal    : 18,
  lal    : 259,
  yinc   : 1642,
  yskip  : 0,
  yofso  : 42,
  yofse  : 42,
  ypix   : 600,
  yiwgto : 2867,
  yiwgte : 819,
  pcl    : 3122659,
  idel   : 0,
};

/* -------- PH -------- PAL -------- */

/* -------- 640x480 -------- */

TVPh1Regs ph_pal_small_a = { /* PH 620x250 Small, Full-Fl, OC 11.68x13.19 */
  adwhs  : 358,  /* hsize: pal 620 +0 */
  adwhe  : 1598, /* hsize: pal 620 +0 */
  xofs   : 159,  /* verified, largest possible */
  xpix   : 320,
  xinc   : 3968, /* 640->620 */
  hlen   : 799,
  fal    : 41,
  lal    : 291,
  yinc   : 2125,
  yskip  : 0,
  yofso  : 76,  /* total: 602 */
  yofse  : 76,
  ypix   : 480,
  yiwgto : 3110,
  yiwgte : 1062,
  pcl    : 1870348,
  idel   : 0,
};

TVPh1Regs ph_pal_medium_a = { /* PH 640x259 +0 Reg, Full-Fl, OC 08.83x10.07 */
  adwhs  : 334,  /* hsize: pal 640 +0 */
  adwhe  : 1614, /* hsize: pal 640 +0 */
  xofs   : 159,  /* verified, largest possible */
  xpix   : 320,
  xinc   : 0,    /* 640->640 */
  hlen   : 799,
  fal    : 37,
  lal    : 296,
  yinc   : 2202,
  yskip  : 0,
  yofso  : 66,
  yofse  : 66,
  ypix   : 480,
  yiwgto : 3148,
  yiwgte : 1100,
  pcl    : 1805104,
  idel   : 0,
};

/* -------- 800x600 -------- */

TVPh1Regs ph_pal_small_b = { /* PH 620x250 Small, Full-Fl, OC 11.68x13.19 */
  adwhs  : 364,  /* hsize: pal 620 +0 */
  adwhe  : 1604, /* hsize: pal 620 +0 */
  xofs   : 223,  /* verified, largest possible */
  xpix   : 400,
  xinc   : 3175,  /* 800->620 */
  hlen   : 1023,
  fal    : 41,
  lal    : 291,
  yinc   : 1701,
  yskip  : 0,
  yofso  : 95,
  yofse  : 95,
  ypix   : 600,
  yiwgto : 2898,
  yiwgte : 850,
  pcl    : 2990569,
  idel   : 0,
};

TVPh1Regs ph_pal_medium_b = { /* PH 640x259 +0 Reg, Full-Fl, OC 08.83x10.07 */
  adwhs  : 346,  /* hsize: pal 640 +0 */
  adwhe  : 1626, /* hsize: pal 640 +0 */
  xofs   : 223,  /* verified, largest possible */
  xpix   : 400,
  xinc   : 3277, /* 800->640 */
  hlen   : 1023,
  fal    : 37,
  lal    : 296,
  yinc   : 1763,
  yskip  : 0,
  yofso  : 83,
  yofse  : 83,
  ypix   : 600,
  yiwgto : 2929,
  yiwgte : 881,
  pcl    : 2887172,
  idel   : 0,
};

TVPh1Regs ph_pal_huge_b = { /* PH 702x288 Full, Full-Fl, OC 00.00x00.00 */
  adwhs  : 284,  /* hsize: pal 702 +0 */
  adwhe  : 1688, /* hsize: pal 702 +0 */
  xofs   : 200,  /* test! */
  xpix   : 400,
  xinc   : 3595, /* 800 -> 702 */
  hlen   : 1023,
  fal    : 22,
  lal    : 310,
  yinc   : 1960,
  yskip  : 0,
  yofso  : 43,
  yofse  : 43,
  ypix   : 600,
  yiwgto : 3027,
  yiwgte : 979,
  pcl    : 2596864,
  idel   : 0,
};

/* -------- SAA7104 Test Master modes -------- */

TVPh2Regs ph_ntsc_test_a = { /* 640x480 NV */
  super : {
  adwhs  : 318,
  adwhe  : 1598,
  xofs   : 100,
  xpix   : 320,
  xinc   : 0,
  hlen   : 761,
  fal    : 32,
  lal    : 246,
  yinc   : 1819,
  yskip  : 0,
  yofso  : 69,
  yofse  : 69,
  ypix   : 672,
  yiwgto : 2957,
  yiwgte : 909,
  pcl    : 2097152,  /* 27.000 MHz */
  fsc    : 0,
  idel   : 1,
  },
  pcle   : 1,
  pcli   : 1,
};
  
TVPh2Regs ph_ntsc_test_b = { /* 800x600 NV */
  super : {
  adwhs  : 331,
  adwhe  : 1571,
  xofs   : 98,
  xpix   : 400,
  xinc   : 3175,
  hlen   : 1221,
  fal    : 30,
  lal    : 244,
  yinc   : 1457,
  yskip  : 0,
  yofso  : 81,
  yofse  : 81,
  ypix   : 696,
  yiwgto : 2776,
  yiwgte : 728,
  pcl    : 4194304,  /* 54.000 MHz */ /* Huh? This is pretty high! */
  fsc    : 0,
  idel   : 6,
  },
  pcle   : 1,
  pcli   : 1,
};

TVPh2Regs ph_ntsc_test_c = { /* 1024x768 NV double freq? */
  super : {
  adwhs  : 344,
  adwhe  : 1588,
  xofs   : 59,
  xpix   : 512,
  xinc   : 2488,
  hlen   : 1139,
  fal    : 33,
  lal    : 243,
  yinc   : 1120,
  yskip  : 0,
  yofso  : 117,
  yofse  : 117,
  ypix   : 768,
  yiwgto : 2688,
  yiwgte : 640,
  pcl    : 2547589,  /* 32.799 MHz * 2 = 65.6 MHz */
  fsc    : 569408543,
  idel   : 0,
  },
  pcle   : 0,
  pcli   : 0,
};

/* ---- */

TVPh2Regs ph_pal_test_a = { /* 640x480 NV */
  super : {
  adwhs  : 385,
  adwhe  : 1625,
  xofs   : 104,
  xpix   : 320,
  xinc   : 3969,
  hlen   : 805,
  fal    : 38,
  lal    : 295,
  yinc   : 2185,
  yskip  : 0,
  yofso  : 69,
  yofse  : 69,
  ypix   : 696,
  yiwgto : 3139,
  yiwgte : 1091,
  pcl    : 1835008,  /* 23.625 MHz */
  idel   : 5,
  },
  pcle   : 1,
  pcli   : 1,
};
  
TVPh2Regs ph_pal_test_b = { /* 800x600 NV */
  super : {
  adwhs  : 349,
  adwhe  : 1617,
  xofs   : 102,
  xpix   : 400,
  xinc   : 3247,
  hlen   : 1031,
  fal    : 36,
  lal    : 297,
  yinc   : 1778,
  yskip  : 0,
  yofso  : 80,
  yofse  : 80,
  ypix   : 696,
  yiwgto : 2935,
  yiwgte : 887,
  pcl    : 2883584,  /* 37.125 MHz */
  idel   : 5,
  },
  pcle   : 1,
  pcli   : 1,
};

TVPh2Regs ph_pal_test_c = { /* 1024x768 NV double freq? */
  super : {
  adwhs  : 360,
  adwhe  : 1604,
  xofs   : 158,
  xpix   : 512,
  xinc   : 2488,
  hlen   : 1199,
  fal    : 37,
  lal    : 290,
  yinc   : 1349,
  yskip  : 0,
  yofso  : 102,
  yofse  : 102,
  ypix   : 768,
  yiwgto : 2721,
  yiwgte : 673,
  pcl    : 2213659,  /* 28.500 MHz * 2 = 57.000 MHz */
  fsc    : 705268427,
  idel   : 0,
  },
  pcle   : 0,
  pcli   : 0,
};

/* -------- SAA7104 Test Slave modes -------- */

TVPh2Regs ph_ntsc_slave_a = { /* 640x480 Slave */
  super : {
  adwhs  : 328,
  adwhe  : 1608,
  xofs   : 86,
  xpix   : 320,
  xinc   : 0,
  hlen   : 1071,
  fal    : 24,
  lal    : 248,
  yinc   : 1824,
  yskip  : 0,
  yofso  : 48,
  yofse  : 48,
  ypix   : 637,
  yiwgto : 2960,
  yiwgte : 912,
  pcl    : 2197224,  /* 28.288 MHz */
  fsc    : 569408543,
  idel   : 6,
  },
  pcle   : 1,
  pcli   : 1,
};

TVPh2Regs ph_ntsc_slave_b = { /* 800x600 Slave */
  super : {
  adwhs  : 320,
  adwhe  : 1604,
  xofs   : 72,
  xpix   : 400,
  xinc   : 3288,
  hlen   : 1071,
  fal    : 27,
  lal    : 248,
  yinc   : 1432,
  yskip  : 0,
  yofso  : 70,
  yofse  : 70,
  ypix   : 637,
  yiwgto : 2793,
  yiwgte : 746,
  pcl    : 3715234,  /* 47.832 MHz */ /* too high ? */
  fsc    : 569408543,
  idel   : 6,
  },
  pcle   : 1,
  pcli   : 1,
};

/* ---- */

TVPh2Regs ph_pal_slave_a = { /* 640x480 Slave, from dump */
  super : {
  adwhs  : 348,
  adwhe  : 1628,
  xofs   : 88,
  xpix   : 320,
  xinc   : 0,
  hlen   : 813,
  fal    : 32,
  lal    : 296,
  yinc   : 2136,
  yskip  : 0,
  yofso  : 87,
  yofse  : 87,
  ypix   : 0,   /* ?? */
  yiwgto : 3116,
  yiwgte : 1068,
  pcl    : 1957085,  /* 25.197 MHz */
  fsc    : 705268427,
  idel   : 5,
  },
  pcle   : 1,
  pcli   : 1,
#if 0
  bs     : 33,  /* !! */
  be     : 29,
  bsta   : 47,
  blckl  : 51,
  blnnl  : 53,
  chps   : 136, /* !! */
  gy     : 26,
  gcd    : 26,
  ccrs   : 1,
  gainu  : 125,
  gainv  : 175,
#endif
};

TVPh2Regs ph_pal_slave_b = { /* 800x600 Slave */
  super : {
  adwhs  : 348,
  adwhe  : 1628,
  xofs   : 96,
  xpix   : 399,
  xinc   : 3294,
  hlen   : 1092,
  fal    : 31,
  lal    : 295,
  yinc   : 1778, /* or: 1781 */
  yskip  : 0,
  yofso  : 64,   /* or: 32 */
  yofse  : 64,   /* or: 32 */
  ypix   : 1023, /* ?? */
  yiwgto : 2934,
  yiwgte : 886,
  pcl    : 2796212,   /* 36.000 MHz */
  fsc    : 705268427,
  idel   : 5,
  },
  pcle   : 1,
  pcli   : 1,
#if 0
  bs     : 33,  /* !! */
  be     : 29,
  bsta   : 47,
  blckl  : 51,
  blnnl  : 53,
  chps   : 136, /* !! */
  gy     : 26,
  gcd    : 26,
  ccrs   : 1,
  gainu  : 125,
  gainv  : 175,
#endif
};

/* -------- -------- */

static int clamp_gain[32] = {-100, 
  -1500/16, -1400/16, -1300/16, -1200/16, -1100/16, -1000/16, 
  -900/16, -800/16, -700/16, -600/16, -500/16, -400/16, -300/16, -200/16,
  -100/16, 0, 100/16, 200/16, 300/16, 400/16, 500/16, 600/16, 700/16, 
  800/16, 900/16, 1000/16, 1100/16, 1200/16, 1300/16, 1400/16, 1500/16};

static int clamp_sharpness[2] = {0, 100};
static int clamp_chroma_bandwidth[3] = {0, 50, 100};
static int clamp_cross_color[4] = {0, 50, 80, 100};

void data_init_ph1 (TVSystem system, TVPh1Regs *r)
{
  if (!r) return;
  r->flags1 = 0;
  r->flags2 = PH_FLAG2_NORMAL;
  switch (system) /* 7.5 IRE setup vs. non-setup */
  {
    case TV_SYSTEM_NTSC_J: 
    case TV_SYSTEM_PAL:
    case TV_SYSTEM_PAL_X:
    case TV_SYSTEM_PAL_60: 
    case TV_SYSTEM_PAL_NC: 
      r->gainu = 125; /* @ 0x5b  100IRE: 125 */
      r->gainv = 175; /* @ 0x5c  100IRE: 175 */
      r->blckl = 51;  /* @ 0x5d  143IRE: 51 = 0x33 */
      r->blnnl = 53;  /* @ 0x5e  143IRE: 53 = 0x35 */
      break; 
    case TV_SYSTEM_NTSC:
    case TV_SYSTEM_PAL_N:  
    case TV_SYSTEM_PAL_M:  
    case TV_SYSTEM_PAL_M60:  
      r->flags1 |= PH_FLAG1_YGS; 
      r->gainu = 118; /* @ 0x5b  92.5IRE: 118 */
      r->gainv = 165; /* @ 0x5c  92.5IRE: 165 */
      r->blckl = 58;  /* @ 0x5d  140IRE: 58 = 0x3a */
      r->blnnl = 46;  /* @ 0x5e  140IRE: 46 = 0x2e */
      break;
    case TV_SYSTEM_NONE:
    case TV_SYSTEM_SECAM:
      break; 
  }
  switch (system) 
  {
    case TV_SYSTEM_NTSC:
      r->flags1 |= PH_FLAG1_FISE;
      r->bs = 25;    /* @ 0x28 nstc: 25 */
      r->be = 29;    /* @ 0x29 ntsc: 29 */
      r->bsta = 63;  /* @ 0x62 92.5IRE/NTSC: 63 */
      break;
    case TV_SYSTEM_NTSC_J: 
      r->flags1 |= PH_FLAG1_FISE;
      r->bs = 25;    /* @ 0x28 pal: 33, nstc: 25 FIXME */
      r->be = 29;    /* @ 0x29 pal: 29, ntsc: 29 FIXME */
      r->bsta = 67;  /* @ 0x62 100IRE/NTSC: 67 */
      break;
    case TV_SYSTEM_PAL:
    case TV_SYSTEM_PAL_X:
    case TV_SYSTEM_PAL_N:  
    case TV_SYSTEM_PAL_NC: 
      r->bs = 25;    /* @ 0x28 pal: 33, nstc: 25 FIXME NV does it.*/
      r->be = 29;    /* @ 0x29 pal: 29, ntsc: 29 */
      r->bsta = 47;  /* @ 0x62 100IRE/PAL: 47 */
      break;
    case TV_SYSTEM_PAL_60: 
    case TV_SYSTEM_PAL_M:  
    case TV_SYSTEM_PAL_M60:  
      r->flags1 |= PH_FLAG1_FISE;
      r->bs = 25;    /* @ 0x28 pal: 33, nstc: 25 FIXME */
      r->be = 29;    /* @ 0x29 pal: 29, ntsc: 29 FIXME */
      r->bsta = 45;  /* @ 0x62 92.5IRE/PAL: 45 */
      break;
    case TV_SYSTEM_NONE:
    case TV_SYSTEM_SECAM:
      break;
  }
  /* Need clock frequency with more precision. Used: 37.037 ns. FSC rounded
     to first 6.5 digits, unless specified in data sheet. FIXME: 27 MHz ! */
  switch (system)
  {
    case TV_SYSTEM_NTSC_J: 
    case TV_SYSTEM_NTSC:
      r->fsc = 569408543; /* 3579455    Hz */
      break;
    case TV_SYSTEM_PAL:
    case TV_SYSTEM_PAL_X:
    case TV_SYSTEM_PAL_60: 
    case TV_SYSTEM_PAL_N:  
      r->fsc = 705268427; /* 4433618.75 Hz */
      break;
    case TV_SYSTEM_PAL_NC: 
      r->fsc = 569807000; /* 3582056.25 Hz FIXME: Better precision */
      break;
    case TV_SYSTEM_PAL_M60: /* unkown */
    case TV_SYSTEM_PAL_M:  
      r->fsc = 569418500; /* 3579611.49 Hz FIXME: Better precision */
      break;
    case TV_SYSTEM_NONE:
    case TV_SYSTEM_SECAM:
      break;
  }
  switch (system)
  {
    case TV_SYSTEM_NTSC_J: 
    case TV_SYSTEM_NTSC:
      r->flags1 |= PH_FLAG1_SCBW; /* NV: SCBW !! ?? */
      r->macro = 0; 
      break;
    case TV_SYSTEM_PAL:
    case TV_SYSTEM_PAL_X:
    case TV_SYSTEM_PAL_60: 
    case TV_SYSTEM_PAL_N:  
    case TV_SYSTEM_PAL_NC: 
    case TV_SYSTEM_PAL_M:  
    case TV_SYSTEM_PAL_M60:  
      r->flags1 |= PH_FLAG1_PAL;
      r->macro = 4; 
      break;
    case TV_SYSTEM_NONE:
    case TV_SYSTEM_SECAM:
      break;
  }
  r->bcy = 16;  /* NV: 0x10 */
  r->bcu = 128; /* NV: 0x80 */
  r->bcv = 128; /* NV: 0x80 */
  r->gy = 26;   /* NV: 0x1a */
  r->gcd = 26;  /* NV: 0x1a */
  r->chps = 0;  /* @ 0x5a NV: 0x00 */
  r->ccrs = 0;  /* No cross color reduction */
  r->flc = 0;   /* NV Interlaced */
  r->phres = 0; /* NV No subcarrier reset */
  /* FIXME idel */
}

void data_init_ph2 (TVSystem system, TVPh2Regs *r)
{
  if (!r) return;
  data_init_ph1 (system, &r->super);
  r->yfil = 1; /* before */
  r->fili = 8;
  r->flags3 = PH_FLAG3_DEF;
}

void data_default_ph (TVSettings *s)
{
  s->tv_hoffset = s->mon_hoffset = 0;
  s->tv_voffset = s->mon_voffset = 0;
  s->brightness_sig = 0;
  s->contrast_sig = 0;
  s->contrast = 0;
  s->saturation_sig = 0;
  s->saturation = 0;
  s->flicker = 50;
  s->flicker_adapt = 0;
  s->luma_bandwidth = 50;
  s->chroma_bandwidth = 50;
  s->sharpness = 50;
  s->cross_color = 0;
  s->phase = 0;
  s->hue = 0;
}

void data_clamp_ph (TVSettings *s, TVRegs *r)
{
  /* FIXME */
  if (s->tv_hoffset <= -100) s->tv_hoffset = -100;
  if (s->tv_hoffset >=  100) s->tv_hoffset =  100;
  if (s->tv_voffset <= -100) s->tv_voffset = -100;
  if (s->tv_voffset >=  100) s->tv_voffset =  100;
  if (s->mon_hoffset <= -100) s->mon_hoffset = -100;
  if (s->mon_hoffset >=  100) s->mon_hoffset =  100;
  if (s->mon_voffset <= -100) s->mon_voffset = -100;
  if (s->mon_voffset >=  100) s->mon_voffset =  100;
  if (s->brightness_sig < -50) s->brightness_sig = -50;
  if (s->brightness_sig >  50) s->brightness_sig = 50;
  if (s->saturation_sig < -50) s->saturation_sig = -50;
  if (s->saturation_sig >  50) s->saturation_sig = 50;
  if (s->phase < -180) s->phase = -180;
  if (s->phase >  180) s->phase = 180;
  s->contrast_sig = 0;
  s->hue = 0;
  if (r) {
    s->flicker = 100 - (r->enc.ph1.yskip * 100 / 4095);
  } else {
    s->flicker = 50; 
  }
  s->flicker_adapt = 0; 
  s->luma_bandwidth = 50;
  s->chroma_bandwidth = data_clamp (s->chroma_bandwidth, 3,
				    clamp_chroma_bandwidth); 
  s->sharpness = data_clamp (s->sharpness, 2, clamp_sharpness); 
  s->cross_color = data_clamp (s->cross_color, 4, clamp_cross_color); 
  s->contrast = data_clamp (s->contrast, 32, clamp_gain); 
  s->saturation = data_clamp (s->saturation, 32, clamp_gain); 
}

void data_setup_ph (TVSettings *s, TVRegs *r)
{
  register int i;
  register float f;

  r->enc.ph1.adwhs += s->tv_hoffset;
  r->enc.ph1.adwhe += s->tv_hoffset;

  i = s->brightness_sig + r->enc.ph1.blckl;
  if (i < 0) i = 0;
  if (i > 63) i = 63;
  r->enc.ph1.blckl = i;  /* 6 bits, 0-63 */
  r->enc.ph1.gy = data_pick (s->contrast, 32, clamp_gain); 
  r->enc.ph1.gcd = data_pick (s->saturation, 32, clamp_gain); 
  f = (1.0 + 0.01 * s->saturation_sig) * r->enc.ph1.gainu;
  if (f < 0.0) f = 0.0;
  if (f > 511.0) f = 511.0;
  r->enc.ph1.gainu = (int) f;
  f = (1.0 + 0.01 * s->saturation_sig) * r->enc.ph1.gainv;
  if (f < 0.0) f = 0.0;
  if (f > 511.0) f = 511.0;
  r->enc.ph1.gainv = (int) f;
  i = (int) (s->phase * 256.0 / 360.0) + r->enc.ph1.chps;
  r->enc.ph1.chps = i & 0xff; /* 8 bit */
  r->enc.ph1.ccrs = data_pick (s->cross_color, 4, clamp_cross_color); 
  switch (data_pick (s->chroma_bandwidth, 3, clamp_chroma_bandwidth)) {
    case 0: r->enc.ph1.flags1 &= ~PH_FLAG1_SCBW; break;
    case 2: r->enc.ph1.flags1 |= PH_FLAG1_SCBW; break;
  }
  /* scbw will be set with tv system; 0 disables, 50 keeps, 100 enables */
  switch (data_pick (s->sharpness, 2, clamp_sharpness)) {
    case 0: r->enc.ph1.flags1 &= ~PH_FLAG1_YFIL; break;
    case 1: r->enc.ph1.flags1 |= PH_FLAG1_YFIL; break;
  }
  r->enc.ph1.macro = (r->enc.ph1.macro & ~0x3) | 
                ((s->flags & TV_DESC_MACROVISION) ? 1 : 0);
  r->portEnc &= ~PORT_PCLK_POLARITY;
  if (s->flags & TV_DESC_COLORFIX) {
    r->portEnc |= PORT_PCLK_LOW;
  } else {
    r->portEnc |= PORT_PCLK_HIGH;
  }
  if (s->flags & TV_DESC_CARRIER_LOCK) {
    r->enc.ph1.phres = 3;
  } else {
    r->enc.ph1.phres = 0;
  }
  if (s->flags & TV_DESC_NONINTERLACED) {
    r->enc.ph1.flc = 1; /* FIXME test */
  } else {
    r->enc.ph1.flc = 0;
  }
  /* FIXME connector */
  switch (s->connector)
  {
    case CONNECT_CONVERT:
      r->enc.ph1.flags2 = PH_FLAG2_CONVERT;
      break;
    case CONNECT_COMPOSITE: 
    case CONNECT_SVIDEO: 
    case CONNECT_BOTH: 
    default:
      r->enc.ph1.flags2 = PH_FLAG2_NORMAL;
      break;
  }
  /* FIXME CAN_CARRIER, ... */
}

/*
bit2 pin27: red,  cr, c(svhs)
bit1 pin28: green,    cvbs, y/vbs(svhs)
bit0 pin30: blue, cb, cvbs(fbas)

fbas:     cvbsen0=1
shvhs:   (cvbsen0=1) cvbsen1=0, vbsen=1, cen=1
convert: (cvbsen0=1) cvbsen1=1, vbsen=0 (cen=1)

no setup:  100IRE
setup:     92.5IRE + 7.5setup 

See table A-1 of CX docs for TV System characteristics

NTSC-M, PAL-M and PAL-N have setup (92.5IRE?)

*/
