/* NVTV 3dfx CRTC data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Data routines for the 3dfx CRTC data.
 *
 */

#include <stddef.h>
#include <stdlib.h>

#include "data_tdfx.h"
#include "data_bt.h"
#include "calc_bt.h"

/* -------- BT ------------------------ */

#define TV_DEF_BT (TV_CAP_MACROVISION | TV_CAP_NONINTERLACED | \
		   TV_CAP_MONOCHROME)

/* -------- BT -------- NTSC -------- */

TVTdfxRegs tdfx_bt_ntsc_small_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 89,       /* tuned h_blanki=126 */
  tvVBlankStart : 75,
};

TVTdfxRegs tdfx_bt_ntsc_normal_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 102,       /* tuned h_blanki=140 */
  tvVBlankStart : 58,
};

TVTdfxRegs tdfx_bt_ntsc_huge_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 101,       /* tuned h_blanki=133 */
  tvVBlankStart : 36,        /* tuned v_blanki=36 */
};

TVTdfxRegs tdfx_bt_ntsc_small_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 29,       /* tuned h_blanki=66 */
  tvVBlankStart	: 86, 
};

TVTdfxRegs tdfx_bt_ntsc_normal_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 0,        /* tuned h_blanki=26 hsynoffset=-12 */
  tvVBlankStart	: 60, 
};

TVTdfxRegs tdfx_bt_ntsc_huge_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 0,        /* tuned h_blanki=26 hsynoffset=-12 */
  tvVBlankStart	: 42,       
};

TVTdfxRegs tdfx_bt_ntsc_normal_c = { 
  HScreenSize   : 720,
  VScreenSize   : 480,
  tvHBlankStart : 116,      /* tuned h_blanki=154 hsynoffset=0 */
  tvVBlankStart	: 75, 
};

TVTdfxRegs tdfx_bt_ntsc_dvd_a = { 
  HScreenSize   : 720,
  VScreenSize   : 480,
  tvHBlankStart : 106,      /* tuned h_blanki=140 */
  tvVBlankStart	: 36, 
};

/* -------- BT -------- PAL -------- */

TVTdfxRegs tdfx_bt_pal_small_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 228,      /* tuned h_blanki=266 */
  tvVBlankStart : 90,
};

TVTdfxRegs tdfx_bt_pal_normal_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 278,      /* tuned h_blanki=315 */
  tvVBlankStart : 57,
};

TVTdfxRegs tdfx_bt_pal_huge_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 260,      /* tuned h_blanki=299/312 */
  tvVBlankStart : 36,
};

TVTdfxRegs tdfx_bt_pal_small_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 98,       /* tuned h_blanki=135/140 */
  tvVBlankStart	: 95, 
};

TVTdfxRegs tdfx_bt_pal_normal_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 138,      /* tuned h_blanki=175/100,140 */
  tvVBlankStart	: 64, 
};

TVTdfxRegs tdfx_bt_pal_large_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 162,      /* tuned h_blanki=190 hsynoffset=-8 */
  tvVBlankStart	: 51, 
};

TVTdfxRegs tdfx_bt_pal_huge_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 63,       /* tuned h_blanki=101 hsynoffset=14 */
  tvVBlankStart	: 46,       /* tuned v_blanki=133 */
};

TVTdfxRegs tdfx_bt_pal_small_c = { 
  HScreenSize   : 768,
  VScreenSize   : 576,
  tvHBlankStart : 166,      /* tuned h_blanki=203 */
  tvVBlankStart	: 71, 
};

TVTdfxRegs tdfx_bt_pal_normal_c = { 
  HScreenSize   : 768,
  VScreenSize   : 576,
  tvHBlankStart : 165,      /* tuned h_blanki=203 */
  tvVBlankStart	: 58, 
};

TVTdfxRegs tdfx_bt_pal_dvd_a = { 
  HScreenSize   : 720,
  VScreenSize   : 576,
  tvHBlankStart : 89,      /* tuned h_blanki=147 hsynoffset=4 */
  tvVBlankStart	: 32,      /* tuned v_blanki=42 */
};

/* -------- BT -------- PAL-60 -------- */

TVTdfxRegs tdfx_bt_pal60_small_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 103,      /* tuned h_blanki=140 */
  tvVBlankStart : 81,       /* tuned v_blank=81 */
};

TVTdfxRegs tdfx_bt_pal60_normal_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 108,      /* tuned h_blanki=145/140 */
  tvVBlankStart : 58,
};

TVTdfxRegs tdfx_bt_pal60_huge_a = { 
  HScreenSize   : 640,
  VScreenSize   : 480,
  tvHBlankStart : 92,       /* tuned h_blanki=126 */
  tvVBlankStart : 23,       /* tuned v_blanki=36 */
};

TVTdfxRegs tdfx_bt_pal60_small_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 28,       /* tuned h_blanki=66 */
  tvVBlankStart	: 86, 
};

TVTdfxRegs tdfx_bt_pal60_normal_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 0,        /* tuned h_blanki=26 (not optimal) */
  tvVBlankStart	: 60, 
};

TVTdfxRegs tdfx_bt_pal60_huge_b = { 
  HScreenSize   : 800,
  VScreenSize   : 600,
  tvHBlankStart : 0,        /* tuned h_blanki=26 (not optimal) */
  tvVBlankStart	: 44,       
};

TVTdfxRegs tdfx_bt_pal60_dvd_a = { 
  HScreenSize   : 720,
  VScreenSize   : 480,
  tvHBlankStart : 106,      /* tuned h_blanki=140 */
  tvVBlankStart	: 36, 
};

/* -------- TDFX -------- Mode list -------- */

typedef struct {
  TVModeSpec spec;
  TVTdfxRegs *tdfx;
  TVBtRegs *bt;
  int hsynoffset;
  int flags;
} TVTemplateTdfxBt;

TVTemplateTdfxBt templ_tdfx_bt [] = {

  {{TV_SYSTEM_NTSC, 640, 480, "Small",  "4:3", 13.79, 13.58},
   &tdfx_bt_ntsc_small_a,  &bt_ntsc_small_a,    0, TV_DEF_BT},
  {{TV_SYSTEM_NTSC, 640, 480, "Normal", "4:3", 10.59, 08.23},
   &tdfx_bt_ntsc_normal_a, &bt_ntsc_normal_a,   0, TV_DEF_BT},
  {{TV_SYSTEM_NTSC, 640, 480, "Huge",   "4:3", 02.46, 01.23}, 
   &tdfx_bt_ntsc_huge_a,   &bt_ntsc_huge_a,     0, TV_DEF_BT},

  {{TV_SYSTEM_NTSC, 800, 600, "Small",  "4:3", 21.62, 11.52}, 
   &tdfx_bt_ntsc_small_b,  &bt_ntsc_small_b,    0, TV_DEF_BT},
  {{TV_SYSTEM_NTSC, 800, 600, "Normal", "4:3", 11.90, 05.35}, 
   &tdfx_bt_ntsc_normal_b, &bt_ntsc_normal_b, -12, TV_DEF_BT},
  {{TV_SYSTEM_NTSC, 800, 600, "Huge",   "4:3", 07.15, 0.004},
   &tdfx_bt_ntsc_huge_b,   &bt_ntsc_huge_b,     0, TV_DEF_BT},

  {{TV_SYSTEM_NTSC, 720, 480, "Normal", "DVD", 08.762, 18.107},
   &tdfx_bt_ntsc_normal_c, &bt_ntsc_normal_c,   0, TV_DEF_BT},
  {{TV_SYSTEM_NTSC, 720, 480, "DVD",    "DVD", 01.245, 01.235},
   &tdfx_bt_ntsc_dvd_a,    &bt_ntsc_dvd_a,      0, TV_DEF_BT},

  {{TV_SYSTEM_PAL, 640, 480, "Small",  "4:3", 16.56, 16.67},
   &tdfx_bt_pal_small_a,  &bt_pal_small_a,    0, TV_DEF_BT},
  {{TV_SYSTEM_PAL, 640, 480, "Normal", "4:3", 12.87, 07.64}, 
   &tdfx_bt_pal_normal_a, &bt_pal_normal_a,   0, TV_DEF_BT},
  {{TV_SYSTEM_PAL, 640, 480, "Huge",   "4:3", 06.22, 00.69},
   &tdfx_bt_pal_huge_a,   &bt_pal_huge_a,     0, TV_DEF_BT},

  {{TV_SYSTEM_PAL, 800, 600, "Small",  "4:3", 14.53, 13.19},
   &tdfx_bt_pal_small_b,  &bt_pal_small_b,    0, TV_DEF_BT},
  {{TV_SYSTEM_PAL, 800, 600, "Normal", "4:3", 10.81, 05.56},
   &tdfx_bt_pal_normal_b, &bt_pal_normal_b,   0, TV_DEF_BT},
  {{TV_SYSTEM_PAL, 800, 600, "Large",  "4:3", 07.461, 02.083},
   &tdfx_bt_pal_large_b,  &bt_pal_large_b,   -8, TV_DEF_BT},
  {{TV_SYSTEM_PAL, 800, 600, "Huge",   "4:3", 00.039, 00.00},
   &tdfx_bt_pal_huge_b,   &bt_pal_huge_b,    14, TV_DEF_BT},

  {{TV_SYSTEM_PAL, 768, 576, "Small",  "4:3", 13.122, 07.986},
   &tdfx_bt_pal_small_c,  &bt_pal_small_c,    0, TV_DEF_BT},
  {{TV_SYSTEM_PAL, 768, 576, "Normal", "4:3", 09.806, 04.514},
   &tdfx_bt_pal_normal_c, &bt_pal_normal_c,   0, TV_DEF_BT},
  {{TV_SYSTEM_PAL, 720, 576, "DVD",    "DVD", 00.208, 00.000},
   &tdfx_bt_pal_dvd_a,    &bt_pal_dvd_a,      4, TV_DEF_BT},

  {{TV_SYSTEM_PAL_60, 640, 480, "Small",  "4:3", 16.76, 14.81},
   &tdfx_bt_pal60_small_a,  &bt_pal60_small_a,    0, TV_DEF_BT},
  {{TV_SYSTEM_PAL_60, 640, 480, "Normal",  "4:3", 10.59, 08.23},
   &tdfx_bt_pal60_normal_a, &bt_pal60_normal_a,   0, TV_DEF_BT},
  {{TV_SYSTEM_PAL_60, 640, 480, "Huge",  "4:3", 01.47, 01.23},
   &tdfx_bt_pal60_huge_a,   &bt_pal60_huge_a,     0, TV_DEF_BT},

  {{TV_SYSTEM_PAL_60, 800, 600, "Small",  "4:3", 21.62, 11.52}, 
   &tdfx_bt_pal60_small_b,  &bt_pal60_small_b,    0, TV_DEF_BT},
  {{TV_SYSTEM_PAL_60, 800, 600, "Normal", "4:3", 11.90, 05.35}, 
   &tdfx_bt_pal60_normal_b, &bt_pal60_normal_b,   0, TV_DEF_BT},
  {{TV_SYSTEM_PAL_60, 800, 600, "Huge",   "4:3", 07.86, 00.82},
   &tdfx_bt_pal60_huge_b,   &bt_pal60_huge_b,     0, TV_DEF_BT},

  {{TV_SYSTEM_PAL_60, 720, 480, "DVD", "DVD", 01.245, 01.235},
   &tdfx_bt_pal60_dvd_a, &bt_pal60_dvd_a,   0, TV_DEF_BT},

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, 0, 0},
};

/* -------- -------- */

static TVMode *modes_tdfx_bt = NULL;

/*
 * count number of modes matching system; TV_SYSTEM_NONE matches all.
 */

int data_count_tdfx_bt (TVSystem system)
{
  int count;
  TVTemplateTdfxBt *t;

  count = 0;
  for (t = templ_tdfx_bt; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (system == TV_SYSTEM_NONE || system == t->spec.system) count++;
  }
  return count;
}

TVMode *data_modes_tdfx_bt (void)
{
  int c;
  TVMode *m;
  TVTemplateTdfxBt *t;

  if (modes_tdfx_bt) return modes_tdfx_bt;
  c = data_count_tdfx_bt (TV_SYSTEM_NONE) + 1;    
  modes_tdfx_bt = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_tdfx_bt;
  for (t = templ_tdfx_bt; t->spec.system != TV_SYSTEM_NONE; t++) {
    m->spec = t->spec;
    m->descFlags = t->flags;
    m->regs.portHost = m->regs.portEnc = PORT_TDFX;
    m->regs.devFlags = DEV_TELEVISION;
    m->regs.crtc.tdfx = *(t->tdfx);
    m->regs.enc.bt = *(t->bt);
    m->regs.enc.bt.hsynoffset = t->hsynoffset;
    data_mux_tdfx_bt (&m->regs.enc.bt);
    data_init_nomux_bt (m->spec.system, &m->regs.enc.bt);
    data_init_tdfx (&m->regs.crtc.tdfx, m->regs.portHost);
    m++;
  }
  m->spec.system = TV_SYSTEM_NONE;
  return modes_tdfx_bt;
}

/* -------- -------- */

void data_init_tdfx (TVTdfxRegs *r, int portHost)
{
  r->tvHBlankEnd = r->tvHBlankStart + r->HScreenSize;
  r->tvVBlankEnd = r->tvVBlankStart + r->VScreenSize;
  r->tvBlankDelay = 0;
  r->tvSyncDelay = 0;
  r->tvLatency = 0;
  if ((portHost & PORT_SYNC_DIR) == PORT_SYNC_MASTER) 
  {
    r->clock = 0;
    r->HDisplay = 0;
    r->HBlankStart = 0;
    r->HSyncStart = 0;
    r->HSyncEnd = 0;
    r->HBlankEnd = 0;
    r->HTotal = 0;
    r->VDisplay = 0;
    r->VBlankStart = 0;
    r->VSyncStart = 0;
    r->VSyncEnd = 0;
    r->VBlankEnd = 0;
    r->VTotal = 0;
  }
  r->flags = 0;
}

void data_make_tdfx (int hdisplay, int hsyncstart, int hsyncend, 
  int htotal, int vdisplay, int vsyncstart, int vsyncend, int vtotal, 
  int dotclock, TVCrtcRegs *crt)
{
  crt->tdfx.clock      = dotclock;
  crt->tdfx.HDisplay   = hdisplay;  
  crt->tdfx.HSyncStart = hsyncstart;
  crt->tdfx.HSyncEnd   = hsyncend;  
  crt->tdfx.HTotal     = htotal;    
  crt->tdfx.VDisplay   = vdisplay;
  crt->tdfx.VSyncStart = vsyncstart;
  crt->tdfx.VSyncEnd   = vsyncend;  
  crt->tdfx.VTotal     = vtotal; 
  crt->tdfx.HScreenSize   = hdisplay;
  crt->tdfx.VScreenSize	  = vdisplay;
  crt->tdfx.tvHBlankStart = 0;
  crt->tdfx.tvHBlankEnd	  = 0;
  crt->tdfx.tvVBlankStart = 0;
  crt->tdfx.tvVBlankEnd   = 0;
  crt->tdfx.tvBlankDelay = 0;
  crt->tdfx.tvSyncDelay = 0;
  crt->tdfx.flags = 0;
  if (dotclock > 135000) {
    crt->tdfx.flags |= TDFX_FLAG_CLOCK2X;
    crt->tdfx.HDisplay    /= 2;
    crt->tdfx.HSyncStart  /= 2;
    crt->tdfx.HSyncEnd    /= 2;
    crt->tdfx.HTotal      /= 2;
    crt->tdfx.HScreenSize /= 2;
  }
}

/* -------- -------- */

void calc_tdfx_btcx (int hres, int vres, TVRegs *r)
{
  register TVTdfxRegs *c = &r->crtc.tdfx;

  r->devFlags = DEV_TELEVISION | DEV_MONITOR;
  c->HScreenSize = hres;
  c->VScreenSize = vres;
  c->tvHBlankStart = r->enc.bt.h_blanki - 38;
  c->tvVBlankStart = r->enc.bt.v_blanki;
  if (c->tvHBlankStart < 0) {
    r->enc.bt.hsynoffset = c->tvHBlankStart;
    c->tvHBlankStart = 0;
  }
  data_init_tdfx (c, PORT_TDFX);
}

void data_calc_tdfx_bt (TVSystem system, int hres, int vres, 
  double hoc, double voc, TVRegs *r)
{
  recalc_bt_custom (system, hres, vres, hoc, voc, r);
  calc_tdfx_btcx (hres, vres, r); 
  data_mux_tdfx_bt (&r->enc.bt);
  data_init_nomux_bt (system, &r->enc.bt);
  r->portHost = r->portEnc = PORT_TDFX;
}

void data_setup_tdfx_bt (TVSettings *s, TVRegs *r)
{
  data_mux_tdfx_bt (&r->enc.bt);
  data_setup_nomux_bt (s, r);
}

/* -------- -------- */

DataCardFunc data_tdfx_func = {
  make: data_make_tdfx,
};

DataFunc data_tdfx_bt_func = {
  modes: data_modes_tdfx_bt,
  defaults: data_default_bt,
  setup: data_setup_tdfx_bt, 
  clamp: data_clamp_bt,
  calc:  data_calc_tdfx_bt,
};

