/* NVTV XBox CRTC/FP data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Data routines for the XBox CRTC/FP data.
 *
 */

#include "local.h" /* before everything else */

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include "data_bt.h"
#include "data_cx.h"
#include "data_ch.h"
#include "data_ph.h"
#include "data_xbox.h"
#include "calc_bt.h"

/* ======== CX ================================ */

#define TV_DEF_CX (TV_CAP_MACROVISION | TV_CAP_NONINTERLACED \
                   | TV_CAP_MONOCHROME)

/* -------- CX -------- NTSC -------- */

/* -------- 640x480 -------- */

TVNvRegs xbox_cx_ntsc_bios = { /* BIOS defaults */
  HDisplay    : 640,
  HBlankStart : 640,
  HBlankEnd   : 728, /* originally 744, but will be modified anyway */
  HSyncStart  : 640,
  HSyncEnd    : 672,
  HTotal      : 728,
  VDisplay    : 480,
  VBlankStart : 480,
  VBlankEnd   : 525,
  VSyncStart  : 492,
  VSyncEnd    : 495,
  VTotal      : 525,
  flags       : 0,
  clock       : 0,
  fp : {
  HDisplay    : 640-1,
  HSyncStart  : 653,
  HSyncEnd    : 685,
  HTotal      : 776-1,
  HValidStart : 0,
  HValidEnd   : 480-1,
  HCrtc       : 600-1,
  VDisplay    : 480-1,
  VSyncStart  : 490,
  VSyncEnd    : 493,
  VTotal      : 525-1,
  VValidStart : 0,
  VValidEnd   : 480-1,
  VCrtc       : 480-1
  }
};

/* -------- CX modes -------- */

typedef struct {
  TVModeSpec spec;
  TVNvRegs *nv;
  TVCxRegs *cx;
  int hostFlags;
  int encFlags;
  int descFlags;
} TVTemplateXBoxCx;

TVTemplateXBoxCx templ_xbox_cx [] = {
  {{TV_SYSTEM_NTSC, 640, 480, "BIOS",  "4:3", 00.00, 00.00},
   &xbox_cx_ntsc_bios,  &cx_ntsc_xbox,  
   PORT_XBOX, PORT_XBOX, TV_DEF_CX},

  {{TV_SYSTEM_NONE, 0, 0, NULL, NULL, 0.0, 0.0}, NULL, NULL, 0},
};

/* -------- -------- */

static TVMode *modes_xbox_cx = NULL;

/*
 * count number of modes matching system; TV_SYSTEM_NONE matches all.
 */

int data_count_xbox_cx (TVTemplateXBoxCx *t, TVSystem system)
{
  int count;

  count = 0;
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (system == TV_SYSTEM_NONE || system == t->spec.system) count++;
  }
  return count;
}

static TVMode *loop_create_xbox_cx (TVMode *m, TVTemplateXBoxCx *t, 
  Bool convert)
{
  for (; t->spec.system != TV_SYSTEM_NONE; t++) {
    if (!convert || t->spec.system == TV_SYSTEM_PAL) 
    {
      m->spec = t->spec;
      if (convert) m->spec.system = TV_SYSTEM_SECAM;
      m->descFlags = t->descFlags;
      m->regs.portHost = t->hostFlags;
      m->regs.portEnc = t->encFlags;
      m->regs.devFlags = DEV_TELEVISION | DEV_FLATPANEL;
      m->regs.crtc.nv = *(t->nv);
      m->regs.enc.cx = *(t->cx);
      data_init_cx (m->spec.system, &m->regs.enc.cx, FALSE);
      data_init_xbox (&m->regs.crtc.nv, m->regs.portHost, m->regs.devFlags);
      if (convert) data_secam_cx (&m->regs.enc.cx);
      m++;
    }
  }
  return m;
}

TVMode *data_modes_xbox_cx (void)
{
  int c;
  TVMode *m;

  if (modes_xbox_cx) return modes_xbox_cx;
  c = data_count_xbox_cx (templ_xbox_cx, TV_SYSTEM_NONE)
    + data_count_xbox_cx (templ_xbox_cx, TV_SYSTEM_PAL)    /* SECAM */
    + 1;                                  /* final zero */
  modes_xbox_cx = (TVMode *) malloc (sizeof (TVMode) * c);
  
  m = modes_xbox_cx;
  m = loop_create_xbox_cx (m, templ_xbox_cx, FALSE);
  m = loop_create_xbox_cx (m, templ_xbox_cx, TRUE);
  m->spec.system = TV_SYSTEM_NONE;
  return modes_xbox_cx;
}

/* ======== ================================ */

void data_init_xbox (TVNvRegs *r, int portHost, int devFlags)
{
  r->HBlankStart = r->HDisplay;
  r->HBlankEnd   = r->HTotal;
  r->VBlankStart = r->VDisplay;
  r->VBlankEnd   = r->VTotal;
  r->latency = 0;
  r->slave.HSyncStart = 0;
  r->slave.HSyncEnd = 0;
  r->slave.HTotal = 0;
  r->slave.VSyncStart = 0;
  r->slave.VSyncEnd = 0;
  r->slave.VTotal = 0;
  r->slave.Unknown = 0;
  r->flags = 0; /* FIXME */
}

void data_move_xbox (TVSettings *s, TVRegs *r)
{
  /* FIXME */
  /* Probably have to adjust FP registers */
}

void data_make_xbox (int hdisplay, int hsyncstart, int hsyncend, 
  int htotal, int vdisplay, int vsyncstart, int vsyncend, int vtotal, 
  int dotclock, TVCrtcRegs *crt)
{
  crt->nv.clock      = dotclock;
  crt->nv.HDisplay   = hdisplay;  
  crt->nv.HSyncStart = hsyncstart;
  crt->nv.HSyncEnd   = hsyncend;  
  crt->nv.HTotal     = htotal;    
  crt->nv.VDisplay   = vdisplay;
  crt->nv.VSyncStart = vsyncstart;
  crt->nv.VSyncEnd   = vsyncend;  
  crt->nv.VTotal     = vtotal;    
  data_init_xbox (&crt->nv, PORT_XBOX, DEV_MONITOR);
}

/* -------- -------- */

void data_calc_xbox_cx (TVSystem system, int hres, int vres, 
  double hoc, double voc, TVRegs *r)
{
  recalc_cx_custom (system, hres, vres, hoc, voc, r);
  calc_nv_btcx (hres, vres, r); 
  data_init_cx (system, &r->enc.cx, 0);
  r->portHost = r->portEnc = PORT_XBOX;
}

/* -------- -------- */

void data_setup_xbox_cx (TVSettings *s, TVRegs *r)
{
  data_move_xbox (s, r); 
  /* FIXME Maybe movement can be done just with FP registers, 
     without changing the encoder regs? */
}

/* -------- -------- */

DataCardFunc data_xbox_func = {
  make: data_make_xbox,
};

/* -------- -------- */

DataFunc data_xbox_cx_func = {
  modes: data_modes_xbox_cx,
  defaults: data_default_cx,
  setup: data_setup_xbox_cx, 
  clamp: data_clamp_cx,
  calc:  data_calc_xbox_cx,
};

