/* NVTV GUI (Chrontel part) -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * The GTK graphical user interface. Chrontel part.
 */

#include "local.h" /* before everything else */

#include <gtk/gtk.h>

#include "gui.h"
#include "gui_ch.h"
#include "backend.h"
#include "data_ch.h"

static GtkAdjustment *update_ch;

static gint gui_timeout_ch_id = -1; /* status timeout */

#define FIELD(b,m) addr:&(b.m), size:sizeof(b.m)
#define FIELD_CH(m) FIELD(gui_regs.enc.ch,m)

/* -------- GUI Masks -------- */

struct mask_ch_freq {
  GtkLabel *clk, *hsyn, *vsyn, *hover, *vover, *aspect;
  GtkLabel *pixels, *lines;
};

struct mask_ch_freq gui_mask_ch_freq;

struct mask_civ {
  GtkLabel *low, *high;
};

struct mask_civ gui_mask_civ;

static GuiRegMask ch_mask1_reg [] = {
  {label:"dmr-ir:",    bits:3,  tick:1, FIELD_CH(dmr_ir)},
  {label:"dmr-vs:",    bits:2,  tick:1, FIELD_CH(dmr_vs)},
  {label:"dmr-sr:",    bits:3,  tick:1, FIELD_CH(dmr_sr)},
  {label:"pll-m:",     bits:9,  tick:1, FIELD_CH(pll_m)},
  {label:"pll-n:",     bits:10, tick:1, FIELD_CH(pll_n)},
  {label:"pllcap:",    bits:1,  tick:1, FIELD_CH(pllcap)},
  {label:"dacg:",      bits:1,  tick:1, FIELD_CH(dacg)},
  {label:"blr:",       bits:8,  tick:1, FIELD_CH(blr)},
  {label:"ce:",        bits:3,  tick:1, FIELD_CH(ce)},
  {label:"sav:",       bits:9,  tick:1, FIELD_CH(sav)},
  {label:"hpr:",       bits:9,  tick:1, FIELD_CH(hpr)},
  {label:"vpr:",       bits:9,  tick:1, FIELD_CH(vpr)},
  {label:"civh:",      bits:2,  tick:1, FIELD_CH(civh)},
  {label:"mode:",      bits:5,  tick:1, FIELD_CH(mode)},
  {label:"macro:",     bits:2,  tick:1, FIELD_CH(macro)},
  {label:NULL}
};

static GuiFlagMask ch_mask1_flag [] = {
  {label:"pd0",    mask:CH_FLAG_DAC_PD0, FIELD_CH(flags)}, 
  {label:"pd1",    mask:CH_FLAG_DAC_PD1, FIELD_CH(flags)}, 
  {label:"poutp",  mask:CH_FLAG_POUTP,   FIELD_CH(flags)}, 
  {label:"aciv",   mask:CH_FLAG_ACIV,    FIELD_CH(flags)}, 
  {label:"cfrb",   mask:CH_FLAG_CFRB,    FIELD_CH(flags)}, 
  {label:"cvbw",   mask:CH_FLAG_CVBW,    FIELD_CH(flags)}, 
  {label:"scart",  mask:CH_FLAG_SCART,   FIELD_CH(flags)}, 
  {label:NULL }
};

static GuiRegMask ch_mask1_twin [] = {
  {label:"fsci:",      bits:32, tick:1, FIELD_CH(fsci)},
  {label:NULL}
};

static GuiRegMask ch_mask2_reg [] = {
  {label:"vbw-flff:",  bits:1,  tick:1, FIELD_CH(vbw_flff)},  
  {label:"vbw-cbw:",   bits:2,  tick:1, FIELD_CH(vbw_cbw)},   
  {label:"vbw-ypeak:", bits:1,  tick:1, FIELD_CH(vbw_ypeak)}, 
  {label:"vbw-ysv:",   bits:2,  tick:1, FIELD_CH(vbw_ysv)},   
  {label:"vbw-ycv:",   bits:1,  tick:1, FIELD_CH(vbw_ycv)},
  {label:"ffr-fc:",    bits:2,  tick:1, FIELD_CH(ffr_fc)},
  {label:"ffr-fy:",    bits:2,  tick:1, FIELD_CH(ffr_fy)},
  {label:"ffr-ft:",    bits:2,  tick:1, FIELD_CH(ffr_ft)},
  {label:NULL}
};

/* -------- -------- */

void reset_ch_cb (GtkWidget *widget, gpointer data)
{
  if ((gui_tv_chip & TV_ENCODER) == TV_CHRONTEL_MODEL1 && gui_act_mode) { 
    gui_regs.enc.ch = gui_act_mode->regs.enc.ch; 
    gui_tv_set ();
    gtk_signal_emit_by_name (GTK_OBJECT (update_ch), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
  }
}

void update_ch_cb (GtkWidget *widget, gpointer data)
{
  if ((gui_tv_chip & TV_ENCODER) == TV_CHRONTEL_MODEL1) { 
    gtk_signal_emit_by_name (GTK_OBJECT (update_ch), "changed");
  }
}

void ch_freq_calc_cb (GtkObject *obj, struct mask_ch_freq *m)
{
  int alo, tlo;
  double ato, tto;
  int tpl, tlf; /* total pixels/line, total lines/field */
  double sc;
  double hoc, voc, aspect;
  double Fref, Fpixel;
  double Fhsyn;
  double Fvsyn;
  int res_x, res_y;
  char s [20];

  static int tlf_pal[8]  = {500,625, -1,750,836, -1,-1,-1};
  static int tlf_ntsc[8] = {420,525,600,630,700,750,-1,-1};
  static double scale[8] = {5/4.0, 1/1.0, 7/8.0, 5/6.0, 3/4.0, 7/10.0, -1, -1};

  if ((gui_tv_chip & TV_ENCODER) != TV_CHRONTEL_MODEL1) return;
  Fref = 14.31818; /* MHz */
  switch (gui_regs.enc.ch.dmr_vs) {
    case 1: /* TV_SYSTEM_NTSC */
    case 3: /* TV_SYSTEM_NTSC_J */
      alo = 480;
      tlo = 262 * 2 + 1;
      ato = 52.65556; /* us */
      tto = 63.55556; /* us */ /* 63.55556;  */
      tlf = tlf_ntsc[gui_regs.enc.ch.dmr_sr];
      break;
    case 0: /* TV_SYSTEM_PAL */
    case 2: /* TV_SYSTEM_PAL_M */
    default:
      alo = 576; 
      tlo = 312 * 2 + 1;
      ato = 52.0; /* us */
      tto = 64.0; /* us */
      tlf = tlf_pal[gui_regs.enc.ch.dmr_sr];
      break;
  }
  switch (gui_regs.enc.ch.dmr_ir) {
    case 0: res_x = 512; res_y = 384; break;
    case 1: res_x = 720; res_y = 400; break;
    case 2: res_x = 640; res_y = 400; break;
    case 3: res_x = 640; res_y = 480; break;
    case 4: res_x = 800; res_y = 600; break;
    case 5: res_x = 720; res_y = alo; break;
    default: res_x = 1; res_y = 1;
  }
  Fpixel = Fref * (gui_regs.enc.ch.pll_n + 2.0) / (gui_regs.enc.ch.pll_m + 2.0);
  sprintf (s, "%3.2f MHz", Fpixel);
  gtk_label_set_text (m->clk, s);

  sc = scale[gui_regs.enc.ch.dmr_sr];
  tpl = (int) ((tto / 2.0) / (1.0 / Fpixel) * sc + 0.5);
  hoc = 1.0 - ((double) res_x / tpl) / ((double) ato / tto);
  voc = 1.0 - ((double) res_y / tlf) / ((double) alo / tlo);
  aspect = 1.0; /* FIXME TODO */

  Fhsyn = 1e3 * Fpixel / tpl;
  snprintf (s, 20, "%3.2f kHz", Fhsyn);
  gtk_label_set_text (m->hsyn, s);

  Fvsyn = 1e3 * Fhsyn / tlf;
  snprintf (s, 20, "%3.2f  Hz", Fvsyn);
  gtk_label_set_text (m->vsyn, s);

  snprintf (s, 20, "%06.3f %%", hoc * 100.0);
  gtk_label_set_text (m->hover, s);
  snprintf (s, 20, "%06.3f %%", voc * 100.0);
  gtk_label_set_text (m->vover, s);
  snprintf (s, 20, "%07.5f", aspect);
  gtk_label_set_text (m->aspect, s);

  snprintf (s, 20, "%3i / %3i", res_x, tpl);
  gtk_label_set_text (m->pixels, s);
  snprintf (s, 20, "%3i / %3i", res_y, tlf);
  gtk_label_set_text (m->lines, s);
}

gint check_civ_cb (struct mask_civ *m)
{
  long status;
  char s[20];

  if ((gui_tv_chip & TV_ENCODER) != TV_CHRONTEL_MODEL1) return TRUE;
  status = back_card->getStatus (0);
  if (status >= 0) {
    snprintf (s, 20, "%9li", status << 6);
    gtk_label_set_text (m->low, s);
    snprintf (s, 20, "%9li", status << 6 | 0x3f); 
    gtk_label_set_text (m->high, s);
  } else {
    gtk_label_set_text (m->low, "<None>");
    gtk_label_set_text (m->high, "<None>");
  }
  return TRUE;
}

void gui_map_ch_cb (GtkWidget *widget, gpointer data)
{
  DPRINTF ("gui_map_ch_cb\n");
#ifndef DISABLE_TIMEOUT
  if (gui_timeout_ch_id == -1) {
    gui_timeout_ch_id = gtk_timeout_add (500 /* ms */, 
      (GtkFunction) check_civ_cb, &gui_mask_civ);
  }
#endif
}

void gui_unmap_ch_cb (GtkWidget *widget, gpointer data)
{
  DPRINTF ("gui_unmap_ch_cb\n");
#ifndef DISABLE_TIMEOUT
  if (gui_timeout_ch_id != -1) {
    gtk_timeout_remove (gui_timeout_ch_id);
    gui_timeout_ch_id = -1;
  }
#endif
}

/* -------- GUI Pages -------- */

GtkWidget *gui_ch_reg1_page (void)
{
  GtkAccelGroup *gui_ch_reg_accel_group;

  gui_ch_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("Chrontel Registers", 
    gui_ch_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_ch), GTK_SIGNAL_FUNC (reset_ch_cb),
    0, 0, 9, ch_mask1_reg, 0, 9, ch_mask1_twin, 7, 0, 9, ch_mask1_flag);
}

GtkWidget *gui_ch_reg2_page (void)
{
  GtkAccelGroup *gui_ch_reg_accel_group;

  gui_ch_reg_accel_group = gtk_accel_group_new ();
  return gui_regs_page ("Chrontel Registers", 
    gui_ch_reg_accel_group, PRINT_CHIP_REGS,
    GTK_OBJECT (update_ch), GTK_SIGNAL_FUNC (reset_ch_cb),
    0, 0, 5, ch_mask2_reg, 0, 0, NULL, 0, 0, 0, NULL);
}

GtkWidget *gui_ch_status_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;

  page = gtk_vbox_new (FALSE, 5);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_ch_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_ch_cb), NULL);
  gtk_container_set_border_width (GTK_CONTAINER (page), 5);

  /* Frequencies CRT / TV */

  frame = gtk_frame_new ("Frequencies");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  gui_mask_ch_freq.clk = 
    gui_mask_label (table, "Monitor dot clock:",   "---.-- MHz", 0,1,2, 0);
  gui_mask_ch_freq.hsyn = 
    gui_mask_label (table, "Monitor horiz. sync:", "---.-- kHz", 0,1,2, 1);
  gui_mask_ch_freq.vsyn = 
    gui_mask_label (table, "Monitor vert. sync:",  "---.--  Hz", 0,1,2, 2);
  gui_mask_ch_freq.hover = 
    gui_mask_label (table, "TV horiz. overscan:",  "--.--  %",   0,1,2, 3);
  gui_mask_ch_freq.vover = 
    gui_mask_label (table, "TV vert. overscan:",   "--.--  %",   0,1,2, 4);
  gui_mask_ch_freq.aspect = 
    gui_mask_label (table, "TV aspect ratio:",     "-.---  ",    0,1,2, 5);
  gui_mask_ch_freq.pixels = 
    gui_mask_label (table, "Pixels (horiz.)",      "--- / ---",  0,1,2, 6);
  gui_mask_ch_freq.lines = 
    gui_mask_label (table, "Lines (vert.)",        "--- / ---",  0,1,2, 7);

  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);

  /* CH Status */

  frame = gtk_frame_new ("Chrontel Status");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  gui_mask_civ.low = 
    gui_mask_label (table, "CIV lower bound:", "---------", 0,1,2, 0);
  gui_mask_civ.high = 
    gui_mask_label (table, "CIV upper bound:", "---------", 0,1,2, 1);

  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);

  return page;
}
  
void gui_ch_init (void)
{
  update_ch = (GtkAdjustment *) gtk_adjustment_new (0, 0, 0, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (update_chip), "changed", 
    GTK_SIGNAL_FUNC (update_ch_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (changed_all), "changed",
    GTK_SIGNAL_FUNC (ch_freq_calc_cb), &gui_mask_ch_freq);
}
