/* NVTV dumper -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Register dump utility.
 *
 */

#include "local.h" /* before everything else */

#include <stdio.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <getopt.h>

#include "xfree.h"
#include "card_direct.h"

/* -------- -------- */

int opt_two  = 0;
int opt_all  = 0;
int opt_regs = 0;
int opt_inc  = -1;
int opt_set  = 0;
int opt_addr = 0;

static const char *short_options = "?2i::ars:";

static struct option long_options[] = {
   {NULL, 0, NULL, 0}
};

/* -------- */

void usage (void)
{
  fprintf (stderr,
      "usage:  nvdump [-ar2] [-i [incr]] <from> <to> ...\n");
  fprintf (stderr,
      "usage:  nvdump -s <addr> <val> <val> ...\n\n");
}

/* -------- -------- */

/* Inc normal = 0x20 */

void dump (char *title, void *mem, ulong from, ulong to, int inc)
{
  ulong i;
  int j;

  if (inc == -1) inc = 0x20;
  if (title) printf (title, from, to);
  for (i = from; i < to; i += inc) {
    printf ("%06lX:", i);
    for (j = 0x00; j < inc; j += 4) {
      printf (" %08lX", MMIO_IN32(mem, i+j));
    }
    printf ("\n");
  }
}

/* 00d220, 00d224 index/data dword */
/* 6013d4, 6013d5 index/data byte  */

void dump_tv (char *title, void *mem, ulong from, ulong to)
{
  int inc = 0x10;
  int index = 0xd220;
  int data  = 0xd224;
  ulong i;
  int j;

  if (title) printf (title, from, to);
  for (i = from; i < to; i += inc) {
    printf ("T %02lX:", i);
    for (j = 0x00; j < inc; j ++) {
      MMIO_OUT32(mem, index, i+j);
      if ((j % 8) == 0) printf (" ");
      printf (" %02lX", MMIO_IN32(mem, data));
    }
    printf ("\n");
  }
}

static long regs_nv11[] = {
  /* PMC */
  0x000000, 0x000004, 0x000100, 0x000140, 0x000160, 0x000200,
  /* PBUS */
  0x001080, 0x001084, 0x001088, 0x00108c,
  0x001090, /* 0x001094, */ 0x00109c,
  0x0010a0, 0x0010a4, 0x0010a8, 0x0010ac,
  0x0010b0, 0x0010b4, 0x0010b8, 0x0010bc,
  0x0010c0, 0x0010c4, 0x0010c8, 0x0010cc,
  0x0010d0, 0x0010d4, 0x0010d8, 
  0x0010e0, 0x0010e4, 0x0010f0,
  0x001100, 0x001140,
  0x001200,
  0x001800, 0x001804, 0x001808, 0x00180c, 
  0x001810, 0x001814, 0x001818, 0x00181c,
  0x001820, 0x001824, 0x001828, 0x00182c,
  0x001830, 0x001834, 0x001838, 0x00183c,
  0x001840, 0x001844, 0x001848, 0x00184c,
  0x001850, 0x001854, 0x001858, 0x00185c,
  0x001860, 0x001864, 0x001868, 0x00186c,
  0x001870, 0x001874, 0x001878, 0x00187c,
  0x001880, 0x001884, 0x001888, 0x00188c,
  0x001890, 0x001894, 0x001898, 0x00189c,
  0x0018a0, 0x0018a4, 0x0018a8, 0x0018ac,
  0x0018b0, 0x0018b4, 0x0018b8, 0x0018bc,
  0x0018c0, 0x0018c4, 0x0018c8, 0x0018cc,
  0x0018d0, 0x0018d4, 0x0018d8, 0x0018dc,
  0x0018e0, 0x0018e4, 0x0018e8, 0x0018ec,
  0x0018f0, 0x0018f4, 0x0018f8, 0x0018fc,
  /* PEXTDEV */
  0x101000,
  /* PCRTC */
  0x600100, 0x600140, 
  0x600800, 0x600804, /* 0x600808, */ 0x60080c, 
  0x600810, 0x600814, 0x600818, 0x60081c,
  0x600830, 0x600834, 0x600838, 
  /* 0x600840, 0x600844, 0x600848, 0x60084c, */
  0x600860, 
  /* PCRTC2 */
  0x602100, 0x602140, 
  0x602800, 0x602804, /* 0x602808, */ 0x60280c, 
  0x602810, 0x602814, 0x602818, 0x60281c,
  0x602830, 0x602834, 0x602838, 
  /* 0x602840, 0x602844, 0x602848, 0x60284c, */
  0x602860, 
  /* PRAMDAC */
  0x680500, 0x680504, 0x680508, 0x68050c, 
  0x680510, 0x680514, /* 0x680518, */ 0x68051c,
  0x680520, 0x680524, 0x680528, 0x68052c,
  0x680530, 0x680534, 0x680538, 0x68053c,
  0x680600, /* 0x680604, */ 0x680608, 0x68060c, 
  0x680610, 0x680614, 0x680618, 0x68061c,
  0x680620, 0x680624, 0x680628, 0x68062c,
  0x680630, 0x680634, 0x680638, 0x68063c,
  0x680700, 0x680704, 0x680708, 0x68070c, 
  0x680710, 0x680714, 0x680718, 0x68071c,
  0x680720, 0x680724, 0x680728, 0x68072c,
  0x680730, 0x680734, 0x680738, 0x68073c,
  0x680800, 0x680804, 0x680808, 0x68080c, 
  0x680810, 0x680814, 0x680818, 0x68081c,
  0x680820, 0x680824, 0x680828, 0x68082c,
  0x680830, 0x680834, 0x680838, 0x68083c,
  0x680840, 0x680844, 0x680848, 0x68084c,
  0x680850, 0x680854, 0x680858, 0x68085c,
  0x680860, 0x680864, 0x680868, 0x68086c,
  0x680870, 0x680874, 0x680878, 0x68087c,
  0x680880, 0x680884, 0x680888, 0x68088c,
  0x680890, 0x680894, 0x680898, 0x68089c,
  /* PRAMDAC2 */
  0x682500, 0x682504, 0x682508, 0x68250c, 
  0x682510, 0x682514, 0x682518, 0x68251c,
  0x682520, 0x682524, 0x682528, 0x68252c,
  0x682530, 0x682534, 0x682538, 0x68253c,
  0x682600, /* 0x682604, */ 0x682608, 0x68260c, 
  0x682610, 0x682614, 0x682618, 0x68261c,
  0x682620, 0x682624, 0x682628, 0x68262c,
  0x682630, 0x682634, 0x682638, 0x68263c,
  0x682700, 0x682704, 0x682708, 0x68270c, 
  0x682710, 0x682714, 0x682718, 0x68271c,
  0x682720, 0x682724, 0x682728, 0x68272c,
  0x682730, 0x682734, 0x682738, 0x68273c,
  0x682800, 0x682804, 0x682808, 0x68280c, 
  0x682810, 0x682814, 0x682818, 0x68281c,
  0x682820, 0x682824, 0x682828, 0x68282c,
  0x682830, 0x682834, 0x682838, 0x68283c,
  0x682840, 0x682844, 0x682848, 0x68284c,
  0x682850, 0x682854, 0x682858, 0x68285c,
  0x682860, 0x682864, 0x682868, 0x68286c,
  0x682870, 0x682874, 0x682878, 0x68287c,
  0x682880, 0x682884, 0x682888, 0x68288c,
  0x682890, 0x682894, 0x682898, 0x68289c,
  /* -- end -- */
  -1
};

void dump_regs (void *mem)
{
  long *p;

  for (p = regs_nv11; *p != -1; p++) {
    dump (NULL, mem, (*p), (*p) + 4, 4);
  }
}

void dump_all (void *mem)
{
  dump ("---- PMC        %06lX-%06lX\n", mem, 0x000000, 0x001000, opt_inc);
  dump ("---- PBUS       %06lX-%06lX\n", mem, 0x001000, 0x002000, opt_inc);
  dump ("---- PEXTDEV    %06lX-%06lX\n", mem, 0x101000, 0x102000, opt_inc);
  dump ("---- PCRTC      %06lX-%06lX\n", mem, 0x600000, 0x601000, opt_inc);
  if (opt_two) {
    dump ("---- PCRTC2     %06lX-%06lX\n", mem, 0x602000, 0x603000, opt_inc);
  }
  dump ("---- PRAMDAC  X %06lX-%06lX\n", mem, 0x680300, 0x6808a0, opt_inc);
  dump ("---- PRAMDAC  Y %06lX-%06lX\n", mem, 0x6808c0, 0x681000, opt_inc);
  if (opt_two) {
    dump ("---- PRAMDAC2 X %06lX-%06lX\n", mem, 0x682300, 0x6828a0, opt_inc);
    dump ("---- PRAMDAC2 Y %06lX-%06lX\n", mem, 0x6828c0, 0x683000, opt_inc);
  }
}

void dump_arg (void *mem, int argc, char *argv[])
{
  ulong from, to;

  while (argc >= 1) {
    if (strcmp (argv[0], "T") == 0) {
      dump_tv ("---- NV17 PTV\n", mem, 0x00, 0x7f);
      argv += 1;
      argc -= 1;
    } else {
      if (argc < 2) break;
      from = strtoul (argv[0], NULL, 16);
      to   = strtoul (argv[1], NULL, 16);
      dump ("---- %06lX-%06lX\n", mem, from, to, opt_inc);
      argv += 2;
      argc -= 2;
    }
  }
}

void set_args (void *mem, ulong addr, int argc, char *argv[])
{
  ulong val;
  int inc = opt_inc;

  if (inc == -1) inc = 0x04;
  while (argc >= 1) {
    val = strtoul (*argv, NULL, 16);
    printf ("%06lX=%08lX     ", addr, val);
    fflush (stdout);
    MMIO_OUT32 (mem, addr, val);
    addr += inc;
    argv++;
    argc--;
  }
  printf ("\n");
}

int main (int argc, char *argv[])
{
  CardPtr main_list;
  CardPtr card;
  int fd;
  void *mem;
  int c = '?';
  int optc = 0;

  opterr = 0;
  while ((c = getopt_long (argc, argv, short_options, 
                long_options, NULL)) != EOF) 
  {
    switch (c)   
    {
      case 'h':
      case '?':
	usage ();
	exit (0);
	break;
      case '2':
	opt_two = 1;
	optc++;
	break;
      case 'a':
	opt_all = 1;
	optc++;
	break;
      case 'r':
	opt_regs = 1;
	optc++;
	break;
      case 'i':
	opt_inc = 0x04;
	optc++;
	if (optarg) opt_inc = (int) strtol (optarg, NULL, 16);
	break;
      case 's':
	opt_set = 1;
	optc++;
	opt_addr = (int) strtoul (optarg, NULL, 16);
	break;
    }
  }
  main_list = NULL;
  scan_cards_pci (&main_list, DEV_MEM);
  for (card = main_list; card; card = card->next) 
  {
    if (card->type == CARD_NVIDIA) 
    {
      printf ("%s (%04X) io=0x%08lX\n", card->name, card->pci_id, 
	      card->reg_base);
      fd = openDevMem (card);
      mem = mapDevMem (card, fd, card->reg_base, 0x1000000);

      if (opt_set) {
	set_args (mem, opt_addr, argc - optind, argv + optind);
      } else if (opt_all || argc - optc <= 1) {
	dump_all (mem);
      } else if (opt_regs) {
	dump_regs (mem);
      } else {
	dump_arg (mem, argc - optind, argv + optind);
      }

      unmapDevMem (card, card->reg_base, 0x1000000);
      closeDevMem (card, fd);
    }
  }
  return 0;
}
