/* NVTV TV common routines -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Header: Common tv-related routines.
 *
 */

#include "local.h" /* before everything else */

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "xf86i2c.h"
#include "tv_common.h"
#include "tv_bt.h"
#include "tv_cx.h"
#include "tv_ch1_7007.h"
#include "tv_ch2_7009.h"
#include "tv_ph1_saa7102.h"
#include "tv_ph2_saa7104.h"
#include "tv_null.h"

/* -------- */

TVState tvState = TV_UNKNOWN;

/* -------- */

/*
 * Detect devices that share the 0x88/0x8a address, i.e. BT/CX/PH/TW
 *
 */

TVChip TVDetectDeviceA (I2CDevPtr dev)
{
  I2CByte x1, x2, y;

  DPRINTF ("TVDetectDeviceA (%1s:%02X)\n", I2C_ID(dev));
#ifdef FAKE_PROBE_ID
  return FAKE_PROBE_ID;
#endif
  tvBusOk = TRUE;  
  TVWriteBus (dev, 0xff, 0x00);
  DPRINTF ("  check LUT (%s)\n", tvBusOk ? "ok" : "err");
  if (!tvBusOk) return TV_BROOKTREE;

  /* So it's not a BT, but may be an CX, in either read mode, or a PH or TW. 
   * Check register 01. It's read only for CX/TW, and reserved for PH, 
   * with default value 00. Although it seems to be used for Macrovision,
   * it's either not writeable or not readable.
   */

  TVWriteBus (dev, 0x01, 0x00); 
  TVReadBus  (dev, 0x00, &x1); 
  TVReadBus  (dev, 0x01, &y); 
  TVReadBus  (dev, 0x00, &x2); 
  DPRINTF ("  a %02X/%02X %02X (%s)\n", x1, x2, y, tvBusOk ? "ok" : "err");
  if (!tvBusOk) return TV_BROOKTREE;
  
  if (y == 0x00) {

    /* It's a PH, TW, or CX in status mode.
     * For PH or TW, reg 0x00 is never zero, so it must be a CX. 
     * Otherwise, test register 1a, which is msm threshold for PH, and
     * a status register for the TW, with at least the lower 4 bits r/o.
     */

    if (x1 == 0x00 || x2 == 0x00) return TV_CONEXANT;
    TVReadBus (dev, 0x1a, &x1); 
    x2 = x1 ^ 0x0f;
    TVWriteBus (dev, 0x1a, x2);
    TVReadBus  (dev, 0x1a, &y); 
    TVWriteBus (dev, 0x1a, x1);
    DPRINTF ("  b %02X/%02X %02X (%s)\n", x1, x2, y, tvBusOk ? "ok" : "err");
    if (!tvBusOk) return TV_NO_CHIP;
    if (y == x2) return TV_PHILIPS;
    return TV_NO_CHIP;

  } else {

    /* It's a TW or CX in read mode.
     * Check registers 28 and 29, which should be identical for the CX,
     * and are unused for the TW98. For the TW99, 29 bits 7-8 are uncritical.
     */

    TVReadBus (dev, 0x28, &x1); 
    TVReadBus (dev, 0x29, &y); 
    DPRINTF ("  c %02X/%02X (%s)\n", x1, y, tvBusOk ? "ok" : "err");
    if (!tvBusOk) return TV_NO_CHIP;
    if (x1 != y) return TV_NO_CHIP;
    x2 = x1 ^ 0xc0;
    TVWriteBus (dev, 0x29, x2); 
    TVReadBus  (dev, 0x28, &x1); 
    TVWriteBus (dev, 0x29, y); 
    DPRINTF ("  d %02X/%02X (%s)\n", x1, x2, tvBusOk ? "ok" : "err");
    if (!tvBusOk) return TV_NO_CHIP;
    if (x1 == x2) return TV_CONEXANT;
    return TV_NO_CHIP;
  }
  return TV_NO_CHIP;
}

/*
 * Detect devices that share the 0xEA/0xEC address, i.e. CH types
 *
 */

TVChip TVDetectDeviceB (I2CDevPtr dev)
{
  I2CByte x1, x2;

  DPRINTF ("TVDetectDeviceB (%1s:%02X)\n", I2C_ID(dev));
#ifdef FAKE_PROBE_ID
  return FAKE_PROBE_ID;
#endif
  tvBusOk = TRUE;  
  TVReadBus  (dev, 0xc0|0x0a, &x1);
  TVWriteBus (dev, 0xc0|0x0a, x1 ^ 1);
  TVReadBus  (dev, 0xc0|0x0a, &x2);
  TVWriteBus (dev, 0xc0|0x0a, x1);
  if (x1 == x2) {
    return TV_CHRONTEL_MODEL2;
  } else {
    return TV_CHRONTEL_MODEL1;
  }
}

/* 

ch7009-12: Only autoincrement/single step mode, not alternating. 
ch7007 etc: lower=alternate, higher=autoinc 

test if 0x0a and 0x0b are equal.

If yes, read 0x4a and 0x4b together.
7009: will be equal (autoinc mode)
7007: will be different (dev and version id)

If no, read 0x0a and 0x0b together.
7009: will be different (autoinc mode)
7007: will be same (no autoinc)

FIXME: Doesn't work -- two reads without autoinc produce wrong result...

Write to 0x4a/0x4b.

7009: Won't change
7007: Will change, so write original values back.

*/

/* -------- */

I2CChainPtr TVFindDevice (I2CChainPtr root, TVChip chip)
{
  I2CChainPtr chain;

  for (chain = root; chain; chain = chain->next)
  {
    if (chain->chip == chip) return chain;
  }
  return NULL;
}

/* FIXME: NVFindTvDevice -> TVFindDevice (pNv->TvChain, ...) */

/* 
 * Create chain of tv chips from list of devices on busses.
 * Process higher numbered busses first. FIXME??
 * If 'all' is true, include all
 * devices, otherwise only correctly detected ones.
 */

I2CChainPtr TVCreateChain (I2CBusPtr busses[], int nbus, Bool all)
{
  int bus;
  I2CDevPtr dev;
  I2CChainPtr chain, root;
  char version[50];
  char *s;
  TVChip chip;

  DPRINTF ("TVCreateChain %i %i\n", nbus, all);
  root = NULL;
  for (bus = 0; bus < nbus; bus++) {
    for (dev = busses[bus]->FirstDev; dev; dev = dev->NextDev) {
      s = NULL;
      chip = TV_NO_CHIP;
      switch (dev->SlaveAddr) {
	case 0x88: 
        case 0x8a: /* FIXME: For test only */
	  chip = TVDetectDeviceA (dev);
	  break;
	case 0xea:
	case 0xec:
	  chip = TVDetectDeviceB (dev);
	  break;
      }
      switch (chip & TV_ENCODER) {
	case TV_CHRONTEL_MODEL1:
	  s = TVDetectChrontel1 (dev, &chip);
	  break;
	case TV_CHRONTEL_MODEL2:
	  s = TVDetectChrontel2 (dev, &chip);
	  break;
	case TV_PHILIPS_MODEL1:
	case TV_PHILIPS_MODEL2:
	  s = TVDetectPhilips (dev, &chip);
	  break; 
	case TV_CONEXANT:
	  s = TVDetectConexant (dev, &chip);
	  break;
	case TV_BROOKTREE:
	  s = TVDetectBrooktree (dev, &chip, tvState);
	  break; 
        default:
	  if (!all) break;
	  snprintf (version, 50, "Unknown chip (%1s:%02X)", I2C_ID(dev));
	  s = version;
      }
      if (s) {
	chain = xcalloc (1, sizeof (I2CChainRec));
	chain->dev = dev;
	chain->next = root;
	chain->chip = chip;
	chain->name = xalloc (strlen(s)+1);
	strcpy (chain->name, s);
	root = chain;
      }
    }
  }
  return root;
}

/*
 * Free the chain of tv chips
 */

void TVDestroyChain (I2CChainPtr root)
{
  I2CChainPtr chain;

  while (root) {
    xfree (root->name);
    chain = root->next;
    xfree (root);
    root = chain;
  }
}

void TVProbeDevice (I2CBusPtr bus, I2CSlaveAddr addr, char *format, ...)
{
  I2CDevPtr dev;
  char *s;
  va_list ap;

#ifndef FAKE_PROBE_ALL
#ifndef FAKE_PROBE_ADDR
  if (xf86I2CProbeAddress(bus, addr))
#else
  if (addr == FAKE_PROBE_ADDR)
#endif
#endif
  {
    dev = xf86CreateI2CDevRec();
    s = xalloc (8); 
    va_start (ap, format);
    vsnprintf (s, 7, format, ap);
    va_end (ap);
    dev->DevName = s;
    dev->SlaveAddr = addr;
    dev->pI2CBus = bus;
    if (!xf86I2CDevInit(dev)) {
      xfree (dev->DevName); 
      xf86DestroyI2CDevRec(dev, TRUE);
    }
  }
}

void TVProbeKnownDevices (I2CBusPtr busses[], int nbus)
{
  int bus;
  
  for (bus = 0; bus < nbus; bus++) {
    TVProbeDevice (busses[bus], 0x88, "%i:%02X", bus, 0x88);
    TVProbeDevice (busses[bus], 0x8A, "%i:%02X", bus, 0x8A);
    TVProbeDevice (busses[bus], 0xEA, "%i:%02X", bus, 0xEA);
    TVProbeDevice (busses[bus], 0xEC, "%i:%02X", bus, 0xEC);
  }
}

void TVProbeAllDevices (I2CBusPtr busses[], int nbus)
{
  I2CSlaveAddr addr;
  int bus;

  for (bus = 0; bus < nbus; bus++) {
    for (addr = 0x00; addr < 0x100; addr += 2) {
      TVProbeDevice (busses[bus], addr, "%1i:%02X", bus, addr);
    }
  }
}

void TVProbeCreateAll (I2CBusPtr busses[], int nbus, I2CChainPtr *chain)
{
  if (*chain) DPRINTF ("TVProbeCreateAll: WARNING! Memory leak\n");
  TVProbeAllDevices (busses, nbus);
  *chain = TVCreateChain (busses, nbus, TRUE);
}

void TVProbeCreateKnown (I2CBusPtr busses[], int nbus, I2CChainPtr *chain)
{
  if (*chain) DPRINTF ("TVProbeCreateKnown: WARNING! Memory leak\n");
  TVProbeKnownDevices (busses, nbus); 
  *chain = TVCreateChain (busses, nbus, FALSE);
#ifdef PROBE_ALL_UNKNOWN
  /* if no dev's found, scan all addresses */
  if (!*chain) {
    TVProbeAllDevices (busses, nbus);
    *chain = TVCreateChain (busses, nbus, TRUE);
  }
#endif
}

void TVDestroyDevices (I2CBusPtr busses[], int nbus)
{
  I2CDevPtr dev;
  int bus;

  for (bus = 0; bus < nbus; bus++) {
    if (busses[bus]) {
      while ((dev = busses[bus]->FirstDev) != NULL) {
	xfree (dev->DevName);
	xf86DestroyI2CDevRec(dev, TRUE); 
      }
    }
  }
}

void TVDestroyBusses (I2CBusPtr busses[], int nbus)
{
  int i;

  for (i = 0; i < nbus; i++) 
  {
    if (busses[i]) xf86DestroyI2CBusRec (busses[i], TRUE, FALSE);
    busses[i] = NULL;
  }
}

void TVSetTvEncoder (TVEncoderObj *encoder, I2CChainPtr chain)
{
  switch (chain->chip & TV_ENCODER) {
    case TV_BROOKTREE:
      *encoder = tvBtTemplate;
      break;
    case TV_CONEXANT:
      *encoder = tvCxTemplate;
      break;
    case TV_CHRONTEL_MODEL1:
      *encoder = tvCh1Template;
      break;
    case TV_CHRONTEL_MODEL2:
      *encoder = tvCh2Template;
      break;
    case TV_PHILIPS_MODEL1:
      *encoder = tvPh1Template;
      break;
    case TV_PHILIPS_MODEL2:
      *encoder = tvPh2Template;
      break;
    default:
      *encoder = tvNullTemplate;
      break;
  }
  encoder->Create (encoder, chain->chip, chain->dev);
}

