/* NVTV Conexant TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Routines to access the Conexant chip registers via the I2C bus.
 *
 */

#include "local.h" /* before everything else */
#include "xf86_ansic.h" 

#include "bitmask.h"
#include "xf86i2c.h"
#include "tv_i2c.h"
#include "tv_cx.h"
#include "tv_bt.h"

/* -------- Conexant -------- */

/* Read hardware config on object creation.
 *
 *  32-6:5 drvs = 0 (default) 
 *  3a-4   14318_xtal = 0 (default)
 */

void TVCxCreate (TVEncoderObj *this, TVChip chip, I2CDevPtr dev)
{
  I2CByte result;

  DPRINTF ("tv create cx\n");
  this->chip = chip;
  this->dev = dev;
  tvBusOk = TRUE;
  TVReadBus (this->dev, 0x32, &result);
  this->hwconfig = SetBitField(result,6:5,6:5);
  TVReadBus (this->dev, 0x3a, &result);
  this->hwconfig |= SetBitField(result,4:4,4:4);
  if (!tvBusOk) {
    fprintf (stderr, "Critical: Error reading Conexant hardware config\n");
    exit (1);
  }
  if (GetBF(this->hwconfig,6:5) != 0) {
    ErrorF ("Warning: Conexant in low voltage configuration!\n");
  }
  if (GetBF(this->hwconfig,4:4) != 0) {
    ErrorF ("Warning: 14.318 MHz crystal detected; timings will be wrong.\n");
  }
}

/* All DACs off, clock output disabled. */

/*  2E-7   hdtv_en = 0 (default)
 *  2E-6   rgb2prpb = 0 (default) (hdtv mode)
 *  2E-5   rpr_sync_dis = 0 (default) (hdtv mode)
 *  2E-4   gy_sync_dis = 0 (default) (hdtv mode)
 *  2E-3   bpb_sync_dis = 0 (default) (hdtv mode)
 *  2E-2   hd_sync_edge = 0 (default) (hdtv mode)
 *  2E-1:0 raster_sel = 0 (default) (hdtv mode)
 *  30-7   sleep_en = 0 (default) (FIXME: Use for TV-Off ?)
 *  30-6   xtal_pad_dis = 0 (default) 
 *  30-5   xtal_bfo_dis = 0 (default) (FIXME: Not needed?)
 *  30-4   pll_keep_alive = 0 (default)
 *  30-3   dis_clki = 0 (default)
 *  30-2   dis_pll = 0 (default)
 *  30-1   dis_clko = 0 (default) (slave mode)
 *  32-7   auto_chk = 1 (default 0)
 *  32-6:5 drvs = 0 (default) (better don't touch)
 *  32-4   setup_hold_adj = 0 (default)
 *  32-3   in_mode[3] = 0/1 (24-bit RGB multiplexed, or YCrCb Alt)
 *  32-2   datdly_re = 0 (default)
 *  32-1   offset_rgb = 0 (default) (hdtv mode)
 *  32-0   csc_sel = 0 (default) (hdtv mode)
 *  6C-7   timing_rst = 0 (default)
 *  6C-6   en_reg_rd = 1 (default 0)
 *  6C-4   blnk_ignore = 0 (default) (ccir mode)
 *  6C-3   en_scart = 0 (default) (vga out_mode)
 *  6C-1:0 fld_mode = 0 (default 2) (slave mode, interlaced input)
 *  BA-7   sreset = 0 (default)
 *  BA-5   slaver = 0 (master mode)
 *  C4-7:6 estatus
 *  C4-5   eccf2
 *  C4-4   eccf1
 *  C4-3   eccgate
 *  C4-2   ecbar
 *  C4-1   dchroma
 *  C4-0   en_out
 *  C6-7   en_blanko = 1 (default) (no blank)
 *  C6-6   en_dot = 0 (default) (internal blanking)
 *  C6-5   fieldi
 *  C6-4   vsynci = 1 (active high) (default 0) (ok for nv)
 *  C6-3   hsynci = 1 (active high) (default 0) (ok for nv)
 *  C6-2:0 in_mode[2:0] = 0/4 (24-bit RGB multiplexed, or YCrCb Alt)
 *  CE     out_mux d-a
 *  D4-7   mode2x = 0 (default)
 *  D4-6   div2
 *  D4-5   en_async = 0 (default)
 *  D4-4:0 (ccr stuff)
 *  D6-6   e656 = 0 (default)
 *  D6-5   blanki = 0 (active low) (default)
 *  D6-4   eblue = 0 (default)
 *  D6-3:2 out_mode = 0 (default: CVBS/Y/C/Y_DLY)
 *  D6-1:0 lumadly = 0 (default)
 *  D8-7   chroma_bw = 0 (default) (USER)
 *  D8-6   by_yccr = 0 (default) (USER)
 *  D8-5:4 pkfil_sel = 0 (default) (USER)
 *  D8-3   field_id = 0 (defualt)
 *  D8-2   cvbsd_inv = 0 (default) 
 *  D8-1   sc_pattern = 0 (default) (USER)
 *  D8-0   prog_sc = 0 (default) (USER)
 */

void TVCxSetPort (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = this->dev;

  DPRINTF ("tv port cx %08X\n", port);

  if ((port & PORT_PCLK_MODE) != PORT_PCLK_MASTER) {
    ERROR ("TVCxSetPort: Only clock master mode supported.\n");
    exit (1);
  }
  if ((port & PORT_SYNC_DIR) == PORT_SYNC_SLAVE) {
    this->hwstate |= SetBit (5);
  } else {
    this->hwstate &= ~SetBit (5);
  }
  TVWriteBus (dev, 0xba, this->hwstate);
  TVWriteBus (dev, 0xc6, 0x00 /* fieldi = 0 */
              | ((port & PORT_FORMAT_MASK_COLOR) ? 0x4 : 0x0) /* in_mode */
  	      | ((port & PORT_BLANK_DIR) ? 
		  ((port & PORT_BLANK_MODE) ? SetBit(6) : 0) : SetBit (7))
	      | ((port & PORT_VSYNC_POLARITY) ? SetBit(4) : 0)
	      | ((port & PORT_HSYNC_POLARITY) ? SetBit(3) : 0));
  TVWriteBus (dev, 0x32, SetBit(7)   /* enable auto_chk */
	      | ((port & PORT_FORMAT_MASK_ALT) ? SetBit(3) : 0)
	      | SetBitField(this->hwconfig,6:5,6:5)); 
  TVWriteBus (dev, 0xd6, 0x00 /* Out_Mode */
	      | ((port & PORT_BLANK_POLARITY) ? SetBit(5) : 0));
}

/* FIXME: See TVBtSetPort, additional BLANK_POLARITY */

void TVCxGetPort (TVEncoderObj *this, int *port)
{
  register I2CDevPtr dev = this->dev;
  I2CByte res;

  DPRINTF ("tv get port cx\n");
  *port = PORT_PCLK_MASTER | PORT_PCLK_HIGH;
  TVReadBus (dev, 0xba, &res);
  *port |= (GetBit(res,5) ? PORT_SYNC_SLAVE : PORT_SYNC_MASTER);
  TVReadBus (dev, 0xc6, &res);
  *port |= (GetBF(res,2:0) == 4 ? PORT_FORMAT_MASK_COLOR : 0)
        | (GetBit(res,4) ? PORT_VSYNC_HIGH : PORT_VSYNC_LOW)
        | (GetBit(res,3) ? PORT_HSYNC_HIGH : PORT_HSYNC_LOW)
        | (GetBit(res,7) ? (PORT_BLANK_OUT | PORT_BLANK_REGION) :
           GetBit(res,6) ? (PORT_BLANK_IN  | PORT_BLANK_DOTCLK) :
	                   (PORT_BLANK_IN  | PORT_BLANK_REGION));
  TVReadBus (dev, 0x32, &res);
  *port |= (GetBit(res,3) ? PORT_FORMAT_MASK_ALT : 0);
  TVReadBus (dev, 0xd6, &res);
  *port |= (GetBit(res,5) ? PORT_BLANK_HIGH : PORT_BLANK_LOW);
}

void TVCxInitRegs (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = this->dev;

  DPRINTF ("tv init cx\n");
  TVCxSetPort (this, port);
  TVWriteBus (dev, 0xba, 0x17);
  TVWriteBus (dev, 0xc4, 0x00);
  TVWriteBus (dev, 0xc6, 0x98); /* En_Blank0, Sync Polarity */
  TVWriteBus (dev, 0xce, 0x18); 
  TVWriteBus (dev, 0xd4, 0x00); /* Mode2x=0 */
  TVWriteBus (dev, 0xd8, 0x00); /* Disable Secam */
  TVWriteBus (dev, 0x2e, 0x00);
  TVWriteBus (dev, 0x30, 0x00);
  TVWriteBus (dev, 0x6c, 0x00);
}

void TVCxSetRegs (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  register I2CDevPtr dev = this->dev;

  DPRINTF ("tv set cx\n");
  TVBtSetRegs (this, r, state);
  TVWriteBus (dev, 0x34, SetBitFlag(r->cx.flags5,CX_FLAG5_ADPT_FF,7)
                       | SetBitField(r->cx.c_altff,1:0,4:3)
                       | SetBitField(r->cx.y_altff,1:0,1:0));
  TVWriteBus (dev, 0x36, SetBitFlag(r->cx.flags5,CX_FLAG5_FFRTN,7)
                       | SetBitFlag(r->cx.flags5,CX_FLAG5_YSELECT,6)
                       | SetBitField(r->cx.c_thresh,2:0,5:3)
                       | SetBitField(r->cx.y_thresh,2:0,2:0));
  TVWriteBus (dev, 0x38, SetBitFlag(r->cx.flags5,CX_FLAG5_PIX_DOUBLE,6)
                       | SetBitFlag(r->cx.flags5,CX_FLAG5_PLL_32CLK,5)
                       | SetBitFlag(r->cx.flags5,CX_FLAG5_DIV2,4)
                       | SetBitField(r->cx.bt.hburst_end,8:8,3:3)
                       | SetBitField(r->cx.bt.hburst_begin,8:8,2:2)
                       | SetBitField(r->cx.bt.v_linesi,10:10,1:1)
                       | SetBitField(r->cx.bt.h_blanki,9:9,0:0));
  TVWriteBus (dev, 0x3c, Set8Bits(r->cx.mcompy));
  TVWriteBus (dev, 0x3e, Set8Bits(r->cx.mcompu));
  TVWriteBus (dev, 0x40, Set8Bits(r->cx.mcompv));
  TVWriteBus (dev, 0x42, Set8Bits(r->cx.msc_db));
  TVWriteBus (dev, 0x44, Set8Bits(r->cx.msc_db >> 8));
  TVWriteBus (dev, 0x46, Set8Bits(r->cx.msc_db >> 16));
  TVWriteBus (dev, 0x48, Set8Bits(r->cx.msc_db >> 24));
  TVWriteBus (dev, 0x4a, Set8Bits(r->cx.dr_limitp));
  TVWriteBus (dev, 0x4c, Set8Bits(r->cx.dr_limitn));
  TVWriteBus (dev, 0x4e, SetBitField(r->cx.dr_limitn,10:8,5:3)
                       | SetBitField(r->cx.dr_limitp,10:8,2:0));
  TVWriteBus (dev, 0x50, Set8Bits(r->cx.db_limitp));
  TVWriteBus (dev, 0x52, Set8Bits(r->cx.db_limitn));
  TVWriteBus (dev, 0x54, SetBitField(r->cx.db_limitn,10:8,5:3)
                       | SetBitField(r->cx.db_limitp,10:8,2:0));
  TVWriteBus (dev, 0x56, Set8Bits(r->cx.filincr)); /* fil4286incr */
  TVWriteBus (dev, 0x58, SetBitField(r->cx.filfsconv,5:0,5:0));
  TVWriteBus (dev, 0x5a, Set8Bits(r->cx.y_off));
  TVWriteBus (dev, 0x5c, Set8Bits(r->cx.hue_adj));
  TVWriteBus (dev, 0x60, SetBitFlag(r->cx.flags5,CX_FLAG5_EWSSF2,7)
                       | SetBitFlag(r->cx.flags5,CX_FLAG5_EWSSF1,6)
                       | SetBitField(r->cx.wsdat,4:1,3:0));
  TVWriteBus (dev, 0x62, SetBitField(r->cx.wsdat,12:5,7:0));
  TVWriteBus (dev, 0x64, SetBitField(r->cx.wsdat,20:13,7:0));
  /* wsdat bit 0 seems not to be used anywhere */
  TVWriteBus (dev, 0x66, Set8Bits(r->cx.wssinc));
  TVWriteBus (dev, 0x68, Set8Bits(r->cx.wssinc >> 8));
  TVWriteBus (dev, 0x6a, SetBitField(r->cx.wssinc,19:16,3:0));
  TVWriteBus (dev, 0xd8, (r->cx.flags4 &  CX_FLAG4_MASK)
                       | SetBitField(r->cx.pkfil_sel,1:0,5:4));
  TVBtMacroMode (this, r);
  xf86usleep(1000); /* BT871 doc says: 1ms minimum */
  /* FIXME: Don't do that always */
  TVWriteBus (dev, 0x6c, 0xC6); /* timing reset */
  xf86usleep(1000); /* not necessary, but won't hurt ... */
}

void TVCxGetRegs (TVEncoderObj *this, TVEncoderRegs *r)
{
  register I2CDevPtr dev = this->dev;
  I2CByte regs[0x100];
  int i;

  DPRINTF ("tv get cx\n");
  for (i = 0x2e; i < 0x100; i += 2) TVReadBus (dev, i, &regs[i]);
  r->cx.bt.h_active     = Set8Bits(regs[0x78])
                        | SetBitField(regs[0x86],6:4,10:8); /* BT: 5:4, 9:8 */
  r->cx.bt.h_blanki     = Set8Bits(regs[0x8c])
                        | SetBitField(regs[0x8e],3:3,8:8)
                        | SetBitField(regs[0x38],0:0,9:9);
  r->cx.bt.h_clki       = Set8Bits(regs[0x8a])
                        | SetBitField(regs[0x8e],2:0,10:8);
  r->cx.bt.h_fract      = Set8Bits(regs[0x88]); 
  r->cx.bt.h_blanko     = Set8Bits(regs[0x80])
                        | SetBitField(regs[0x9a],7:6,9:8);
  r->cx.bt.h_clko       = Set8Bits(regs[0x76])
                        | SetBitField(regs[0x86],3:0,11:8);
  r->cx.bt.v_activei    = Set8Bits(regs[0x94])
                        | SetBitField(regs[0x96],3:2,9:8);
  r->cx.bt.v_blanki     = Set8Bits(regs[0x92]); 
  r->cx.bt.v_linesi     = Set8Bits(regs[0x90])
                        | SetBitField(regs[0x96],1:0,9:8)
                        | SetBitField(regs[0x38],1:1,10:10);
  r->cx.bt.v_activeo    = Set8Bits(regs[0x84])
                        | SetBitField(regs[0x86],7:7,8:8);
  r->cx.bt.v_blanko     = Set8Bits(regs[0x82]); 
  r->cx.bt.v_scale      = Set8Bits(regs[0x98])
                        | SetBitField(regs[0x9a],5:0,13:8);
  r->cx.bt.pll_fract    = Set8Bits(regs[0x9c]) 
                        | Set8Bits(regs[0x9e]) << 8; 
  r->cx.bt.pll_int      = SetBitField(regs[0xa0],5:0,5:0); 
  r->cx.bt.hsynoffset   = Set8Bits(regs[0x6e]) 
                        | SetBitField(regs[0x70],7:6,9:8);
  r->cx.bt.vsynoffset   = Set8Bits(regs[0x72])
                        | SetBitField(regs[0x74],5:3,10:8);
  r->cx.bt.hsynwidth    = SetBitField(regs[0x70],5:0,5:0); 
  r->cx.bt.vsynwidth    = SetBitField(regs[0x74],2:0,2:0); 
  r->cx.bt.hsync_width  = Set8Bits(regs[0x7a]); 
  r->cx.bt.hburst_begin = Set8Bits(regs[0x7c])
                        | SetBitField(regs[0x38],2:2,8:8);
  r->cx.bt.hburst_end   = Set8Bits(regs[0x7e])
                        | SetBitField(regs[0x38],3:3,8:8);
  /* r->cx.bt.v_blank_dly  = SetBitField(regs[0x8e],4:4,0:0); */
  r->cx.bt.sync_amp     = Set8Bits(regs[0xa4]); 
  r->cx.bt.bst_amp      = Set8Bits(regs[0xa6]); 
  r->cx.bt.mcr          = Set8Bits(regs[0xa8]); 
  r->cx.bt.mcb          = Set8Bits(regs[0xaa]); 
  r->cx.bt.my           = Set8Bits(regs[0xac]); 
  r->cx.bt.msc          = (unsigned long) Set8Bits(regs[0xae]) 
                        | (unsigned long) Set8Bits(regs[0xb0]) << 8
                        | (unsigned long) Set8Bits(regs[0xb2]) << 16
                        | (unsigned long) Set8Bits(regs[0xb4]) << 24; 
  r->cx.bt.phase_off  = Set8Bits(regs [0xb6]);
  r->cx.bt.f_selc     = SetBitField(regs [0xc8],5:3,2:0);
  r->cx.bt.f_sely     = SetBitField(regs [0xc8],2:0,2:0);
  r->cx.bt.ccoring    = SetBitField(regs [0xcc],5:3,2:0);
  r->cx.bt.cattenuate = SetBitField(regs [0xcc],2:0,2:0);
  r->cx.bt.ycoring    = SetBitField(regs [0xca],5:3,2:0);
  r->cx.bt.yattenuate = SetBitField(regs [0xca],2:0,2:0);
  r->cx.bt.clpf       = SetBitField(regs [0x96],7:6,1:0);
  r->cx.bt.ylpf       = SetBitField(regs [0x96],5:4,1:0);
  r->cx.bt.out_muxa   = SetBitField(regs [0xce],1:0,1:0);
  r->cx.bt.out_muxb   = SetBitField(regs [0xce],3:2,1:0);
  r->cx.bt.out_muxc   = SetBitField(regs [0xce],5:4,1:0);
  r->cx.bt.out_muxd   = SetBitField(regs [0xce],7:6,1:0);
  r->cx.bt.flags1 = GetBitFlag(regs [0xa2],0,BT_FLAG1_NI_OUT)		
             	  | GetBitFlag(regs [0xa2],1,BT_FLAG1_SETUP)		
             	  | GetBitFlag(regs [0xa2],2,BT_FLAG1_625LINE)	
             	  | GetBitFlag(regs [0xa2],3,BT_FLAG1_VSYNC_DUR)	
             	  | GetBitFlag(regs [0xa2],4,BT_FLAG1_DIS_SCRESET)	
             	  | GetBitFlag(regs [0xa2],5,BT_FLAG1_PAL_MD)		
                  | GetBitFlag(regs [0xa2],6,BT_FLAG1_ECLIP)
                  | GetBitFlag(regs [0xa2],7,CX_FLAG1_FM);
  r->cx.bt.flags2 = GetBitFlag(regs [0xc8],7,BT_FLAG2_DIS_YFLPF)
             	  | GetBitFlag(regs [0xc8],6,BT_FLAG2_DIS_FFILT)
             	  | GetBitFlag(regs [0xca],7,BT_FLAG2_DIS_GMUSHY)
             	  | GetBitFlag(regs [0xca],6,BT_FLAG2_DIS_GMSHY)
             	  | GetBitFlag(regs [0xcc],7,BT_FLAG2_DIS_GMUSHC)
             	  | GetBitFlag(regs [0xcc],6,BT_FLAG2_DIS_GMSHC)
             	  | GetBitFlag(regs [0xc4],1,BT_FLAG2_DIS_CHROMA);
  r->cx.bt.flags3 = SetBitField(regs [0xba],3:0,3:0);
  r->cx.bt.macro  = 0;
  r->cx.flags4    = regs [0xd8] & CX_FLAG4_MASK;
  r->cx.flags5    = GetBitFlag(regs [0x38],6,CX_FLAG5_PIX_DOUBLE)
                  | GetBitFlag(regs [0x38],5,CX_FLAG5_PLL_32CLK)
                  | GetBitFlag(regs [0x38],4,CX_FLAG5_DIV2)
  		  | GetBitFlag(regs [0x34],7,CX_FLAG5_ADPT_FF)
  		  | GetBitFlag(regs [0x36],7,CX_FLAG5_FFRTN)
  		  | GetBitFlag(regs [0x36],6,CX_FLAG5_YSELECT)
  		  | GetBitFlag(regs [0x60],7,CX_FLAG5_EWSSF2)
  		  | GetBitFlag(regs [0x60],6,CX_FLAG5_EWSSF1);
  r->cx.c_altff   = SetBitField(regs [0x34],4:3,1:0);
  r->cx.y_altff   = SetBitField(regs [0x34],1:0,1:0);
  r->cx.c_thresh  = SetBitField(regs [0x36],5:3,2:0);
  r->cx.y_thresh  = SetBitField(regs [0x36],2:0,2:0);
  r->cx.mcompy    = Set8Bits(regs [0x3c]);
  r->cx.mcompu    = Set8Bits(regs [0x3e]);
  r->cx.mcompv    = Set8Bits(regs [0x40]);
  r->cx.msc_db    = (unsigned long) Set8Bits(regs [0x42])
                  | (unsigned long) Set8Bits(regs[0x44]) << 8
                  | (unsigned long) Set8Bits(regs[0x46]) << 16
                  | (unsigned long) Set8Bits(regs[0x48]) << 24; 
  r->cx.dr_limitp = Set8Bits(regs[0x4a]) 
                  | SetBitField(regs[0x4e],2:0,10:8);
  r->cx.dr_limitn = Set8Bits(regs[0x4c]) 
                  | SetBitField(regs[0x4e],5:3,10:8);
  r->cx.db_limitp = Set8Bits(regs[0x50]) 
                  | SetBitField(regs[0x54],2:0,10:8);
  r->cx.db_limitn = Set8Bits(regs[0x52]) 
                  | SetBitField(regs[0x54],5:3,10:8);
  r->cx.filfsconv = SetBitField(regs[0x58],5:0,5:0);
  r->cx.filincr   = Set8Bits(regs[0x56]);
  r->cx.wsdat     = SetBitField(regs[0x60],3:0,4:1)
                  | SetBitField(regs[0x62],7:0,12:5)
                  | SetBitField(regs[0x64],7:0,20:13);
  r->cx.wssinc    = (unsigned long) Set8Bits(regs[0x66]) 
		  | (unsigned long) Set8Bits(regs[0x68]) << 8;
  r->cx.y_off     = Set8Bits(regs[0x5a]);
  r->cx.hue_adj   = Set8Bits(regs[0x5c]);
  r->cx.pkfil_sel = SetBitField(regs[0xd8],5:4,1:0);
}

void TVCxSetState (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  DPRINTF ("tv state cx\n");
  TVBtSetState (this, r, state);
  /* FIXME: 6c en_reg_rd=1 6c-5 ffcbar */
}

long TVCxGetStatus (TVEncoderObj *this, int index)
{ 
  I2CByte result;

  TVReadBus (this->dev, 0x28, &result); 
  return result | ((tvState == TV_ON) ? 0x100 : 0);
}

TVConnect TVCxGetConnect (TVEncoderObj *this)
{
  register I2CDevPtr dev = this->dev;
  I2CByte result;
 
  tvBusOk = TRUE;  
  TVWriteBus (dev, 0xba, 0x40); 
  xf86usleep(50000); /* BT871 doc says: 20ms - 50ms */
  TVReadBus (dev, 0x06, &result); 
  DPRINTF ("cx conn %02x (%s)\n", result, tvBusOk ? "ok" : "err");
  TVWriteBus (dev, 0xba, 0x00); 
  /* FIXME. (But next SetTVState will write correct values.) */
  if (tvBusOk) {
    switch (result & 0xe0) {
      case 0x80 : return CONNECT_COMPOSITE; break;
      case 0x60 : return CONNECT_SVIDEO; break;
      case 0x20 : return CONNECT_CONVERT; break;
      case 0x40 : return CONNECT_CONVERT; break;
      default: 
	ErrorF("Strange Conexant connection status %02x\n", result & 0xe0); 
      case 0xe0 : return CONNECT_BOTH; break;
    }
  } else {
    return CONNECT_NONE;
  }
}

/*
 * Check for Conexant chip on device dev. Return version string if found, 
 * NULL otherwise. Make sure it cannot be a Philips chip, otherwise
 * register 0x6c (HTRIG) and register 0x28 might be changed to an illegal 
 * value.
 */

char *TVDetectConexant (I2CDevPtr dev, TVChip *encoder)
{
  I2CByte result;
  char *chip;
  static char version [50];

#ifdef FAKE_PROBE_CONEXANT
  *encoder = TV_CONEXANT;
  snprintf (version, 50, "Conexant Fake (%1s:%02X)", I2C_ID(dev));
  return version;
#else
#ifdef FAKE_I2C
  return NULL;
#endif
#endif  
  tvBusOk = TRUE;  
  TVWriteBus (dev, 0x6c, 0x46); /* enable reg rd */

#if 0 //// FIXME remove
  TVWriteBus (dev, 0x28, 0x55); /* FIXME this will change CNTRL2 in TW99 */
  TVReadBus (dev, 0x28, &result); 
  DPRINTF ("cx check (%s)\n", tvBusOk ? "ok" : "err");
  if (!tvBusOk || result != 0x55) return NULL;
  TVWriteBus (dev, 0x28, 0x00); 
#endif

  TVReadBus (dev, 0x00, &result);
  DPRINTF ("cx check %02x (%s)\n", result, tvBusOk ? "ok" : "err");
  if (!tvBusOk) return NULL;
  *encoder = TV_CONEXANT;
  switch (result & 0xe0) {
    case 0x40 : chip = "25870"; break; /* Accipiter, no MV     */
    case 0x60 : chip = "25871"; break; /* Accipiter, with MV   */
    case 0x80 : chip = "25872"; break; /* Aquila Lite, no MV   */
    case 0xa0 : chip = "25873"; break; /* Aquila Lite, with MV */
    case 0xc0 : chip = "25874"; break; /* Aquila, no MV        */
    case 0xe0 : chip = "25875"; break; /* Aquila, with MV      */
    default : chip = NULL; break;
  }
  if (chip) {
    snprintf (version, 50, "Conexant CX%s Rev %i (%1s:%02X)", 
	      chip, result & 0x1f, I2C_ID(dev));
    return version;
  } else {
    return NULL;
  }
}

/* maxclock is specified for "standard mode", whatever that means */
/* Apparently, in 3:2 mode it can be larger. Maybe in async mode, too? */

TVEncoderObj tvCxTemplate = {
  chip: TV_CONEXANT, dev: NULL, minClock: 0, maxClock: 53333, 
  Create:     TVCxCreate,
  InitRegs:   TVCxInitRegs, 
  SetRegs:    TVCxSetRegs, 
  GetRegs:    TVCxGetRegs, 
  SetPort:    TVCxSetPort,
  GetPort:    TVCxGetPort,
  SetState:   TVCxSetState,
  GetConnect: TVCxGetConnect, 
  GetStatus:  TVCxGetStatus
};
