/* NVTV i810 TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Routines to control the TV and the I2C bus on Intel i810 etc. cards.
 *
 */

#include "local.h" /* before everything else */
#include "xfree.h" 

#include "mmio.h"
#include "bitmask.h"
#include "tv_i2c.h"
#include "tv_i810.h"
#include "xf86i2c.h"

/* -------- Registers -------- */

#define I810_WR32(p,i,d)  MMIO_OUT32(((p)->MMIOBase), (i), (d))
#define I810_RD32(p,i)    MMIO_IN32(((p)->MMIOBase), (i))
#define I810_H_RD32(p,h,i)   MMIO_IN32(((p)->MMIOBase), ((i)+0x1000*(h)))
#define I810_H_WR32(p,h,i,d) MMIO_OUT32(((p)->MMIOBase), ((i)+0x1000*(h)), (d))

#define I810_GPIOA	0x05010
#define I810_GPIOB	0x05014
#define I830_GPIOC	0x05018  /* M_I2C */

#define I810_DCLK(i)	(0x06000 + (i)*4)
#define I810_DCLK_0D	0x06000
#define I810_DCLK_1D	0x06004
#define I810_DCLK_2D	0x06008
#define I810_LCD_CLKD	0x0600c
#define	I810_DCLK_0DS	0x06010
#define	I810_PWR_CLKC	0x06014

#define	I810_LCDTV_C 	0x60018
#define	I810_OVRACT  	0x6001c
#define	I810_BCLRPAT 	0x60020
#define	I810_LCD_CLKD	0x0600c

/*  This is an attempt to explain the i830 registers as far as I can 
 *  guess them. Some of it may be severely wrong.
 *
 *  1. Architecture overview
 *
 *  The i830 emulates two PCI cards. The register of the PCI cards seem
 *  to shadow each other, and the X driver only uses the first card.
 *
 *  There are several parts used for graphic output:
 *
 *  Display planes A,B,C --> Pipes A,B --> Analog/Digital Ports  
 *  I2C control for TV encoder.
 *
 *  Some registers may be double buffered, and may need a write to another
 *  register to commit.
 *
 *  2. I2C control
 *
 *  There are 5 I2C ports: DDC1, I2C, DDC2, M_DDC, M_I2C. The DDC ports
 *  are used for EDID display information transfer from the monitor
 *  (typically at address 0xA0). On the I2C port (nvtv bus 0) there is a 
 *  device at 0x04, maybe a flat panel controller. The TV encoder chip
 *  is on M_I2C (bus 1). On the i810, the TV encoder is on bus 0.
 *
 *  0x05010 GPIOA General purpose I/O A, one of the DDC ports.
 *  0x05014 GPIOB General purpose I/O B, I2C port.
 *  0x05018 GPIOC General purpose I/O C, M_I2C port.
 *    31:13 reserved
 *       12 GPIO1 data in
 *       11 GPIO1 data value
 *       10 GPIO1 data mask (1=write data)
 *        9 GPIO1 direction value (0=input, 1=output)
 *        8 GPIO1 direction mask (1=write direction)
 *     7: 5 reserved
 *        4 GPIO0 data in
 *        3 GPIO0 data value
 *        2 GPIO0 data mask
 *        1 GPIO0 direction value
 *        0 GPIO0 direction mask
 *
 *  GPIO1 is the data pin, GPIO0 is the clock pin. Maybe there are even 
 *  more GPIO registers?
 *
 *  3. Display planes
 *
 *  There are (at least) 3 display planes:
 *
 *  A primary display plane
 *  B 'sprite' plane (for specific laptop display information?)
 *  C overlay plane (can also be directly sent in YUV format to DVO)
 * 
 *  0x70180 DSPACTRL Display plane A control
 *  0x71180 DSPBCTRL Display plane B control
 *       31 plane enable (1=enabled)
 *       30 gamma correction enable (0=bypass, 1=enabled)
 *    29:26 pixformat (2=8BPP, 4=15_16BPP, 5=16BPP, 6=32BPP_NO_ALPHA, 7=32BPP) 
 *       25 stereo enable (1=enabled)
 *       24 select pipe (0=A, 1=B)
 *       23 reserved
 *       22 source key enable (1=enabled)
 *    21:20 pixel multiplicity (0=normal, 1=doubleline, 2=3=reserved)
 *       19 reserved
 *       18 stereo polarity (0=first, 1=second)
 *    17-16 reserved
 *       15 alpha transfer mode (1=enabled) ; B only
 *    14- 1 reserved
 *        0 sprite_above (0=display A, 1=overlay) ; B only
 *   
 *  Note: 8BPP is indexed, 15_16BPP is 5-5-5, 16BPP is 5-6-5, 
 *    32BPP_NO_ALPHA is X:8:8:8, and 32BPP is 8:8:8:8.
 *
 *  Note: Stereo requires two start (base) addresses, at least for plane B.
 *
 *  I don't understand why a display plane has to connect to a pipe,
 *  unless the display plane cannot be read by both pipes simultaniosly
 *  (which would be bad, because it would mean no "TwinView" feature.)
 *
 *  0x70184 DSPABASE Display plane A base address
 *  0x71184 DSPBBASE Display plane B base address
 *
 *  0x70188 DSPASTRIDE Display plane A stride (= pitch)
 *  0x71188 DSPBSTRIDE Display plane B stride (= pitch)
 *    31-16 UNKNOWN, 15CF/0001
 *     ?- 0 stride
 *
 *  4. Pipes
 *
 *  There are two pipes, which control the timing. 
 *
 *  0x60000 HTOTAL_A
 *  0x60004 HBLANK_A
 *  0x60008 HSYNC_A
 *  0x6000c VTOTAL_A
 *  0x60010 VBLANK_A
 *  0x60014 VSYNC_A
 *  0x6001c PIPEASRC
 *    26:16 horizontal (= HTOTAL displayed)
 *    10: 0 vertical   (= VTOTAL displayed)
 *  0x60020 BCLRPAT border color pattern
 *
 *  Identical registers for pipe B at 0x61000-0x61020
 *  On the i810, there also was the LCDTV_C register, which is probably
 *  not any longer supported.
 *
 *  0x60018 LCDTV_C LCD/TV-Out control register
 *
 *  0x70008 PIPEACONF
 *       31 enable (1=enabled)
 *       30 width (1=double, 0=single)
 *    29:27 reserved
 *       25 lock (1=locked) ; unclear
 *       24 (0=palette, 1=gamma)
 *    23: 0 reserved
 *  0x71008 PIPEBCONF
 *       31 enable (1=enabled)
 *    30:25 reserved
 *       24 (0=palette, 1=gamma)
 *    23: 0 reserved
 * 
 *  0x71400 UNKNOWN (= 0020008e)
 *
 *  5. Analog/Digital Ports
 *
 *  There are three dedicated Digital Video Ports and one Analog Port.
 *  DVOB and C are multixplexed with the AGP pins. The local flat panel
 *  (LFP) is normally connected to DVOA. DVOB and DVOC may be combined
 *  into one 24bit wide port. DVO port signals are:
 *
 *  CLK/CLK#  differental clock output up to 165 MHz
 *  D[11:0]   12bit data
 *  HSYNC     outgoing HSync with programmable polarity
 *  VSYNC     outgoing VSync with programmable polarity
 *  BLANK#    outgoing blank period or border period signal
 *  INTR      incoming interrupt (DVOA and DVOBC)
 *  CLKINT    incoming clock or second interrupt (DVOA and DVOBC)
 *  FLD/STL   incoming TV field (to synchronize overlay) or FP stall signal 
 *
 *  The DPLL registers control the phase locked loops that convert the
 *  incoming to the outgoing clock, or possibly some internal clock to
 *  the outgoing clock. They are associated to the corresponding pipe.
 *
 *  0x06014 DPLL_A          
 *  0x06018 DPLL_B    
 *    31:28 UNKNOWN (M or N ?)
 *    27:24 reserved
 *       23 post divisor (loop divide flag?)
 *       22 reserved
 *    21:16 post divisor (P ?)
 *       15 reserved
 *    14:13 UNKNOWN (=0 for LCD, =2 for TV ?)
 *    12: 0 reserved
 *
 *  0x61100 ADPA 
 *       31 DAC enable (1=enabled) (=monitor enabled)
 *       30 Pipe select (0=A, 1=B)
 *    29:16 reserved
 *       15 Use_VGA_HVPolarity (0=Sets HVPolarity)
 *    14:12 UNKNOWN
 *       11 VSync control disable (1=disabled)
 *       10 HSync control disable (1=disabled)
 *     9: 5 reserved
 *        4 VSync active (0=low, 1=high)
 *        3 HSync active (0=low, 1=high)
 *     2: 0 reserved
 * 
 *  For the 3 DVO ports, only the enable bit is known. Other bits may
 *  include interrupt handling, polarity control for blank and sync signals,
 *  configuration of clock vs. second interrupt and field vs. stall.
 *
 *  0x61120 DVOA                    
 *  0x61140 DVOB                    
 *  0x61160 DVOC                    
 *       31 DVO enable (1=enabled)
 *       30 Pipe select [?] (0=A, 1=B)
 *       29 UNKNOWN, set for Monitor/LFP xinerama
 *    15: 0 UNKNOWN (maybe identical to ADPA??)
 *       14 UNKNOWN, set for LFP1
 *        8 UNKNOWN, clear LFP, set for TV
 *        7 UNKNOWN, set for LFP1 and TV
 *        4 UNKNOWN, set for LFP12 (VSync active high?) clear for TV
 *        3 UNKNOWN, set for LFP12 (HSync active high?) clear for TV
 *        2 UNKNOWN, set for LFP12 TV
 *          			
 * Here LFPx refer to flat panel dumps. [?] Means a guess.
 *    
 *  0x61124 DVOA_SRCDIM
 *  0x61144 DVOB_SRCDIM
 *  0x61164 DVOC_SRCDIM
 *    23:12 (not needed ?)
 *    11: 0 (not needed ?)
 *          			    
 *  0x61180 LVDS meaning unknown (not needed ?)
 *
 *  6. Other registers
 *
 *  0x020cc UNKNOWN = 000c000c / 0000004c
 *
 *  There are now two watermark registers, with a format different from
 *  the I810 FW_BLC register.
 * 
 *  0x020d8 Watermark/Burst = 01080108 
 *  0x020dc Watermark/Burst = 00000108 
 *    31:27 reserved
 *    26:24 Burst B
 *    23:21 reserved
 *    20:16 Watermark B
 *    15:12 reserved
 *    11: 8 Burst A
 *     7: 6 reserved
 *     5: 0 Watermark A
 *   */

/* -------- I2C Driver Routines -------- */

/* bit locations in the register */

#define I2C_SCL_IN    (1 << 4)
#define I2C_SDA_IN    (1 << 12)

static void I810TvI2CGetBits(I2CBusPtr b, int *clock, int *data)
{
  I810Ptr pI810 = I810PTR(xf86Screens[b->scrnIndex]);
  register CARD32 val;

  I810_WR32(pI810, b->DriverPrivate.val, 0x0);
  val = I810_RD32(pI810, b->DriverPrivate.val);

  *clock = (val & I2C_SCL_IN) != 0;
  *data  = (val & I2C_SDA_IN) != 0;

  DEBUG(ErrorF("I810TvI2CGetBits(%p) val=0x%x, returns clock %d, data %d\n",
	       b, val, *clock, *data));
}

static void I810TvI2CPutBits(I2CBusPtr b, int clock, int data)
{
  I810Ptr pI810 = I810PTR(xf86Screens[b->scrnIndex]);

  if (clock) {
    I810_WR32(pI810, b->DriverPrivate.val, 0x5);
    I810_WR32(pI810, b->DriverPrivate.val, 0x4);
  } else { 
    I810_WR32(pI810, b->DriverPrivate.val, 0x7);
    I810_WR32(pI810, b->DriverPrivate.val, 0x6);
  }
  
  if (data) {
    I810_WR32(pI810, b->DriverPrivate.val, 0x500);
    I810_WR32(pI810, b->DriverPrivate.val, 0x400);
  } else { 
    I810_WR32(pI810, b->DriverPrivate.val, 0x700);
    I810_WR32(pI810, b->DriverPrivate.val, 0x600);
  }
  
  DEBUG(ErrorF("I810TvI2CPutBits(%p, %d, %d) val=0x%x\n", 
	       b, clock, data, val));
}

I2CBusPtr I810TvI2CInit (ScrnInfoPtr pScrn, char *name, int private)
{
  I2CBusPtr I2CPtr;

  I2CPtr = xf86CreateI2CBusRec();
  if (I2CPtr) {
    I2CPtr->BusName    = name;
    I2CPtr->scrnIndex  = pScrn->scrnIndex;
    I2CPtr->I2CPutBits = I810TvI2CPutBits;
    I2CPtr->I2CGetBits = I810TvI2CGetBits;
    /* Not sure about the delays. Matthew's code has HOLD_TIME = 100, and
     * uses that for all timings. The values used are about 100x larger 
     * than the default, so it should work, hopefully.
     */
    /* FIXME: Make these shorter, and see if it still works */
    I2CPtr->AcknTimeout  = 500; /* usec, default: 5 */
    I2CPtr->HoldTime     = 100; /* usec, default: 5 */
    I2CPtr->BitTimeout   = 500; /* usec, default: 5 */
    I2CPtr->ByteTimeout  = 500; /* usec, default: 5 */
    I2CPtr->AcknTimeout  = 500; /* usec, default: 5 */
    I2CPtr->StartTimeout = 500; /* usec, default: 5 */
    I2CPtr->RiseFallTime = 100; /* usec, default: 2 */
    I2CPtr->DriverPrivate.val = private; /* port addr */
  }
  return I2CPtr;
}

Bool I810TvBusInit (ScrnInfoPtr pScrn)
{
  Bool ok;
  I810Ptr pI810 = I810PTR(pScrn);
  int bus;

  pI810->TvChain = NULL;
  pI810->TvDev = NULL;
  /* Must be named TV0, TV1, etc.; otherwise I2C_ID won't work */
  if (pI810->arch.major == 0x30) {
    pI810->TvMaxBus = 2;
    pI810->TvBusses[0] = I810TvI2CInit (pScrn, "TV0", I810_GPIOB);
    pI810->TvBusses[1] = I810TvI2CInit (pScrn, "TV1", I830_GPIOC);
  } else {
    pI810->TvMaxBus = 1;
    pI810->TvBusses[0] = I810TvI2CInit (pScrn, "TV0", I810_GPIOB);
  }
  ok = TRUE;
  for (bus = 0; bus < pI810->TvMaxBus; bus++) {
    if (!xf86I2CBusInit(pI810->TvBusses[bus])) {
      ErrorF ("Cannot init Bus %i\n", bus);
      ok = FALSE;
    }
  }
  return ok;
}

/* -------- CRT -------- */

void I810SetTvRegs (I810Ptr pI810, int head, TVI810Regs *r)
{
  I810_H_WR32 (pI810, head, 0x60000, SetBitField (r->tvHDisplay,10:0,10:0) 
                                   | SetBitField (r->tvHTotal,11:0,27:16));
  I810_H_WR32 (pI810, head, 0x60004, SetBitField (r->tvHBlankStart,11:0,11:0) 
                                   | SetBitField (r->tvHBlankEnd,11:0,27:16));
  I810_H_WR32 (pI810, head, 0x60008, SetBitField (r->tvHSyncStart,11:0,11:0)
                                   | SetBitField (r->tvHSyncEnd,11:0,27:16));
  I810_H_WR32 (pI810, head, 0x6000C, SetBitField (r->tvVDisplay,10:0,10:0)
                                   | SetBitField (r->tvVTotal,11:0,27:16));
  I810_H_WR32 (pI810, head, 0x60010, SetBitField (r->tvVBlankStart,11:0,11:0)
                                   | SetBitField (r->tvVBlankEnd,11:0,27:16));
  I810_H_WR32 (pI810, head, 0x60014, SetBitField (r->tvVSyncStart,11:0,11:0)
                                   | SetBitField (r->tvVSyncEnd,11:0,27:16));
}

void I810GetTvRegs (I810Ptr pI810, int head, TVI810Regs *r)
{
  r->tvHDisplay    = SetBitField (I810_H_RD32 (pI810,head,0x60000),10: 0,10:0);
  r->tvHTotal      = SetBitField (I810_H_RD32 (pI810,head,0x60000),27:16,11:0);
  r->tvHBlankStart = SetBitField (I810_H_RD32 (pI810,head,0x60004),11: 0,11:0);
  r->tvHBlankEnd   = SetBitField (I810_H_RD32 (pI810,head,0x60004),27:16,11:0);
  r->tvHSyncStart  = SetBitField (I810_H_RD32 (pI810,head,0x60008),11: 0,11:0);
  r->tvHSyncEnd    = SetBitField (I810_H_RD32 (pI810,head,0x60008),27:16,11:0);
  r->tvVDisplay    = SetBitField (I810_H_RD32 (pI810,head,0x6000C),10: 0,10:0);
  r->tvVTotal      = SetBitField (I810_H_RD32 (pI810,head,0x6000C),27:16,11:0);
  r->tvVBlankStart = SetBitField (I810_H_RD32 (pI810,head,0x60010),11: 0,11:0);
  r->tvVBlankEnd   = SetBitField (I810_H_RD32 (pI810,head,0x60010),27:16,11:0);
  r->tvVSyncStart  = SetBitField (I810_H_RD32 (pI810,head,0x60014),11: 0,11:0);
  r->tvVSyncEnd    = SetBitField (I810_H_RD32 (pI810,head,0x60014),27:16,11:0);
}

void I810SetClock (I810Ptr pI810, int index, int m, int n, int p, int vco)
{
  register CARD32 val;

  I810_WR32 (pI810, I810_DCLK(index), SetBitField (n,9:0,25:16)
	                            | SetBitField (m,9:0,9:0));
  /* FIXME for index 3, bit 0 of lcdtv_c must be zero */
  val = I810_RD32 (pI810, I810_DCLK_0DS);
  val &= ~ (0x7c << index * 8);
  val |= (p & 0x7) << (index * 8 + 4) | (vco & 0x1) << (index * 8 + 2);
  I810_WR32 (pI810, I810_DCLK_0DS, val);
}

/* -------- Device enable/disable -------- */

int I810GetDevFlags (I810Ptr pI810)
{
  if (GetBit (I810_RD32(pI810, I810_LCDTV_C), 31)) {
    /* FIXME: may also be flat panel */
    return DEV_TELEVISION;
  } else {
    return DEV_MONITOR;
  }
}

/* -------- Device routines -------- */

void I810SetTvDevice (I810Ptr pI810, I2CChainPtr chain, Bool init)
{
  DPRINTF ("i810 set dev %p %i\n", chain, init);
  TVSetTvEncoder (&pI810->tvEncoder, chain);
  tvBusOk = TRUE;
  pI810->tvEncoder.InitRegs (&pI810->tvEncoder, PORT_I810);
}

/*
 * Free chain and device records of all tv chips
 */

void I810DestroyDevices (I810Ptr pI810)
{
  if (!pI810) return;
  if (pI810->TvChain) TVDestroyChain (pI810->TvChain);
  pI810->TvChain = NULL;
  pI810->TvDev = NULL;
  TVDestroyDevices (pI810->TvBusses, pI810->TvMaxBus);
}

/*
 * Free chain, device records, and all busses
 */

void I810DestroyBusses (I810Ptr pI810)
{
  if (!pI810) return;
  I810DestroyDevices (pI810);
  TVDestroyBusses (pI810->TvBusses, pI810->TvMaxBus);
}

void I810ProbeTvDevices (I810Ptr pI810)
{
  if (!pI810) return;
  I810UpdateTvState (pI810);
  tvState = TV_OFF;
  TVProbeCreateKnown (pI810->TvBusses, pI810->TvMaxBus, &pI810->TvChain); 
  /* FIXME: Is below necessary? */
  pI810->TvDev = pI810->TvChain ? pI810->TvChain->dev : NULL;
}

/* -------- -------- */

void I810UpdateTvState (I810Ptr pI810)
{
#ifndef FAKE_MMIO
  if (GetBit (I810_RD32(pI810, I810_LCDTV_C), 31)) {
    tvState = TV_ON;
  } else {
    if (tvState != TV_BARS) tvState = TV_OFF;
  }
#endif
}

void I810SetTestImage (I810Ptr pI810, TVEncoderRegs *r)
{
  /* FIXME chip registers ... */
  I810UpdateTvState (pI810);
  if (tvState != TV_ON) {
    /* Only do bars if not in tv-mode */
    pI810->tvEncoder.SetState (&pI810->tvEncoder, r, TV_BARS);
  }
}

void I810SetTvMode (I810Ptr pI810, TVRegs *r)
{
  TVState oldState, newState; /* hack for now, FIXME */

  I810UpdateTvState (pI810);
  oldState = tvState;
  newState = (r->devFlags & DEV_TELEVISION) ? TV_ON : TV_OFF;
  DPRINTF ("i810 set tv mode %i -> %i\n", oldState, newState);

  if (oldState == TV_ON && newState != TV_ON) 
  {
    I810_WR32 (pI810, I810_PWR_CLKC, 0x103);
    I810_WR32 (pI810, I810_LCDTV_C, 0x20000006); /* turn off tv */
  }
  if (oldState != newState) {
    pI810->tvEncoder.SetState (&pI810->tvEncoder, &r->enc, newState);
  }
  if (newState != TV_OFF) 
  {
    pI810->tvEncoder.SetPort (&pI810->tvEncoder, r->portEnc);
    pI810->tvEncoder.SetRegs (&pI810->tvEncoder, &r->enc, newState);
  }
  if (oldState != TV_ON && newState == TV_ON) {
    /* border enable flag in lcdtv_c */
    I810SetTvRegs (pI810, 0, &r->crtc.i810);
    I810SetClock (pI810, 3, 10, 1, 3, 0); /* m=10 n=1 p=3 vco=0 -> f=1 */
    I810_WR32 (pI810, I810_PWR_CLKC, 0x102);
    I810_WR32 (pI810, I810_LCDTV_C, 0xa0000007); /* turn on tv, turn off crt */
  }
  tvState = newState;
}

long I810GetTvStatus (I810Ptr pI810, int index)
{
  long result; 

  I810UpdateTvState (pI810);
  tvBusOk = TRUE;
  result = pI810->tvEncoder.GetStatus (&pI810->tvEncoder, index);
  return result;
}

TVConnect I810GetTvConnect (I810Ptr pI810)
{
  TVConnect result;

  I810UpdateTvState (pI810);
  tvBusOk = TRUE;
  result = pI810->tvEncoder.GetConnect (&pI810->tvEncoder);
  return result;
}

#if 0

{
  register long l;

  /* LCD/TV clock: m=10, n=1, p=3 (/8) just reflects clock from DVO */
  /* Intel PLL: Fclk = 4 * (Fref * (m+2)/(n+2)) >> p */
  /* I810CalcVCLK in xfree/.../driver/i810_driver.c */

  [0x60018/4] = ; /* lcdtv_c control */
  /* 0xa0000007 -> turn on tv, turn off crt */
  /* 0x20000006 -> turn off tv */
  /* vsync, hsync active low. blank# active high */
  /* bit7: border enable */

  [0x6001c/4]  /* ovract  overlay active start/end   -- not used */
  [0x60020/4]  /* bclrpat   border color */

  [0x600c/4] = n << 16 | m; /* lcd_clkd */
  l = [0x6010/4]; /* dclk_0ds */
  l &= ~0x7c000000;
  l |= p << 28 | 0 << 26; /* vco=0 -> divide by 4*m */
  [0x6010/4] = l; /* dclk_0ds */
  [0x6014/4] = ?; /* pwr_clkc */
  /* 0x102; -> turn on tv, turn off crt */
  /* 0x103; -> if crt_enable */
}

#endif

/*

06=C0/3F cfrb=1 m/s*=1
0D=80/0F only

0E=0B/F0 00=60/00 01=80/3F 02=6C/00 04=45/30 05=2E/00 06=C0/3F 07=64/00 
08=18/03 0C=33/00 0D=80/0F 0E=40/3F 0F=22/C0 10=00/0F 11=00/07 12=00/00 
13=40/18 14=0D/00 15=14/00 16=38/00 17=40/3F 18=13/00 19=A0/00 1A=20/00 
1B=0A/F0 1C=0E/C0 1D=07/F0 1E=0C/F0 1F=04/F0 20=10/EF 21=00/FE 2A=40/80 
2B=10/00 2C=84/00 2D=84/00 2E=BB/00 2F=9B/00 30=64/00 31=0A/00 32=EE/00 
33=26/80 34=2A/80 35=7E/00 36=9A/00 37=3F/80 38=E0/00 39=06/80 3A=04/00 
3B=FE/00 3C=7E/00 3D=00/F8 0A=19/00 08=00/FD 0B=30/00 08=01/FE 

*/

/*

DCLK0 25.175 MHz n=3 m=19   p= /16  vco= /4m
DCLK1 28.322 MHz n=16 m=83  p= /16  vco= /4m

*/

/* I830:

All but DDC1 are explicitely said to be a GMBUS (form of SMBUS ?)
DVOA   uses DDC2/I2C
DVOB/C use  M_DDC, M_I2C

05014 GPIOB controls probably I2C, with a device at 0x04 (LFP ?)

00006000= 00021207 ? DCLK_0D
00006004= 00031406 ? DCLK_1D
00006010= 0000888b ? DCLK_0DS
00006040= 00021207 FPA0
00006044= 00021207 FPA1
00006014= 808b0000 DPLL_A
00006048= 00021207 ? FPA2
0000604c= 00021207 ? FPA3 
00006018= 808b0000 DPLL_B
00070080= 00000000  
00070180= 00000000 DSPACTRL (= pipe A)
00071180= 01000000 DSPBCTRL (= pipe B)
00071280= 00000000
00071400= 80000000  
00060000= 031f027f HTOTAL_A  800,640
00060004= 03170287 HBLANK_A
00060008= 02ef028f HSYNC_A
0006000c= 020c01df VTOTAL_A  525,480
00060010= 020401e7 VBLANK_A
00060014= 01eb01e9 VSYNC_A
0006001c= 027f01df PIPEASRC  640x480
00061000= 031f027f HTOTAL_B  800,640
00061004= 03170287 HBLANK_B
00061008= 02ef028f HSYNC_B
0006100c= 020c01df VTOTAL_B  525,480
00061010= 020401e7 VBLANK_B
00061014= 01eb01e9 VSYNC_B
0006101c= 027f01df PIPEBSRC  640x480
000020d8= 01080108 ? FW_BLC Fifo Watermark/Burst Length Control
000020dc= 00000108 ? pipe B watermark 
000020cc= 000c000c
00070030= 00017e5f ?
00070008= 80000000 PIPEACONF (enable, no gamma)
00071008= 80000000 PIPEBCONF (enable, no gamma)
00071400= 0020008e
00061100= 80008018 ADPA (enable, pipe_a, vga_hvpol, sync_enable, ...)

(at least dump those registers...)

*/
