/* NVTV NV TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_tdfx.c,v 1.6 2002/03/24 17:42:48 dthierbach Exp $
 *
 * Contents:
 *
 * Routines to control the TV and the I2C bus on a NVidia card, as well
 * as other CRT/TV mode related register programming.
 *
 * Xbox Support added -- Milosch Meriac <xboxlinux@meriac.de>
 *
 */

#include "local.h" /* before everything else */
#include "xfree.h" 
#include "xf86_ansic.h"

#include <stdio.h>
#include <string.h>

#include "mmio.h"
#include "bitmask.h"
#include "xf86i2c.h"
#include "tv_nv.h"
#include "tv_i2c.h"
#include "nv_type.h"

#ifdef  XBOX_SUPPORT
#include <sys/io.h>
#include "xbox.h"
#endif /*XBOX_SUPPORT*/


/* -------- Crt access -------- */ 

/* these should go to driver/nv/nv_setup.c in the XFree driver;
   maybe they should be unified with the direct use of VGA_* elsewhere
   in the driver */

U008 readCrtNv (NVPtr pNv, int head, int reg)
{
#ifndef FAKE_CRTC
  VGA_WR08(pNv->riva.PCIO, CRT_INDEX(head), reg);
  return VGA_RD08(pNv->riva.PCIO, CRT_DATA(head));
#else
  FPRINTF ("%02x? ", reg);
  return 0;
#endif
}

void writeCrtNv (NVPtr pNv, int head, int reg, U008 val)
{
#ifdef FAKE_CRTC
  if (reg == 0x1f) {
#endif
  VGA_WR08(pNv->riva.PCIO, CRT_INDEX(head), reg);
  VGA_WR08(pNv->riva.PCIO, CRT_DATA(head), val);
#ifdef FAKE_CRTC
  }
  FPRINTF ("%02x=%02x ", reg, val);
#endif
}

void orCrtNv (NVPtr pNv, int head, int reg, U008 val)
{
#ifndef FAKE_CRTC
  register U008 tmp;

  VGA_WR08(pNv->riva.PCIO, CRT_INDEX(head), reg);
  tmp = VGA_RD08(pNv->riva.PCIO, CRT_DATA(head));
  VGA_WR08(pNv->riva.PCIO, CRT_DATA(head), tmp | val);
#else
  FPRINTF ("%02x|%02x ", reg, val);
#endif
}

void andCrtNv (NVPtr pNv, int head, int reg, U008 val)
{
#ifndef FAKE_CRTC
  register U008 tmp;

  VGA_WR08(pNv->riva.PCIO, CRT_INDEX(head), reg);
  tmp = VGA_RD08(pNv->riva.PCIO, CRT_DATA(head));
  VGA_WR08(pNv->riva.PCIO, CRT_DATA(head), tmp & val);
#else
  FPRINTF ("%02x&%02x ", reg, val);
#endif
}

void andOrCrtNv (NVPtr pNv, int head, int reg, U008 and, U008 or)
{
#ifndef FAKE_CRTC
  register U008 tmp;

  VGA_WR08(pNv->riva.PCIO, CRT_INDEX(head), reg);
  tmp = VGA_RD08(pNv->riva.PCIO, CRT_DATA(head));
  VGA_WR08(pNv->riva.PCIO, CRT_DATA(head), (tmp & and) | or);
#else
  FPRINTF ("%02x&%02x|%02x ", reg, and, or);
#endif
}

inline void unlockCrtNv (NVPtr pNv, int head)
{
#ifndef FAKE_CRTC
  writeCrtNv (pNv, head, 0x1f, 0x57); /* unlock extended registers */
#endif
}

inline void lockCrtNv (NVPtr pNv, int head)
{
#ifndef FAKE_CRTC
#if 0 /**/
  writeCrtNv (pNv, head, 0x1f, 0x99); /* lock extended registers */
#endif
#endif
}

/* the following routines are from nv_setup.c, but they take
   an NVPtr instead of a vgaHWPtr. */

void NVWriteGr(NVPtr pNv, CARD8 index, CARD8 value)
{
    VGA_WR08(pNv->riva.PVIO, VGA_GRAPH_INDEX, index);
    VGA_WR08(pNv->riva.PVIO, VGA_GRAPH_DATA,  value);
}

CARD8 NVReadGr(NVPtr pNv, CARD8 index)
{
    VGA_WR08(pNv->riva.PVIO, VGA_GRAPH_INDEX, index);
    return (VGA_RD08(pNv->riva.PVIO, VGA_GRAPH_DATA));
}

void NVWriteSeq(NVPtr pNv, CARD8 index, CARD8 value)
{
    VGA_WR08(pNv->riva.PVIO, VGA_SEQ_INDEX, index);
    VGA_WR08(pNv->riva.PVIO, VGA_SEQ_DATA,  value);
}

CARD8 NVReadSeq(NVPtr pNv, CARD8 index)
{
    VGA_WR08(pNv->riva.PVIO, VGA_SEQ_INDEX, index);
    return (VGA_RD08(pNv->riva.PVIO, VGA_SEQ_DATA));
}

void NVWriteAttr(NVPtr pNv, int head, CARD8 index, CARD8 value)
{
    volatile CARD8 tmp;

    tmp = MMIO_H_IN8(pNv->riva.PCIO, head, 
		     VGA_IOBASE_COLOR + VGA_IN_STAT_1_OFFSET);
#if 0
    if (pVga->paletteEnabled)
        index &= ~0x20;
    else
#endif
        index |= 0x20;
    MMIO_H_OUT8(pNv->riva.PCIO, head, VGA_ATTR_INDEX,  index);
    MMIO_H_OUT8(pNv->riva.PCIO, head, VGA_ATTR_DATA_W, value);
}

CARD8 NVReadAttr(NVPtr pNv, int head, CARD8 index)
{
    volatile CARD8 tmp;

    tmp = MMIO_H_IN8(pNv->riva.PCIO, head, 
		     VGA_IOBASE_COLOR + VGA_IN_STAT_1_OFFSET);
#if 0
    if (pVga->paletteEnabled)
        index &= ~0x20;
    else
#endif
        index |= 0x20;
    MMIO_H_OUT8(pNv->riva.PCIO, head, VGA_ATTR_INDEX, index);
    return MMIO_H_IN8(pNv->riva.PCIO, head, VGA_ATTR_DATA_R);
}

void NVWriteMiscOut(NVPtr pNv, CARD8 value)
{
    VGA_WR08(pNv->riva.PVIO, VGA_MISC_OUT_W, value);
}

CARD8 NVReadMiscOut(NVPtr pNv)
{
    return (VGA_RD08(pNv->riva.PVIO, VGA_MISC_OUT_R));
}

void NVDisablePalette (NVPtr pNv, int head)
{
    volatile CARD8 tmp;

    tmp = VGA_RD08(pNv->riva.PCIO + head * HEAD, 
		   VGA_IOBASE_COLOR + VGA_IN_STAT_1_OFFSET);
    VGA_WR08(pNv->riva.PCIO + head * HEAD, VGA_ATTR_INDEX, 0x20);
}

void NVDacDelay (NVPtr pNv, int head)
{
  volatile CARD8 tmp;

  tmp = VGA_RD08(pNv->riva.PCIO + head * HEAD, 
		 VGA_IOBASE_COLOR + VGA_IN_STAT_1_OFFSET);
  tmp = VGA_RD08(pNv->riva.PCIO + head * HEAD, 
		 VGA_IOBASE_COLOR + VGA_IN_STAT_1_OFFSET);
}

void NVWriteDacMask (NVPtr pNv, int head, CARD8 value)
{
    VGA_WR08(pNv->riva.PDIO + head * HEAD, VGA_DAC_MASK, value);
}

CARD8 NVReadDacMask (NVPtr pNv, int head)
{
    return (VGA_RD08(pNv->riva.PDIO + head * HEAD, VGA_DAC_MASK));
}

void NVWriteDacReadAddr (NVPtr pNv, int head, CARD8 value)
{
    VGA_WR08(pNv->riva.PDIO + head * HEAD, VGA_DAC_READ_ADDR, value);
}

void NVWriteDacWriteAddr (NVPtr pNv, int head, CARD8 value)
{
    VGA_WR08(pNv->riva.PDIO + head * HEAD, VGA_DAC_WRITE_ADDR, value);
}

void NVWriteDacData (NVPtr pNv, int head, CARD8 value)
{
    VGA_WR08(pNv->riva.PDIO + head * HEAD, VGA_DAC_DATA, value);
}

CARD8 NVReadDacData (NVPtr pNv, int head)
{
    return (VGA_RD08(pNv->riva.PDIO + head * HEAD, VGA_DAC_DATA));
}


/* -------- I2C Driver Routines -------- */

#ifdef XBOX_SUPPORT

/* FIXME TODO: Make this a proper I2C driver with local state */

/* It seems like the XBox always needs the subaddress to transmit
   or receive a byte. This may break some things, and makes it
   difficult to access some part of the PH encoder ... - Dirk */

static unsigned char XboxCmdBuffer;
static int XboxCmdIdx=0;
static int XboxAddr;

static Bool XBoxI2CPutByte(I2CDevPtr d, I2CByte data)
{
    int res;
    
    DPRINTF("I2C: put addr=0x%02x data=0x%02x\n",XboxAddr,data);
    switch(XboxCmdIdx)
    {
	case 0:
	    XboxCmdBuffer=data;
	    XboxCmdIdx++;
	    return TRUE;
	case 1:
	    res=I2CTransmitCmdData(XboxAddr,XboxCmdBuffer,data,FALSE);  
	    if(res<0)
	        fprintf(stdout,"I2C: transmit error (0x%08x)\n",res);
	    XboxCmdBuffer=0;
	    XboxCmdIdx=0;	    
	    return res>=0;	     
	default:
    	    fprintf(stdout,"I2C: output error !\n");	
	    XboxCmdBuffer=0;
	    XboxCmdIdx=0;	
	    return FALSE;
    }
    return FALSE;
}

static Bool XBoxI2CGetByte(I2CDevPtr d, I2CByte *data, Bool last)
{
    int res;
    
    DPRINTF("I2C: get addr=0x%02x\n",XboxAddr);
    
    if(XboxCmdIdx==1)
    {	
	res=I2CTransmitCmdGetReturn(XboxAddr,XboxCmdBuffer);
	*data = res>=0 ? res : 0xFF;
        if(res<0)
	    fprintf(stdout,"I2C: recv error (0x%08x)\n",res);	
	else
	{
	    DPRINTF("I2C: recv data=0x%02x\n",*data);
	}
    }        	
    else
    {
        DPRINTF("I2C: unexpected data index=%i\n",XboxCmdIdx);
	res=-1;
    }
    
    XboxCmdIdx=0;
    XboxCmdBuffer=0;
    return res>=0;
}

static void XBoxI2CStop(I2CDevPtr d)
{
    DPRINTF("I2C: stop\n");
    XboxCmdBuffer=0;            
    XboxCmdIdx=0;
}

/* FIXME Why is this necessary? Because of the former multiple busses? 
   If yes, then it can go away now. - Dirk */

static Bool XBoxI2CAddress(I2CDevPtr d, I2CSlaveAddr addr)
{
    XboxAddr=addr&~1;
    return (XboxAddr==0x8A)&&(d->pI2CBus->BusName[2]=='0');
}

I2CBusPtr XBoxI2CInit (ScrnInfoPtr pScrn, char *name)
{
  I2CBusPtr I2CPtr;

  I2CPtr = xf86CreateI2CBusRec();
  if (I2CPtr) {
    I2CPtr->BusName    = name;
    I2CPtr->scrnIndex  = pScrn->scrnIndex;
    I2CPtr->I2CPutByte = XBoxI2CPutByte;
    I2CPtr->I2CGetByte = XBoxI2CGetByte;
    I2CPtr->I2CAddress = XBoxI2CAddress;
    I2CPtr->I2CStop    = XBoxI2CStop;
    I2CPtr->AcknTimeout = 5;
    I2CPtr->DriverPrivate.ptr = NULL; /* FIXME TODO private state */
  }
  return I2CPtr;
}

/* FIXME TODO: There is probably only one I2C bus on the XBox, so
   include number of busses in state, and use only one. */

Bool XBoxBusInit (ScrnInfoPtr pScrn)
{
  Bool ok;
  NVPtr pNv = NVPTR(pScrn);

  pNv->TvChain = NULL;
  pNv->TvHead = 0;
  /* Must be named XB0, XB1, etc.; otherwise I2C_ID won't work */
  pNv->TvMaxBus = 1;
  pNv->TvBusses[0] = XBoxI2CInit (pScrn, "XB0");
  
  ok = TRUE;
  if (!xf86I2CBusInit(pNv->TvBusses[0])) {
      ErrorF ("Cannot init XBox bus\n");
      ok = FALSE;
  }
  return ok;
}

#else /* XBOX_SUPPORT */

Bool XBoxBusInit (ScrnInfoPtr pScrn)
{
  return FALSE;
}

#endif /* XBOX_SUPPORT */

/* FIXME: These depend on TvHead */

static void NVTvI2CGetBits(I2CBusPtr b, int *clock, int *data)
{
  NVPtr pNv = NVPTR(xf86Screens[b->scrnIndex]);
  unsigned char val;

  /* Get the result. */
  VGA_WR08(pNv->riva.PCIO, CRT_INDEX(pNv->TvHead), b->DriverPrivate.val);
  val = VGA_RD08(pNv->riva.PCIO, CRT_DATA(pNv->TvHead));

  *clock = (val & DDC_SCL_READ_MASK) != 0;
  *data  = (val & DDC_SDA_READ_MASK) != 0;

  DEBUG(ErrorF("NVTvI2CGetBits(%p) val=0x%x, returns clock %d, data %d\n",
	       b, val, *clock, *data));
}

static void NVTvI2CPutBits(I2CBusPtr b, int clock, int data)
{
    NVPtr pNv = NVPTR(xf86Screens[b->scrnIndex]);
    unsigned char val;

    VGA_WR08(pNv->riva.PCIO, CRT_INDEX(pNv->TvHead), b->DriverPrivate.val+1);
    val = VGA_RD08(pNv->riva.PCIO, CRT_DATA(pNv->TvHead)) & 0xf0;
    if (clock)
        val |= DDC_SCL_WRITE_MASK;
    else
        val &= ~DDC_SCL_WRITE_MASK;

    if (data)
        val |= DDC_SDA_WRITE_MASK;
    else
        val &= ~DDC_SDA_WRITE_MASK;

    VGA_WR08(pNv->riva.PCIO, CRT_INDEX(pNv->TvHead), b->DriverPrivate.val+1);
    VGA_WR08(pNv->riva.PCIO, CRT_DATA(pNv->TvHead), val | 0x1);
    
    DEBUG(ErrorF("NVTvI2CPutBits(%p, %d, %d) val=0x%x\n", 
		 b, clock, data, val));
}

static void NVTvI2CUDelay (I2CBusPtr b, int usec)
{
  NVPtr pNv = NVPTR(xf86Screens[b->scrnIndex]);
  U032 h1, h2, l1, l2;

#ifdef FAKE_MMIO
  return;
#endif
  l1 = MMIO_IN32 (pNv->riva.PTIMER, 0x400);
  h1 = MMIO_IN32 (pNv->riva.PTIMER, 0x410);
  l2 = l1 + usec * 1000;
  h2 = h1;
  if (l2 < l1) h2++; /* overflow */
  do {
    l1 = MMIO_IN32 (pNv->riva.PTIMER, 0x400);
    h1 = MMIO_IN32 (pNv->riva.PTIMER, 0x410);
  } while (h1 < h2 || (h1 == h2 && l1 < l2));
}

Bool NVIsTimerActive (NVPtr pNv)
{
  U032 h1, h2, l1, l2;

  /* FIXME check PMC device enable? Maybe enable it during it? */
  l1 = MMIO_IN32 (pNv->riva.PTIMER, 0x400);
  h1 = MMIO_IN32 (pNv->riva.PTIMER, 0x410);
  l2 = MMIO_IN32 (pNv->riva.PTIMER, 0x400);
  h2 = MMIO_IN32 (pNv->riva.PTIMER, 0x410);
  if (l1 != l2 || h1 != h2) {
    DPRINTF ("NVIsTimerActive: TRUE\n");
    return TRUE;
  }
  DPRINTF ("NVIsTimerActive: FALSE\n");
  return FALSE;
}

I2CBusPtr NVTvI2CInit (ScrnInfoPtr pScrn, char *name, int private)
{
  I2CBusPtr I2CPtr;

  I2CPtr = xf86CreateI2CBusRec();
  if (I2CPtr) {
    I2CPtr->BusName    = name;
    I2CPtr->scrnIndex  = pScrn->scrnIndex;
    I2CPtr->I2CPutBits = NVTvI2CPutBits;
    I2CPtr->I2CGetBits = NVTvI2CGetBits;
    if (NVIsTimerActive (NVPTR(pScrn))) {
      I2CPtr->I2CUDelay  = NVTvI2CUDelay;
    }
    I2CPtr->AcknTimeout = 5;
    I2CPtr->DriverPrivate.val = private; /* CRTC index for I2C Bus */
  }
  return I2CPtr;
}

Bool NVTvBusInit (ScrnInfoPtr pScrn)
{
  Bool ok;
  NVPtr pNv = NVPTR(pScrn);
  int bus;

  pNv->TvChain = NULL;
  pNv->TvHead = 0;
  /* Must be named TV0, TV1, etc.; otherwise I2C_ID won't work */
  pNv->TvMaxBus = 3;
  pNv->TvBusses[0] = NVTvI2CInit (pScrn, "TV0", 0x3e);
  pNv->TvBusses[1] = NVTvI2CInit (pScrn, "TV1", 0x36);
  pNv->TvBusses[2] = NVTvI2CInit (pScrn, "TV2", 0x50);
  
  ok = TRUE;
  for (bus = 0; bus < pNv->TvMaxBus; bus++) {
    if (!xf86I2CBusInit(pNv->TvBusses[bus])) {
      ErrorF ("Cannot init Bus %i\n", bus);
      ok = FALSE;
    }
  }
  return ok;
}

/* -------- CRT -------- */

/* The crt register settings are verified to be the same as the NVidia 
   driver settings, at least for the GeForce2 MX. This includes +1 instead
   of +0 (as in the X 'nv' driver) for horizSync. I do not know if all this 
   works for older/newer cards, since the bitranges might be different.  */

/* FIXME: May use disp_end_skew to horizontal move image? */

void NVSetCrtRegs (NVPtr pNv, int head, TVNvRegs *r)
{
  int horizDisplay    = (r->HDisplay/8)    - 1;
  int horizBlankStart = (r->HBlankStart/8) - 1;
  int horizSyncStart  = (r->HSyncStart/8)  + 1; /* verified */
  int horizSyncEnd    = (r->HSyncEnd/8)    + 1; /* verified */
  int horizBlankEnd   = (r->HBlankEnd/8)   - 1;
  int horizTotal      = (r->HTotal/8)      - 5;
  int vertDisplay     =  r->VDisplay       - 1;
  int vertBlankStart  =  r->VBlankStart    - 1;
  int vertSyncStart   =  r->VSyncStart     - 1;
  int vertSyncEnd     =  r->VSyncEnd       - 1;
  int vertBlankEnd    =  r->VBlankEnd      - 1;
  int vertTotal       =  r->VTotal         - 2;

  unlockCrtNv (pNv, head);
#ifdef FAKE_CRTC
  FPRINTF ("\ncrt[");
#endif
  writeCrtNv (pNv, head, 0x00, Set8Bits(horizTotal) );
  writeCrtNv (pNv, head, 0x01, Set8Bits(horizDisplay));
  writeCrtNv (pNv, head, 0x02, Set8Bits(horizBlankStart));
  writeCrtNv (pNv, head, 0x03, 
	      SetBitField(horizBlankEnd,4:0,4:0) 
	    | SetBit(7)); 
  /* index 03; 6:5 disp_end_skew = 0; 7 vert_retrace_acces = 1 */
  writeCrtNv (pNv, head, 0x04, Set8Bits(horizSyncStart));
  writeCrtNv (pNv, head, 0x05, 
	      SetBitField(horizBlankEnd,5:5,7:7)
	    | SetBitField(horizSyncEnd,4:0,4:0));
  writeCrtNv (pNv, head, 0x06, Set8Bits(vertTotal));
  writeCrtNv (pNv, head, 0x07, 
	      SetBitField(vertTotal,8:8,0:0)
	    | SetBitField(vertDisplay,8:8,1:1)
	    | SetBitField(vertSyncStart,8:8,2:2)
	    | SetBitField(vertDisplay,8:8,3:3)
	    | SetBit(4) /* line_compare_8 */
	    | SetBitField(vertTotal,9:9,5:5)
	    | SetBitField(vertDisplay,9:9,6:6)
	    | SetBitField(vertSyncStart,9:9,7:7));
  writeCrtNv (pNv, head, 0x08, 0x00); 
  writeCrtNv (pNv, head, 0x09, 
	      SetBitField(vertBlankStart,9:9,5:5)
	    | SetBit(6) /* line_compare_9 */
	    | SetBitFlag(r->flags,NV_FLAG_DOUBLE_SCAN,7));
  /* index 09: 4:0 multi_scan (num of lines-1) */
  writeCrtNv (pNv, head, 0x10, Set8Bits(vertSyncStart));
  writeCrtNv (pNv, head, 0x11, 
	      SetBitField(vertSyncEnd,3:0,3:0)
	    | SetBit(5));
  /* index 11: 5 disable_irq2 */
  writeCrtNv (pNv, head, 0x12, Set8Bits(vertDisplay));
  writeCrtNv (pNv, head, 0x14, 0x00); /* underline, double word */
  writeCrtNv (pNv, head, 0x15, Set8Bits(vertBlankStart));
  writeCrtNv (pNv, head, 0x16, Set8Bits(vertBlankEnd));
  /* FIXME */
  /* orCrtNv (pNv, head, 0x17, 0xc3); */ /* mode control */
  writeCrtNv (pNv, head, 0x18, 0xff); /* line_compare_7_0 */
  writeCrtNv (pNv, head, 0x1a, 0x00
                           | (r->HDisplay < 1280) ? 0x04 : 0x00);
  /* NV: 38, 3c / bc */
  /* 1c ?? */
  /* 28 framebuffer format !! */
  /* FIXME TODO: Arbitration 0 (0x1b), 1 (0x20) */
  andOrCrtNv (pNv, head, 0x25, 0x1f,
	      SetBitField(vertTotal,10:10,0:0)
	    | SetBitField(vertDisplay,10:10,1:1)
	    | SetBitField(vertSyncStart,10:10,2:2)
	    | SetBitField(vertBlankStart,10:10,3:3)
	    | SetBitField(horizBlankEnd,6:6,4:4)); 
  /* index 25, bit 5: offset_11 */
  andOrCrtNv (pNv, head, 0x2D, ~0x18, /* bit 5-4 */
	      SetBitField(horizTotal,8:8,0:0)
	    | SetBitField(horizDisplay,8:8,1:1)
	    | SetBitField(horizBlankStart,8:8,2:2)
	    | SetBitField(horizSyncStart,8:8,3:3));
  switch (pNv->arch.exact) 
  {
    case 0x11:
    case 0x20:
    case 0x2a:
      writeCrtNv (pNv, head, 0x41, 
	      SetBitField(vertTotal,     11:11,0:0)
	    | SetBitField(vertDisplay,   11:11,2:2)
	    | SetBitField(vertSyncStart, 11:11,4:4)
	    | SetBitField(vertBlankStart,11:11,6:6));
      break;
    case 0x17:
    case 0x18: 
    case 0x25:
    case 0x28:
    case 0x1f:
    case 0x30:
      writeCrtNv (pNv, head, 0x41, 
	      SetBitField(vertTotal,     12:11,0:0)
	    | SetBitField(vertDisplay,   12:11,2:2)
	    | SetBitField(vertSyncStart, 12:11,4:4)
	    | SetBitField(vertBlankStart,12:11,6:6));
      writeCrtNv (pNv, head, 0x55, 
	      SetBitField(horizTotal,     9:9,0:0)
	    | SetBitField(horizDisplay,   9:9,2:2)
	    | SetBitField(horizBlankStart,9:9,4:4)
	    | SetBitField(horizBlankEnd,  7:7,6:6));
      writeCrtNv (pNv, head, 0x56, 
	      SetBitField(horizSyncStart, 9:9,0:0)
	    | SetBitField(horizSyncEnd,   5:5,2:2)
	    | SetBitField(vertBlankEnd,   8:8,4:4));
      /* FIXME bit 7 ?? */
      break;
  }
#ifdef FAKE_CRTC
  FPRINTF ("]\n");
#endif
  lockCrtNv (pNv, head);
}

void NVGetCrtRegs (NVPtr pNv, int head, TVNvRegs *r)
{
  int horizDisplay;
  int horizBlankStart;
  int horizSyncStart;
  int horizSyncEnd;
  int horizBlankEnd;
  int horizTotal;
  int vertDisplay;
  int vertBlankStart;
  int vertSyncStart;
  int vertSyncEnd;
  int vertBlankEnd;
  int vertTotal;
  int bit;

  unlockCrtNv (pNv, head);
  horizTotal      = Set8Bits(readCrtNv(pNv, head, 0x00)) 
                  | SetBitField(readCrtNv(pNv, head, 0x2d),0:0,8:8);
  horizDisplay    = Set8Bits(readCrtNv(pNv, head, 0x01)) 
                  | SetBitField(readCrtNv(pNv, head, 0x2d),1:1,8:8);
  horizBlankStart = Set8Bits(readCrtNv(pNv, head, 0x02))
                  | SetBitField(readCrtNv(pNv, head, 0x2d),2:2,8:8);
  horizBlankEnd   = SetBitField(readCrtNv(pNv, head, 0x03),4:0,4:0)
                  | SetBitField(readCrtNv(pNv, head, 0x05),7:7,5:5)
                  | SetBitField(readCrtNv(pNv, head, 0x25),4:4,6:6);
  horizSyncStart  = Set8Bits(readCrtNv(pNv, head, 0x04)) 
                  | SetBitField(readCrtNv(pNv, head, 0x2d),3:3,8:8);
  horizSyncEnd    = SetBitField(readCrtNv(pNv, head, 0x05),4:0,4:0);

  vertTotal       = Set8Bits(readCrtNv(pNv, head, 0x06)) 
                  | SetBitField(readCrtNv(pNv, head, 0x07),0:0,8:8)
                  | SetBitField(readCrtNv(pNv, head, 0x07),5:5,9:9)
                  | SetBitField(readCrtNv(pNv, head, 0x25),0:0,10:10);
  vertDisplay     = Set8Bits(readCrtNv(pNv, head, 0x12)) 
	          | SetBitField(readCrtNv(pNv, head, 0x07),1:1,8:8)
	          | SetBitField(readCrtNv(pNv, head, 0x07),6:6,9:9)
	          | SetBitField(readCrtNv(pNv, head, 0x25),1:1,10:10);
  vertBlankStart  = Set8Bits(readCrtNv(pNv, head, 0x15))
	          | SetBitField(readCrtNv(pNv, head, 0x07),3:3,8:8)
	          | SetBitField(readCrtNv(pNv, head, 0x09),5:5,9:9)
	          | SetBitField(readCrtNv(pNv, head, 0x25),3:3,10:10);
  vertBlankEnd    = Set8Bits(readCrtNv(pNv, head, 0x16));
  vertSyncStart   = Set8Bits(readCrtNv(pNv, head, 0x10))
	          | SetBitField(readCrtNv(pNv, head, 0x07),2:2,8:8)
	          | SetBitField(readCrtNv(pNv, head, 0x07),7:7,9:9)
	          | SetBitField(readCrtNv(pNv, head, 0x25),2:2,10:10);
  vertSyncEnd     = SetBitField(readCrtNv(pNv, head, 0x11),3:0,3:0);

  bit = 0;
  switch (pNv->arch.exact) 
  {
    case 0x11:
    case 0x20:
    case 0x2a:
      vertTotal      |= SetBitField(readCrtNv (pNv, head, 0x41),0:0,11:11);
      vertDisplay    |= SetBitField(readCrtNv (pNv, head, 0x41),2:2,11:11);
      vertSyncStart  |= SetBitField(readCrtNv (pNv, head, 0x41),4:4,11:11);
      vertBlankStart |= SetBitField(readCrtNv (pNv, head, 0x41),6:6,11:11);
      break;
    case 0x17:
    case 0x18: 
    case 0x25:
    case 0x28:
    case 0x1f:
    case 0x30:
      vertTotal      |= SetBitField(readCrtNv (pNv, head, 0x41),0:0,12:11);
      vertDisplay    |= SetBitField(readCrtNv (pNv, head, 0x41),2:2,12:11);
      vertSyncStart  |= SetBitField(readCrtNv (pNv, head, 0x41),4:4,12:11);
      vertBlankStart |= SetBitField(readCrtNv (pNv, head, 0x41),6:6,12:11);
      horizTotal      |= SetBitField(readCrtNv (pNv, head, 0x55),0:0,9:9);
      horizDisplay    |= SetBitField(readCrtNv (pNv, head, 0x55),2:2,9:9);
      horizBlankStart |= SetBitField(readCrtNv (pNv, head, 0x55),4:4,9:9);
      horizBlankEnd   |= SetBitField(readCrtNv (pNv, head, 0x55),6:6,7:7);
      horizSyncStart  |= SetBitField(readCrtNv (pNv, head, 0x56),0:0,9:9);
      horizSyncEnd    |= SetBitField(readCrtNv (pNv, head, 0x56),2:2,5:5);
      vertBlankEnd    |= SetBitField(readCrtNv (pNv, head, 0x56),4:4,8:8);
      bit++;
      break;
  }

  /* It is not clear if blank end is relative to blank start or
     total. I have chosen total here, since blank end is equal to
     total normally, anyway. */

  horizBlankEnd  |= SetBitField (horizTotal,8:7,8:7);
  while (horizBlankEnd < horizTotal) horizBlankEnd += SetBit (7+bit);

  horizSyncEnd  |= SetBitField (horizSyncStart,8:5,8:5);
  while (horizSyncEnd < horizSyncStart) horizSyncEnd += SetBit (5+bit);

  vertBlankEnd  |= SetBitField (vertTotal,10:8,10:8);
  while (vertBlankEnd < vertTotal) vertBlankEnd += SetBit (8+bit);

  vertSyncEnd  |= SetBitField (vertSyncStart,10:4,10:4);
  while (vertSyncEnd < vertSyncStart) vertSyncEnd += SetBit (4);

  r->HDisplay    = (horizDisplay    + 1) * 8;   
  r->HBlankStart = (horizBlankStart + 1) * 8;
  r->HBlankEnd   = (horizBlankEnd   + 1) * 8;  
  r->HSyncStart	 = (horizSyncStart  - 1) * 8; 
  r->HSyncEnd  	 = (horizSyncEnd    - 1) * 8;   
  r->HTotal    	 = (horizTotal      + 5) * 8;     
     
  r->VDisplay    = vertDisplay    + 1;
  r->VBlankStart = vertBlankStart + 1;
  r->VBlankEnd   = vertBlankEnd   + 1;
  r->VSyncStart  = vertSyncStart  + 1;
  r->VSyncEnd    = vertSyncEnd    + 1;
  r->VTotal      = vertTotal      + 2;

  r->clock = 0;  
  lockCrtNv (pNv, head);
}

void NVGetTvRegs (NVPtr pNv, int head, TVNvRegs *r)
{
  unlockCrtNv (pNv, head);
  r->latency = SetBitField(readCrtNv(pNv, head, 0x28),5:3,2:0);
  r->slave.VTotal     = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x720);
  r->slave.VSyncStart = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x724);
  r->slave.VSyncEnd   = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x728);
  r->slave.HTotal     = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x72c);
  r->slave.HSyncStart = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x730);
  r->slave.HSyncEnd   = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x734);
  r->slave.Unknown    = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x738);
  /* FIXME: VTotal only bits 10-0, HTotal only bits 10-0/11-0 */
  lockCrtNv (pNv, head);
}

void NVSetTvRegs (NVPtr pNv, int head, TVNvRegs *r)
{
  /* don't set latency */
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x720, r->slave.VTotal);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x724, r->slave.VSyncStart);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x728, r->slave.VSyncEnd);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x72c, r->slave.HTotal);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x730, r->slave.HSyncStart);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x734, r->slave.HSyncEnd);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x738, r->slave.Unknown);
}

void NVGetFpRegs (NVPtr pNv, int head, TVNvFpRegs *r)
{
  r->HDisplay    = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x820);
  r->HSyncStart  = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x82c);
  r->HSyncEnd    = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x830);
  r->HTotal      = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x824);
  r->HValidStart = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x834);
  r->HValidEnd   = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x838);
  r->HCrtc       = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x828);
  r->VDisplay    = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x800);
  r->VSyncStart  = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x80c);
  r->VSyncEnd    = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x810);
  r->VTotal      = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x804);
  r->VValidStart = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x814);
  r->VValidEnd   = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x818);
  r->VCrtc       = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x808);
}

void NVSetFpRegs (NVPtr pNv, int head, TVNvFpRegs *r)
{
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x820, r->HDisplay);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x82c, r->HSyncStart); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x830, r->HSyncEnd); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x824, r->HTotal); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x834, r->HValidStart); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x838, r->HValidEnd); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x828, r->HCrtc); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x800, r->VDisplay); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x80c, r->VSyncStart);  
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x810, r->VSyncEnd);    
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x804, r->VTotal);      
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x814, r->VValidStart); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x818, r->VValidEnd); 
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x808, r->VCrtc); 
}

/* FIXME: Hack: Using SyncRegs for TV BlankRegs */

void NVSetBlankRegs (NVPtr pNv, int head, TVNvRegs *r)
{
  DPRINTF ("nv blank h=%i-%i v=%i-%i\n", 
	   r->HSyncStart, r->HSyncEnd, r->VSyncStart, r->VSyncEnd);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x704, r->VSyncStart);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x708, r->VSyncEnd);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x70c, r->HSyncStart);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x710, r->HSyncEnd);
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x714, 0x00000000);
}

/* All registers (except actual "switching") */

void NVGetAllRegs (NVPtr pNv, int head, TVRegs *r)
{
  r->devFlags = NVGetDevFlags (pNv, head);
  NVGetCrtRegs (pNv, head, &r->crtc.nv);
  r->crtc.nv.clock = NVGetClock (pNv, head);
  NVGetTvRegs (pNv, head, &r->crtc.nv);
}

void NVSetAllRegs (NVPtr pNv, int head, TVRegs *r)
{
  NVSetCrtRegs (pNv, head, &r->crtc.nv);
  NVModifyClock (pNv, head, r->crtc.nv.clock);
  if (r->devFlags & DEV_TELEVISION)
    NVSetTvRegs (pNv, head, &r->crtc.nv);
}

void NVSetCrtLayout (NVPtr pNv, int head, int addr, int ofs)
{
  unlockCrtNv (pNv, head);
    /* crt offset (= crt line increment)? in 0x13,8bit 0x19,10:8,5:7 */

  writeCrtNv (pNv, head, 0x0c, SetBitField(addr,15:8,7:0));
  writeCrtNv (pNv, head, 0x0d, Set8Bits(addr));
  writeCrtNv (pNv, head, 0x13, Set8Bits(ofs));
  writeCrtNv (pNv, head, 0x19, SetBitField(addr,20:16,4:0)
                           | SetBitField(ofs,10:8,7:5));
#if 0 /* FIXME */
  andOrCrtNv (pNv, head, 0x25, ~0xd0); /* VERTICAL_EXTRA */
  andOrCrtNv (pNv, head, 0x2d, ~0xd0); /* HORIZONTAL_EXTRA */
  writeCrtNv (pNv, head, 0x42, ); /* START_ADDR */
#else
  andCrtNv (pNv, head, 0x25, (U008) ~0xd0); /* VERTICAL_EXTRA  */
  andCrtNv (pNv, head, 0x2d, (U008) ~0xd0); /* HORIZONTAL_EXTRA */
  andCrtNv (pNv, head, 0x42, (U008) ~0x0f);
#endif
  lockCrtNv (pNv, head);
}

void NVGetCrtLayout (NVPtr pNv, int head, int *paddr, int *pofs)
{
  int addr, ofs;

  unlockCrtNv (pNv, head);
  addr = Set8Bits(readCrtNv (pNv, head, 0x0d))
       | SetBitField(readCrtNv (pNv, head, 0x0c),7:0,15:8)
       | SetBitField(readCrtNv (pNv, head, 0x19),4:0,20:16)
       | SetBitField(readCrtNv (pNv, head, 0x2d),7:5,25:23)
       | SetBitField(readCrtNv (pNv, head, 0x25),7:6,27:26)
       | SetBitField(readCrtNv (pNv, head, 0x42),3:0,31:28);
  ofs = Set8Bits(readCrtNv (pNv, head, 0x13))
      | SetBitField(readCrtNv (pNv, head, 0x19),7:5,10:8)
      | SetBitField(readCrtNv (pNv, head, 0x25),5:5,11:11);
  lockCrtNv (pNv, head);
  if (paddr) *paddr = addr;
  if (pofs)  *pofs  = ofs;
}

/* -------- Clock / PLL -------- */

long NVCalcMNP (NVArch *arch, long targetFreq, int *mOut, int *nOut, int *pOut)
{
  long bestFreq;
  long bestDelta;
  long delta;
  int minVco, maxVco;
  int maxM;
  int n, m, p;
  long realFreq;

  /* FIXME TODO: adjust minVco and maxVco, if maxVco != 2 * minVco */
  minVco = arch->minVco;
  maxVco = arch->maxVco;
  for (n = 0; arch->freqM[n] != 0 && targetFreq >= arch->freqM[n]; n++);
  maxM = arch->maxM[n];
  bestDelta = -1;
  bestFreq = 0;
  for (p = 0; p <= arch->maxP; p++) {
    if ((targetFreq << p) >= minVco && (targetFreq << p) <= maxVco)
    {
      for (m = arch->minM; m <= maxM; m++) {
	/* round both n and p during divisions */
	n = (((targetFreq << p) * m + 
	      arch->crystalFreq/2) / arch->crystalFreq);
	if (n > 255) continue;
	realFreq = (((arch->crystalFreq * n + m/2) / m) + 
		    ((1 << p) >> 1)) >> p;
	if (realFreq > targetFreq)
	  delta = realFreq - targetFreq;
	else
	  delta = targetFreq - realFreq;
	if (bestDelta == -1 || delta < bestDelta)
	{
	  *mOut	 = m;
	  *nOut	 = n;
	  *pOut	 = p;
	  bestDelta = delta;
	  bestFreq  = realFreq;
	  /* Accept immediately within 0.5% relative error if above freqM 
             limit, to keep m as low as possible */
	  if (delta == 0 || 
	      (targetFreq >= arch->freqM[0] && (targetFreq / delta > 200)))
	  {
	    return bestFreq;
	  }
	}
      }
    }
  }
  return bestFreq;
}

void NVProgClock (NVPtr pNv, int head, int m, int n, int p)
{
  int reg;
  register CARD32 val;

  DPRINTF ("nv set mnp %i m=%i n=%i p=%i\n", head, m, n, p);
  /* Modify post divisor only after PLL has settled */
  reg = (head==1) ? 0x520 : 0x508;
  val = MMIO_IN32 (pNv->riva.PRAMDAC, reg);
  val &= ~SetBitField(-1,15:0,15:0);
  val |= SetBitField(m,7:0,7:0) | SetBitField(n,7:0,15:8);
  MMIO_OUT32 (pNv->riva.PRAMDAC, reg, val);
  xf86usleep (1); /* 1 us */
  val &= ~SetBitField(-1,18:16,18:16);
  val |= SetBitField(p,2:0,18:16);
  MMIO_OUT32 (pNv->riva.PRAMDAC, reg, val);
}

long NVGetClock (NVPtr pNv, int head)
{
  int n, m, p;
  register U032 l;
  register long clock;

#ifdef FAKE_MMIO
  return 25000;
#endif
  l = MMIO_IN32 (pNv->riva.PRAMDAC, (head==1) ? 0x520 : 0x508);
  m = SetBitField (l,7:0,7:0);
  n = SetBitField (l,15:8,7:0);
  p = SetBitField (l,18:16,2:0);
  clock = (((pNv->arch.crystalFreq * n + m/2) / m) + ((1 << p) >> 1)) >> p;
  DPRINTF ("nv get clock pll=%08X, m=%i n=%i p=%i, Fref=%li Fclk=%li kHz\n", 
	   l, m, n, p, pNv->arch.crystalFreq, clock);
  return clock;
}

long NVSetClock (NVPtr pNv, int head, long clock)
{
  int m, n, p;

  DPRINTF ("nv set clock %li\n", clock);
  if (clock == 0) return 0;

  clock = NVCalcMNP (&pNv->arch, clock, &m, &n, &p);
  NVProgClock (pNv, head, m, n, p);
  return clock;
}

long NVModifyClock (NVPtr pNv, int head, long clock)
{
  long oldClock, newClock;
  int m, n, p;

  DPRINTF ("nv mod clock %li\n", clock);
  if (clock == 0) return 0;
  oldClock = NVGetClock (pNv, head);
  newClock = NVCalcMNP (&pNv->arch, clock, &m, &n, &p);
  DPRINTF ("nv mod old=%li new=%li\n", oldClock, newClock);
  /* FIXME */
  /* first check against old pll frequency. If better than newly
     calc'd frequency, don't change (may have different limits). */
  NVProgClock (pNv, head, m, n, p);
  return newClock;
}

/* -------- Device enable/disable -------- */

int NVGetDevFlags (NVPtr pNv, int head)
{
  int devFlags;

  unlockCrtNv (pNv, head);
  devFlags = 0;

  /* FIXME: Check that display is enabled at all */
  /* TODO: can use crt 54 bit 7-6 to check if internal TV is active */

  if (readCrtNv (pNv, head, 0x28) & 0x80) {
    /* CRTC is slaved. */
    if (readCrtNv (pNv, head, 0x33) & 0x01) {
      devFlags |= DEV_FLATPANEL;
    } else {
      devFlags |= DEV_TELEVISION;
    }
  }

  if ((MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x608) & 0x00010000) == 0) { 
    devFlags |= DEV_MONITOR;
  }

  /* FIXME: Check data_src instead */
/*
  if ((MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x608) & 0x00010000) == 0) { 
    devFlags |= DEV_OVERLAY;
  }
*/

  lockCrtNv (pNv, head);
  return devFlags;
}

void NVEnableMonitor (NVPtr pNv, int head, Bool state)
{
  if (state) {
    DPRINTF ("nv enable monitor\n");

#if 0 /* FIXME experimental */
    /* reset attr ff */
    MMIO_H_IN8 (pNv->riva.PCIO, pNv->TvHead, 0x3da); 
    /* enable disp */
    MMIO_H_OUT8 (pNv->riva.PCIO, pNv->TvHead, 0x3c0, 0x10 | 0x20); 
#endif
#if 1 /* FIXME experimental */
    /* TEST_CONTROL Pwrdown DAC */
    MMIO_H_AND32 (pNv->riva.PRAMDAC, head, 0x608, ~0x00010000); 
#endif
#if 1 /* FIXME standard */
    andCrtNv (pNv, pNv->TvHead, 0x1A, 0x3f);    /* Enable vsync, hsync */
#endif

  } else {

    DPRINTF ("nv disable monitor\n");
#if 1
    /* TEST_CONTROL Pwrdown DAC */
    MMIO_H_OR32 (pNv->riva.PRAMDAC, head, 0x608, 0x00010000); 
#endif
#if 0
    /* reset attr ff */
    MMIO_H_IN8 (pNv->riva.PCIO, pNv->TvHead, 0x3da); 
    /* disable disp */
    MMIO_H_OUT8 (pNv->riva.PCIO, pNv->TvHead, 0x3c0, 0x00);
#endif
#if 0 /* turns both off */
    DPRINTF ("Seq 0x01=%02X\n", NVReadSeq (pNv, 0x01));
    NVWriteSeq (pNv, 0x01, NVReadSeq (pNv, 0x01) | 0x20); /* display off */
#endif
#if 1 /* FIXME standard */
    orCrtNv (pNv, pNv->TvHead, 0x1A, 0x80); /* disable hsync */
#endif
  }
}

/* -------- Device routines -------- */

void NVUpdateTvState (NVPtr pNv);

void NVSetTvDevice (NVPtr pNv, I2CChainPtr chain, Bool init)
{
  DPRINTF ("nv set dev %p %i\n", chain, init);
  TVSetTvEncoder (&pNv->tvEncoder, chain);
  if (init) {
    unlockCrtNv (pNv, pNv->TvHead);
    if (readCrtNv (pNv, pNv->TvHead, 0x28) & 0x80) {
      /* Already in TV mode, don't init */
      tvState = TV_ON;
    } else {
      tvBusOk = TRUE;
      pNv->tvEncoder.InitRegs (&pNv->tvEncoder, PORT_NVIDIA);
      if (!tvBusOk) ERROR ("I2C Bus error\n");
    }
    lockCrtNv (pNv, pNv->TvHead);
  }
}

/*
 * Free chain and device records of all tv chips
 */

void NVDestroyDevices (NVPtr pNv)
{
  if (!pNv) return;
  if (pNv->TvChain) TVDestroyChain (pNv->TvChain);
  pNv->TvChain = NULL;
  TVDestroyDevices (pNv->TvBusses, pNv->TvMaxBus);
}

/*
 * Free chain, device records, and all busses
 */

void NVDestroyBusses (NVPtr pNv)
{
  if (!pNv) return;
  NVDestroyDevices (pNv);
  TVDestroyBusses (pNv->TvBusses, pNv->TvMaxBus);
}

void NVProbeTvDevices (NVPtr pNv)
{
  if (!pNv) return;
  unlockCrtNv (pNv, pNv->TvHead); 
  NVUpdateTvState (pNv);
  TVProbeCreateKnown (pNv->TvBusses, pNv->TvMaxBus, &pNv->TvChain); 
}

/* -------- Multiple heads -------- */

void NVAssocFeature (NVPtr pNv, int head, int mask)
{
  int h;

  DPRINTF ("NVAssoc %i %08X\n", head, mask);
  MMIO_H_OR32 (pNv->riva.PCRTC, head, 0x860, mask);
  for (h = 0; h < pNv->arch.heads; h++) {
    if (h != head) {
      MMIO_H_AND32 (pNv->riva.PCRTC, h, 0x860, ~mask);
    }
  }
}

void NVSetVideoHead (NVPtr pNv, int head)
{
  if (pNv->arch.heads == 1) return;
  NVAssocFeature (pNv, head, 0x00001000); /* Video Scaler */
}

int NVSetTvHead (NVPtr pNv, int head)
{
  int h;
  int tv;

  if (pNv->arch.heads == 1) return head;
  /* FIXME: Don't set if one head is in TV mode ... test this on FP */
  /* FIXME: This should be checked somewhere else... */
  tv = -1;
  for (h = 0; h < pNv->arch.heads; h++) {
    unlockCrtNv (pNv, h);
    if (((readCrtNv (pNv, h, 0x28) & 0x80) != 0) &&
	((readCrtNv (pNv, h, 0x33) & 0x01) == 0)) tv = h;
    lockCrtNv (pNv, h);
  }
  if (tv >= 0) {
    tvState = TV_ON;
    pNv->TvHead = tv;
  } else {
    tvState = TV_OFF;
    pNv->TvHead = head;
    NVAssocFeature (pNv, head, 0x00000110); /* TV & DDC */
  }
  return pNv->TvHead;
}

void NVCopyHead (NVPtr pNv, int fromHead, int toHead)
{
  DPRINTF ("NVCopyHead %i -> %i %08lX %08lX\n", fromHead, toHead,
	   MMIO_H_IN32 (pNv->riva.PCRTC, fromHead, 0x800), 
	   MMIO_H_IN32 (pNv->riva.PCRTC, fromHead, 0x804));
  /* NV_PCRTC_START */
  MMIO_H_OUT32 (pNv->riva.PCRTC, toHead, 0x800, 
    MMIO_H_IN32 (pNv->riva.PCRTC, fromHead, 0x800)); 
  /* NV_PCRTC_CONFIG */
  MMIO_H_OUT32 (pNv->riva.PCRTC, toHead, 0x804, 
    MMIO_H_IN32 (pNv->riva.PCRTC, fromHead, 0x804));
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, toHead, 0x600, 
    MMIO_H_IN32 (pNv->riva.PRAMDAC, fromHead, 0x600));
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, toHead, 0x848, 
    MMIO_H_IN32 (pNv->riva.PRAMDAC, fromHead, 0x848));
  MMIO_H_OR32 (pNv->riva.PRAMDAC, toHead, 0x880, 0x10000000);
  /* FIXME !! */
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, toHead, 0x814, 
    MMIO_H_IN32 (pNv->riva.PRAMDAC, fromHead, 0x814));
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, toHead, 0x818, 
    MMIO_H_IN32 (pNv->riva.PRAMDAC, fromHead, 0x818));
  unlockCrtNv (pNv, fromHead);
  unlockCrtNv (pNv, toHead);
#if 0
  writeCrtNv (pNv, toHead, 0x44, 0x03); /* FIXME Test */
  writeCrtNv (pNv, toHead, 0x21, 0xfa); /* FIXME Test */
  writeCrtNv (pNv, toHead, 0x26, 0x14); /* FIXME Test */
  writeCrtNv (pNv, toHead, 0x22, readCrtNv (pNv, fromHead, 0x22));
  writeCrtNv (pNv, toHead, 0x23, readCrtNv (pNv, fromHead, 0x23));
#endif
  andOrCrtNv (pNv, toHead, 0x28, ~0x3, 0x3 & readCrtNv (pNv, fromHead, 0x28));
  /* FIXME LCD off */
  writeCrtNv (pNv, toHead, 0x33, 0x00);
  lockCrtNv (pNv, fromHead);
  lockCrtNv (pNv, toHead);
}

void NVCopyGrSeq (NVPtr pNv, int fromHead, int toHead)
{
  DPRINTF ("NVCopyGrSeq %i -> %i\n", fromHead, toHead);
}

void NVInitGrSeq (NVPtr pNv)
{
  NVWriteSeq(pNv, 0x01, 0x01);	/* reenable display */
  NVWriteSeq(pNv, 0x02, 0x0f);	/* if depth != 1 */
  NVWriteSeq(pNv, 0x03, 0x00);	
  NVWriteSeq(pNv, 0x04, 0x0E);	/* if depth < 8 */
  NVWriteSeq(pNv, 0x00, 0x03);	/* End Reset */
  NVWriteGr(pNv, 0x00, 0x00);
  NVWriteGr(pNv, 0x01, 0x00);
  NVWriteGr(pNv, 0x02, 0x00);
  NVWriteGr(pNv, 0x03, 0x00);
  NVWriteGr(pNv, 0x04, 0x00);    /* depth != 1 */
  NVWriteGr(pNv, 0x05, 0x40);    /* depth != 1 && depth != 4 */
  NVWriteGr(pNv, 0x06, 0x05);
  NVWriteGr(pNv, 0x07, 0x0f);
  NVWriteGr(pNv, 0x08, 0xff);
}

void NVInitDac (NVPtr pNv, int head)
{
  int i;

  NVWriteDacMask (pNv, head, 0xFF);
  NVWriteDacWriteAddr (pNv, head, 0x00);
  for (i = 0; i < 256; i++) {
    NVWriteDacData (pNv, head, i);
    NVDacDelay (pNv, head);
    NVWriteDacData (pNv, head, i);
    NVDacDelay (pNv, head);
    NVWriteDacData (pNv, head, i);
    NVDacDelay (pNv, head);
  }
  NVDisablePalette (pNv, head);
}

void NVInitAttr (NVPtr pNv, int head)
{
  int i;

  DPRINTF ("NVInitAttr\n");
  /* reset attr ff */
  MMIO_H_IN8 (pNv->riva.PCIO, head, 0x3da); 
  for (i = 0; i <= 0x0f; i++) {
    MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, i | 0x20);
    MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, i);
  }
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x10 | 0x20); /* enable disp */
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x41); /* mode control */
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x11 | 0x20); 
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x00); /* overscan value */
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x12 | 0x20); 
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x0f); /* plane enable */
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x13 | 0x20); 
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x00); /* pel panning */
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x14 | 0x20); 
  MMIO_H_OUT8 (pNv->riva.PCIO, head, 0x3c0, 0x00); /* color select */
}

void NVCopyCursor (NVPtr pNv, int fromHead, int toHead)
{
  DPRINTF ("NVCopyCursor %i -> %i\n", fromHead, toHead);
  /* NV_PCRTC_CURSOR */
  MMIO_H_OUT32 (pNv->riva.PCRTC, toHead, 0x80c,
    MMIO_H_IN32 (pNv->riva.PCRTC, fromHead, 0x80c));
  /* NV_PCRTC_CURSOR_CONFIG */
  MMIO_H_OUT32 (pNv->riva.PCRTC, toHead, 0x810,
    MMIO_H_IN32 (pNv->riva.PCRTC, fromHead, 0x810));
  /* CURSOR_CNTRL */
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, toHead, 0x320,
    MMIO_H_IN32 (pNv->riva.PRAMDAC, fromHead, 0x320));
  /* CURSOR_DATA */
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, toHead, 0x324,
    MMIO_H_IN32 (pNv->riva.PRAMDAC, fromHead, 0x324));
  /* CURSOR_DATA */
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, toHead, 0x328,
    MMIO_H_IN32 (pNv->riva.PRAMDAC, fromHead, 0x328));
  unlockCrtNv (pNv, fromHead);
  unlockCrtNv (pNv, toHead);
  writeCrtNv (pNv, toHead, 0x2f, readCrtNv (pNv, fromHead, 0x2f));
  writeCrtNv (pNv, toHead, 0x30, readCrtNv (pNv, fromHead, 0x30));
  writeCrtNv (pNv, toHead, 0x31, readCrtNv (pNv, fromHead, 0x31));
  lockCrtNv (pNv, fromHead);
  lockCrtNv (pNv, toHead);
}

void NVSetCursorPos (NVPtr pNv, int head, int x, int y)
{
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, head, 0x300, 
    SetBitField(x,11:0,11:0) | SetBitField(y,11:0,27:16));
}
  
void NVGetCursorPos (NVPtr pNv, int head, int *x, int *y)
{
  register int v;

  v = MMIO_H_IN32 (pNv->riva.PRAMDAC, head, 0x300);
  *x = SetBitField(v,11:0,11:0);
  *y = SetBitField(v,27:16,11:0);
}

void NVSetVideoPointOut (NVPtr pNv, int x, int y)
{
  MMIO_OUT32 (pNv->riva.PRAMDAC, 0x948,
    SetBitField(x,11:0,11:0) | SetBitField(y,11:0,27:16));
  MMIO_OUT32 (pNv->riva.PRAMDAC, 0x94C, 
    SetBitField(x,11:0,11:0) | SetBitField(y,11:0,27:16));
}
  
void NVGetVideoPointOut (NVPtr pNv, int *x, int *y)
{
  register int v;

  v = MMIO_IN32 (pNv->riva.PRAMDAC, 0x948);
  *x = SetBitField(v,11:0,11:0);
  *y = SetBitField(v,27:16,11:0);
}

Bool NVVertIntrEnabled (NVPtr pNv, int head) 
{
  if (MMIO_H_IN32 (pNv->riva.PCRTC, head, 0x140) & SetBit(0)) {
    return TRUE;
  } else {
    return FALSE;
  }
}
  
void NVCopyArb (NVPtr pNv, int fromHead, int toHead)
{
  /* copy aribtration registers */
  unlockCrtNv (pNv, fromHead);
  unlockCrtNv (pNv, toHead);
  writeCrtNv (pNv, toHead, 0x20, readCrtNv (pNv, fromHead, 0x2f));
  writeCrtNv (pNv, toHead, 0x1b, readCrtNv (pNv, fromHead, 0x30));
  lockCrtNv (pNv, fromHead);
  lockCrtNv (pNv, toHead);
}

/* -------- Digital Video Output (DVO) Ports -------- */

/* This seems to be wildly different for different architectures.
   Some seem to be able to use both DVO ports on the same head? 
   Also, it seems necessary to associate the second TV port.
*/

void NVEnableDVO_NV17 (NVPtr pNv, int head, int path)
{
  int h;

  for (h = 0; h < pNv->arch.heads; h++) {
    unlockCrtNv (pNv, h);
    if (h == head) {
      orCrtNv (pNv, h, 0x59, 0x01);
      if (path & 0x1) {
	orCrtNv (pNv, h, 0x33, 0x10);
      }
    } else {
      andCrtNv (pNv, h, 0x59, ~0x01);
      if (path & 0x1) {
	andCrtNv (pNv, h, 0x33, ~0x10);
      }    
    }
    lockCrtNv (pNv, h);
  }
}

void NVEnableDVO_NV18 (NVPtr pNv, int head, int path)
{
  int h;

  for (h = 0; h < pNv->arch.heads; h++) {
    unlockCrtNv (pNv, h);
    if (h == head) {
      if (path & 0x1) {
	orCrtNv (pNv, h, 0x33, 0x10); // ???
	orCrtNv (pNv, h, 0x59, 0x01);
      } else {
	andCrtNv (pNv, h, 0x59, ~0x01);
      }
      if (path & 0x2) {
	orCrtNv (pNv, h, 0x33, 0x20); // ???
	orCrtNv (pNv, h, 0x9f, 0x10);
      } else {
	andCrtNv (pNv, h, 0x9f, ~0x10);
      }
    } else {
      if (path & 0x1) {
	andCrtNv (pNv, h, 0x59, ~0x01);
      }    
      if (path & 0x2) {
      } else {
	andCrtNv (pNv, h, 0x9f, ~0x10);
      } 
    }
    lockCrtNv (pNv, h);
  }
}

void NVEnableDVO_NV25 (NVPtr pNv, int head, int path)
{
  int h;

  for (h = 0; h < pNv->arch.heads; h++) {
    unlockCrtNv (pNv, h);
    if (h == head) {
      orCrtNv (pNv, h, 0x59, 0x01);
      if (path & 0x1) {
	orCrtNv (pNv, h, 0x33, 0x10);
      } else {
	andCrtNv (pNv, h, 0x33, ~0x10);
      }
      if (path & 0x2) {
	orCrtNv (pNv, h, 0x33, 0x20);
      } else {
	andCrtNv (pNv, h, 0x33, ~0x20);
      }
    } else {
      if (path & 0x1) {
	andCrtNv (pNv, h, 0x33, ~0x10);
      } else {
	orCrtNv (pNv, h, 0x33, 0x10);
      }    
      if (path & 0x2) {
	andCrtNv (pNv, h, 0x33, ~0x20);
      } else {
	orCrtNv (pNv, h, 0x33, 0x20);
      } 
    }
    lockCrtNv (pNv, h);
  }
}

void NVEnableDVO_NVX (NVPtr pNv, int head, int path)
{
  int h;

  for (h = 0; h < pNv->arch.heads; h++) {
    unlockCrtNv (pNv, h);
    if (h == head) {
      orCrtNv (pNv, h, 0x59, 0x01);
      if (path & 0x1) {
	orCrtNv (pNv, h, 0x33, 0x10);
      }
      if (path & 0x2) {
	orCrtNv (pNv, h, 0x33, 0x20);
      }
    } else {
      if (path & 0x1) {
	andCrtNv (pNv, h, 0x33, ~0x10);
      }    
      if (path & 0x2) {
	andCrtNv (pNv, h, 0x33, ~0x20);
      } 
    }
    lockCrtNv (pNv, h);
  }
}

void NVEnableDVO (NVPtr pNv, int head, int path)
{
  switch (pNv->arch.exact) 
  {
    case 0x17:
      NVEnableDVO_NV17 (pNv, head, path);
      break;
    case 0x18:
      NVEnableDVO_NV18 (pNv, head, path);
      break;
    case 0x25:
    case 0x28:
      NVEnableDVO_NV25 (pNv, head, path);
      break;
    case 0x1f:
    case 0x30:
      NVEnableDVO_NVX (pNv, head, path);
      break;
    case 0x2a:
      break;
  }
}

void NVDisableDVO (NVPtr pNv, int head, int path)
{
  unlockCrtNv (pNv, head);
  switch (pNv->arch.exact) 
  {
    case 0x17:
    case 0x20: /* FIXME ??? */
    case 0x25:
    case 0x28:
      andCrtNv (pNv, head, 0x59, ~0x01);
      break;
    case 0x18:
    case 0x1f:
    case 0x30:
      if (path & 0x1) andCrtNv (pNv, head, 0x59, ~0x01);
      if (path & 0x2) andCrtNv (pNv, head, 0x9f, ~0x10);
      break;
    case 0x2a:
      break;
    case 0x11: /* FIXME ??? */
    default:
      andCrtNv (pNv, head, 0x33, ~0x10);
      break;
  }
  lockCrtNv (pNv, head);
}

/* -------- Get Port / Encoder Registers -------- */

void NVGetEncRegsPort (NVPtr pNv, TVEncoderRegs *r, int *port)
{
  tvBusOk = TRUE;
  pNv->tvEncoder.GetPort (&pNv->tvEncoder, port);
  pNv->tvEncoder.GetRegs (&pNv->tvEncoder, r);
  if (!tvBusOk) ERROR ("I2C Bus error\n");
}

void NVGetPort (NVPtr pNv, int *port)
{
  register CARD32 tv_setup;
  
  *port |= PORT_VSYNC_HIGH | PORT_HSYNC_HIGH | 
           PORT_BLANK_LOW  | PORT_BLANK_OUT  | 
           PORT_PCLK_HIGH  | PORT_PCLK_MASTER | PORT_FORMAT_RGB;
  tv_setup = MMIO_H_IN32 (pNv->riva.PRAMDAC, pNv->TvHead, 0x700);
  *port |= (GetBit(tv_setup,0) ? PORT_SYNC_MASTER : PORT_SYNC_SLAVE);
}

/* -------- -------- */

/* Intermediate interface to TV. This might not be needed for the
 * XFree driver, and should be moved to a different file in that case.
 */

static int nvCrtIntr;

inline void prepCrtNv (NVPtr pNv, int head)
{
  // unlockCrtNv (pNv, head);
  nvCrtIntr = readCrtNv (pNv, head, 0x11) & 0x10;
  // andCrtNv (pNv, head, 0x11, 0xef);   /* disable vert intr */
  // orCrtNv (pNv, head, 0x21, 0x80);    /* "virtual" changes */
  writeCrtNv (pNv, head, 0x21, 0xfa);
}

inline void commitCrtNv (NVPtr pNv, int head)
{
  // andCrtNv (pNv, head, 0x21, 0x7f);      /* commit changes */
  // orCrtNv (pNv, head, 0x11, nvCrtIntr);  /* enable vert intr again */
}

/* FIXME merge this with dev flags and flat panel stuff */

void NVUpdateTvState (NVPtr pNv)
{
  if (tvState != TV_UNKNOWN) return; /* FIXME remove */
  if (readCrtNv (pNv, pNv->TvHead, 0x28) & 0x80) {
    tvState = TV_ON;
  } else {
    if (tvState != TV_BARS) tvState = TV_OFF;
  }
}

void NVSetTestImage (NVPtr pNv, TVEncoderRegs *r)
{
  /* FIXME chip registers ... */
  DPRINTF ("nv tv testimage\n");
  unlockCrtNv (pNv, pNv->TvHead);
  if (! (readCrtNv (pNv, pNv->TvHead, 0x28) & 0x80)) {
    /* Only do bars if not in tv-mode */
    pNv->tvEncoder.SetState (&pNv->tvEncoder, r, TV_BARS);
  }
  lockCrtNv (pNv, pNv->TvHead);
}

/* FIXME: Check that if not changing mode, only registers are set,
   i.e. crt/tv regs in tv-mode and crt regs in non-tv mode

   Use a "safe mode" flag to avoid switching devices if the kernel
   controls them?
*/

/* FIXME: Cleanup and split state to support overlay */

/* For NV11, the DATA_SRC fields does not seem to be supported.
   It is possible to "turn off" the overlay with COMP_SRC, but the
   PCLK and the DAC data seem always to be routed to the TV port,
   even if crt 28 is not set. This is strange, because it should
   be at least possible to switch to the VIP.
*/

void NVSetTvMode (NVPtr pNv, TVRegs *r)
{
  TVState newstate;

  DPRINTF ("NVSetTvMode { ");
  /* FIXME: Don't switch state if not necessary */
  NVUpdateTvState (pNv);
  /* FIXME: Just convert to old 'state' for now */
  newstate = (r->devFlags & DEV_TELEVISION) ? TV_ON : TV_OFF;
  DPRINTF ("nv tv %02X state %i -> %i\n", r->devFlags, tvState, newstate);

  unlockCrtNv (pNv, pNv->TvHead);
  /* FIXME Test this */
  if (pNv->TvHead == 1 && tvState != TV_ON) {
    DPRINTF ("init 2nd ");
    unlockCrtNv (pNv, 0);
    writeCrtNv (pNv, 0, 0x44, 0x03); 
    NVInitGrSeq (pNv);
    NVInitDac (pNv, pNv->TvHead);
    writeCrtNv (pNv, 0, 0x44, 0x00); 
    NVCopyArb (pNv, 0, pNv->TvHead);
    NVInitAttr (pNv, pNv->TvHead);
  }

#if 0 /* FIXME experimental NV4 blank color */
  MMIO_H_OUT32 (pNv->riva.PRAMDAC, pNv->TvHead, 0x714, 0x00000000);
#endif
#if 1 /* FIXME experimental NV3/NV4 overscan color */
  /* crt index 2a */
#endif

#if 0 /* FIXME experimental */
  /* FIXME Head ... doesn't work yet ... */
  /* Experimental: Wait for vert retrace... */
  MMIO_H_OUT8 (pNv->riva.PCIO, pNv->TvHead, 0x3c4, 0x01);
  MMIO_H_OUT8 (pNv->riva.PCIO, pNv->TvHead, 0x3c5, 0x01);
  /* Wait until in image */
  while (MMIO_H_IN8 (pNv->riva.PCIO, pNv->TvHead, 0x3da) & 0x08);
  /* Wait until retrace starts */
  while (! MMIO_H_IN8 (pNv->riva.PCIO, pNv->TvHead, 0x3da) & 0x08);
#if 0
  /* Wait until again in image */
  while (MMIO_H_IN8 (pNv->riva.PCIO, pNv->TvHead, 0x3da) & 0x08);
#endif
#endif /* FIXME */

  tvBusOk = TRUE;
  if (tvState == TV_ON && newstate != TV_ON) 
  {
    register CARD32 coeff;

    prepCrtNv (pNv, pNv->TvHead);
    andCrtNv (pNv, pNv->TvHead, 0x28, 0x7f);
    commitCrtNv (pNv, pNv->TvHead);

    coeff = MMIO_IN32 (pNv->riva.PRAMDAC, 0x50C);
    switch (pNv->TvHead)
    {
      case 0:
	coeff &= ~0x00030000; /* VS_PCLK_TV = None */
	break;
      case 1:
	coeff &= ~0x000c0000; /* VS_PCLK_TV2 = None */
	break;
      default:
	/* FIXME error */
        break;
    }
    MMIO_OUT32 (pNv->riva.PRAMDAC, 0x50C, coeff); /* PLL Coeff Sel */

    /* Reset data path */
    MMIO_H_AND32 (pNv->riva.PRAMDAC, pNv->TvHead, 0x700, ~0x00001270);

    NVEnableMonitor (pNv, pNv->TvHead, TRUE);

#if 0 /* FIXME experimental */
    {
      int tmp;
      int i;
      U032 a, b;

      /* FIXME Head ... */
      /* Disable slave CRTC mode */
      tmp = readCrtNv (pNv, pNv->TvHead, 0x28) & 0x7b; 
      writeCrtNv (pNv, pNv->TvHead, 0x28, tmp); 
      for (i = 0; i < 25; i++) {
	/* NV_PCRTC_RASTER */
	a = MMIO_H_IN32 (pNv->riva.PCRTC, pNv->TvHead, 0x808); 
	xf86usleep (2000); /* 2ms */
	b = MMIO_H_IN32 (pNv->riva.PCRTC, pNv->TvHead, 0x808); 
	DPRINTF ("off %08x %08x\n", a, b);
	if (a == b) {
	  writeCrtNv (pNv, pNv->TvHead, 0x28, tmp | 0x80); 
	  xf86usleep (1000); /* 2ms */
	  writeCrtNv (pNv, pNv->TvHead, 0x28, tmp); 
	} else {
	  break;
	}
      }
    }
#endif
  }
  if (newstate != TV_OFF) 
  {
    register CARD32 tv_setup;
    
    /* power down flatpanel */
    /* FIXME this is a guess */
    if (pNv->arch.heads == 1) {
      /* For one head, power down FPClk and TMDS */
      MMIO_H_OR32 (pNv->riva.PRAMDAC, pNv->TvHead, 0x880, 0x30000000);
    } else {
      /* For multiple heads, power down FPClk only */
      /* Powering down TMDS even on the other head causes problems */
      MMIO_H_OR32 (pNv->riva.PRAMDAC, pNv->TvHead, 0x880, 0x10000000);
    }

    /* TV_SETUP: At least for NV11, other head *must* be in same mode */ 
    /* This might cause problems with the flatpanel ... */

    /* FIXME: If changing sync polarity, should do it the right 
       way round, and allow a delay to let the lines become tristate */

    tv_setup = 0
      | ((r->devFlags & DEV_OVERLAY) ? 0x1110 : 0)
/* FIXME test this below */ 
/*
      | ((r->portHost & PORT_VSYNC_POLARITY) ? 0 : SetBit(4))
      | ((r->portHost & PORT_HSYNC_POLARITY) ? 0 : SetBit(3))
*/
      | ((r->portHost & PORT_SYNC_DIR) ? 0 : SetBit(0));
    MMIO_H_OUT32 (pNv->riva.PRAMDAC, pNv->TvHead, 0x700, tv_setup);
    if (pNv->arch.heads == 2) {
      MMIO_H_OUT32 (pNv->riva.PRAMDAC, pNv->TvHead ^ 1, 0x700, tv_setup);
    }
    pNv->tvEncoder.SetPort (&pNv->tvEncoder, r->portEnc);
    pNv->tvEncoder.SetRegs (&pNv->tvEncoder, &r->enc, newstate);
  }
  if (tvState != newstate) {
    pNv->tvEncoder.SetState (&pNv->tvEncoder, &r->enc, newstate);
  }

  prepCrtNv (pNv, pNv->TvHead);
  if (r->devFlags & DEV_OVERLAY) {
    NVSetBlankRegs (pNv, pNv->TvHead, &r->crtc.nv); 
  } else {
    NVSetAllRegs (pNv, pNv->TvHead, r);
  }

  /* FIXME disable DVO ? */

  if (newstate == TV_ON) 
  {
    register CARD32 coeff;

    andCrtNv (pNv, pNv->TvHead, 0x33, ~0x01);
    NVEnableDVO (pNv, pNv->TvHead, 0x1); /* 1st DVO port */

    /* For NV17orBetter, i.e. NV17, NV18, NV25, NV30: */
    /* crt 53, crt 54, 6808c0 */
    if ((pNv->arch.exact >= 0x17 && pNv->arch.exact < 0x20) ||
	(pNv->arch.exact >= 0x25 && pNv->arch.exact < 0x2A) ||
        (pNv->arch.exact >= 0x30)) 
    {
      writeCrtNv (pNv, pNv->TvHead, 0x53, 0x00); 
      writeCrtNv (pNv, pNv->TvHead, 0x54, 0x00); 
      /* Bit 8 of 68x8c0 must be zero on the TV head. It is unclear what
         this bit means, so we set it to one on the other head (like
	 the resulting driver settings). */
      MMIO_H_AND32 (pNv->riva.PRAMDAC, pNv->TvHead, 0x8c0, ~SetBit(8));
      MMIO_H_OR32 (pNv->riva.PRAMDAC, pNv->TvHead ^ 1, 0x8c0, SetBit(8));
    }
    if (pNv->arch.exact == 0x25) {
      /* FIXME only in slave mode?? */
      writeCrtNv (pNv, pNv->TvHead, 0x52, 0x00); 
      /* FIXME Not sure, maybe 0x44 ?? */
      writeCrtNv (pNv, pNv->TvHead, 0x58, 0x83); /* FIXME Test this */

    }

    /* Check for proper clock frequency if PORT_PCLK_SLAVE */
    if ((r->portHost & PORT_PCLK_MODE) != PORT_PCLK_MASTER) {
      long clock = NVGetClock (pNv, pNv->TvHead);
      if (clock < pNv->tvEncoder.minClock || clock > pNv->tvEncoder.maxClock)
      {
	ERROR ("NVSetTvMode: slave clock out of range\n");
	r->portHost &= ~PORT_PCLK_MODE; /* Master mode */
      }
    }

    /* FIXME TODO: Only one display is driven by the external clock. 
       This will probably not work for flatpanels. */

    coeff = MMIO_IN32 (pNv->riva.PRAMDAC, 0x50C);
    coeff &= ~0x011f0000; /* TVCLK_SRC=Ext, VS_PCLK_TV 1&2, TV_CLK_RATIO=DB1 */
    /* FIXME: DoublePix doesn't work yet */
    coeff |= SetBitFlag(r->crtc.nv.flags,NV_FLAG_DOUBLE_PIX,24);
    switch (pNv->TvHead)
    {
      case 0:
	coeff &= ~0x10000000; /* VCLK_RATIO = DB1 */
	if ((r->portHost & PORT_PCLK_MODE) == PORT_PCLK_MASTER) {
	  if (r->devFlags & DEV_OVERLAY) {
	    coeff |=  0x00010000; /* VS_PCLK_TV = VSCLK */
	  } else {
	    coeff |=  0x00030000; /* VS_PCLK_TV = BOTH */
	  }
	}
	break;
      case 1:
	coeff &= ~0x20000000; /* VCLK2_RATIO = DB1 */
	if ((r->portHost & PORT_PCLK_MODE) == PORT_PCLK_MASTER) {
	  if (r->devFlags & DEV_OVERLAY) {
	    coeff |=  0x00040000; /* VS_PCLK_TV2 = VSCLK */
	  } else {
	    coeff |=  0x000c0000; /* VS_PCLK_TV2 = BOTH */
	  }
	}
	break;
      default:
	/* FIXME error */
        break;
    }
    MMIO_OUT32 (pNv->riva.PRAMDAC, 0x50C, coeff); /* PLL Coeff Sel */
    if (! (r->devFlags & DEV_OVERLAY)) {
      andOrCrtNv (pNv, pNv->TvHead, 0x28, (U008) ~0xf8, 
	      (U008) (0x80 | SetBitField(r->crtc.nv.latency,2:0,5:3)));
      if (! (r->devFlags & DEV_MONITOR)) 
	NVEnableMonitor (pNv, pNv->TvHead, FALSE);
    }
  }
  commitCrtNv (pNv, pNv->TvHead);
  if (!tvBusOk) ERROR ("I2C Bus error\n");
  tvState = newstate;
  lockCrtNv (pNv, pNv->TvHead);
  DPRINTF ("} NVSetTvMode\n");
}

/* This is a much simpler version for the XBox. There is no state,
   just the crtc/fp/encoder registers are changed. */

void XBoxSetTvMode (NVPtr pNv, TVRegs *r)
{
  DPRINTF ("XBoxSetTvMode { ");
  if (r->devFlags == (DEV_TELEVISION | DEV_FLATPANEL)) {
    tvState = TV_ON;
    unlockCrtNv (pNv, pNv->TvHead);
    tvBusOk = TRUE;
    pNv->tvEncoder.SetPort (&pNv->tvEncoder, r->portEnc);
    pNv->tvEncoder.SetRegs (&pNv->tvEncoder, &r->enc, TV_ON);
    if (!tvBusOk) ERROR ("I2C Bus error\n");
    prepCrtNv (pNv, pNv->TvHead);
    NVSetCrtRegs (pNv, pNv->TvHead, &r->crtc.nv);
    NVSetFpRegs (pNv, pNv->TvHead, &r->crtc.nv.fp);
    commitCrtNv (pNv, pNv->TvHead);
    lockCrtNv (pNv, pNv->TvHead);
  }
  DPRINTF ("} XBoxSetTvMode\n");
}

long NVGetTvStatus (NVPtr pNv, int index)

{
  long result; 

  unlockCrtNv (pNv, pNv->TvHead);
  NVUpdateTvState (pNv);
  tvBusOk = TRUE;
  result = pNv->tvEncoder.GetStatus (&pNv->tvEncoder, index);
  if (!tvBusOk) ERROR ("I2C Bus error\n");
  lockCrtNv (pNv, pNv->TvHead);
  return result;
}

TVConnect NVGetTvConnect (NVPtr pNv)
{
  TVConnect result;

  unlockCrtNv (pNv, pNv->TvHead);
  NVUpdateTvState (pNv);
  tvBusOk = TRUE;
  result = pNv->tvEncoder.GetConnect (&pNv->tvEncoder);
  if (!tvBusOk) ERROR ("I2C Bus error\n");
  lockCrtNv (pNv, pNv->TvHead);
  return result;
}

/* 

      head  TVCLK VCLK VCLK2
<=400  0     DB2  DB1
       1     DB2       DB1
>400   0     DB1  DB1
       1     DB1       DB1

TVCLK doubles pixels.

*/

/*

CIO:

0x3c0: ARX index data_write
0x3c1: AR data   data_read
0x3c2: INP0
0x3da: INP0_WRITE (MONO/COLOR)  ; w: reset ARX to write
0x3ca: INP0_READ

VIO:
0x102: VSE1
0x3c2: MISC_WR
0x3c3: VSE2
0x3c4: SRX index
0x3c5: SR data
0x3cc: MISC_RD
0x3ce: GRX index
0x3cf: GR data


3c2.w bit 1: MISC enable cpu access to vidmem
3c2.w bit 1: MISC enable/disable display RAM
3c2.w bit 4: MISC disable video

3c3   bit 0: VGA enable/disable
Description: Disables access to display memmory and the other
             VGA's ports

3c4.01 bit 5: display off -- also TV disp off (only show b/w vertical lines)

*/

/*

1 port  for NV11, NV20, NV17
2 ports for NV25, NV18, NV1F, NV30

DVO ports: [NV17 or better] = NV17, NV18, NV25, NV30

Q: When to initialize the second head?
A: Not when it's already in TV mode.

 */
