/* NVTV Philips Model2 TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_ph.c,v 1.12 2002/04/18 06:02:42 dthierbach Exp $
 *
 * Contents:
 *
 * Routines to access the Philips Model2 (SAA7104 etc.) encoder chip 
 * registers via the I2C bus.
 *
 */

#include "local.h" /* before everything else */

#include "bitmask.h"
#include "tv_chip.h"
#include "tv_i2c.h"
#include "tv_ph1_saa7102.h"
#include "tv_ph2_saa7104.h"

void TVPh2SetPort (TVEncoderObj *this, int port)
{
  TVPhSetPort (this, port);
  if ((port & PORT_PCLK_POLARITY) == PORT_PCLK_HIGH) {
    TVWriteBus (this->dev, 0x54, 0x02); /* edge1: falling, edge2: rising */
  } else {
    TVWriteBus (this->dev, 0x54, 0x00); /* edge1: raising, edge2: falling */
  }
  /* FIXME TODO correct polarity with bit 0 */
}

void TVPh2GetPort (TVEncoderObj *this, int *port)
{
  I2CByte res;

  DPRINTF ("ph1 get port\n");
  TVPhGetPort (this, port);
  TVReadBus (this->dev, 0x54, &res); 
  *port |= ((res & 0x3) == 0x00 ? PORT_PCLK_LOW : PORT_PCLK_HIGH);
}

void TVPh2InitRegs (TVEncoderObj *this, int port)
{
  register I2CDevPtr dev = this->dev;

  DPRINTF ("ph2 init %04X\n", this->chip);
  TVPh2SetPort (this, port);
  TVPhInitRegs (this);
  TVWriteBus (dev, 0x37, 0x12);  /* NV: 0x37= 0x12 */ 
  TVWriteBus (dev, 0x54, 0x02); /* NV */
}

void TVPh2SetRegs (TVEncoderObj *this, TVEncoderRegs *r, TVState state)
{
  TVPh1SetRegs (this, r, state);
  TVWriteBus (this->dev, 0x37, 0x02 /* igain=1 */
	       	       | SetBitFlag(r->ph2.flags3,PH_FLAG3_XINT,0) 
	       	       | SetBitFlag(r->ph2.flags3,PH_FLAG3_YUPSC,6)
	       	       | SetBitField(r->ph2.yfil,1:0,5:4)
	       	       | SetBitField(r->ph2.pcli,1:0,1:0));
  TVWriteBus (this->dev, 0x84, 0xc0 /* dclk=single, pclsy=sync */
	       	       | SetBitFlag(r->ph2.flags3,PH_FLAG3_IFRA,5) 
	       	       | SetBitFlag(r->ph2.flags3,PH_FLAG3_IFBP,4)
	       	       | SetBitField(r->ph2.pcle,1:0,3:2)
	       	       | SetBitField(r->ph2.pcli,1:0,1:0));
  TVWriteBus (this->dev, 0x85, SetBitFlag(r->ph2.flags3,PH_FLAG3_EIDIV,7) 
	               | SetBitField(r->ph2.fili,3:0,3:0));
}

void TVPh2GetRegs (TVEncoderObj *this, TVEncoderRegs *r)
{
  I2CByte res;

  TVPh1GetRegs (this, r);
  r->ph2.flags3 = 0;
  TVReadBus (this->dev, 0x37, &res); 
  r->ph2.flags3 |= GetBitFlag(res,0,PH_FLAG3_XINT) 
	       	|  GetBitFlag(res,6,PH_FLAG3_YUPSC);
  r->ph2.yfil   = SetBitField(res,5:4,1:0);
  r->ph2.pcli   = SetBitField(res,1:0,1:0);
  TVReadBus (this->dev, 0x84, &res); 
  r->ph2.flags3 |= GetBitFlag(res,5,PH_FLAG3_IFRA) 
	       	|  GetBitFlag(res,4,PH_FLAG3_IFBP);
  r->ph2.pcle	= SetBitField(res,3:2,1:0);
  r->ph2.pcli	= SetBitField(res,1:0,1:0);
  TVReadBus (this->dev, 0x85, &res); 
  r->ph2.flags3 |= GetBitFlag(res,7,PH_FLAG3_EIDIV);
  r->ph2.fili   = SetBitField(res,3:0,3:0);
}

TVEncoderObj tvPh2Template = {
  chip: TV_PHILIPS_MODEL2, dev: NULL, minClock: 10000, maxClock: 44000, 
  Create:     TVPhCreate,
  InitRegs:   TVPh2InitRegs, 
  SetRegs:    TVPh2SetRegs, 
  GetRegs:    TVPh2GetRegs, 
  SetPort:    TVPh2SetPort,
  GetPort:    TVPh2GetPort,
  SetState:   TVPh1SetState,
  GetConnect: TVPh1GetConnect, 
  GetStatus:  TVPh1GetStatus
};

/* SAA7104: 

(iferr bferr in status)
(edge/slot via port)

maxClock 85 MHz 
PLL between ca. 40 Mhz and 85 Mhz (before divide)
PAL/NTSC up to 1280x1024 (1920x1080 interlaced)
HDTV up to 1920x1080 interlaced, 1280x720 progressive
I2C at 400kHz (also 7102)
MV rev 7.01, rev 6.1, rev 1.03 (525p)
Anti-Flicker 5 lines
2 FIFOs (one for upsampling)
seperate CSYNC pin (also 7102)

Different registers:

1b: 6=msa 5=msoe ; automatic monitor sense, ms pin TVD output enable
2d: 2=cvbsen2 ; pin RED... normal or CVBS
37: 6=yupsc 5=yfil1 4=yfil0 2=gzoom 1=igain 0=xint ; Input path control
    (nv=12: yfil=1 (before), igain=1)
3a: 5=syntv
54: 5=gpval 4=gpen 1=edge 0=slot
61: 5=inpi
62: 7=rtce
6e: 7=nvtrig ; vtrig sign
7c: 5=ttxo
80: 3=iferr 2=bferr ; input fifo overflow, buffer fifo overflow (for yupsc)
84: 7=dclk 6=pclsy 5=ifra 4=ifbp 3=pcle1 2=pcle0 1=pcli1 0=pcli0
    (nv=e_: dclk=single, pclsy=sync, ifra=reset, ifbp=active)
    (nv=_5: e:i=2:2, nv=_0: e:i=1:1)
85: 7=eidiv 3=fili3 2=fili2 1=fili1 0=fili0
    (nv=88: input dividers on, threshold=8)
96: 2=hsl is n/a
99: 3=hlen11
d0-dc: hdtv stuff

*/

