// see License.txt for copyright and terms of use

#include "qual.h"
#include "qual_global.h"

#include "qual_dataflow_visitor.h" // dataFlow_refUnify
#include "qual_cmd.h"           // QualCmd
#include "qual_annot.h"         // findQualAndCheck, attachOneLiteralToQvar
#include "qual_value_children.h" // treeContainsExternQvars
#include "qual_libqual_iface.h" // init_libqual, LibQual::
#include "qual.gr.gen.h"        // CCParse_Qual
#include "oink_util.h"
#include "cc_print.h"

#include "xml_reader.h"         // XmlReaderManager, XmlLexer
#include "xml_type_reader.h"    // XmlTypeReader
#include "qual_xml_value_reader.h" // XmlValueReader_Q
#include "qual_xml_value_writer.h" // XmlValueWriter_Q
#include "xmlhelp.h"            // xmlPrintPointer

#include "sobjset.h"            // SObjSet

#include <sstream>
#include <memory>

#include "archive_srz.h"        // ArchiveSerializer/ArchiveDeserializer
#include "archive_srz_format.h"

// temporary: the qvars marked as extern. Only used for assertions, which will
// go away.
// static SObjSet<LibQual::Type_qualifier*> externVisitedSet_global;

// phony XML-alike tags as a sanity check that s18n is okay
static std::string const QUAL_BOF_MAGIC = "<QUAL>";
static std::string const QUAL_EOF_MAGIC = "</QUAL>";

void Qual::visitInSerializationOrder(ValueVisitor &visitor)
{
  IdentityManager idmgr;
  XmlValueWriter_Q::XVWQ_SerializeOracle srzOracle;
  int depth = 0;

  XmlValueWriter_Q valueVisitor(idmgr,
                                (ASTVisitor*)NULL,
                                &visitor, NULL, depth, false, &srzOracle);
  serialize_abstrValues_stream(valueVisitor, NULL);

  xassert(depth == 0);
}

// QualEcrUpdateVisitor **********

// class QualEcrUpdateVisitor : public ValueVisitor {

// public:
//   virtual bool preVisitValue(Value *obj) { qa(obj)->ecrUpdate(); return true; }

//   // do nothing
//   virtual void postVisitValue(Value *obj) {}
//   virtual bool preVisitVariable(Variable_O *var) { return true; }
//   virtual void postVisitVariable(Variable_O *var) {}
// };

// Update all qvars to point to their ECRs.  This is a necessary step before
// eliminate_quals_links().  (Previously it was okay to delay this step until
// serialization because eliminate_quals_links() did not actually deallocate
// any qvars.)
void Qual::updateEcrAll() {
  printStage("   updateEcrAll");
  // QualEcrUpdateVisitor visitor;
  // visit all nodes, including buckets
  // visitInSerializationOrder(visitor, true);

  // visit all values (order doesn't matter)

  FOREACH_VALUE(iter) {
    Value *v = iter.data<Value>();
    qa(v)->ecrUpdate();
  }
}

// quarl 2006-05-15
//   This visitor is not working because it doesn't visit everything the
//   pretty-printer visits.  See qa_removeDeadQvars in qual_annot.cc.

// // QualDeadQvarRemoveVisitor **********

// class QualDeadQvarRemoveVisitor : public ValueVisitor {

// public:
//   virtual bool preVisitValue(Value *obj) { qa(obj)->removeQvarIfDead(); return true; }

//   // do nothing
//   virtual void postVisitValue(Value *obj) {}
//   virtual bool preVisitVariable(Variable_O *var) { return true; }
//   virtual void postVisitVariable(Variable_O *var) {}
// };

void Qual::removeDeadQvars() {
  printStage("   removeDeadQvars");

  // quarl 2006-05-15
  //   the pretty-printer traversal is hitting nodes that I can't reach
  //   through visitInSerializationOrder() for some reason; do this for now.
  //   Eventually we should get rid of this map so then we'll have to get the
  //   traversals to match up.

  // quarl 2006-05-19
  //   We now traverse the (newly introduced) list of all values
  //   Value::valueList, not the map.

  FOREACH_VALUE(iter) {
    Value *v = iter.data<Value>();
    qa(v)->removeQvarIfDead();
  }
}


// QualSerialization_ValueVisitor ****************

class QualSerialization_ValueVisitor : public ValueVisitor {
  ostream &mapOut;
  int numSerializedQvars;
  LibQual::s18n_context *serializationCtxt;
  SObjSet<LibQual::Type_qualifier*> serializedQvars; // NOTE: only used for assertion
  IdentityManager &idmgr;

  public:
  QualSerialization_ValueVisitor (IdentityManager &idmgr0,
                                  ostream &mapOut0, LibQual::s18n_context *serializationCtxt0)
    : mapOut(mapOut0)
    , numSerializedQvars(0)
    , serializationCtxt(serializationCtxt0)
    , idmgr(idmgr0)
  {}

  int getNumSerializedQvars() { return numSerializedQvars; }

  // serialize the map edge
  virtual bool preVisitValue(Value *obj);
  // do nothing
  virtual void postVisitValue(Value *obj) {}

  virtual bool preVisitVariable(Variable_O *var) {
    xassert(serializeVar_O(var));
    return true;
  }
  virtual void postVisitVariable(Variable_O *var) {}
};

bool QualSerialization_ValueVisitor::preVisitValue(Value *value) {
  QualAnnot *annot = qa(value);
  // LibQual::check_all_edge_sets();

  // count the qvars; ensure non-duplicate counting; NOTE: we have to
  // do this before mapping it through LibQual::ecr_qual_noninline()
  // otherwise the count doesn't correspond to what was marked as
  // extern
  xassert(annot->hasQv());
  LibQual::Type_qualifier *qv = annot->qv();

  // printf("## QualSerialization_ValueVisitor: value=%p, qa=%p, qv=%p (%p) %s\n",
  //        value, annot, qv, LibQual::ecr_qual_noninline(qv), name_qual(qv));
  // fflush(stdout);

  // quarl: I think that it's OK to see qvar more than once: this can happen
  // if we're re-serializing an already serialized, compacted, deserialized
  // file, where quals have been merged.

  // this should have been done in updateEcrAll()
  xassert(qv == LibQual::ecr_qual_noninline(qv));

  // Check that it was marked extern earlier.  Note that extern_qual(qv)
  // doesn't do a proper subset check because it goes through ecr.

  // xassert(externVisitedSet_global.contains(qv) && "6578c45f-42a6-4bcb-a1cd-c9d49fb2b0aa");
  xassert(LibQual::extern_qual(qv) && "d61f1ed1-a219-4e5b-b6cb-7c11bea92d73");

  // dsw: this was inserted by Rob; I think it is just to check we are
  // not serializing a constant
  xassert(!LibQual::constant_qual(qv));

  // quarl 2006-06-26
  //    Check that the qvar wasn't compacted away, since we now serialize the
  //    map for non-extern qvars as well?
  xassert(!LibQual::is_dead_qual(qv));

  // check that the qvar was serialized earilier
  xassert(LibQual::s18n_lookup_serialized(serializationCtxt, qv));

  // LOUD-SERIALIZATION
//   printf("map edge annot->qv==%p\n", (void const*)(annot->qv()));
//   Value_gdb(value);

  // TODO: integrate into value.xml
  outputXmlPointer(mapOut, "VL", idmgr.uniqueId(value));
  mapOut << " -> ";
    // FIX: get rid of the reinterpret_cast when Rob implements unique
    // ids for us
  outputXmlPointer(mapOut, "QV", reinterpret_cast<xmlUniqueId_t>(qv));
  mapOut << '\n';

  ++numSerializedQvars;
  return true;
}


// QvarSerializationContextHolder ****************

class QvarSerializationContextHolder {
  public:
  LibQual::s18n_context *ctxt;

  QvarSerializationContextHolder(void *stream)
    : ctxt(LibQual::s18n_new_context_io_custom(stream, readf, writef))
  {}

  ~QvarSerializationContextHolder() {
    LibQual::s18n_destroy_context(ctxt);
  }

private:
  static int writef(void* stream, void* buf, int len) {
    ostream& o = *( (ostream*)stream );
    o.write((char*) buf, len);
    return o ? len : -1;
  }

  static int readf(void* stream, void* buf, int len) {
    istream&i = *( (istream*)stream );
    i.read((char*) buf, len);
    return i ? len : -1;
  }
};

// Module coloring ****************

// get the module for this location
static StringRef moduleForLoc(SourceLoc loc, bool &dueToDefault) {
  char const *filename = sourceLocManager->getFile(loc);
  StringRef module = file2module.queryif(filename);
  dueToDefault = false;
  if (!module) {
    dueToDefault = true;
    USER_ASSERT(defaultModule, loc,
                "Default module is needed for file '%s' but none was given.", filename);
    module = defaultModule;
  }
  return module;
}

static void colorWithModule_alloc
  (Value *value, SourceLoc loc, char const *name, char const *astNode)
{
  xassert(value);
  bool dueToDefault;
  StringRef module = moduleForLoc(loc, dueToDefault);

  // get the qualifier
  stringBuilder qconstName("$");
  qconstName << module;
  qconstName << "_alloc";
  LibQual::Type_qualifier *qconst = LibQual::find_qual(qconstName.c_str());
  USER_ASSERT(qconst, value->loc, "Literal '%s' not in lattice", strdup(qconstName.c_str()));

  // tell the user what we are doing
  if (oinkCmd->report_colorings) {
    cout << sourceLocManager->getFile(loc) << ":" << sourceLocManager->getLine(loc);
    cout << " " << astNode;
    if (name) cout << " '" << name << "'";
    else cout << " <no-name>";
    cout << " colored " << qconstName;
    if (dueToDefault) {
      cout << " by default";
    }
    cout << endl;
  }

  // color the value with the module
  qa(value)->attachOneLiteral(value->loc, qconst);
}

static void colorWithModule_access
  (Value *value, SourceLoc loc, char const *name, char const *astNode)
{
  xassert(value);
  bool dueToDefault;
  StringRef module = moduleForLoc(loc, dueToDefault);

  // get the qualifier
  stringBuilder qconstName("$");
  qconstName << module;
  qconstName << "_access";
  LibQual::Type_qualifier *qconst = LibQual::find_qual(qconstName.c_str());
  USER_ASSERT(qconst, value->loc, "Literal '%s' not in lattice", strdup(qconstName.c_str()));

  // tell the user what we are doing
  if (oinkCmd->report_colorings) {
    cout << sourceLocManager->getFile(loc) << ":" << sourceLocManager->getLine(loc);
    cout << " " << astNode;
    if (name) cout << " '" << name << "'";
    else cout << " <no-name>";
    cout << " colored " << qconstName;
    if (dueToDefault) {
      cout << " by default";
    }
    cout << endl;
  }

  // color the value with the module
  qa(value)->attachOneLiteral(value->loc, qconst);
}

static void colorWithModule_otherControl
  (Value *value, SourceLoc loc, char const *name, char const *astNode)
{
  xassert(value);
  bool dueToDefault;
  StringRef module = moduleForLoc(loc, dueToDefault);

  // tell the user what we are doing
  if (oinkCmd->report_colorings) {
    cout << sourceLocManager->getFile(loc) << ":" << sourceLocManager->getLine(loc);
    cout << " " << astNode;
    if (name) cout << " '" << name << "'";
    else cout << " <no-name>";
    cout << endl;
  }

  // for each source module not equal to us, make the otherControl target
  SFOREACH_OBJLIST(char, moduleList, iter) {
    StringRef srcModule = iter.data();
    if (srcModule == module) continue;

    // get the qualifier
    stringBuilder qconstName("$");
    qconstName << srcModule;
    qconstName << "_otherControl";
    LibQual::Type_qualifier *qconst = LibQual::find_qual(qconstName.c_str());
    USER_ASSERT(qconst, value->loc, "Literal '%s' not in lattice", strdup(qconstName.c_str()));

    // print out a line for each
    if (oinkCmd->report_colorings) {
      cout << "\tcolored " << qconstName;
      if (dueToDefault) {
        cout << " by default";
      }
      cout << endl;
    }

    // NOTE: this is a value-level qualifier on a pointer
    xassert(value->isPointerValue());

    // color the value with the module
    qa(value)->attachOneLiteral(value->loc, qconst);
  }
}

static void colorWithModule_otherWrite
  (Value *value, SourceLoc loc, char const *name, char const *astNode)
{
  xassert(value);
  bool dueToDefault;
  StringRef module = moduleForLoc(loc, dueToDefault);

  // tell the user what we are doing
  if (oinkCmd->report_colorings) {
    cout << sourceLocManager->getFile(loc) << ":" << sourceLocManager->getLine(loc);
    cout << " " << astNode;
    if (name) cout << " '" << name << "'";
    else cout << " <no-name>";
    cout << endl;
  }

  // for each source module not equal to us, make the write target
  SFOREACH_OBJLIST(char, moduleList, iter) {
    StringRef srcModule = iter.data();
    if (srcModule == module) continue;

    // get the qualifier
    stringBuilder qconstName("$");
    qconstName << srcModule;
    qconstName << "_otherWrite";
    LibQual::Type_qualifier *qconst = LibQual::find_qual(qconstName.c_str());
    USER_ASSERT(qconst, value->loc, "Literal '%s' not in lattice", strdup(qconstName.c_str()));

    // print out a line for each
    if (oinkCmd->report_colorings) {
      cout << "\tcolored " << qconstName;
      if (dueToDefault) {
        cout << " by default";
      }
      cout << endl;
    }

    // color the value with the module
    qa(value)->attachOneLiteral(value->loc, qconst);
  }
}

static void colorWithModule_otherAccess
  (Value *value, SourceLoc loc, char const *name, char const *astNode)
{
  xassert(value);
  bool dueToDefault;
  StringRef module = moduleForLoc(loc, dueToDefault);

  // tell the user what we are doing
  if (oinkCmd->report_colorings) {
    cout << sourceLocManager->getFile(loc) << ":" << sourceLocManager->getLine(loc);
    cout << " " << astNode;
    if (name) cout << " '" << name << "'";
    else cout << " <no-name>";
    cout << endl;
  }

  // for each source module not equal to us, make the write target
  SFOREACH_OBJLIST(char, moduleList, iter) {
    StringRef srcModule = iter.data();
    if (srcModule == module) continue;

    // get the qualifier
    stringBuilder qconstName("$");
    qconstName << srcModule;
    qconstName << "_otherAccess";
    LibQual::Type_qualifier *qconst = LibQual::find_qual(qconstName.c_str());
    USER_ASSERT(qconst, value->loc, "Literal '%s' not in lattice", strdup(qconstName.c_str()));

    // print out a line for each
    if (oinkCmd->report_colorings) {
      cout << "\tcolored " << qconstName;
      if (dueToDefault) {
        cout << " by default";
      }
      cout << endl;
    }

    // color the value with the module
    qa(value)->attachOneLiteral(value->loc, qconst);
  }
}

// Qual_ModuleAlloc_Visitor ****************

// visit all of the allocation points
class Qual_ModuleAlloc_Visitor : private ASTVisitor, public WalkEnv {
  public:
  LoweredASTVisitor loweredVisitor; // use this as the argument for traverse()

  bool color_alloc;          // ref-level coloring of allocated memory
  bool color_otherControl;   // value-level coloring of other-control memory

  SObjSet<E_funCall*> seenAllocators; // allocators that we saw
  SObjSet<E_funCall*> castedAllocators; // allocators that were inside a cast expression

  public:
  Qual_ModuleAlloc_Visitor(TranslationUnit *tunit0, bool color_alloc0, bool color_otherControl0)
    : WalkEnv(tunit0)
    , loweredVisitor(this)
    , color_alloc(color_alloc0)
    , color_otherControl(color_otherControl0)
  {}

  virtual bool visitTopForm(TopForm *obj);
  virtual bool visitStatement(Statement *obj);
  virtual bool visitDeclarator(Declarator *);
  virtual void postvisitExpression(Expression *);

  virtual bool subVisitE_funCall(E_funCall *);
  virtual bool subVisitE_cast(E_cast *);
  virtual bool subVisitE_new(E_new *);
};

static StringRef funCallName(E_funCall *obj) {
  Expression *func0 = obj->func->skipGroups();
  if (!func0->isE_variable()) return NULL;
  return func0->asE_variable()->name->getName();
}

// is this a function call to an allocator; FIX: we should actually be
// checking the gcc __attribute__(()) for this instead
static bool isAllocator(E_funCall *obj, SourceLoc loc) {
  StringRef funcName = funCallName(obj);
  if (!funcName) return false;
  if (streq(funcName, "alloca")) {
    userReportWarning(loc, "We can't handle alloca");
  }
  return
    streq(funcName, "malloc")  ||
    streq(funcName, "realloc") ||
    streq(funcName, "calloc")  ||
    streq(funcName, "strdup");
}

bool Qual_ModuleAlloc_Visitor::visitTopForm(TopForm *obj) {
  loc = obj->loc;
  return true;
}

bool Qual_ModuleAlloc_Visitor::visitStatement(Statement *obj) {
  loc = obj->loc;
  return true;
}

bool Qual_ModuleAlloc_Visitor::visitDeclarator(Declarator *obj) {
  Value *varValue = asVariable_O(obj->var)->abstrValue()->asRval();
  if (color_alloc) {
    colorWithModule_alloc(varValue, varValue->loc, obj->var->name, "Declarator");
  }
  if (color_otherControl) {
    if (varValue->isPointerValue()) {
      colorWithModule_otherControl(varValue, varValue->loc, obj->var->name, "Declarator");
    }
  }
  return true;
}

void Qual_ModuleAlloc_Visitor::postvisitExpression(Expression *obj) {
  switch(obj->kind()) {         // roll our own virtual dispatch
  default: break;               // expression kinds for which we do nothing
  case Expression::E_FUNCALL:     subVisitE_funCall(obj->asE_funCall());         break;
  case Expression::E_CAST:        subVisitE_cast(obj->asE_cast());               break;
  case Expression::E_NEW:         subVisitE_new(obj->asE_new());                 break;
  }
}

bool Qual_ModuleAlloc_Visitor::subVisitE_funCall(E_funCall *obj) {
  if (isAllocator(obj, loc)) {
    // record the allocator so we can look for missed allocators later
    seenAllocators.add(obj);
  }
  return true;
}

bool Qual_ModuleAlloc_Visitor::subVisitE_cast(E_cast *obj) {
  // see if it is a cast from an allocator
  Expression *expr0 = obj->expr->skipGroups();
  if (expr0->isE_funCall() && isAllocator(expr0->asE_funCall(), loc)) {
    if (color_alloc) {
      // This is subtle: attach the color to the 1) ref value of the 2)
      // thing pointed to by 3) the cast expresion; thanks to Matt for
      // help on this.
      Value *castValue = obj->abstrValue
        ->getAtValue()            // color the value pointed-to, not the pointer
        ->asRval();
      colorWithModule_alloc(castValue, castValue->loc,
                            funCallName(expr0->asE_funCall()), "E_cast-allocator");
    }
    if (color_otherControl) {
      // This is subtle: attach the color to the 1) pointer value of
      // the 2) the cast expresion; thanks to Matt for help on this.
      Value *castValue = obj->abstrValue->asRval();
      if (castValue->isPointerValue()) {
        colorWithModule_otherControl(castValue, castValue->loc,
                                        funCallName(expr0->asE_funCall()), "E_cast-allocator");
      }
    }
    // check off the allocator so we can look for missed allocators later
    castedAllocators.add(expr0->asE_funCall());
  }
  return true;
}

bool Qual_ModuleAlloc_Visitor::subVisitE_new(E_new *obj) {
  // attach the color to the expression ref value
  Value *value0 = obj->abstrValue->asRval();
  // FIX: could print the type name here
  if (color_alloc) {
    // This is subtle: attach the color to the 1) ref value of the 2)
    // thing pointed to by 3) the new expresion; thanks to Matt for
    // help on this.
    Value *newValue = obj->abstrValue
      ->getAtValue()    // color the value pointed-to, not the pointer
      ->asRval();
    colorWithModule_alloc(newValue, newValue->loc, NULL, "E_new");
  }
  if (color_otherControl) {
    xassert(value0->isPointerValue());
    colorWithModule_otherControl(value0, value0->loc, NULL, "E_new");
  }
  return true;
}

// Qual_ModuleOtherWrite_Visitor ****************

// Write points:
// E_assign
//      =
//      *= and others
// E_effect
//      ++
//      --
// Initializer
//      IN_expr: int x = 3;
//      IN_compound: int x[] = { 1, 2 }
// * C++ only:
// Initializer
//      IN_ctor
// MemberInit
//      member initializers
// Handler
//      catch clause

// visit all of the write points
class Qual_ModuleOtherWrite_Visitor : private ASTVisitor, public WalkEnv {
  public:
  LoweredASTVisitor loweredVisitor; // use this as the argument for traverse()

  public:
  Qual_ModuleOtherWrite_Visitor(TranslationUnit *tunit0)
    : WalkEnv(tunit0)
    , loweredVisitor(this)
  {}

  virtual bool visitTopForm(TopForm *obj);
  virtual bool visitStatement(Statement *obj);
  virtual bool visitDeclarator(Declarator *obj);
  virtual bool visitMemberInit(MemberInit *obj);
  virtual bool visitHandler(Handler *obj);
  virtual void postvisitExpression(Expression *);

  virtual bool subVisitE_effect(E_effect *);
  virtual bool subVisitE_assign(E_assign *);
};

bool Qual_ModuleOtherWrite_Visitor::visitTopForm(TopForm *obj) {
  loc = obj->loc;
  return true;
}

bool Qual_ModuleOtherWrite_Visitor::visitStatement(Statement *obj) {
  loc = obj->loc;
  return true;
}

bool Qual_ModuleOtherWrite_Visitor::visitDeclarator(Declarator *obj) {
  Value *varValue = asVariable_O(obj->var)->abstrValue()->asRval();
  if (obj->init) {
    if (IN_ctor *ctor = dynamic_cast<IN_ctor*>(obj->init)) {
      if (!obj->ctorStatement) {
        // IN_ctor for a simple type that does not have an implicit ctor,
        // such as an int.
        xassert(!varValue->t()->isCompoundType());
        USER_ASSERT(ctor->args->count()==1, loc,
                    "Non-compound type must have one-arg MemberInit"
                    " (e5a12447-7229-4e2c-bc23-db855954f06e)");
        colorWithModule_otherWrite(varValue, varValue->loc, obj->var->name, "Declarator-IN_ctor");
      }
      // otherwise, the ctorStatement was handled above.
    } else if (
               // get rid of gcc unused warning
//                 IN_expr *expr =
               dynamic_cast<IN_expr*>(obj->init)) {
      if (!obj->ctorStatement) {
        // **** from the dataflow analysis:
        //
        // This is wrong, it could be calling a ctor.  Check for the
        // ctor and do a function call instead here if there is one.
        // 9sep2003: update: Scott isn't sure about the semantics of
        // IN_expr; keep doing this for now
        //
        // UPDATE: the ctorStatement above handles this
        //
        // UPDATE: FIX: GATED_EDGE: I need to omit this edge for the
        // special case of a string literal being used to initialize
        // an array.  The real answer is that a constructor for the
        // array should be called with the string literal that
        // *copies* it to the array; note that while a string literal
        // is an array of 'char const'-s it can be used to initialize
        // an array of (nonconst) chars.  That is, this is fine;
        // 'fmt[0]' is writable:
        //
        //   static char fmt[80] = "abort %d (%x); MSCP free pool: %x;";
        // however this is not; it is unspecified what happens if you
        // write 'fmt[0]':
        //   static char *fmt = "abort %d (%x); MSCP free pool: %x;";
        //          if (! (expr->e->isE_stringLit() && varValue->isStringType()) ) {
        //          if (! (expr->e->isE_stringLit()) ) {
        //            dfe.eDataFlow_normal(expr->e->abstrValue, varValue, loc);
        //          }
        colorWithModule_otherWrite(varValue, varValue->loc, obj->var->name, "Declarator-IN_expr");
      }
    } else if (
               // get rid of gcc unused warning
//                 IN_compound *ci =
               dynamic_cast<IN_compound*>(obj->init)) {
      // **** from the dataflow analysis:
      //
      // FIX: I think it is more elegant and local here to use the
      // declType.
      // Scott puts this in, though he doesn't say that he knows its
      // right, so I'll let it go through.
      //        xassert(!obj->ctorStatement);

//       try {
//         compoundInit(&dfe, loc, varValue, ci, oneAssignmentDataFlow, reportUserErrorDataFlow);
//       } catch (UserError &err) {
//         // dsw: there are some compound initializers situations where
//         // gcc just gives a warning instead of an error, such as if
//         // you put too many elements into an array:
//         // oink/Test/too_many_initializers1.c; here I turn the error
//         // into a warning so we can at least process such files
//         userReportWarning
//           (loc, stringc << "above error turned into a warning; "
//            "this is an unsoundness in the dataflow");
//       }

      // FIX: we should actually set it to deeply write
      userReportWarning(loc, "unimplemented: compound initializers should be a deep write");
      colorWithModule_otherWrite(varValue, varValue->loc, obj->var->name,
                                 "Declarator-IN_compound");
    } else xfailure("can't happen");
  }
  return true;
}

bool Qual_ModuleOtherWrite_Visitor::visitMemberInit(MemberInit *obj) {
  if (!obj->ctorStatement) {
    // MemberInit for a simple type that does not have an implicit
    // ctor, such as an int.
    // FIX: make this instance-specific; (did I mean to imitate this?)
//        obj->receiver->type->asRval()->asCVAtomicValue()->getInstanceSpecificValue();
    Value *memberValue0 = asVariable_O(obj->member)->abstrValue();
    xassert(!memberValue0->t()->isCompoundType());
    USER_ASSERT(obj->args->count()==1, loc,
                "Non-compound type must have one-arg MemberInit"
                " (7ec11921-4016-4649-8120-2ad50f9a73ce)");
    colorWithModule_otherWrite(memberValue0, memberValue0->loc, obj->member->name, "MemberInit");
  }
  return true;
}

bool Qual_ModuleOtherWrite_Visitor::visitHandler(Handler *obj) {
  if (obj->globalVar) {
    Value *tgt0 = asVariable_O(obj->typeId->decl->var)->abstrValue()->asRval();
    colorWithModule_otherWrite(tgt0, tgt0->loc, obj->typeId->decl->var->name, "Handler");
  } else {
    xassert(!obj->globalDtorStatement);
  }
  return true;
}

void Qual_ModuleOtherWrite_Visitor::postvisitExpression(Expression *obj) {
  switch(obj->kind()) {         // roll our own virtual dispatch
  default: break;               // expression kinds for which we do nothing
  case Expression::E_EFFECT: subVisitE_effect(obj->asE_effect()); break;
  case Expression::E_ASSIGN: subVisitE_assign(obj->asE_assign()); break;
  }
}

bool Qual_ModuleOtherWrite_Visitor::subVisitE_effect(E_effect *obj) {
  // NOTE: In the dataflow analysis, we flow from the
  // obj->expr->abstrValue to the obj->abstrValue; however it is the
  // obj->expr->abstrValue that is being mutated
  xassert(obj->expr->abstrValue->isReferenceValue());
  Value *value0 = obj->expr->abstrValue->asRval();
  char const *name0 = NULL;
  if (obj->expr->skipGroups()->isE_variable()) {
    name0 = obj->expr->skipGroups()->asE_variable()->name->getName();
  }
  colorWithModule_otherWrite(value0, value0->loc, name0, "E_effect");
  return true;
}

bool Qual_ModuleOtherWrite_Visitor::subVisitE_assign(E_assign *obj) {
  // should not get targets here which are compound types or
  // references to such, since it should have been resolved into a
  // call to an operator assign, which always exists implicitly.
  Value *tgt = obj->target->abstrValue;
  // remove l-value layer of indirection for ASSIGNMENT (not for
  // INITIALIZATION).
  USER_ASSERT(tgt->isReferenceValue(), loc, "Left side of assignment must be an lvalue");
  Value *tgt1 = tgt->asRval();
  char const *name0 = NULL;
  if (obj->target->skipGroups()->isE_variable()) {
    name0 = obj->target->skipGroups()->asE_variable()->name->getName();
  }
  colorWithModule_otherWrite(tgt1, tgt1->loc, name0, "E_assign");
  return true;
}

// Qual_ModuleAccess_Visitor ****************

// an access point is any L-value expression (an expression with Ref
// type).

// visit all of the write points
class Qual_ModuleAccess_Visitor : private ASTVisitor, public WalkEnv {
  public:
  LoweredASTVisitor loweredVisitor; // use this as the argument for traverse()

  bool otherAccess;             // color with our access or other accesses?

  public:
  Qual_ModuleAccess_Visitor(TranslationUnit *tunit0, bool otherAccess0)
    : WalkEnv(tunit0)
    , loweredVisitor(this)
    , otherAccess(otherAccess0)
  {}

  virtual bool visitTopForm(TopForm *obj);
  virtual bool visitStatement(Statement *obj);

  virtual void postvisitExpression(Expression *);
};

bool Qual_ModuleAccess_Visitor::visitTopForm(TopForm *obj) {
  loc = obj->loc;
  return true;
}

bool Qual_ModuleAccess_Visitor::visitStatement(Statement *obj) {
  loc = obj->loc;
  return true;
}

void Qual_ModuleAccess_Visitor::postvisitExpression(Expression *obj) {
  Value *exprValue = obj->abstrValue;
  if (exprValue->isReferenceValue()) {
    if (otherAccess) {
      colorWithModule_otherAccess(exprValue->asRval(), exprValue->loc, NULL, "Expression");
    } else {
      colorWithModule_access(exprValue->asRval(), exprValue->loc, NULL, "Expression");
    }
  }
}

// class MarkDecltorsInstanceSpecificVisitor ****************

#if DEBUG_INSTANCE_SPECIFIC_VALUES
bool MarkDecltorsInstanceSpecificVisitor::visitDeclarator(Declarator *obj) {
  Value *declValue = obj->abstrValue;
  if (!declValue->isFunctionValue()) {
    instanceSpecificValues.add(declValue);
    dataDeclaratorValues.add(declValue);
  }
  return true;
}
#endif

// class NameASTNodesVisitor ****************

bool NameASTNodesVisitor::visitExpression(Expression *ex) {
  // pretty print AST to name
  stringBuilder sb;
  StringBuilderOutStream out0(sb);
  CodeOutStream codeOut(out0);
  PrintEnv env(typePrinterOink, &codeOut);
  ex->print(env);
  codeOut.finish();

  // attach the name to the abstract value of the AST
  Value *v = ex->abstrValue;
  if (v) {
    qa(v)->nameTree(sb);
  }
  return true;                  // do the sub-expressions also
}

bool NameASTNodesVisitor::visitDeclarator(Declarator *obj) {
  Variable_O *var = asVariable_O(obj->var);
  return !var->filteredOut();
}

bool NameASTNodesVisitor::visitFunction(Function *obj) {
  Variable_O *var = asVariable_O(obj->nameAndParams->var);
  return !var->filteredOut();
}

// Qual ****************

void Qual::configure() {
  if (qualCmd->inference && !qualCmd->config) {
    userFatalError(SL_UNKNOWN, "Since you are doing inference, you must supply a config file.\n");
  }
  init_libqual(qualCmd->config);
}


// Deserialization ****************

/* virtual */
void Qual::deserialize_1archive(ArchiveDeserializer *arc, XmlReaderManager &manager)
{
  // NOTE: the XmlReaderManager is shared across both de-serializations below:
  // the deserialization of value_qual.map needs the XmlReaderManager::id2obj.

  deserialize_formatVersion(arc);
  deserialize_files(arc, manager);
  deserialize_abstrValues(arc, manager);
  deserialize_valueQualMapAndQualGraph(arc, manager);
}

// virtual, overrides Oink
void Qual::printStats()
{
  LibQual::Qual_print_stats();

  cout << endl;

  Oink::printStats();

  size_t values = 0;
  size_t values_with_names = 0;
  size_t values_with_qvars = 0;

  FOREACH_VALUE(iter) {
    Value *v = iter.data<Value>();
    QualAnnot *a = qa(v);
#if !DELAY_SET_NAMES
    if (a->vname) ++values_with_names;
#endif
    if (a->hasQv()) ++values_with_qvars;
    ++values;
  }
  xassert(values == vFac->getNumValues());

#if DELAY_SET_NAMES
  extern size_t getNumQvarsWithNames(); // TODO XXX KLUDGE
  values_with_names = getNumQvarsWithNames();
#endif

  cout << "Values with names: " << values_with_names << endl;
  cout << "Values with qvars: " << values_with_qvars << endl;
}

// virtual, overrides Oink
void Qual::printSizes()
{
#define P(T) cout << "  sizeof(" #T ") = " << sizeof(T) << endl
  Oink::printSizes();

  cout << endl;

  cout << "Qual:\n";

  P(CVAtomicValue_Q);
  P(PointerValue_Q);
  P(ReferenceValue_Q);
  P(FunctionValue_Q);
  P(ArrayValue_Q);
  P(PointerToMemberValue_Q);

  cout << endl;

  P(CVAtomicValue_QualAnnot);
  P(PointerValue_QualAnnot);
  P(ReferenceValue_QualAnnot);
  P(FunctionValue_QualAnnot);
  P(ArrayValue_QualAnnot);
  P(PointerToMemberValue_QualAnnot);

  cout << endl;

  P(CVAtomicValue_Q::MyStruct);
  P(PointerValue_Q::MyStruct);
  P(ReferenceValue_Q::MyStruct);
  P(FunctionValue_Q::MyStruct);
  P(ArrayValue_Q::MyStruct);
  P(PointerToMemberValue_Q::MyStruct);

  cout << endl;

  LibQual::Qual_print_sizes();
#undef P
}

void Qual::srzFormat(ArchiveSrzFormat &srzfmt)
{
  Oink::srzFormat(srzfmt);                          // delegate to super class

  srzfmt.opt("instance_sensitive", qualCmd->instance_sensitive);
  srzfmt.opt("poly", qualCmd->poly);
}

void Qual::deserialize_abstrValues(ArchiveDeserializer *arc, XmlReaderManager &manager) {
  // use heap-allocated readers because if we get an exception, then the
  // underlying ASTList will 'delete' it upon stack unwinding.
  XmlTypeReader *typeReader = new XmlTypeReader;
  manager.registerReader(typeReader);
  XmlValueReader_Q *valueReader = new XmlValueReader_Q(*typeReader);
  manager.registerReader(valueReader);

  deserialize_abstrValues_toLinker(arc, manager);

  manager.unregisterReader(typeReader);
  manager.unregisterReader(valueReader);
}

// read " -> " from input stream; set badbit on error.
static inline istream& readArrow(istream&i) {
  char c1, c2;
  i >> ws >> c1 >> c2 >> ws;
  if (!i || c1 != '-' || c2 != '>') {
    i.setstate(ios::badbit);
  }
  return i;
}

// Parse a line containing "valueId -> qvarId".  Throws exception on error.
static inline void parseValueQualMapLine
  (std::string const& line, std::string & valueId, unsigned int & qvarId)
{
  std::istringstream input(line);

  char c1, c2;
  input >> valueId >> readArrow >> c1 >> c2 >> qvarId;

  if (input.fail() || !input.eof() || c1 != 'Q' || c2 != 'V') {
    userFatalError(SL_UNKNOWN, "parse error while reading value_qual.map line '%s'", line.c_str());
  }
}

// Read a line. Returns true on success and false on eof.  Throws exception on
// read error.
static inline bool readValueQualMapLine(istream& in, std::string& line) {
  std::getline(in, line);

  if (in.eof() && line.empty()) {
    return false;
  }

  if (!in || line.empty()) {
    perror("failed while reading value_qual.map");
    throw UserError(USER_ERROR_ExitCode);
  }

  return true;
}

// Rob: "for each qualifier address on deserialized abstract values
// replace it using s18n_lookup_deserialized(...)"
void Qual::deserialize_valueQualMap_stream
  (istream& in, XmlReaderManager &manager, LibQual::s18n_context *serializationCtxt)
{
  std::string line;
  while(readValueQualMapLine(in, line)) {
    // read old value -> old qvar

    std::string oldValueId;
    unsigned int oldQvarPtr;
    parseValueQualMapLine(line, oldValueId, oldQvarPtr);

    // convert old value -> new value
    Value *value = (Value*) manager.getNodeById(oldValueId.c_str());
    if (!value) {
      printf("failed to find Value object while reading value_qual.map\n");
      breaker();
      throw UserError(USER_ERROR_ExitCode);
    }
    // FIX: should probably check that this is one of the kinds of Values
    //        int valueKind = manager.id2kind.get(oldValueId);

    // convert old qvar -> new qvar
    LibQual::Type_qualifier *newQvarPtr = NULL;
    int foundIt = LibQual::s18n_lookup_deserialized
      (serializationCtxt, (void*) oldQvarPtr, (void**) &newQvarPtr);
    if (!foundIt) {
      userFatalError(SL_UNKNOWN, "failed to find qvar %x while reading value_qual.map",
                     oldQvarPtr);
    }
    xassert(newQvarPtr);

    // make QualAnnot to mediate between Value and qvar and insert it
    // annotateWithQualAnnot(value);
    static_cast<QualAnnot*>(value->getAnnotation0())->setQv_deserialize(newQvarPtr);
  }
}

// expect string EXPECTED, else set badbit on IN.
static inline istream& matchMarker(istream& in, std::string const& expected) {
  char * buf = (char*) alloca(expected.size());
  in.read(buf, expected.size());
  if (0 != memcmp(buf, expected.c_str(), expected.size())) {
    in.setstate(ios::badbit);
  }
  return in;
}

// expect string EXPECTED, else throw user error.
static inline void expectMarker(istream& in, std::string const& expected, const char* fname) {
  if (!matchMarker(in, expected)) {
    userFatalError(SL_UNKNOWN, "input error reading '%s', did not find %s marker",
                   fname, expected.c_str());
  }
}

static inline void expectEOF(istream& in, const char* fname) {
  in.peek(); // peek to force setting eofbit if we're at EOF.
  if (!in.eof()) {
    userFatalError(SL_UNKNOWN, "input error reading '%s', did not reach end of file",
                   fname);
  }
}

void Qual::deserialize_valueQualMapAndQualGraph
  (ArchiveDeserializer* arc, XmlReaderManager &manager)
{
  // de-serialize the quals and the map

  istream& inValueQualMap = arc->input("value_qual.map");
  // std::stringstream tmpMapIn;
  // tmpMapIn.get(* inValueQualMap.rdbuf() );

  istream& inQual = arc->input("qual.dat");

  // check that we're reading the right file.
  expectMarker(inQual, QUAL_BOF_MAGIC, arc->curFname());

  // printStage("      reading quals");
  QvarSerializationContextHolder qctxt(&inQual);
  LibQual::merge_quals_graph(qctxt.ctxt); // merge in the file contents

  // printStage("      reading value-qual map");
  deserialize_valueQualMap_stream(inValueQualMap, manager, qctxt.ctxt);

  // check that we've read exactly what we expect to read.
  expectMarker(inQual, QUAL_EOF_MAGIC, arc->curFname());
  expectEOF(inQual, arc->curFname());

  // LibQual::check_all_edge_sets();
}


// Serialization ****************

// Mark the backend qualifier for all linker-visible abstract values as
// extern.  NOTE: This must be done before running compact_quals_graph().
void Qual::markExternVars() {
  printStage("markExternVars");
  // FIX: someday we should do the optimization where if there are no
  // other occurances of a variable in any other translation units
  // that it is no longer marked as external.

  // Remember: valueWriter is now a semi-generic visitor with a crappy name.

  {
    // 1. mark everything as non-extern.

    LibQual::set_extern_all_qual(false);
  }

  {
    // 2. mark linker-visibles as extern.

    IdentityManager idmgr;
    QualAnnot_ExternVisit visitor(/* externness */ true);
    bool indent = false;
    int depth = 0;
    // "Serialize" the values.
    //
    // We DO NOT use a XVWQ_SerializeOracle because we are the ones
    // setting "externness", and XVWQ_SerializeOracle looks at whether
    // there are extern qvars.  However, we do need to avoid
    // Variable-s that are not getFilteredKeep(), so we use the
    // XVW_SerializeOracle.
    XmlValueWriter::XVW_SerializeOracle filterKeepSrzOracle;
    XmlValueWriter_Q valueWriter(idmgr,
                                 (ASTVisitor*)NULL,
                                 &visitor,
                                 NULL, // no serialization
                                 depth,
                                 indent,
                                 &filterKeepSrzOracle
      );

    TailList<Variable_O> externVars;

    linker.getOrderedExternVars(NULL, externVars);
    valueWriter.toXml_externVars(&externVars);

    xassert(depth == 0);

    this->numExternQvars = visitor.getCount();
  }
}

void Qual::compactifyGraph() {
  // Compress the qualifier graph.  This reduces the size of the graph which
  // is good when serialized, though it requires more memory to do the
  // compaction.
  //
  // Compaction requires all externally visible qualifiers to be marked
  // 'extern'.

  printStage("   compact_quals_graph");
  LibQual::compact_quals_graph();

  // quarl 2006-05-14
  //    Update Type_qualifier pointers for all abstract values to point to
  //    their ECRs.  This is required before eliminate_quals_links().  This
  //    step used to be done during serialization after
  //    eliminate_quals_links() -- that was unclean and prevented
  //    eliminate_quals_links() from deleting anything; it is better to do it
  //    here, and it doesn't take much time to do an extra list traversal.
  //
  //    We really only need to do this for linker-visible values, but it's
  //    fast to update all.

  updateEcrAll();

  // After compaction, many qualifier nodes may be unified together.  This
  // passes over the graph, changing all references to those nodes to point
  // directly to the current ecr.
  printStage("   eliminate_quals_links");
  LibQual::eliminate_quals_links();
  // LibQual::check_all_edge_sets();

  // quarl 2006-05-15
  //    Remove pointers to the dead qvars so that the backend can get rid of
  //    them (currently it doesn't actually deallocate them, but it's still
  //    necessary because otherwise we'll try to use them in serialization,
  //    and their edge sets are no longer valid).
  removeDeadQvars();
}

int Qual::serialize_valuesAndMap(ArchiveSerializer* arc, LibQual::s18n_context *serializationCtxt) {
  // write to a string first, since we can't interleave output to two archive
  // files
  ostream& mapOut = arc->outputAlt("value_qual.map");
  ostream& valueOut = arc->output("value.xml");

  IdentityManager idmgr;

  // set up to serialize the qvars as a side-effect
  QualSerialization_ValueVisitor qSerValVisitor(idmgr, mapOut, serializationCtxt);
  bool indent = tracingSys("xmlPrintAST-indent");
  int depth = 0;              // shared depth counter between printers

  XmlValueWriter_Q::XVWQ_SerializeOracle srzOracle;
  // serialize the values
  {
    XmlValueWriter_Q valueWriter(idmgr,
                                 (ASTVisitor*)NULL,
                                 &qSerValVisitor, &valueOut, depth, indent, &srzOracle);
    serialize_abstrValues_stream(valueWriter, serializeVar_O);
    xassert(depth == 0);
  }

  return qSerValVisitor.getNumSerializedQvars();
}

// actually serialize everything
void Qual::serialize_results() {
  printStage("serialize_results");
  // FIX: should be able to get rid of this one; it is only there so
  // you can ask if something has any qualifier literals that haven't
  // been attached; there shouldn't be any at this point
  Restorer<bool> restorer(value2typeIsOn, true);
  try {
    ArchiveSerializerP arc = archiveSrzManager->getArchiveSerializer(qualCmd->srz);

    serialize_formatVersion(arc.get());

    serialize_files(arc.get());

    // should have been set from result of markExternVars().
    xassert(this->numExternQvars != -1);

    ostream& outQual = arc->output("qual.dat");
    outQual << QUAL_BOF_MAGIC;

    QvarSerializationContextHolder qctxt(&outQual);
    // LibQual::check_all_edge_sets();
    printStage("   serialize_quals_graph");
    LibQual::serialize_quals_graph(qctxt.ctxt);

    // write a magic marker so we can check on deserialization
    outQual << QUAL_EOF_MAGIC;

    // LOUD-SERIALIZATION
//      printf("\nVALUE ****************\n");
    printStage("   serialize_valuesAndMap");
    int numSerializedQvars = serialize_valuesAndMap(arc.get(), qctxt.ctxt);

    // NOTE: this is a very important assertion; do not remove it.  If
    // this is failing, 1) print the qvars marked as extern by turning
    // on the print statement in QualAnnot_ExternVisit::setExtern, and
    // 2) the qvars serialized by turning on the print statement in
    // QualSerialization_ValueVisitor::preVisitValue; both are marked
    // LOUD-SERIALIZATION.
    if (numExternQvars != numSerializedQvars) {
      userFatalError(SL_UNKNOWN, "INTERNAL ERROR: numExternQvars=%d, numSerializedQvars=%d",
                     numExternQvars, numSerializedQvars);
    }
    xassert(numExternQvars == numSerializedQvars);

  } catch(ArchiveIOException & e) {
    userFatalError(SL_UNKNOWN, "Failed to serialize %s: %s", qualCmd->srz.c_str(), e.error.c_str());
  }
}

bool Qual::varUsedInDataflow(Variable_O *var) {
  bool myAnswer = asVariable_Q(var)->treeContainsQvars();
  // bool parentAnswer = Oink::varUsedInDataflow(var);

  // // myAnswer and parentAnswer should be the same.  run with both for a while
  // // to make sure; then we can drop this function altogether.
  // printf("## varUsedInDataflow: myAnswer=%d, parentAnswer=%d\n",
  //        myAnswer, parentAnswer);
  // xassert(myAnswer == parentAnswer);

  // it turns out treeContainsQvars() and Oink::varUsedInDataflow() are not
  // the same, because Oink::varUsedInDataflow() doesn't take into account
  // funky qualifiers.

  return myAnswer;
}

void Qual::unifyVars(Variable_O *v1, Variable_O *v2, SourceLoc loc) {
  dfe.eDataFlow_refUnify(v1->abstrValue(), v2->abstrValue(), loc);
}


// Stages ****************

// callback for traversal using anyCtorSatisfiesF() to mark values as
// instance-specific
#if DEBUG_INSTANCE_SPECIFIC_VALUES
bool markValueInstanceSpecific(Value *t) {
//    cout << "Marking as instance-specific " << t->toString() << endl;
  instanceSpecificValues.add(t);
  return false;                 // keep traversing; a true would cause the search to prune
}
#endif

#if DEBUG_INSTANCE_SPECIFIC_VALUES
void markAsInstanceSpecific(Variable *var0) {
  Variable_Q *var = asVariable_Q(var0);
  if (!var->hasFlag(DF_TYPEDEF)) return;
  if (!var->type->isCompoundType()) return;
  CompoundType *ct = var->type->asCompoundType();
  for (StringRefMap<Variable>::Iter iter = ct->getVariableIter();
       !iter.isDone();
       iter.adv()) {
    Variable_O *var = asVariable_O(iter.value());
    if (!var->getReal()) continue; // only real vars get Values
    Value *v = var->abstrValue(); // NOTE: do NOT use getInstanceSpecificValue
    // I think namespaces don't have a type
    if (v && !v->isFunctionValue()) {
      // FIX: this is a violation of the notion of anyCtorSatisfies as
      // taking a predicate that does not modify the Value in
      // question.
      v->anyCtorSatisfiesF( (ValuePredicate*) markValueInstanceSpecific );
    }
  }
}
#endif

void Qual::markInstanceSpecificValues_stage() {
#if DEBUG_INSTANCE_SPECIFIC_VALUES
  printStage("markInstanceSpecificValues");
  Restorer<bool> restorer(value2typeIsOn, true);
  // NOTE: I don't think that I want to visit just the
  // non-filtered-out ones here; this marking is in support of an
  // assertion that I think should visit all the real vars.
  VisitRealVars visMarkInstSpec(markAsInstanceSpecific); // VAR-TRAVERSAL
  visitVarsMarkedRealF(builtinVars, visMarkInstSpec);
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);

    // for every variable that is a CompoundType, mark as
    // instance-specific the value tree of all of its data members
    visitRealVarsF(unit, visMarkInstSpec);

    // for all the Declarators, mark its value instance-specific; we want
    // to make sure that we use the value of the var of the Declarator
    // instead
    MarkDecltorsInstanceSpecificVisitor env;
    unit->traverse(env.loweredVisitor);
  }
#endif
}

void Qual::moduleAlloc_stage() {
  printStage("moduleAlloc");
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    Qual_ModuleAlloc_Visitor env(unit,
                                 true, // label alloc memory
                                 false // don't label other-control memory
                                 );
    unit->traverse(env.loweredVisitor);
    // check that we didn't hit any allocators that were not down inside
    // a cast
    for(SObjSetIter<E_funCall*> seenIter(env.seenAllocators); !seenIter.isDone(); seenIter.adv()) {
      E_funCall *call0 = seenIter.data();
      if (!env.castedAllocators.contains(call0)) {
        SourceLoc loc = call0->abstrValue->loc; // this is just luck that I have this
        userReportWarning(loc, "(Access) Allocator that was not inside a cast expression");
      }
    }
  }
}


void Qual::moduleOtherControl_stage() {
  printStage("moduleOtherControl");
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    Qual_ModuleAlloc_Visitor env(unit,
                                 false, // don't label alloc memory
                                 true // label other-control memory
                                 );
    unit->traverse(env.loweredVisitor);
    // check that we didn't hit any allocators that were not down inside
    // a cast
    for(SObjSetIter<E_funCall*> seenIter(env.seenAllocators); !seenIter.isDone(); seenIter.adv()) {
      E_funCall *call0 = seenIter.data();
      if (!env.castedAllocators.contains(call0)) {
        SourceLoc loc = call0->abstrValue->loc; // this is just luck that I have this
        userReportWarning(loc, "(Trust) Allocator that was not inside a cast expression");
      }
    }
  }
}

void Qual::moduleOtherWrite_stage() {
  printStage("moduleOtherWrite");
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    Qual_ModuleOtherWrite_Visitor env(unit);
    unit->traverse(env.loweredVisitor);
  }
}

void Qual::moduleAccess_stage() {
  printStage("moduleAccess");
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    Qual_ModuleAccess_Visitor env(unit,
                                  false // color OUR accesses, NOT others
                                  );
    unit->traverse(env.loweredVisitor);
  }
}

void Qual::moduleOtherAccess_stage() {
  printStage("moduleOtherAccess");
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    Qual_ModuleAccess_Visitor env(unit,
                                  true // color OTHER accesses, NOT ours
                                  );
    unit->traverse(env.loweredVisitor);
  }
}

void Qual::moduleAnalysis_stages() {
  // access analysis
  if (qualCmd->module_access) {
    if (qualCmd->module_write) { // exclusive with write analysis
      throw UserError(USER_ERROR_ExitCode,
                      "module-access and module-write don't make sense together");
    }
    if (!(defaultModule || file2module.isNotEmpty())) {
      throw UserError(USER_ERROR_ExitCode, "to use module analyses, you must supply modules");
    }
    moduleAlloc_stage();
    moduleOtherAccess_stage();
  }

  if (qualCmd->module_write) {
    xassert(!qualCmd->module_access); // exclusive with write analysis
    if (!(defaultModule || file2module.isNotEmpty())) {
      throw UserError(USER_ERROR_ExitCode, "to use module analyses, you must supply modules");
    }
    moduleAlloc_stage();
    moduleOtherWrite_stage();
  }

  // trust analysis
  if (qualCmd->module_trust) {
    if (!(defaultModule || file2module.isNotEmpty())) {
      throw UserError(USER_ERROR_ExitCode, "to use module analyses, you must supply modules");
    }
    moduleOtherControl_stage();
    moduleAccess_stage();
  }
}

static void setGlobalTreeOnVar(Variable *var) {
  asVariable_Q(var)->setGlobalTree();
}

void Qual::qualCompile_setGlobalness()
{
  printStage("   setGlobalness");
  VisitRealVars_filter visSetGlobalTree(setGlobalTreeOnVar); // VAR-TRAVERSAL
  visitVarsMarkedRealF_filtered(builtinVars, visSetGlobalTree);
  // TODO: factor this loop into a macro
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    visitRealVarsF_filtered(unit, visSetGlobalTree);
  }
}

void Qual::qualCompile_nameExpressions()
{
#if !DELAY_SET_NAMES
  if (qualCmd->name_expressions) {
    printStage("   nameExpressions");
    doNameExpressions();
  }
#endif
}

static void nameTreeOnVar(Variable *var) {
  xassert(!asVariable_O(var)->filteredOut());
  asVariable_Q(var)->nameTree();
}

void Qual::doNameVars()
{
  VisitRealVars_filter visNameTree(nameTreeOnVar); // VAR-TRAVERSAL
  visitVarsMarkedRealF_filtered(builtinVars, visNameTree);
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    visitRealVarsF_filtered(unit, visNameTree);
  }
}

void Qual::doNameExpressions()
{
  // name the values that annotate the ast
  NameASTNodesVisitor nv;
  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    unit->traverse(nv.loweredVisitor);
  }
}

void Qual::doNameUnnamed()
{
  // name things the nameTree pass didn't reach.
  FOREACH_VALUE(iter) {
    Value *v = iter.data<Value>();
    QualAnnot *a = qa(v);
    a->maybeSetNameMissingIfMissing();
  }
}

void Qual::qualCompile_nameVars()
{
#if !DELAY_SET_NAMES
  // name the values that annotate qvars
  if (qualCmd->name_vars) {
    printStage("   nameVars");
    doNameVars();
  }
#endif
}

// XXX FIXME KLUDGE
extern Qual *theQual;
void Qual_doAssignNames()
{
  theQual->doAssignNames();
}

void Qual::doAssignNames()
{
  printStage("      assign names");
  Restorer<bool> restorer(value2typeIsOn, true);
  doNameVars();
  doNameExpressions();
  doNameUnnamed();
}

// FIX: class VisitRealVars_ValueVisitor should go into value.cc/.h
// but I have trouble including cc_ast_aux.h into value.h; since this
// is the only place it is used I put it here for now.

// carry a ValueVisitor to all of the real Variables
class VisitRealVars_ValueVisitor : public VisitRealVars_filter {
  public:
  // data
  ValueVisitor *valueVis;

  // tor
  explicit VisitRealVars_ValueVisitor(ValueVisitor *valueVis0)
    : VisitRealVars_filter(NULL), valueVis(valueVis0)
  {}

  // only visits each Variable once
  virtual void visitVariableIdem(Variable *var0);
};

void VisitRealVars_ValueVisitor::visitVariableIdem(Variable *var0) {
  Variable_O *var = asVariable_O(var0);
  if (!var->hasFlag(DF_TYPEDEF)) return;
  Value *v = var->abstrValue();
  if (valueVis->preVisitValue(v)) {
    valueVis->postVisitValue(v);
  }
}

void Qual::qualCompile_qualVisitation() {
  printStage("   qualVisitation");
  supportedOperators.add("_op_deref", NULL);
  supportedOperators.add("_op_gt", NULL);

  // This has to wait until after the cqual infrastructure has been
  // initialized.
  thrownQv = make_qvar_CQUAL(
#if !DELAY_SET_NAMES
    "generic exception",
#endif
    SL_UNKNOWN, 0, 1);
  FOREACH_ASTLIST(char, qualCmd->catchQuals, iter) {
    LibQual::Type_qualifier *qconst = findQualAndCheck(iter.data());
    // FIX: would be nice to make a "command line loc" rather than UNKNOWN
    attachOneLiteralToQvar(SL_UNKNOWN, qconst, thrownQv);
  }

  // also do the compound containment pointers: to/from
  // struct/class/union members and their whole container
  StructuralFlow_ValueVisitor *sfValueVis = NULL;
  VisitRealVars_ValueVisitor *sfRealVarValueVis = NULL; // VAR-TRAVERSAL
  if (qualCmd->flow_compoundUp || qualCmd->flow_compoundDown ||
      qualCmd->flow_pointUp || qualCmd->flow_pointDown ||
      qualCmd->flow_refUp || qualCmd->flow_refDown) {

    // set up to visit the values in the AST
    sfValueVis = new StructuralFlow_ValueVisitor(dfe);
    sfValueVis->compoundUp   = qualCmd->flow_compoundUp;
    sfValueVis->compoundDown = qualCmd->flow_compoundDown;
    sfValueVis->pointUp      = qualCmd->flow_pointUp;
    sfValueVis->pointDown    = qualCmd->flow_pointDown;
    sfValueVis->refUp        = qualCmd->flow_refUp;
    sfValueVis->refDown      = qualCmd->flow_refDown;

    // set up to visit all the Variables
    sfRealVarValueVis = new VisitRealVars_ValueVisitor(sfValueVis);
    // visit the builtinVars
    visitVarsMarkedRealF_filtered(builtinVars, *sfRealVarValueVis);
  }

  foreachSourceFile {
    File *file = files.data();
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    // do qual visitation
    QualVisitor env(unit, *this, dfe);
    unit->traverse(env.loweredVisitor);
    if (sfValueVis) {
      xassert(sfRealVarValueVis);
      FindValueASTVisitor fvastvis(*sfValueVis);
      unit->traverse(fvastvis.loweredVisitor);
      visitRealVarsF_filtered(unit, *sfRealVarValueVis);
    }
  }

  if (sfValueVis) delete sfValueVis;
  if (sfRealVarValueVis) delete sfRealVarValueVis;
}

void Qual::qualCompile_stage() {
  printStage("qualCompile");
  Restorer<bool> restorer(value2typeIsOn, true);

  qualCompile_setGlobalness();
  qualCompile_nameExpressions();
  qualCompile_nameVars();
  qualCompile_qualVisitation();
  // optional backend integrity check
  // LibQual::check_all_edge_sets();
}

bool Qual::finishQuals_stage() {
  // NOTE: print quals graph gets done here by the backend if the flag
  // to do it was set during command line parsing
  printStage("finish quals");

#if DELAY_SET_NAMES
  // quarl 2006-06-26
  //    For DELAY_SET_NAMES, we need to run before finish_quals because it's
  //    going to request names.
  updateEcrAll();
#else
#endif
  bool qualifierInconsistency = LibQual::finish_quals();
  // updateEcrAll(); // see below markExtern_stage() why this is commented for now
  haveRunFinishQuals = true;
  return qualifierInconsistency;
}

void Qual::markExtern_stage() {
  Restorer<bool> restorer(value2typeIsOn, true);
  markExternVars();
  // TODO: apparently this has to run after markExternVars, perhaps because it
  // is creating new variables?!
  updateEcrAll();
}

void Qual::compactify_stage() {
  if (qualCmd->doCompactifyGraph()) {
    printStage("compactify stage");
    compactifyGraph();
  }
}
