// see License.txt for copyright and terms of use

#include "qual_cmd.h"
#include "qual_libqual_iface.h" // LibQual::
#include "oink_cmd_util.h"      // HANDLE_FLAG
#include "oink_util.h"
#include "oink_global.h"        // globalLang
#include <cstring>              // strdup
#include <cstdlib>              // atoi

QualCmd::QualCmd()
  : config                   (NULL)
  , hotspots                 (0)
  , print_quals_graph        (false)
  , strict_const             (false)
  , casts_preserve           (false)
  , use_const_subtyping      (true) // default is "on" says the manual
  , flow_sensitive           (false)

  // in cqual the default is "off" from empirical observation, not the
  // manual, but the graph it generates is deceptive when off, so I
  // turn it on by default
  , ugly                     (true)

  , poly                     (false)

  , inference                (true)
  , print_trans_qual         (false)
  , name_expressions         (true)
  , name_vars                (true)
  , name_if_missing          (true)
  , casts_preserve_below_functions(true) // Jeff says the default is to not do this; Revolution!

  , flow_compoundUp          (false)
  , flow_compoundDown        (false)
  , flow_pointUp             (false)
  , flow_pointDown           (false)
  , flow_refUp               (false)
  , flow_refDown             (false)

  , explain_errors           (true)
  , max_errors               (100)
  , ret_inf                  (true)
  , name_with_loc            (false)
  , name_with_serialno       (false)

  , merge_ref_ptr_qvars      (true)

  , compactify_graph         (CG_IFSRZ)

  , module_access            (false)
  , module_write             (false)
  , module_trust             (false)
{}

void QualCmd::readEnvironment()
{
  OinkCmd::readEnvironment();
  const char *env_config = getenv("QUALCC_CONFIG");
  if (env_config && *env_config)
    config = strdup(env_config);
}

void QualCmd::readOneArg(int &argc, char **&argv) {
  int old_argc = argc;
  OinkCmd::readOneArg(argc, argv);
  if (old_argc != argc) return; // the superclass read one so we don't

  char *arg = argv[0];
  if (streq(arg, "-q-config")) {
    shift(argc, argv);
    config = shift(argc, argv);
    return;
  }
  if (streq(arg, "-q-hotspots")) {
    shift(argc, argv);
    hotspots = atoi(shift(argc, argv));
    return;
  }

  HANDLE_FLAG(print_quals_graph, "-fq-", "print-quals-graph");
  HANDLE_FLAG(strict_const, "-fq-", "strict-const");
  HANDLE_FLAG(casts_preserve, "-fq-", "casts-preserve");
  HANDLE_FLAG(use_const_subtyping, "-fq-", "use-const-subtyping");
  HANDLE_FLAG(ugly, "-fq-", "ugly");
  HANDLE_FLAG(poly, "-fq-", "poly");

  if (streq(arg, "-q-catch-qual")) {
    shift(argc, argv);
    catchQuals.append(strdup(shift(argc, argv)));
    return;
  }
  HANDLE_FLAG(inference, "-fq-", "inference");
  HANDLE_FLAG(print_trans_qual, "-fq-", "print-trans-qual");
  HANDLE_FLAG(name_expressions, "-fq-", "name-expressions");
  HANDLE_FLAG(name_vars, "-fq-", "name-vars");
  HANDLE_FLAG(name_if_missing, "-fq-", "name-if-missing");

  if (streq(arg, "-fq-names")) {
    shift(argc, argv);
    name_expressions = true;
    name_vars = true;
    name_if_missing = true;
    return;
  }
  if (streq(arg, "-fq-no-names")) {
    shift(argc, argv);
    name_expressions = false;
    name_vars = false;
    name_if_missing = false;
    return;
  }

  HANDLE_FLAG(casts_preserve_below_functions, "-fq-", "casts-preserve-below-functions");

  HANDLE_FLAG(flow_compoundUp, "-fq-", "flow-compoundUp");
  HANDLE_FLAG(flow_compoundDown, "-fq-", "flow-compoundDown");
  HANDLE_FLAG(flow_pointUp, "-fq-", "flow-pointUp");
  HANDLE_FLAG(flow_pointDown, "-fq-", "flow-pointDown");
  HANDLE_FLAG(flow_refUp, "-fq-", "flow-refUp");
  HANDLE_FLAG(flow_refDown, "-fq-", "flow-refDown");

  HANDLE_FLAG(explain_errors, "-fq-", "explain-errors");

  if (streq(arg, "-q-max-errors")) {
    shift(argc, argv);
    max_errors = atoi(shift(argc, argv));
    return;
  }

  HANDLE_FLAG(ret_inf, "-fq-", "ret-inf");
  HANDLE_FLAG(name_with_loc, "-fq-", "name-with-loc");
  HANDLE_FLAG(name_with_serialno, "-fq-", "name-with-serialno");

  HANDLE_FLAG(merge_ref_ptr_qvars, "-fq-", "merge-ref-ptr-qvars");

  // HANDLE_FLAG(compactify_graph, "-fq-", "compactify-graph");
  if (streq(arg, "-fq-compactify-graph") ||
      streq(arg, "-fq-compactify-graph=yes") ||
      streq(arg, "-fq-compactify-graph=ifsrz"))
  {
    shift(argc, argv);
    compactify_graph = CG_IFSRZ;
    return;
  }
  if (streq(arg, "-fq-no-compactify-graph") ||
      streq(arg, "-fq-compactify-graph=never") ||
      streq(arg, "-fq-compactify-graph=no"))
  {
    shift(argc, argv);
    compactify_graph = CG_NEVER;
    return;
  }
  if (streq(arg, "-fq-compactify-graph=always"))
  {
    shift(argc, argv);
    compactify_graph = CG_ALWAYS;
    return;
  }

  HANDLE_FLAG(module_access, "-fq-", "module-access");
  HANDLE_FLAG(module_write, "-fq-", "module-write");
  HANDLE_FLAG(module_trust, "-fq-", "module-trust");
}

void QualCmd::dump() {
  OinkCmd::dump();
  printf("q-config '%s'\n", config);
  printf("q-hotspots: %d\n", hotspots);
  printf("fq-print-quals-graph: %s\n", boolToStr(print_quals_graph));
  printf("fq-strict-const: %s\n", boolToStr(strict_const));
  printf("fq-casts-preserve: %s\n", boolToStr(casts_preserve));
  printf("fq-use-const-subtyping: %s\n", boolToStr(use_const_subtyping));
  printf("fq-ugly: %s\n", boolToStr(ugly));
  printf("fq-poly: %s\n", boolToStr(poly));

  // qual arguments
  printf("q-catch-qual:\n");
  FOREACH_ASTLIST(char, catchQuals, iter) {
    printf("\t%s\n", iter.data());
  }
  printf("fq-inference: %s\n", boolToStr(inference));
  printf("fq-print-trans-qual: %s\n", boolToStr(print_trans_qual));
  printf("fq-name-expressions: %s\n", boolToStr(name_expressions));
  printf("fq-name-vars: %s\n", boolToStr(name_vars));
  printf("fq-name-if_missing: %s\n", boolToStr(name_if_missing));
  printf("fq-casts-preserve-below-functions: %s\n",
         boolToStr(casts_preserve_below_functions));

  printf("fq-flow-compoundUp: %s\n", boolToStr(flow_compoundUp));
  printf("fq-flow-compoundDown: %s\n", boolToStr(flow_compoundDown));
  printf("fq-flow-pointUp: %s\n", boolToStr(flow_pointUp));
  printf("fq-flow-pointDown: %s\n", boolToStr(flow_pointDown));
  printf("fq-flow-refUp: %s\n", boolToStr(flow_refUp));
  printf("fq-flow-refDown: %s\n", boolToStr(flow_refDown));

  printf("fq-explain-errors: %s\n", boolToStr(explain_errors));
  printf("q-max-errors: %d\n", max_errors);
  printf("fq-ret-inf: %s\n", boolToStr(ret_inf));
  printf("fq-name-with-loc: %s\n", boolToStr(name_with_loc));
  printf("fq-name-with-serialno: %s\n", boolToStr(name_with_serialno));

  printf("fq-merge-ref-ptr-qvars: %s\n", boolToStr(merge_ref_ptr_qvars));

  printf("fq-compactify-graph: %s\n",
         compactify_graph == CG_ALWAYS ? "always" :
         compactify_graph == CG_IFSRZ ? "ifsrz" :
         compactify_graph == CG_NEVER ? "never" :
         "?!");

  printf("fq-module-access: %s\n", boolToStr(module_access));
  printf("fq-module-write: %s\n", boolToStr(module_write));
  printf("fq-module-trust: %s\n", boolToStr(module_trust));
}

void QualCmd::printHelp() {
  OinkCmd::printHelp();
  printf
    ("%s",
     "\nqualcc/legacy-cqual flags that take an argument:\n"
     "  -q-config <file>         : set the configuration file to <file>\n"
     "                                 (can also set $QUALCC_CONFIG)\n"
     "  -q-hotspots <num>        : set the number of hotspots to <num>\n"
     "\n"
     "qual/legacy-cqual boolean flags; precede by '-fq-no-' for the negative sense.\n"
     "  -fq-print-quals-graph    : print out quals.dot of inference graph\n"          //
     "  -fq-strict-const         : enforce const, rather than allow const inference\n"
     "  -fq-casts-preserve       : inference goes through casts, rather than blocking\n"
     "  -fq-use-const-subtyping  : no back edge below pointers if pointing to const\n"
     "  -fq-ugly                 : put address of qvar into quals.dot node names\n"
     "  -fq-poly                 : do polymorphic analysis\n"
     "\n"
     "qual flags that take an argument:\n"
     "  -q-catch-qual <qname>    : attach qname to singleton exception variable\n"
     "  -q-max-errors            : max errors to print; 0 for unlimited (default 80)\n"
     "\n"
     "qual boolean flags; preceded by '-fq-no-' for the negative sense.\n"
     "  -fq-inference            : do inference\n"
     "  -fq-print-trans-qual     : in pretty printing, annotate inferred qualifiers\n"
     "  -fq-name-expressions     : name expressions\n"
     "  -fq-name-vars            : name variables\n"
     "  -fq-name-if-missing      : name otherwise unnamed objects\n"
     "  -fq-names                : control all other -fq-name flags\n"
     "  -fq-casts-preserve-below-functions : -fcasts-preserve works below functions\n"
     "  -fq-flow-compoundUp      : insert var to container edges\n"
     "  -fq-flow-compoundDown    : insert container to var edges\n"
     "  -fq-flow-pointUp         : insert value to pointer/array edges\n"
     "  -fq-flow-pointDown       : insert pointer/array to value edges\n"
     "  -fq-flow-refUp           : insert value to ref edges\n"
     "  -fq-flow-refDown         : insert ref to value edges\n"
     "  -fq-explain-errors       : print bad dataflow path when one is found\n"
     "  -fq-ret-inf              : return the inference result in the exit code\n"
     "  -fq-name-with-loc        : put location info on qualifier variable names\n"
     "  -fq-merge-ref-ptr-qvars  : merge qvars of refs and ptrs (vs unify)\n"
     "  -fq-compactify-graph     : compactify graph before serialization\n"
     "  -fq-compactify-graph=always : compactify graph even if not serializing\n"
     "\n"
     "module analysis: look for violations of the module boundaries.\n"
     "  -fq-module-access        : other module accesses a module's memory\n"
     "  -fq-module-write         : other module writes a module's memory\n"
     "  -fq-module-trust         : access through a pointer in another's control"
     "");
}

// push the state out to other places where it is needed; return value
// is true unless there is an error
void QualCmd::initializeFromFlags() {
  OinkCmd::initializeFromFlags();

  // hotspots
  if (hotspots < 0) {
    throw UserError(USER_ERROR_ExitCode,
                    stringc << "hotspots must be non-negative: " << hotspots);
  }
  LibQual::num_hotspots = hotspots;

  // print_quals_graph
  LibQual::flag_print_quals_graph = print_quals_graph ?1:0;

  // ugly
  LibQual::flag_ugly = ugly ?1:0;

  // poly
  LibQual::flag_poly = poly ?1:0;

  // explain errors
  LibQual::flag_explain_errors = explain_errors ?1:0;

  // max errors
  LibQual::flag_max_errors = max_errors;

  // check for inconsistent command line combinations
  if (exit_after_parse && print_quals_graph) {
    throw UserError(USER_ERROR_ExitCode,
                    "You may not use -fq-exit-after-parse and -fprint-quals-graph together.");
  }

  if (exit_after_typecheck && print_quals_graph) {
    throw UserError(USER_ERROR_ExitCode,
                    "You may not use -fq-exit-after-typecheck and -fprint-quals-graph together.");
  }

  if (exit_after_elaborate && print_quals_graph) {
    throw UserError(USER_ERROR_ExitCode,
                    "You may not use -fq-exit-after-elaborate and -fprint-quals-graph together.");
  }

  if (instance_sensitive && print_trans_qual) {
    throw UserError(USER_ERROR_ExitCode,
                    "You may not use -fo-instance-sensitive and -fq-print-trans-qual together.");
  }

  if (instance_sensitive && globalLang.isCplusplus) {
    throw UserError(USER_ERROR_ExitCode,
                    "You may not use -fo-instance-sensitive in C++ mode.");
  }

  if (!pretty_print && print_trans_qual) {
    throw UserError(USER_ERROR_ExitCode,
                    "You must use -fo-pretty-print with -fq-print-trans-qual.");
  }

  // NOTE: flattenInputFiles() has already been called
  // this is off because I am now using the same input file mechanism for the .qdir files
//    if (srz && inputFiles.count() > 1) {
//      throw UserError(USER_ERROR_ExitCode,
//                      "You may not use -q-srz with more than one input file.");
//    }
}
