/**
 * ===========================================
 * LibFonts : a free Java font reading library
 * ===========================================
 *
 * Project Info:  http://reporting.pentaho.org/libfonts/
 *
 * (C) Copyright 2006-2007, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * $Id: ITextFontStorage.java 3523 2007-10-16 11:03:09Z tmorgner $
 * ------------
 * (C) Copyright 2006-2007, by Pentaho Corporation.
 */

package org.jfree.fonts.itext;

import java.util.HashMap;

import com.lowagie.text.pdf.BaseFont;
import org.jfree.fonts.merge.CompoundFontIdentifier;
import org.jfree.fonts.registry.FontContext;
import org.jfree.fonts.registry.FontIdentifier;
import org.jfree.fonts.registry.FontMetrics;
import org.jfree.fonts.registry.FontRecord;
import org.jfree.fonts.registry.FontRegistry;
import org.jfree.fonts.registry.FontStorage;
import org.jfree.fonts.registry.FontKey;
import org.jfree.fonts.truetype.TrueTypeFontIdentifier;

/**
 * Creation-Date: 22.07.2007, 17:54:43
 *
 * @author Thomas Morgner
 */
public class ITextFontStorage implements FontStorage
{
  private ITextFontRegistry registry;
  private BaseFontSupport baseFontSupport;
  private FontKey lookupKey;
  private HashMap knownMetrics;

  public ITextFontStorage()
  {
    this.lookupKey = new FontKey();
    this.knownMetrics = new HashMap();

    this.registry = new ITextFontRegistry();
    this.registry.initialize();

    this.baseFontSupport = new BaseFontSupport(registry);
  }

  public ITextFontStorage(final ITextFontRegistry registry)
  {
    this.lookupKey = new FontKey();
    this.knownMetrics = new HashMap();
    this.registry = registry;
    this.baseFontSupport = new BaseFontSupport(registry);
  }

  public ITextFontStorage(final ITextFontRegistry registry, final String encoding)
  {
    this.lookupKey = new FontKey();
    this.knownMetrics = new HashMap();
    this.registry = registry;
    this.baseFontSupport = new BaseFontSupport(registry, encoding);
  }

  public FontRegistry getFontRegistry()
  {
    return registry;
  }

  public FontMetrics getFontMetrics(final FontIdentifier rawRecord, final FontContext context)
  {
    if (rawRecord == null)
    {
      throw new NullPointerException();
    }
    if (context == null)
    {
      throw new NullPointerException();
    }

    lookupKey.setAliased(context.isAntiAliased());
    lookupKey.setFontSize(context.getFontSize());
    lookupKey.setIdentifier(rawRecord);
    lookupKey.setFractional(context.isFractionalMetrics());

    final FontMetrics cachedMetrics = (FontMetrics) knownMetrics.get(lookupKey);
    if (cachedMetrics != null)
    {
      return cachedMetrics;
    }

    final CompoundFontIdentifier compoundFontIdentifier;
    final FontIdentifier record;
    if (rawRecord instanceof CompoundFontIdentifier)
    {
      compoundFontIdentifier = (CompoundFontIdentifier) rawRecord;
      record = compoundFontIdentifier.getIdentifier();
    }
    else
    {
      record = rawRecord;
      compoundFontIdentifier = null;
    }

    final String fontName;
    final boolean bold;
    final boolean italic;
    if (record instanceof FontRecord)
    {
      final FontRecord fontRecord = (FontRecord) record;
      fontName = fontRecord.getFamily().getFamilyName();
      if (compoundFontIdentifier != null)
      {
        bold = compoundFontIdentifier.isBoldSpecified();
        italic = compoundFontIdentifier.isItalicsSpecified();
      }
      else
      {
        bold = fontRecord.isBold();
        italic = fontRecord.isItalic();
      }
    }
    else if (record instanceof TrueTypeFontIdentifier)
    {
      final TrueTypeFontIdentifier ttfFontRecord = (TrueTypeFontIdentifier) record;
      fontName = ttfFontRecord.getFontName();
      if (compoundFontIdentifier != null)
      {
        bold = compoundFontIdentifier.isBoldSpecified();
        italic = compoundFontIdentifier.isItalicsSpecified();
      }
      else
      {
        bold = false;
        italic = false;
      }
    }
    else
    {
      throw new IllegalArgumentException("Unknown font-identifier type encountered.");
    }

    final BaseFont baseFont = baseFontSupport.createBaseFont
        (fontName, bold, italic, context.getEncoding(), context.isEmbedded());

    final FontMetrics metrics = new BaseFontFontMetrics(baseFont, (float) context.getFontSize());
    final FontKey key = new FontKey(rawRecord, context.isAntiAliased(),
        context.isFractionalMetrics(), context.getFontSize());
    knownMetrics.put(key, metrics);
    return metrics;
  }

  public BaseFontSupport getBaseFontSupport()
  {
    return baseFontSupport;
  }
}
