/**
 * ===========================================================
 * LibRepository : a free Java content repository access layer
 * ===========================================================
 *
 * Project Info:  http://reporting.pentaho.org/librepository/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * DefaultMimeRegistry.java
 * ------------
 */

package org.pentaho.reporting.libraries.repository;

import java.io.Serializable;

import org.pentaho.reporting.libraries.base.util.StringUtils;

/**
 * The default-mime registry contains a list of well-known file types and returns mime-information for them.
 * This implementation recognizes a couple of image types and CSS, XML and HTML files. The content is recognized
 * by its filename, not by its actual content.
 *
 * @author Thomas Morgner
 */
public class DefaultMimeRegistry implements MimeRegistry, Serializable
{
  private static final long serialVersionUID = 2815922456302361614L;

  /**
   * Default Constructor.
   */
  public DefaultMimeRegistry()
  {
  }

  /**
   * Queries the mime-type for a given content-item. Some repositories store mime-type information along with
   * the content data, while others might resort to heuristics based on the filename or actual data stored in the
   * item.
   *
   * @param item the content item for which Mime-Data should be queried.
   * @return the mime-type never null.
   */
  public String getMimeType(final ContentItem item)
  {
    final String name = item.getName();
    if (name == null)
    {
      return "application/octet-stream";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".png"))
    {
      return "image/png";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".jpg"))
    {
      return "image/jpeg";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".jpeg"))
    {
      return "image/jpeg";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".gif"))
    {
      return "image/gif";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".pdf"))
    {
      return "application/pdf";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".txt"))
    {
      return "text/plain";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".html"))
    {
      return "text/html";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".htm"))
    {
      return "text/html";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".xml"))
    {
      return "text/xml";
    }
    if (StringUtils.endsWithIgnoreCase(name, ".css"))
    {
      return "text/css";
    }
    return "application/octet-stream";
  }

  /**
   * Returns the default suffix for files with the given content type.
   *
   * @param mimeType the mime-type for which a suffix is queried.
   * @return the suffix, never null.
   */
  public String getSuffix(final String mimeType)
  {
    // needs 'libMagic'
    if ("image/png".equals(mimeType))
    {
      return "png";
    }
    if ("image/jpeg".equals(mimeType))
    {
      return "jpg";
    }
    if ("image/jpg".equals(mimeType))
    {
      return "jpg";
    }
    if ("image/gif".equals(mimeType))
    {
      return "gif";
    }
    if ("text/html".equals(mimeType))
    {
      return "html";
    }
    if ("text/xml".equals(mimeType))
    {
      return "xml";
    }
    if ("text/plain".equals(mimeType))
    {
      return "txt";
    }
    if ("text/css".equals(mimeType))
    {
      return "css";
    }
    if ("application/pdf".equals(mimeType))
    {
      return "pdf";
    }
    return "dat";
  }
}
