/**
 * VampirTrace
 * http://www.tu-dresden.de/zih/vampirtrace
 *
 * Copyright (c) 2005-2011, ZIH, TU Dresden, Federal Republic of Germany
 *
 * Copyright (c) 1998-2005, Forschungszentrum Juelich, Juelich Supercomputing
 *                          Centre, Federal Republic of Germany
 *
 * See the file COPYING in the package base directory for details
 **/
package widgets;
import help.Config;
import help.Convertions;

import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;

import javax.swing.BorderFactory;
import javax.swing.JFormattedTextField;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;


/****
 * This class is a component that consists of a slider element and textbox to 
 * manage values of type float.
 */
public class FloatPanel extends javax.swing.JPanel{
	/**
	 * generated serialVersionUID
	 */
	private static final long serialVersionUID = -2827378741605769471L;
	private int min;
	private int max;
	private int curent;
	private double realCurent;
	private double realMin;
	private double realMax;
	public JFormattedTextField text = new JFormattedTextField();
	private boolean changeable = true;
	JSlider slider = null;
	JPanel panel = new JPanel(new FlowLayout());
	
	public FloatPanel(){;}
	
	/****
	 * initialize all components of this class
	 * @param min minimum value that can be set
	 * @param max maximum value that can be set
	 * @param curent default value 
	 */
	public FloatPanel(double min, double max, double curent)
	{
		this.changeable=false;
		this.min = 0;
		this.max = 1000;
		this.realMin = min;
		this.realMax = max;
		this.realCurent = curent;
		this.curent = getCurent(curent);
		panel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
		this.slider = new JSlider(JSlider.HORIZONTAL,this.min, this.max,this.curent);
		text.setText(String.valueOf(Convertions.round(this.realCurent)));
		this.changeable=true;
		getPanel();
		this.add(this.slider);
		this.add(this.text);
		
	}
	
	/***
	 * this method is used by FloatPanel and initialize
	 * the EventListener of the components
	 */
	public void getPanel()
	{

		slider.setMajorTickSpacing(500);
		slider.setMinorTickSpacing(100);
		slider.setPaintTicks(true);
		panel.add(slider);
		text.setColumns(Integer.toString((int)Math.ceil(this.realMax)).length()+4);
		
		// ******* Event Listener ********* /
		text.addFocusListener(new FocusListener(){
			public void focusLost(FocusEvent e){
				if(changeable)
					FloatPanel.this.changeSlider(FloatPanel.this.text.getText().trim());
				}
			public void focusGained(FocusEvent e){;}
		});
		
		text.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				if(changeable)
					FloatPanel.this.changeSlider(FloatPanel.this.text.getText().trim());
			}
		});
		text.getDocument().addDocumentListener(new DocumentListener(){
			public void changedUpdate(DocumentEvent e){;}
			
			public void insertUpdate(DocumentEvent e){
				FloatPanel.this.text.getDocument().removeDocumentListener(this);
				if(changeable)
					FloatPanel.this.changeSlider(FloatPanel.this.text.getText().trim());
				FloatPanel.this.text.getDocument().addDocumentListener(this);
				}
			
			public void removeUpdate(DocumentEvent e){				
				FloatPanel.this.text.getDocument().removeDocumentListener(this);
				if(changeable)
					FloatPanel.this.changeSlider(FloatPanel.this.text.getText().trim());
				FloatPanel.this.text.getDocument().addDocumentListener(this);
				}
		});
		
		slider.addChangeListener(new ChangeListener(){
			public void stateChanged(ChangeEvent e){
				if(changeable)
					FloatPanel.this.changeText(FloatPanel.this.slider.getValue());
			}
		});
		panel.add(text);
		
	}
	
	
	/***
	 * converts the value of the text field to the value of the slider
	 * @param realCurent value of the text field
	 * @return value of the slider
	 */
	private int getCurent(double realCurent)
	{	
		if(realCurent == this.realMax)
			return this.max;
		if(realCurent == this.realMin)
			return this.min;

		double min,max;
		double help;
		min = (double) this.min;
		max = (double) this.max;
		
		help = (max-min);
		help = help/(realMax-realMin);
		help = help * realCurent;
		return (int) help;
	}
	
	/***
	 * converts the value of the slider to the value of the text field
	 * @param curent value of the slider
	 * @return value of the text field
	 */
	private double getRealCurent(double curent)
	{
		if(curent == this.max)
			return this.realMax;
		if(curent == this.min)
			return this.realMin;
		
		double min,max;
		double help;
		min = (double) this.min;
		max = (double) this.max;
		help = (realMax-realMin);
		help = help/(max-min);
		help = help*curent;
		return help;
	}

	/***
	 * sets the value of the slider element
	 * @param text number of the value 
	 */
	public void changeSlider(String text )
	{
		this.changeable = false;
		if(this.text.getText().compareTo(text) != 0)
			this.text.setText(text);

		double realCurent = Convertions.getDouble(text);
			
		try{	
			if(realCurent > this.realMax)
			{
				realCurent = this.realMax;
				this.text.setText(String.valueOf(Convertions.round(realCurent)));
			}
			if(realCurent < this.realMin)
			{
				realCurent = this.realMin;
				this.text.setText(String.valueOf(Convertions.round(realCurent)));
			}
		}catch(Exception ex)
		{
			Config.errorHandler(ex);
		}
		
			int curent = getCurent(realCurent);
			if(this.slider.getValue() != curent)
				this.slider.setValue(curent);
			this.curent = curent;
			this.realCurent = realCurent;
			
			this.changeable = true;
	}
	

	/***
	 * sets the value of the text field
	 * @param curent value 
	 */
	public void changeText(double curent)
	{
		this.changeable = false;
		
		double longCurent = getRealCurent(curent);
		this.curent = (int)curent;
		this.realCurent = longCurent;
		String text = String.valueOf(Convertions.round(longCurent)).toString();
		if(this.text.getText().compareTo(text) != 0)
			this.text.setText(text);
		this.changeable = true;
	}
	
}
