/**
 * VampirTrace
 * http://www.tu-dresden.de/zih/vampirtrace
 *
 * Copyright (c) 2005-2011, ZIH, TU Dresden, Federal Republic of Germany
 *
 * Copyright (c) 1998-2005, Forschungszentrum Juelich, Juelich Supercomputing
 *                          Centre, Federal Republic of Germany
 *
 * See the file COPYING in the package base directory for details
 **/
package widgets;

import help.Config;
import help.Convertions;

import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;

import javax.swing.BorderFactory;
import javax.swing.JFormattedTextField;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

/****
 * This class is a component that consists of a slider element and textbox to 
 * to manage values of type integer.
 */

public class IntegerPanel extends javax.swing.JPanel {
	/**
	 * generated serialVersionUID
	 */
	private static final long serialVersionUID = -4102871509982834770L;
	private int min;
	private int max;
	private int curent;
	private long realCurent;
	private long realMin;
	private long realMax;
	public JFormattedTextField text = new JFormattedTextField();
	private boolean changeable = true;
	JSlider slider = null;
	JPanel panel = new JPanel(new FlowLayout());
	
	public IntegerPanel(){;}
	
	/****
	 * initialize all components of this class
	 * @param min minimum value that can be set
	 * @param max maximum value that can be set
	 * @param curent default value 
	 */
	public IntegerPanel(long min, long max, long curent)
	{
		this.changeable=false;
		this.min = 0;
		this.max = 100000;
		this.realMin = min;
		this.realMax = max;
		this.realCurent = curent;
		this.curent = getCurent(curent);
		panel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
		this.slider = new JSlider(JSlider.HORIZONTAL,this.min, this.max,this.curent);
		text.setText(String.valueOf(this.realCurent));
		this.changeable=true;
		getPanel();
		this.add(this.slider);
		this.add(this.text);
	}
	
	/***
	 * this method is used by IntegerPanel and initialize
	 * the EventListener of the components
	 */
	public void getPanel()
	{
		slider.setMajorTickSpacing(50000);
		slider.setMinorTickSpacing(10000);
		slider.setPaintTicks(true);
		slider.setPaintLabels(false);
		
		this.add(slider);
		text.setColumns(Long.toString(this.realMax).length());
		
		// ******* Event Listener ********* /
		text.addFocusListener(new FocusListener(){
			public void focusLost(FocusEvent e){
				if(changeable)
					IntegerPanel.this.changeSlider(IntegerPanel.this.text.getText().trim());
				}
			public void focusGained(FocusEvent e){;}
		});
		
		text.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				if(changeable)
					IntegerPanel.this.changeSlider(IntegerPanel.this.text.getText().trim());
			}
		});
		text.getDocument().addDocumentListener(new DocumentListener(){
			public void changedUpdate(DocumentEvent e){;}
			
			public void insertUpdate(DocumentEvent e){
				IntegerPanel.this.text.getDocument().removeDocumentListener(this);
				if(changeable)
					IntegerPanel.this.changeSlider(IntegerPanel.this.text.getText().trim());
				IntegerPanel.this.text.getDocument().addDocumentListener(this);
				}
			
			public void removeUpdate(DocumentEvent e){				
				IntegerPanel.this.text.getDocument().removeDocumentListener(this);
				if(changeable)
					IntegerPanel.this.changeSlider(IntegerPanel.this.text.getText().trim());
				IntegerPanel.this.text.getDocument().addDocumentListener(this);
				}
		});
		
		slider.addChangeListener(new ChangeListener(){
			public void stateChanged(ChangeEvent e){
				if(changeable)
					IntegerPanel.this.changeText(IntegerPanel.this.slider.getValue());
			}
		});
		this.add(text);
		
	}
	
	/***
	 * converts the value of the text field to the value of the slider
	 * @param realCurent value of the text field
	 * @return value of the slider
	 */
	private int getCurent(long realCurent)
	{	
		if(realCurent == this.realMax)
			return this.max;
		if(realCurent == this.realMin)
			return this.min;

		long min,max;
		double help;
		min = (long) this.min;
		max = (long) this.max;
		
		help = (max-min);
		help = help/(realMax-realMin);
		help = help * realCurent;
		return (int) help;
	}
	
	/***
	 * converts the value of the slider to the value of the text field
	 * @param curent value of the slider
	 * @return value of the text field
	 */
	private long getRealCurent(int curent)
	{
		if(curent == this.max)
			return this.realMax;
		if(curent == this.min)
			return this.realMin;
		
		long min,max;
		double help;
		min = (long) this.min;
		max = (long) this.max;
		help = (realMax-realMin);
		help = help/(max-min);
		help = help*curent;
		return (long)help;
	}

	/***
	 * sets the value of the slider element
	 * @param text number of the value 
	 */
	public void changeSlider(String text )
	{
		this.changeable = false;
		
		if(this.text.getText().compareTo(text) != 0)
			this.text.setText(text);
			

		long realCurent = Convertions.getLong(text);
			
		try{	
			if(realCurent > this.realMax)
			{
				realCurent = this.realMax;
				this.text.setText(String.valueOf(realCurent));
			}
			if(realCurent < this.realMin)
			{
				realCurent = this.realMin;
				this.text.setText(String.valueOf(realCurent));
			}
		}catch(Exception ex)
		{
			Config.errorHandler(ex);
		}
		
			int curent = getCurent(realCurent);
			if(this.slider.getValue() != curent)
				this.slider.setValue(curent);
			this.curent = curent;
			this.realCurent = realCurent;
			
			this.changeable = true;
	}

	/***
	 * sets the value of the text field
	 * @param curent value 
	 */
	public void changeText(int curent)
	{
		this.changeable = false;
		
		long longCurent = getRealCurent(curent);
		this.curent = curent;
		this.realCurent = longCurent;
		String text = String.valueOf(longCurent).toString();
		if(this.text.getText().compareTo(text) != 0)
			this.text.setText(text);
		this.changeable = true;
				
	}
	
}
