/* moov.c
 * Copyright (C) 2001 QT4Linux and OpenQuicktime Teams
 *
 * This file is part of OpenQuicktime, a free QuickTime library.
 *
 * Based on QT4Linux by Adam Williams.
 *
 * OpenQuicktime is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * OpenQuicktime is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "funcprotos.h"
#include "openquicktime.h"
#include <zlib.h>




int quicktime_moov_init(quicktime_moov_t *moov)
{
	int i;

	moov->total_tracks = 0;
	for(i = 0 ; i < MAXTRACKS; i++) moov->trak[i] = 0;
	quicktime_mvhd_init(&(moov->mvhd));
	quicktime_udta_init(&(moov->udta));
	quicktime_ctab_init(&(moov->ctab));
	return 0;
}

int quicktime_moov_delete(quicktime_moov_t *moov)
{
	int i;
	while(moov->total_tracks) quicktime_delete_trak(moov);
	quicktime_mvhd_delete(&(moov->mvhd));
	quicktime_udta_delete(&(moov->udta));
	quicktime_ctab_delete(&(moov->ctab));
	return 0;
}

void quicktime_moov_dump(quicktime_moov_t *moov)
{
	int i;
	printf("movie\n");
	quicktime_mvhd_dump(&(moov->mvhd));
	quicktime_udta_dump(&(moov->udta));
	for(i = 0; i < moov->total_tracks; i++)
		quicktime_trak_dump(moov->trak[i]);
	quicktime_ctab_dump(&(moov->ctab));
}


#define QT_zlib 0x7A6C6962

int quicktime_read_moov(quicktime_t *file, quicktime_moov_t *moov, quicktime_atom_t *parent_atom)
{
  /* mandatory mvhd */
  quicktime_atom_t leaf_atom;
  
  do
    {
      quicktime_atom_read_header(file, &leaf_atom);
      
      if(quicktime_atom_is(&leaf_atom, "cmov")) 
	{
	  quicktime_atom_t compressed_atom;

	  unsigned char *cmov_buf = 0;
	  unsigned char *moov_buf = 0;
	  longest cmov_sz, tlen;
	  int moov_sz;
	  int cmov_ret = 0;
	  long cmov_comp = 0;
	  
	  quicktime_atom_read_header(file, &compressed_atom);
	  
	  if(quicktime_atom_is(&compressed_atom, "dcom"))
	    {
	      quicktime_atom_t compressed_type_atom;
	      int zlibfourcc;
	      longest offset;
	      

	      quicktime_read_char32(file, (char *)&zlibfourcc);
	      zlibfourcc = quicktime_atom_read_size((char *)&zlibfourcc);

	      if(zlibfourcc != QT_zlib)
		printf("Header not compressed with zlib\n");

	      if(compressed_atom.size - 4 > 0)
		{
		  offset = file->ftell_position + compressed_atom.size - 4;
		  file->quicktime_fseek(file, offset);
		}
	    }
	  quicktime_atom_read_header(file, &compressed_atom);

	  if(quicktime_atom_is(&compressed_atom, "cmvd"))
	    {
	      z_stream zstrm;
	      int zret;

	      /* read how large uncompressed moov will be */
	      quicktime_read_char32(file, (char *)&moov_sz);
	      moov_sz = quicktime_atom_read_size((char *)&moov_sz);
	      cmov_sz = compressed_atom.size - 4;

	      /* Allocate buffer for compressed header */
	      cmov_buf = (unsigned char *)malloc( cmov_sz );
	      if (cmov_buf == 0) 
		{
		  fprintf(stderr, "QT cmov: malloc err 0");
		  exit(1);
		}
	      /* Read in  compressed header */

	      tlen = file->quicktime_read_data(file, (char*)cmov_buf, cmov_sz);
	      
	      if (tlen != 1)
		{ 
		  fprintf(stderr,"QT cmov: read err tlen %llu\n", tlen);
		  free(cmov_buf);
		  return 0;
		}
	      
	      /* Allocate buffer for decompressed header */
	      moov_sz += 16; /* slop?? */
	      moov_buf = (unsigned char *)malloc( moov_sz );
	      if (moov_buf == 0) 
		{
		  fprintf(stderr,"QT cmov: malloc err moov_sz %u\n", moov_sz);
		  exit(1);
		}
	      
	      zstrm.zalloc          = (alloc_func)0;
	      zstrm.zfree           = (free_func)0;
	      zstrm.opaque          = (voidpf)0;
	      zstrm.next_in         = cmov_buf;
	      zstrm.avail_in        = cmov_sz;
	      zstrm.next_out        = moov_buf;
	      zstrm.avail_out       = moov_sz;
	      
	      zret = inflateInit(&zstrm);
	      if (zret != Z_OK)
		{ fprintf(stderr,"QT cmov: inflateInit err %d\n",zret);
		break;
		}
	      zret = inflate(&zstrm, Z_NO_FLUSH);
	      if ((zret != Z_OK) && (zret != Z_STREAM_END))
		{ fprintf(stderr,"QT cmov inflate: ERR %d\n",zret);
		break;
		}
	      else {
		FILE *DecOut;

		DecOut = fopen("Out.bin", "w");
		fwrite(moov_buf, 1, moov_sz, DecOut);
		fclose(DecOut);
	      }
	      moov_sz = zstrm.total_out;
	      zret = inflateEnd(&zstrm);
	      
	      file->decompressed_buffer_size = moov_sz;
	      file->decompressed_buffer = (char*)moov_buf;
	      file->decompressed_position = 8; // Passing the first moov

	      
	      
	    } // end of "cmvd"
	  /*		  
			  if (cmov_buf) free(cmov_buf);
			  if (moov_buf) free(moov_buf);
			  if (cmov_ret == 0) return(cmov_ret); //failed or unsupported */
	} /* end of cmov */
      else		
	if(quicktime_atom_is(&leaf_atom, "mvhd"))
	  {
	    quicktime_read_mvhd(file, &(moov->mvhd));
	  }
	else
	  if(quicktime_atom_is(&leaf_atom, "clip"))
	    {
	      quicktime_atom_skip(file, &leaf_atom);
	    }
	  else
	    if(quicktime_atom_is(&leaf_atom, "trak"))
	      {
		quicktime_trak_t *trak = quicktime_add_trak(moov);	
		quicktime_read_trak(file, trak, &leaf_atom);
	      }
	    else
	      if(quicktime_atom_is(&leaf_atom, "udta"))
		{
		  quicktime_read_udta(file, &(moov->udta), &leaf_atom);
		  quicktime_atom_skip(file, &leaf_atom);
		}
	      else
		if(quicktime_atom_is(&leaf_atom, "ctab"))
		  {
		    quicktime_read_ctab(file, &(moov->ctab));
		  }
		else
		  quicktime_atom_skip(file, &leaf_atom);

      //      printf("quicktime_read_moov 0x%llx 0x%llx\n", quicktime_position(file), parent_atom->end);
    }while((quicktime_position(file) < parent_atom->end && file->decompressed_buffer==NULL)
	   || (quicktime_position(file) < file->decompressed_buffer_size && file->decompressed_buffer!=NULL));
    
  return 0;
}

void quicktime_write_moov(quicktime_t *file, quicktime_moov_t *moov)
{
	quicktime_atom_t atom;
	int i;
	long longest_duration = 0;
	long duration, timescale;
	int result;


// Try moov header immediately
	file->mdat.atom.end = quicktime_position(file);
	result = quicktime_atom_write_header(file, &atom, "moov");

// Disk full.  Rewind and try again
	if(result)
	{
		quicktime_set_position(file, file->mdat.atom.end - (longest)0x100000);
		file->mdat.atom.end = quicktime_position(file);
		quicktime_atom_write_header(file, &atom, "moov");
	}

/* get the duration from the longest track in the mvhd's timescale */
	for(i = 0; i < moov->total_tracks; i++)
	{
		quicktime_trak_fix_counts(file, moov->trak[i]);
		quicktime_trak_duration(moov->trak[i], &duration, &timescale);

		duration = (long)((float)duration / timescale * moov->mvhd.time_scale);

		if(duration > longest_duration)
		{
			longest_duration = duration;
		}
	}
	moov->mvhd.duration = longest_duration;
	moov->mvhd.selection_duration = longest_duration;

	quicktime_write_mvhd(file, &(moov->mvhd));
	quicktime_write_udta(file, &(moov->udta));
	for(i = 0; i < moov->total_tracks; i++)
	{
		quicktime_write_trak(file, moov->trak[i], moov->mvhd.time_scale);
	}
	/*quicktime_write_ctab(file, &(moov->ctab)); */

	quicktime_atom_write_footer(file, &atom);
// Rewind to end of mdat
	quicktime_set_position(file, file->mdat.atom.end);
}

void quicktime_update_durations(quicktime_moov_t *moov)
{
	
}

int quicktime_shift_offsets(quicktime_moov_t *moov, longest offset)
{
	int i;
	for(i = 0; i < moov->total_tracks; i++)
	{
		quicktime_trak_shift_offsets(moov->trak[i], offset);
	}
	return 0;
}
