#!/bin/sh

# These are the only variables that need to be changed:
opera_version='9.24-20071015.6'
debug_level='0' # From 0 to 2

# Everything else should look after itself:
str_progname="install.sh"
str_defaultprefix="/usr"
OPERADESTDIR=''

debug_msg()
{
    # This function outputs a stylish debug message if debug is enabled
    # $1 = debuglevel
    # $2 = message

    if test "$debug_level" -gt "$1"; then echo " $str_progname: debug: $2"; fi
}

error()
{
    # This function displays a error message and exits abnormally.
    # Arg_1 = Error Type
    # Arg_2 = Error Message

    debug_msg 0 'in error()'

    case $1 in

	missing)
	    echo " $str_progname: missing parameter(s)" >&2
	    echo
	    usage
	    ;;

	invalid)
	    echo " $str_progname: invalid parameter(s)" >&2
	    echo
	    usage
	    ;;

	abort)
	    echo " $str_progname: installation aborted by user" >&2
	    ;;

	conflict)
	    echo " $str_progname: conflicting parameters" >&2
	    echo
	    usage
	    ;;

	access)
	    echo " $str_progname: You do not have write access to $2, please change to appropriate user." >&2
	    ;;

        uname)
	    echo " $str_progname: OS cannot be determined without the program \"uname\", aborting..." >&2
	    ;;

	sed)
	    echo " $str_progname: Install script will not work without the program \"sed\" aborting..." >&2
	    ;;
	os)
	    echo " $str_progname: Unrecognizeed platform ($machine) and OS ($os), aborting..." >&2
	    echo " $str_progname: Please send output of uname -a to packager\@opera.com" >&2
	    ;;

	*)
	    debug_msg 0 "Error message $1 not defined, aborting..."
	    ;;
    esac
    exit 1
}

guess_os()
{
    # This function tries to guess which OS the install Machine is running.
    # other ideas are $OSTYPE, arch
    # Return = $os

    os=`uname -s` || error 'uname'
    case $os in
	FreeBSD|NetBSD|DragonFly) os=AnyBSD;;
	SunOS*) os=SunOS;;
    esac
    case $os in AnyBSD|OpenBSD) str_defaultprefix="/usr/local";; esac

    machine=`uname -m`
    case $machine in
	# Canonicalize architecture names with aliases:
	sun*|sparc*) machine=sparc;;
	amd64) machine=x86_64;;
	i[3456]86|i86pc) machine=x86;;
    esac
}

check_prefixes()
{
    # This function checks system enviroment for install prefix locations
    # and sets prefixed installation paths.

    export prefix exec_prefix wrapperdir docdir sharedir plugindir man_dir

    debug_msg 0 "in check_prefixes()"
    debug_msg 1 "enviroment = $prefix $exec_prefix $wrapperdir $docdir $sharedir $plugindir $man_dir"

    verbose="0"

    guess_os

    if test "$#" -ne 0; then parse_input "$@"; fi
}

set_prefix()
{
    # Arg 1 = new prefix
    prefix="$1"

    test "${prefix}" || prefix="${str_defaultprefix}"

    share_dir="${prefix}/share/opera"
    case "$os" in
	AnyBSD|OpenBSD)
	    exec_dir="${share_dir}/bin"
	    plugin_dir="${share_dir}/plugins"
	    man_dir="$prefix/man"
	    ;;
	*)
	    exec_dir="${prefix}/lib/opera/$opera_version"
	    plugin_dir="${prefix}/lib/opera/plugins"
	    man_dir="$prefix/share/man"
	    ;;
    esac
    doc_dir="${prefix}/share/doc/opera"
    wrapper_dir="${prefix}/bin"
}

set_destdir()
{
    test "$1" || return 1
    OPERADESTDIR="$1"
    exec_dir="${OPERADESTDIR}${exec_dir}"
    wrapper_dir="${OPERADESTDIR}${wrapper_dir}"
    doc_dir="${OPERADESTDIR}${doc_dir}"
    man_dir="${OPERADESTDIR}${man_dir}"
    share_dir="${OPERADESTDIR}${share_dir}"
    plugin_dir="${OPERADESTDIR}${plugin_dir}"
}

usage()
{
    # This function displays the help information.

    debug_msg 0 "in useage()"

    echo "Usage: ${str_progname} -s"
    echo "  or:  ${str_progname} -f DEST"
    test "$1" && echo "  or:  ${str_progname} -f EXEC_DEST WRAPPER_DEST DOC_DEST SHARE_DEST PLUGIN_DEST MAN_DEST"
    if test "$1";
	then echo "  or:  ${str_progname} --prefix=DEST [--exec_prefix=] [--wrapperdir=] [--docdir=] [--sharedir=] [--plugindir=] [--mandir=]"
	else echo "  or:  ${str_progname} --prefix=DEST"
    fi
    echo "  or:  ${str_progname} --exec_prefix=EXEC_DEST --wrapperdir=WRAPPER_DEST --docdir=DOC_DEST --mandir=MAN_DEST --sharedir=SHARE_DEST --plugindir=PLUGIN_DEST"
    test "$1" && echo
    test "$1" && echo ' or by enviroment variables:'
    test "$1" && echo "  or:  prefix=DEST ${str_progname}"
    test "$1" && echo "  or:  export prefix=DEST; ${str_progname}"
    echo
    echo 'Install Opera files to standard or user defined locations.'
    echo
    echo '  -i, --interactive            Interactive mode (default)'
    echo
    echo '  -s, --standard               install to standard locations'
    echo '  -f, --force                  install to user defined location(s)'
    echo
    test "$1" && echo '      $prefix'
    echo '      --prefix=                install all files to directory'
    echo
    test "$1" && echo '      $exec_prefix'
    echo '      --exec_prefix=           install Opera binaries to directory'
    test "$1" && echo '      $wrapperdir'
    echo '      --wrapperdir=            install Opera wrapper script to directory'
    test "$1" && echo '      $docdir'
    echo '      --docdir=                install Opera documentation to directory'
    test "$1" && echo '      $man_dir'
    echo '      --mandir=                install Opera manual page to directory'
    test "$1" && echo '      $sharedir'
    echo '      --sharedir=              install Opera shared files to directory'
    test "$1" && echo '      $plugindir'
    echo '      --plugindir=             install Opera plugin files to directory'
    echo
    echo '  -v, --verbose                output which files are copied'
    echo '  -vv                          output info on each executed command'
    echo
    echo '  -V, --version                output version information and exit'
    echo '  -h, --help                   display this help and exit'
    echo '      --help=long              show advanced functions'
    echo
    echo 'If you choose to do a standard locations install, files will be put into'
    echo '/usr/bin, /usr/share/doc/opera and /usr/share/opera.'
    echo
    echo 'However, if you choose to install to user defined locations, you must either'
    echo '  specify one directory (all files will be put in this directory), or'
    echo '  specify five directories (you may enter the same directory several times).'
}

parse_input()
{
    # This function parses trough command line parameters
    # and sets install flags and selected installation paths.

    debug_msg 0 "in parse_input()"
    debug_msg 1 "args = $*"


while test ! -z "$1"
do
case $1 in

    -h|--help) usage; exit 0 ;;

    --help=long) usage long; exit 0 ;;

    -V|--version) version; exit 0 ;;

    -v|--verbose)
	verbose='1'
	shift
	;;

    -vv)
	verbose='2'
	shift
	;;

    -i|--interactive)
	if test -z "$flag_mode"
	then
	    flag_mode='--interactive'
	    shift
	else error 'conflict'
	fi
	;;


    -s|--standard)
	if test -z "$flag_mode"
	then
	    flag_mode='--standard'
	    shift
	else error 'conflict'
	fi
	;;

    -f|--force)
	if test -z "$flag_mode"
	then
	    flag_mode='--force'
	    shift
	    if test -z "$1"
	    then error 'missing'
	    elif test -z "$2"
	    then
		str_paramprefix=`echo "$1" | sed -e 's/--prefix=//'`
		echo "Only one destination parameter found, all files will be installed to ${str_paramprefix}"
		echo 'Do you want to continue [y/n]?'
		read continue
		if test "${continue}" != 'y' && test "${continue}" != 'Y'
		then error 'abort'
		else shift
		fi
	    elif test -z "$6"
	    then error 'missing'
	    else
		str_paramexec="$1"
		str_paramwrapper="$2"
		str_paramdoc="$3"
		str_paramshare="$4"
		str_paramplugin="$5"
		str_paramman="$6"
		shift 6
	    fi
	else error 'conflict'
	fi
	;;

    DESTDIR=*)
	    param=`echo "$1" | sed -e 's/DESTDIR=//'`
	    shift
	    test "${param}" && str_paramdestdir="$param"
	    ;;

    --DESTDIR=*)
	    param=`echo "$1" | sed -e 's/--DESTDIR=//'`
	    shift
	    test "${param}" && str_paramdestdir="$param"
	    ;;

    --prefix=*)
	    param=`echo "$1" | sed -e 's/--prefix=//'`
	    shift
	    if test "${param}"
	    then
		if test "$flag_mode" = '--prefix=' ||  test -z "$flag_mode"
		then
		    flag_mode='--prefix='
		    str_paramprefix="${param}"
		else error 'conflict'
		fi
	    fi
	    ;;

    --exec_prefix=*)
	    param=`echo "$1" | sed -e 's/--exec_prefix=//'`
	    shift
	    if test "${param}"
	    then
		if test "$flag_mode" = '--prefix=' || test -z "$flag_mode"
		then
		    flag_mode='--prefix='
		    str_paramexec="${param}"
		else error 'conflict'
		fi
	    fi
	    ;;

    --wrapperdir=*)
	    param=`echo "$1" | sed -e 's/--wrapperdir=//'`
	    shift
	    if test "${param}"
	    then
		if test "$flag_mode" = '--prefix=' || test -z "$flag_mode"
		then
		    flag_mode='--prefix='
		    str_paramwrapper="${param}"
		else error 'conflict'
		fi
	    fi
	    ;;

    --mandir=*)
	    param=`echo "$1" | sed -e 's/--mandir=//'`
	    shift
	    if test "${param}"
	    then
		if test "$flag_mode" = '--prefix=' || test -z "$flag_mode"
		then
		    flag_mode='--prefix='
		    str_paramman="${param}"
		else error 'conflict'
		fi
	    fi
	    ;;

    --docdir=*)
	    param=`echo "$1" | sed -e 's/--docdir=//'`
	    shift
	    if test "${param}"
	    then
		if test "$flag_mode" = '--prefix=' || test -z "$flag_mode"
		then
		    flag_mode='--prefix='
		    str_paramdoc="${param}"
		else error 'conflict'
		fi
	    fi
	    ;;

    --sharedir=*)
	    param=`echo "$1" | sed -e 's/--sharedir=//'`
	    shift
	    if test "${param}"
	    then
		if test "$flag_mode" = '--prefix=' || test -z "$flag_mode"
		then
		    flag_mode='--prefix='
		    str_paramshare="${param}"
		else error 'conflict'
		fi
	    fi
	    ;;

    --plugindir=*)
	    param=`echo "$1" | sed -e 's/--plugindir=//'`
	    shift
	    if test "${param}"
	    then
		if test "$flag_mode" = '--prefix=' || test -z "$flag_mode"
		then
		    flag_mode='--prefix='
		    str_paramplugin="${param}"
		else error 'conflict'
		fi
	    fi
	    ;;

     *) error 'invalid';;
esac
done
    debug_msg 1 "flag_mode = $flag_mode"
}

set_os_spesific()
{
    # This function determines which commands and parameters will work on a given platform.

    case "$os" in
	SunOS)
	    cpf=''    # SunOS cp overwrites by default (no -f force option)
	    cpv=''    # SunOS cp (no -v verbose option)
	    chmodv='' # SunOS chmod (no -v verbose option)
	    mkdirv='' # SunOS mkdir (no -v verbose option)
	    mvv=''    # SunOS mv (no -v verbose option)
	;;

	AnyBSD)
		cpf='-f'
		if test "$verbose" -gt '1'
		then
		    chmodv='-v'
		    mkdirv='-v'
		fi
		if test "$verbose" -gt '0'
		then
		    mvv='-v'
		    cpv='-v'
		fi
	;;

	OpenBSD)
	    cpf='-f'
	    mkdirv=''
	    chmodv=''
	    cpv=''
	    mvv=''
	;;

	Linux)
		cpf='-f'
		if test "$verbose" -gt "1"
		then
		    chmodv='-v'
		    mkdirv='--verbose'
		else # in case of csh
		    chmodv=''
		    mkdirv=''
		fi
		if test "$verbose" -gt "0"
		then
		    mvv='-v'
		    cpv='-v'
		else # in case of csh
		    mvv=''
		    cpv=''
		fi
	;;

	*) error 'os';;
    esac
    debug_msg 1 "cpf = $cpf"
    debug_msg 1 "cpv = $cpv"
    debug_msg 1 "chmodv = $chmodv"
    debug_msg 1 "mkdirv = $mkdirv"

    # Common
    mkdirp='-p'
    cpp='-p'
    cpR='-R'
    lns='-s'
}

version()
{
    # This function displays the version information.

    debug_msg 0 'in version()'

    echo "${str_progname} (Opera Software ASA) 3.98"
    echo 'Maintained by Opera packaging team <packager@opera.com>'
    echo
    echo 'Copyright (C) 2001-2006 Opera Software ASA.'
    echo 'This is free software; There is NO warranty; not even'
    echo 'for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.'
}

con_firm()
{
    str_options="[ y,n | yes,no ]"
    test "$2" && str_options="[ y,n,c | yes,no,cancel ]"
    echo "$1 ${str_options} ?"
    while true
    do
      read str_answer
      case "${str_answer}" in

	    ''|[yY]|[yY][eE][sS])
		return 0
		;;

	    [nN]|[nN][oO])
		return 1
		;;

	    [aAcCqQ]|[aA][bB][oO][rR][tT]|[cC][aA][nN][cC][eE][lL]|[qQ][uU][iI][tT])
		if test "$2"
		then error "abort"
		else echo 'Invalid answer, try again:'
		fi
		;;

	    *)
		echo 'Invalid answer, try again:'
		;;

      esac
    done
}

ver_bose()
{
    if test "${verbose}" = "0"; then return 1; fi
}

select_installation_type()
{
    # This function parses installation flags and sets final installation paths.

    debug_msg 0 'in select_installation_type()'
    debug_msg 1 "flag_mode = ${flag_mode}"

    if test -z "${flag_mode}"
    then
	if test "${str_paramdestdir}"
	then flag_mode='--prefix='
	else flag_mode='--interactive'
	fi
    fi

    set_prefix "${prefix}"
    test "${exec_prefix}" && exec_dir="${exec_prefix}"
    test "${wrapperdir}"  && wrapper_dir="${wrapperdir}"
    test "${docdir}"      && doc_dir="${docdir}"
    test "${sharedir}"    && share_dir="${sharedir}"
    test "${plugindir}"   && plugin_dir="${plugindir}"

    case $flag_mode in

	--interactive)
	    if ver_bose; then echo 'Running interactive installation...'; fi
	    interactive_install
	    ;;

	--standard)
	    if ver_bose; then echo 'Running standard installation...'; fi
	    set_prefix "${str_defaultprefix}"
	    set_destdir "${str_paramdestdir}"
	    ;;

	--force)
	    if ver_bose; then echo 'Running forced installation...'; fi
	    set_prefix "${str_paramprefix}"
	    set_destdir "${str_paramdestdir}"
	    ;;

	--prefix=)
	    if ver_bose; then echo 'Running prefixed installation...'; fi
	    set_prefix "${str_paramprefix}"
	    test "${str_paramexec}"     && exec_dir="${str_paramexec}"
	    test "${str_paramwrapper}"  && wrapper_dir="${str_paramwrapper}"
	    test "${str_paramdoc}"      && doc_dir="${str_paramdoc}"
	    test "${str_paramman}"      && man_dir="${str_paramman}"
	    test "${str_paramshare}"    && share_dir="${str_paramshare}"
	    test "${str_paramplugin}"   && plugin_dir="${str_paramplugin}"
	    set_destdir "${str_paramdestdir}"
	    ;;
    esac
}

can_write_to()
{
    # This function checks write access on paths
    # Returns (0 true writeable) or (1 false unwriteable)

    # Arg1 = directory to test

    debug_msg 0 'in can_write_to()'
    for arg # implicitly in "$@"
    do
	debug_msg 1 " $arg"
	test_folder="$arg"

	# If directory doesn't exist go up once and test again
	while test ! -d "${test_folder}"
	do
	    temp_folder=`echo ${test_folder} | sed -e 's:/[^/]*$::'`
	    # If nothing removed, avoid infinite loop
	    if test "${temp_folder}" = "${test_folder}"; then break; fi
	    test_folder="${temp_folder}"
	    debug_msg 1 " ${test_folder}"
	done

	if test -w "${test_folder}"
	then
	    debug_msg 1 "User \"${USERNAME}\" has write access to [${test_folder}]"
	else
	    debug_msg 1 "User \"${USERNAME}\" can't write to [${test_folder}]"
	    return 1
	fi
    done
}

cannot_write_to()
{
    # Needed for the while loop in prompt_path because Solaris sh doesn't like ! as negation.
    if can_write_to "$@"
    then return 1
    else return 0
    fi
}

confirm_paths()
{
    while true
    do
      echo
      echo "Files will be installed as follows:"
      echo "-----------------------------------------------------------"
      echo " Wrapper Script : ${wrapper_dir}"
      echo " Binaries       : ${exec_dir}"
      echo " Plugins        : ${plugin_dir}"
      echo " Shared files   : ${share_dir}"
      echo " Documentation  : ${doc_dir}"
      echo " Manual page    : ${man_dir}"
      echo "-----------------------------------------------------------"
      if con_firm "Is this correct" "cancel"
      then return 0
      else change_paths
      fi
    done
}

csh_set()
{
    # This funtion enables csh syntax for the set command.
    # Read more at http://zsh.sunsite.dk/Intro/intro_12.html
    eval "$1$2$3"
}

chop()
{
    str_toremove="$1"
    str_varname="$2"
    eval str_from=\$"${str_varname}"
    str_removed=""

    while test "$str_toremove"
    do
	str_toremove=`echo "${str_toremove}"|sed -e 's/.//'`
	str_removed="${str_removed}`echo "${str_from}"|sed -e 's/\(.\).*/\1/'`"
	str_from=`echo "${str_from}"|sed -e 's/.//'`
    done

    test "${str_removed}" = "$1" && eval ${str_varname}="${str_from}"
}

prompt_path()
{
    # This function suggests a path and test new if user changes
    # Arg 1 = type of files
    # Arg 2 = path to test
    # Arg 3 = variable to modify

    if test "$1" = 'prefix'
    then echo "Enter installation prefix [${prefix}]"
    else echo "Enter path for the Opera $1 [$2]"
    fi

    read a_path

    test "${a_path}" || a_path="$2"
    while cannot_write_to "${a_path}"
    do
	echo "User \"${USERNAME}\" does not have write access to [${a_path}]."
	if test "$1" = 'prefix'
	then echo "Enter installation prefix [$2]"
	else echo "Enter path for the Opera $1 [$2]"
	fi
	read a_path
	if test -z "${a_path}"; then return; fi
    done
    if test "$1" = 'prefix'
    then set_prefix "${a_path}"
    else csh_set  "$3"="${a_path}"
    fi
}

change_paths()
{
    prompt_path "prefix" "${prefix}" 'prefix'
    prompt_path "wrapper script" "${wrapper_dir}" "wrapper_dir"
    prompt_path "binaries" "${exec_dir}" "exec_dir"
    prompt_path "plugins" "${plugin_dir}" "plugin_dir"
    prompt_path "shared files" "${share_dir}" "share_dir"
    prompt_path "documentation" "${doc_dir}" "doc_dir"
    prompt_path "manual page" "${man_dir}" "man_dir"
}

interactive_install()
{
    # This function starts of checking if you have access to
    # prefix location
    # default locations (/usr)
    # ${HOME}

    # Test current prefixed locations.
    if test -n "${prefix}"
    then
	if can_write_to "${exec_dir}" "${wrapper_dir}" "${doc_dir}" "${share_dir}" "${plugin_dir}" "${man_dir}"
	then
	  # Prefix location success -> Recommend prefix install
	    if ver_bose; then echo "User \"${USERNAME}\" has write access to current prefixed locations."; fi
	    confirm_paths
	    return
	else
	  # Prefix location failed -> Try default
	    if ver_bose
	    then
		echo "User \"${USERNAME}\" does not have write access to current prefixed locations."
		echo "Trying default locations, prefix [$str_defaultprefix]..."
	    fi
	fi
    fi # prefix

    set_prefix "${str_defaultprefix}"

    if can_write_to "${exec_dir}" "${wrapper_dir}" "${doc_dir}" "${share_dir}" "${plugin_dir}" "${man_dir}"
    then
	    # Default location success -> Recommend standard install
	if ver_bose; then echo "User \"${USERNAME}\" has write access to default locations. Prefix [${prefix}]"; fi
    else
	    # Default location failed -> Try other prefixes
	if ver_bose; then echo "User \"${USERNAME}\" does not have write access to default locations. Prefix [${prefix}]"; fi
	if test -z "${HOME}"
	then
	    echo " $str_progname: Enviroment variable \"${HOME}\" not set, user \"${USERNAME}\" has no homefolder?" >&2
	    echo "Not trying locations with prefix [~/.]"
	else
	    if ver_bose; then echo "Trying prefix [${HOME}]..."; fi
	    set_prefix "${HOME}"

	    if can_write_to "${exec_dir}" "${wrapper_dir}" "${doc_dir}" "${share_dir}" "${plugin_dir}" "${man_dir}"
	    then
		# Prefix [~/.] success -> Recommend home install
		if ver_bose; then echo "User \"${USERNAME}\" has write access to locations with prefix [${prefix}]."; fi
	    fi
	fi
    fi
    confirm_paths
}


backup()
{
    # This function makes a backup of a file or directory to use in future rollback
    # This function is in development
    # 1 = file or directory
    # 2 = location in backup
    # 3 = folder to create

    debug_msg 0 "in backup()"

    rollback_dir="${share_dir}/uninst/rollback_${opera_version}"
    if test -f "$1" -o -d "$1"
    then
	if can_write_to "${rollback_dir}"
	then
	    if test ! -d ${rollback_dir}; then mkdir $mkdirv $mkdirp ${rollback_dir}; fi
	    if test "$3"; then
		if test ! -d ${rollback_dir}/$3; then mkdir $mkdirv $mkdirp ${rollback_dir}/$3; fi
		location="$3/$2"
	    else
    		location="$2"
	    fi
	    if mv $mvv $mvf $1 ${rollback_dir}/${location} && ver_bose
	    then
		echo
		echo "Backup of:"
		echo " [$1] can be found at"
		echo " [${rollback_dir}/${location}]"
	    fi # ver_bose
	else echo "Can't make backup of [$1] for future rollback." >&2
	fi
    else debug_msg 1 "[$1] does not exist"
    fi
}

generate_wrapper()
{

    # This function generates the wrapper script with correct opera path

    debug_msg 0 "in generate_wrapper()"

    case "${machine}:${os}" in
	x86:Linux|x86_64:Linux|x86:AnyBSD|x86_64:AnyBSD|x86:OpenBSD)
	    wrapper_ibmjava="
	    IBMJava2-142/jre \\
	    IBMJava2-141/jre \\
	    IBMJava2-14/jre \\
	    IBMJava2-131/jre \\"
	    wrapper_sunjava_machine="i386"
	;;

	sparc:SunOS)
	    wrapper_sunjava_machine="sparc"
	;;

	x86:SunOS)
	    wrapper_sunjava_machine="i386"
	;;

	sparc:Linux)
	    wrapper_sunjava_machine="sparc"
	;;

	mips*:Linux)
	;;

	ppc:Linux|ppc64:Linux)
	    wrapper_ibmjava="
	    IBMJava2-ppc-14/jre \\
	    IBMJava2-ppc-131/jre \\"
	;;


	*:*)
		error 'os'
	;;
    esac
    wrapper_file="${wrapper_dir}/opera"

    wrapper_contain="#!/bin/sh

# Required for Session Management
case \$0 in /*) OPERA_SCRIPT_PATH=\$0;; *) OPERA_SCRIPT_PATH=`/bin/pwd`/\$0;; esac
export OPERA_SCRIPT_PATH

# Location of the Opera binaries
OPERA_BINARYDIR=${str_localdirexec}
export OPERA_BINARYDIR

# Parse commandline parameters
toset=
_QTSTYLE_FROM_CMDLINE_='NO'
for arg
do
  if test \"\$toset\"
  then
      eval \$toset=\"\$arg\"
      export \$toset
      toset=
  else
      case \$arg in
          --binarydir*|-binarydir*) toset=OPERA_BINARYDIR ;;
          --personaldir*|-personaldir*|--pd*|-pd*) toset=OPERA_PERSONALDIR ;;
          -style)
              _QTSTYLE_FROM_CMDLINE_='YES'
              ;;
      esac
  fi
done
test -z \"\$toset\" || die Missing parameter for \$arg

test \"\${_QTSTYLE_FROM_CMDLINE_}\" = 'NO' && test -f \"\$HOME/.qt/qtrc\" && _OPERA_QTSTYLE=\`sed -n 's/^style=//p' \$HOME/.qt/qtrc\` && test \"\$_OPERA_QTSTYLE\" && set -- \"\$@\" -style \"\$_OPERA_QTSTYLE\"


# Add ending path divider \"/\" if missing, so execution won't fail
if test \`echo \${OPERA_BINARYDIR} | sed -e 's/.*[^\/]$/NO/'\` = 'NO'; then
OPERA_BINARYDIR=\"\${OPERA_BINARYDIR}/\"
fi
if test ! -f \${OPERA_BINARYDIR}opera; then
    echo \"The Opera binary is not located at \\\"\${OPERA_BINARYDIR}opera\\\".\"
    echo \"Please modify the wrapper script at \\\"\${0}\\\".\"
    exit 1
elif test ! -x \"\${OPERA_BINARYDIR}opera\"; then
    echo \"You do not have execute rights on \\\"\${OPERA_BINARYDIR}opera\\\", please ask the sysadmin to chmod +x it.\"
    exit 1
fi

# Opera enviroment
if test \"\${OPERA_DIR}\" = '' ; then
  if test -d ${str_localdirshare} ; then
    OPERA_DIR=${str_localdirshare}
  else
    echo \"OPERA_DIR unset and not found at expected location (${str_localdirshare})\"
    exit 1
  fi
fi

OPERA_LD_PRELOAD=\"\${LD_PRELOAD}\"
export OPERA_LD_PRELOAD

# Native Java enviroment
if test -f \"\${OPERA_PERSONALDIR}/javapath.txt\"; then
    INIJAVA=\`cat \${OPERA_PERSONALDIR}/javapath.txt\`
    if test -f \"\${INIJAVA}/libjava.so\"; then OPERA_JAVA_DIR=\"\${INIJAVA}\"; fi
fi

if test ! \"\${OPERA_JAVA_DIR}\"; then
    if test -f \"\${HOME}/.opera/javapath.txt\"; then
        INIJAVA=\`cat \${HOME}/.opera/javapath.txt\`
        if test -f \"\${INIJAVA}/libjava.so\"; then OPERA_JAVA_DIR=\"\${INIJAVA}\"; fi
    fi
fi

if test ! \"\${OPERA_JAVA_DIR}\"; then
    if test -f \"\${JAVA_HOME}/lib/${wrapper_sunjava_machine}/libjava.so\"; then OPERA_JAVA_DIR=\"\${JAVA_HOME}/lib/${wrapper_sunjava_machine}\"; fi
fi

if test ! \"\${OPERA_JAVA_DIR}\"; then
    if test -f \"\${JAVA_HOME}/jre/lib/${wrapper_sunjava_machine}/libjava.so\"; then OPERA_JAVA_DIR=\"\${JAVA_HOME}/jre/lib/${wrapper_sunjava_machine}\"; fi
fi

if test ! \"\${OPERA_JAVA_DIR}\"; then

    PREFIXES=\"
	/usr
	/usr/java
	/usr/lib
	/usr/local
	/opt\"

    for SUNJAVA in \\
	java-1.5.0-sun-1.5.0.06 \\
	java-1.5.0-sun-1.5.0.06/jre \\
	java-1.5.0-sun-1.5.0.05 \\
	java-1.5.0-sun-1.5.0.05/jre \\
	java-1.5.0-sun-1.5.0.04 \\
	java-1.5.0-sun-1.5.0.04/jre \\
	jre1.5.0_06 \\
	jdk1.5.0_06/jre \\
	jre1.5.0_05 \\
	jdk1.5.0_05/jre \\
	jre1.5.0_04 \\
	jdk1.5.0_04/jre \\
	jre1.5.0_03 \\
	jdk1.5.0_03/jre \\
	jre1.5.0_02 \\
	jdk1.5.0_02/jre \\
	jre1.5.0_01 \\
	jdk1.5.0_01/jre \\
	j2re1.4.2_06 \\
	j2sdk1.4.2_06/jre \\
	j2re1.4.2_04 \\
	j2sdk1.4.2_04/jre \\
	j2re1.4.2_03 \\
	j2sdk1.4.2_03/jre \\
	j2re1.4.2_02 \\
	j2sdk1.4.2_02/jre \\
	j2re1.4.2_01 \\
	j2sdk1.4.2_01/jre \\
	j2re1.4.2 \\
	j2sdk1.4.2/jre \\
	j2re1.4.1_01 \\
	j2re1.4.1 \\
	SUNJava2-1.4.1 \\
	BlackdownJava2-1.4.1/jre \\
	j2re1.4.0_01 \\
	j2sdk1.4.0_01/jre \\
	j2re1.4.0 \\
	jre1.4.0 \\
	j2se/1.4/jre \\
	j2se/1.3/jre \\
	j2se/jre \\
	jre1.3.1_15 \\
	jre1.3.1_04 \\
	jre1.3.1_02 \\
	jre1.3.1_01 \\
	j2re1.3.1 \\
	jre1.3.1 \\
	j2re1.3 \\
	j2se/1.3/jre \\
	SunJava2-1.3/jre \\
	java2re \\
	jdk1.2.2/jre \\
	jdk1.2/jre \\
	jre \\
	java \\
	; do
	for PREFIX in \${PREFIXES}; do
	    if test -f \"\${PREFIX}/\${SUNJAVA}/lib/${wrapper_sunjava_machine}/libjava.so\"; then OPERA_JAVA_DIR=\"\${PREFIX}/\${SUNJAVA}/lib/${wrapper_sunjava_machine}\" && break; fi
	done
	if test \"\${OPERA_JAVA_DIR}\"; then break; fi
    done"

if test -n "${wrapper_ibmjava}"; then
    wrapper_contain="${wrapper_contain}
    if test ! \"\${OPERA_JAVA_DIR}\"; then
	for IBMJAVA in \\${wrapper_ibmjava}
	    ; do
	    for PREFIX in \${PREFIXES}; do
		if test -f \"\${PREFIX}/\${IBMJAVA}/bin/libjava.so\"; then OPERA_JAVA_DIR=\"\${PREFIX}/\${IBMJAVA}/bin\" && break; fi
	    done
	if test \"\${OPERA_JAVA_DIR}\"; then break; fi
	done
    fi"
fi

    wrapper_contain="${wrapper_contain}
fi

if test \"\${OPERA_JAVA_DIR}\"; then
    LD_LIBRARY_PATH=\"\${OPERA_JAVA_DIR}:\${OPERA_JAVA_DIR}/native_threads:\${OPERA_JAVA_DIR}/client:\${OPERA_JAVA_DIR}/classic\${LD_LIBRARY_PATH:+:}\${LD_LIBRARY_PATH}\"
    export LD_LIBRARY_PATH OPERA_JAVA_DIR
fi

# Workarounds for the \"preloaded libXt\" problem.
LD_PRELOAD=\"libjvm.so:libawt.so:\${OPERA_LD_PRELOAD}\"
export LD_PRELOAD
# To disable the first workaround, comment the next line.
JAVA_WORKAROUND=\`\${OPERA_BINARYDIR}/works 2>/dev/null\`

if test \"\${JAVA_WORKAROUND}\" = 'works'; then
    OPERA_FORCE_JAVA_ENABLED=\"1\"
else
    LD_PRELOAD=\"libjava.so:\${LD_PRELOAD}\"
    # To disable the second workaround, comment the next line.
    IBMJAVA131_WORKAROUND=\`\${OPERA_BINARYDIR}/works 2>/dev/null\`
    if test \"\${IBMJAVA131_WORKAROUND}\" = 'works'; then
	OPERA_FORCE_JAVA_ENABLED=\"1\"
    else
	LD_PRELOAD=\"\${OPERA_LD_PRELOAD}\"
    fi
fi
export LD_PRELOAD OPERA_FORCE_JAVA_ENABLED

# Acrobat Reader
for BINDIR in \\
    /usr/local/Acrobat[45]/bin \\
    /usr/lib/Acrobat[45]/bin \\
    /usr/X11R6/lib/Acrobat[45]/bin \\
    /opt/Acrobat[45]/bin \\
    /usr/Acrobat[45]/bin \\
    ; do
    if test -d \${BINDIR} ; then PATH=\${PATH}:\${BINDIR}; fi
done

# Exporting the enviroment
export OPERA_DIR PATH

LD_LIBRARY_PATH=\"\${OPERA_BINARYDIR}\${LD_LIBRARY_PATH:+:}\${LD_LIBRARY_PATH}\"
export LD_LIBRARY_PATH

# Spellchecker needs to find libaspell.so.15
for LIBASPELL_DIR in \\
    /usr/local/lib \\
    /opkg/lib \\
; do
    if test -f \"\${LIBASPELL_DIR}/libaspell.so.15\"; then
        LD_LIBRARY_PATH=\"\${LD_LIBRARY_PATH}:\${LIBASPELL_DIR}\"
    fi
done"

case "${os}" in
	AnyBSD|OpenBSD)
wrapper_contain="${wrapper_contain}

# Make sure the compat libraries are found
test -d /usr/local/lib/compat/ && LD_LIBRARY_PATH=\"\${LD_LIBRARY_PATH}:/usr/local/lib/compat/\"
"
	;;
    SunOS)
wrapper_contain="${wrapper_contain}

# Make sure libstdc++.so.6 is found on Solaris 10.
test -d /usr/sfw/lib/ && LD_LIBRARY_PATH=\"\${LD_LIBRARY_PATH}:/usr/sfw/lib/\"
"
	;;
esac

wrapper_contain="${wrapper_contain}

# Running Opera
exec \"\${OPERA_BINARYDIR}opera\" \"\$@\"
"

echo "${wrapper_contain}" > ${wrapper_file}
}

generate_opera_policy()
{
    policy_contain="
// Standard extensions get all permissions by default

grant codeBase \"file:\${java.home}/lib/ext/*\" {
	permission java.security.AllPermission;
};

// Opera package classes get all permissions
grant codebase \"file://${str_localdirshare}/java//opera.jar\" {
	permission java.security.AllPermission;
};

// default permissions granted to all domains

grant {
	// Allows any thread to stop itself using the java.lang.Thread.stop()
	// method that takes no argument.
	// Note that this permission is granted by default only to remain
	// backwards compatible.
	// It is strongly recommended that you either remove this permission
	// from this policy file or further restrict it to code sources
	// that you specify, because Thread.stop() is potentially unsafe.
	// See \"http://java.sun.com/notes\" for more information.
	permission java.lang.RuntimePermission \"stopThread\";

	// allows anyone to listen on un-privileged ports
	permission java.net.SocketPermission \"localhost:1024-\", \"listen\";

	// \"standard\" properies that can be read by anyone

	permission java.util.PropertyPermission \"java.version\", \"read\";
	permission java.util.PropertyPermission \"java.vendor\", \"read\";
	permission java.util.PropertyPermission \"java.vendor.url\", \"read\";
	permission java.util.PropertyPermission \"java.class.version\", \"read\";
	permission java.util.PropertyPermission \"os.name\", \"read\";
	permission java.util.PropertyPermission \"os.version\", \"read\";
	permission java.util.PropertyPermission \"os.arch\", \"read\";
	permission java.util.PropertyPermission \"file.separator\", \"read\";
	permission java.util.PropertyPermission \"path.separator\", \"read\";
	permission java.util.PropertyPermission \"line.separator\", \"read\";

	permission java.util.PropertyPermission \"java.specification.version\", \"read\";
	permission java.util.PropertyPermission \"java.specification.vendor\", \"read\";
	permission java.util.PropertyPermission \"java.specification.name\", \"read\";

	permission java.util.PropertyPermission \"java.vm.specification.version\", \"read\";
	permission java.util.PropertyPermission \"java.vm.specification.vendor\", \"read\";
	permission java.util.PropertyPermission \"java.vm.specification.name\", \"read\";
	permission java.util.PropertyPermission \"java.vm.version\", \"read\";
	permission java.util.PropertyPermission \"java.vm.vendor\", \"read\";
	permission java.util.PropertyPermission \"java.vm.name\", \"read\";
};
"
    echo "${policy_contain}" > "${share_dir}/java/opera.policy"
}

run_install()
{
    # This function copies files to selected locations and

    debug_msg 0 "in run_install()"

    can_write_to ${exec_dir} || error access ${exec_dir}
    can_write_to ${wrapper_dir} || error access ${wrapper_dir}
    can_write_to ${doc_dir} || error access ${doc_dir}
    can_write_to ${man_dir} || error access ${man_dir}
    can_write_to ${share_dir} || error access ${share_dir}
    can_write_to ${plugin_dir} || error access ${plugin_dir}

    str_localdirexec="${exec_dir}"
    str_localdirshare="${share_dir}"
    str_localdirplugin="${plugin_dir}"

    chop "${OPERADESTDIR}" "str_localdirexec"
    chop "${OPERADESTDIR}" "str_localdirshare"
    chop "${OPERADESTDIR}" "str_localdirplugin"

    backup ${wrapper_dir}/opera opera

    # Executable
    debug_msg 1 "Executable"
    mkdir $mkdirv $mkdirp $exec_dir/
    chmod $chmodv 755 $exec_dir

    rm -f $exec_dir/opera
    cp $cpv $cpf bin/opera $exec_dir/opera
    chmod $chmodv 755 $exec_dir/opera

    # spellcheck
    if test -f lib/spellcheck.so
    then
	rm -f $exec_dir/spellcheck.so
	cp $cpv $cpf lib/spellcheck.so $exec_dir/spellcheck.so
	chmod $chmodv 755 $exec_dir/spellcheck.so
    fi

    # dymanic executeable works
    if test -f bin/works
    then
	rm -f $exec_dir/works
	cp $cpv $cpf bin/works $exec_dir/works
	chmod $chmodv 755 $exec_dir/works
    fi

    # Wrapper
    debug_msg 1 "Wrapper"
    mkdir $mkdirv $mkdirp $wrapper_dir/
    chmod $chmodv 755 $wrapper_dir

    #cp $cpv $cpf wrapper.sh $wrapper_dir/opera
    generate_wrapper
    chmod $chmodv 755 $wrapper_dir/opera

    # Manual page
    debug_msg 1 "Manual page"

    mkdir $mkdirv $mkdirp ${man_dir}
    chmod $chmodv 755 ${man_dir}
    mkdir $mkdirv $mkdirp ${man_dir}/man1
    chmod $chmodv 755 ${man_dir}/man1
    cp $cpv $cpf man/opera.1 ${man_dir}/man1

    # Documentation
    debug_msg 1 "Documentation"

    mkdir $mkdirv $mkdirp $doc_dir/
    chmod $chmodv 755 $doc_dir
    cp $cpv $cpf LICENSE $doc_dir

    # Shared files
    debug_msg 1 "Shared files"

    mkdir $mkdirv $mkdirp $share_dir/config/
    chmod $chmodv 755 $share_dir/config
    cp $cpv $cpf $cpR config/* $share_dir/config/

    mkdir $mkdirv $mkdirp $share_dir/images/
    chmod $chmodv 755 $share_dir/images
    cp $cpv $cpf $cpR images/* $share_dir/images/

    if test -d skin
    then
	mkdir $mkdirv $mkdirp $share_dir/skin/
	chmod $chmodv 755 $share_dir/skin
	cp $cpv $cpf $cpR skin/* $share_dir/skin/
    fi

    if test -d ini
    then
	mkdir $mkdirv $mkdirp $share_dir/ini/
	chmod $chmodv 755 $share_dir/ini
	cp $cpv $cpf $cpR ini/* $share_dir/ini/
	if test -f $share_dir/ini/pluginpath.ini
	then (
	    echo
	    echo '; locally installed:'
	    echo "${str_localdirplugin}=1"
	    ) >> $share_dir/ini/pluginpath.ini
	fi
    fi

    mkdir $mkdirv $mkdirp $share_dir/locale/
    chmod $chmodv 755 $share_dir/locale

    if ls locale/* >/dev/null 2>/dev/null
    then cp $cpv $cpf $cpR locale/* $share_dir/locale/
    fi

    mkdir $mkdirv $mkdirp $share_dir/styles/
    chmod $chmodv 755 $share_dir/styles
    cp $cpv $cpf $cpR styles/* $share_dir/styles/

    mkdir $mkdirv $mkdirp $share_dir/styles/user/
    chmod $chmodv 755 $share_dir/styles/user
    cp $cpv $cpf $cpR styles/user/* $share_dir/styles/user/

    cp $cpv $cpf html40_entities.dtd $share_dir/
    cp $cpv $cpf chartables*.bin $share_dir/
    cp $cpv $cpf opera6.adr $share_dir/
    cp $cpv $cpf *.ini $share_dir/ >/dev/null 2>/dev/null
    if test -f lngcode.txt
    then cp $cpv $cpf lngcode.txt $share_dir/
    fi

    # SVG fonts
    if test -f svg-mo.dat
    then cp $cpv $cpf svg-*.dat $share_dir/
    fi

    # Java
    mkdir $mkdirv $mkdirp $share_dir/java/
    chmod $chmodv 755 $share_dir/java

    cp $cpv $cpf java/opera.jar $share_dir/java/
    if test -f java/lc.jar; then cp $cpv $cpf java/lc.jar $share_dir/java/; fi
    generate_opera_policy

    # Plug-in files
    debug_msg 1 "Plug-in files"

    plugin_support='no'

    mkdir $mkdirv $mkdirp $plugin_dir/
    chmod $chmodv 755 $plugin_dir

    if test -f plugins/operapluginwrapper
    then
	rm -f $plugin_dir/operapluginwrapper
	cp $cpv $cpf plugins/operapluginwrapper $plugin_dir/
	chmod $chmodv 755 $plugin_dir/operapluginwrapper
	plugin_support='yes'
    fi

    if test -f plugins/libnpp.so
    then
	rm -f $plugin_dir/libnpp.so
	cp $cpv $cpf plugins/libnpp.so $plugin_dir/
	chmod $chmodv 755 $plugin_dir/libnpp.so
    else plugin_support='no'
    fi

    if test -f lib/missingsyms.so
    then
	rm -f $exec_dir/missingsyms.so
	cp $cpv $cpf lib/missingsyms.so $exec_dir/
	chmod $chmodv 755 $exec_dir/missingsyms.so
    else echo "This package does not support Netscape Realplayer Plug-in."
    fi

    if test -f plugins/operaplugincleaner
    then
	rm -f $plugin_dir/operaplugincleaner
	cp $cpv $cpf plugins/operaplugincleaner $plugin_dir/
	chmod $chmodv 755 $plugin_dir/operaplugincleaner
    fi

    if test "$plugin_support" = 'no'
    then echo "This package does not contain support for Netscape Plug-ins."
    fi

    # Voice
    if test -f plugins/vxmplugin.so.0.1
    then
	rm -f $exec_dir/vxmlplugin.so.0.1
	cp $cpv $cpf plugins/vxmlplugin.so.0.1 $exec_dir/
	chmod $chmodv 755 $exec_dir/vxmlplugin.so.0.1
    fi

    if test -z "${OPERADESTDIR}"
    then
	# System wide configuration files
	config_dir="/etc"
	if can_write_to "$config_dir"
	then
	    echo
	    echo "System wide configuration files:"
	    echo "  $config_dir/opera6rc"
	    echo "  $config_dir/opera6rc.fixed"
	    echo " would be ignored if installed with the prefix \"$prefix\"."
	    if con_firm "Do you want to install them in $config_dir"
	    then
		backup $config_dir/opera6rc opera6rc config
		backup $config_dir/opera6rc.fixed opera6rc.fixed config
		cp $cpv $cpf config/opera6rc $config_dir
		cp $cpv $cpf config/opera6rc.fixed $config_dir
	    fi
	else
	    echo
	    echo "User \"${USERNAME}\" does not have write access to $config_dir"
	    echo " System wide configuration files:"
	    echo "  $config_dir/opera6rc"
	    echo "  $config_dir/opera6rc.fixed"
	    echo " were not installed."
	fi

	# Shorcuts and Icons
	bool_icons=1 # install icons by default

	if test "${flag_mode}" = "--force" -o "${flag_mode}" = "--prefix="
	then
	    echo
	    echo "Shortcut icons will be ignored if installed with the prefix \"$prefix\"."
	    con_firm "Do you want to (try to) install them in default locations" || bool_icons=0
	fi

	if test "${bool_icons}" -ne 0
	then xdg
	fi

    fi # OPERADESTDIR

# Add uninstallation here

    if ver_bose
    then
	echo
	echo 'Installation completed. Enjoy !'
	if test "$flag_mode" = "--prefix="
	then
	    echo
	    echo "You've completed a prefixed installation."
	    echo
	else
	    if test "$flag_mode" = "--force"
	    then
		echo
		echo "You've completed a forced installation."
		echo
	    fi
	fi
	echo "Be sure to include $wrapper_dir in your PATH or invoke it as"
	echo "$wrapper_dir/opera or ./opera; and include $man_path in your MANPATH"
	echo "to make 'man opera' work, or invoke 'man -M $man_path opera'"
    fi # ver_bose
}

generate_wmconfig()
{
    # arg1 = location

    wmconfig_file="$1/opera"

    wmconfig_contain='opera name "Opera"
opera description "Opera Web Browser"
opera icon "opera.xpm"
opera mini-icon "opera.xpm"
opera exec "opera &"
opera group "Internet"'

    echo "${wmconfig_contain}" > ${wmconfig_file}
    chmod $chmodv 644 ${wmconfig_file}
}

generate_desktop()
{
    # arg1 = location
    # arg2 = type

    desktop_file="${1}/opera.desktop"
    desktop_contain='[Desktop Entry]'

    if test ${2}; then
	if test "${2}" = "xdg"; then
	    desktop_contain="${desktop_contain}
Version=1.0
TryExec=opera"
	fi

	desktop_contain="${desktop_contain}
Encoding=UTF-8
Name=Opera
Name[af]=opera
Name[eo]=Opero
Name[zu]=I Opera
GenericName=Web browser
GenericName[bs]=Web preglednik
GenericName[de]=Web-Browser
GenericName[eo]=TTT-rigardilo
GenericName[es]=Navegador web
GenericName[et]=Veebibrauser
GenericName[eu]=Web arakatzailea
GenericName[fi]=WWW-selain
GenericName[fr]=Un navigateur web
GenericName[is]=Vafri
GenericName[it]=Browser Web
GenericName[nl]=webbrowser
GenericName[nn]=Nettlesar
GenericName[pt]=Navegador Web
GenericName[pt_BR]=Navegador
GenericName[ro]=Navigator de web
GenericName[sl]=Spletni brskalnik
GenericName[ven]=Buronza ya Webu
GenericName[xh]=Umkhangeli Zincwadi Zokubhaliweyo
GenericName[zu]=Umkhangeli zincwadi we Web
Exec=opera %u
Terminal=false"

# Application is not a category, according to
# http://standards.freedesktop.org/menu-spec/menu-spec-latest.html
#
# but I include it anyway to fix
# http://bugs.gentoo.org/show_bug.cgi?id=90508
#
# Other possible categories:
# Email, Presentation, IRCClient, FileTransfer, P2P, News, WebDevelopment

	if test "${2}" = "xdg"; then
	    desktop_contain="${desktop_contain}
Categories=Application;Qt;Network;WebBrowser;X-Ximian-Main;X-Ximian-Toplevel
Icon=opera.png"
	else
	    desktop_contain="${desktop_contain}
Icon=opera"
	fi

	desktop_contain="${desktop_contain}
MimeType=text/html;text/xml;application/xhtml+xml"
    else
	desktop_contain="${desktop_contain}
Name=Opera
Exec=opera
Icon=opera.xpm
Terminal=0"
    fi

    desktop_contain="${desktop_contain}
Comment=Web Browser
Type=Application"

    echo "${desktop_contain}" > ${desktop_file}
    chmod $chmodv 644 ${desktop_file}
}

generate_mdk_menu()
{
    # arg1 = location
    mdk_menu_file="$1/opera"

    mdk_menu_contain="?package(opera):command=\"${wrapper_dir}/opera\" \\
icon=\"opera.png\" \\
needs=\"X11\" \\
section=\"Networking/WWW\" \\
title=\"Opera\" \\
longtitle=\"Opera Web Browser\" \\
mimetypes=\"text/html,text/plain,text/vnd.csr,text/vnd.wap.wml,text/wml,text/xml,text/xhtml\" \\
accept_url=\"true\" \\
multiple_files=\"false\""

    echo "${mdk_menu_contain}" > ${mdk_menu_file}
    chmod $chmodv 644 ${mdk_menu_file}
}

icons()
{
    # This function searches for common icon paths.

    debug_msg 0 "in icons()"

    if test ! -d /usr/share/icons
    then
	if test -w /usr/share
	then
	    mkdir $mkdirv $mkdirp /usr/share/icons/
	    chmod $chmodv 755 /usr/share/icons
	    cp $cpv $share_dir/images/opera.xpm /usr/share/icons/opera.xpm
	fi
    elif test -w /usr/share/icons
    then cp $cpv $share_dir/images/opera.xpm /usr/share/icons/opera.xpm
    fi

    if test ! -d /usr/share/pixmaps
    then
	if test -w /usr/share
	then
	    mkdir $mkdirv $mkdirp /usr/share/pixmaps/
	    chmod $chmodv 755 /usr/share/pixmaps
	    cp $cpv $share_dir/images/opera.xpm /usr/share/pixmaps/opera.xpm
	fi
    elif test -w /usr/share/pixmaps/
    then cp $cpv $share_dir/images/opera.xpm /usr/share/pixmaps/opera.xpm
    fi

    if test ! -d /etc/X11/wmconfig/
    then
	if test -w /etc/X11
	then
	    mkdir $mkdirv $mkdirp /etc/X11/wmconfig/
	    chmod $chmodv 755 /etc/X11/wmconfig
	    generate_wmconfig /etc/X11/wmconfig
	fi
    elif test -w /etc/X11/wmconfig/
    then generate_wmconfig /etc/X11/wmconfig
    fi

    if test -d /etc/X11/applnk/
    then
	if test ! -d /etc/X11/applnk/Internet/
	then
	    if test -w /etc/X11/applnk
	    then
		mkdir $mkdirv $mkdirp /etc/X11/applnk/Internet/
		chmod $chmodv 755 /etc/X11/applnk/Internet
		generate_desktop /etc/X11/applnk/Internet
	    fi
	elif test -w /etc/X11/applnk/Internet
	then generate_desktop /etc/X11/applnk/Internet
	fi
    fi
}

gnome()
{
    # This function searches for common gnome icon paths.
    debug_msg 1 "in gnome()"

    if test -d /opt/gnome/
    then
	# /opt/gnome share
	if test -d /opt/gnome/share
	then
	    # /opt/gnome icon
	    if test ! -d /opt/gnome/share/pixmaps/
	    then
		if test -w /opt/gnome/share
		then
		    mkdir $mkdirv $mkdirp /opt/gnome/share/pixmaps/
		    chmod $chmodv 755 /opt/gnome/share/pixmaps
		    cp $cpv $share_dir/images/opera.xpm /opt/gnome/share/pixmaps/opera.xpm
		fi
	    elif test -w /opt/gnome/share/pixmaps
	    then cp $cpv $share_dir/images/opera.xpm /opt/gnome/share/pixmaps/opera.xpm
	    fi
	    # end /opt/gnome icon

	    # /opt/gnome link
	    if test -d /opt/gnome/share/gnome/apps/
	    then
		if test -d /opt/gnome/share/gnome/apps/Internet/
		then
		    if test -w /opt/gnome/share/gnome/apps/Internet
		    then generate_desktop /opt/gnome/share/gnome/apps/Internet
		    fi
		elif test -d /opt/gnome/share/gnome/apps/Networking/WWW/
		then
		    if test -w /opt/gnome/share/gnome/apps/Networking/WWW
		    then generate_desktop /opt/gnome/share/gnome/apps/Networking/WWW
		    fi
		elif test -w /opt/gnome/share/gnome/apps
		then
		    mkdir $mkdirv $mkdirp /opt/gnome/share/gnome/apps/Internet/
		    chmod $chmodv 755 /opt/gnome/share/gnome/apps/Internet
		    generate_desktop /opt/gnome/share/gnome/apps/Internet
		fi
	    fi
	    # end /opt/gnome link
	fi
	# end /opt/gnome share

    elif test -d /usr/share/gnome/
    then
        # /usr/share/gnome icon
        if test ! -d /usr/share/gnome/pixmaps/
	then
	    if test -w /usr/share/gnome
	    then
		mkdir $mkdirv $mkdirp /usr/share/gnome/pixmaps/
		chmod $chmodv 755 /usr/share/gnome/pixmaps
		cp $cpv $share_dir/images/opera.xpm /usr/share/gnome/pixmaps/opera.xpm
	    fi
	elif test -w /usr/share/gnome/pixmaps
	then cp $cpv $share_dir/images/opera.xpm /usr/share/gnome/pixmaps/opera.xpm
	fi
	# end /usr/share/gnome icon

	# /usr/share/gnome link
	if test -d /usr/share/gnome/apps/
	then
	    if test -d /usr/share/gnome/apps/Internet/
	    then
		if test -w /usr/share/gnome/apps/Internet
		then generate_desktop /usr/share/gnome/apps/Internet
		fi
	    elif test -d /usr/share/gnome/apps/Networking/WWW/
	    then
		if test -w /usr/share/gnome/apps/Networking/WWW
		then generate_desktop /usr/share/gnome/apps/Networking/WWW
		fi
	    elif test -w /usr/share/gnome/apps
	    then
		mkdir $mkdirv $mkdirp /usr/share/gnome/apps/Internet/
		chmod $chmodv 755 /usr/share/gnome/apps/Internet
		generate_desktop /usr/share/gnome/apps/Internet
	    fi
	fi
	# end /usr/share/gnome link
    fi
    # Add ximian here
}

kde()
{
    # This function searches for common kde2 and kde 3 icon paths.
    debug_msg 1 "in kde()"

    if test -d /opt/kde$1/share
    then
	DIR_HI=/opt/kde$1/share/icons/hicolor
	if test -d "$DIR_HI" -a -w "$DIR_HI"
	then
	    if test -d "$DIR_HI"/48x48/apps -a -w "$DIR_HI"/48x48/apps
	    then cp $cpv $share_dir/images/opera_48x48.png $DIR_HI/48x48/apps/opera.png
	    fi
	    if test -d "$DIR_HI"/32x32/apps -a -w "$DIR_HI"/32x32/apps
	    then cp $cpv $share_dir/images/opera_32x32.png $DIR_HI/32x32/apps/opera.png
	    fi
	    if test -d "$DIR_HI"/22x22/apps -a -w "$DIR_HI"/22x22/apps
	    then cp $cpv $share_dir/images/opera_22x22.png $DIR_HI/22x22/apps/opera.png
	    fi
	fi

	DIR_LO=/opt/kde$1/share/icons/locolor
	if test -d $DIR_LO -a -w $DIR_LO
	then
	    if test -d $DIR_LO/32x32/apps -a -w $DIR_LO/32x32/apps
	    then cp $cpv $share_dir/images/opera_32x32.png $DIR_LO/32x32/apps/opera.png
	    fi
	    if test -d $DIR_LO/22x22/apps -a -w $DIR_LO/22x22/apps
	    then cp $cpv $share_dir/images/opera_22x22.png $DIR_LO/22x22/apps/opera.png
	    fi
	    if test -d $DIR_LO/16x16/apps -a -w $DIR_LO/16x16/apps
	    then cp $cpv $share_dir/images/opera_16x16.png $DIR_LO/16x16/apps/opera.png
	    fi
	fi

	if test -d /opt/kde$1/share/applnk/
	then
	    if test ! -d /opt/kde$1/share/applnk/Internet/ -a -w /opt/kde$1/share/applnk
	    then
		mkdir $mkdirv $mkdirp /opt/kde$1/share/applnk/Internet/
		chmod $chmodv 755 /opt/kde$1/share/applnk/Internet
	    fi
	    if test -w /opt/kde$1/share/applnk/Internet
	    then generate_desktop /opt/kde$1/share/applnk/Internet $1
	    fi
	fi
    fi

    if test -d /usr/share/applnk/Networking
    then
	if test ! -d /usr/share/applnk/Networking/WWW/ -a -w /usr/share/applnk/Networking
	then
	    mkdir $mkdirv $mkdirp /usr/share/applnk/Networking/WWW/
	    chmod $chmodv 755 /usr/share/applnk/Networking/WWW
	fi
	if test -w /usr/share/applnk/Networking/WWW
	then generate_desktop /usr/share/applnk/Networking/WWW $1
	fi
    fi
}

kde1()
{
    # This function searches for common kde1 icon paths.
    debug_msg 1 "in kde1()"

    if test -d /opt/kde/share
    then
	if test ! -d /opt/kde/share/icons/ -a -w /opt/kde/share
	then
	    mkdir $mkdirv $mkdirp /opt/kde/share/icons/
	    chmod $chmodv 755 /opt/kde/share/icons
	fi
	if test -w /opt/kde/share/icons
	then cp $cpv $share_dir/images/opera.xpm /opt/kde/share/icons/opera.xpm
	fi

	if test ! -d /opt/kde/share/applnk/Internet/ -a -w /opt/kde/share/applnk
	then
	    mkdir $mkdirv $mkdirp /opt/kde/share/applnk/Internet/
	    chmod $chmodv 755 /opt/kde/share/applnk/Internet
	fi
	if test -w /opt/kde/share/applnk/Internet
	then generate_desktop /opt/kde/share/applnk/Internet
	fi

    elif test -d /usr/share/applnk/Internet/ -a -w /usr/share/applnk/Internet
    then generate_desktop /usr/share/applnk/Internet
    fi
}

mandrake()
{
    if test -x /usr/bin/update-menus
    then
	test ! -d /usr/share/menu -a -w /usr/share && mkdir $mkdirv $mkdirp /usr/share/menu/
	test -d /usr/share/menu && generate_mdk_menu /usr/share/menu

	test ! -d /usr/share/icons -a -w /usr/share && mkdir $mkdirv $mkdirp /usr/share/icons/ && chmod $chmodv 755 /usr/share/icons
	test -d /usr/share/icons && cp $cpv $share_dir/images/opera_32x32.png /usr/share/icons/opera.png

	test ! -d /usr/share/icons/mini -a -w /usr/share/icons && mkdir $mkdirv $mkdirp /usr/share/icons/mini/ && chmod $chmodv 755 /usr/share/icons/mini
	test -d /usr/share/icons/mini && cp $cpv $share_dir/images/opera_16x16.png /usr/share/icons/mini/opera.png

	test ! -d /usr/share/icons/large -a -w /usr/share/icons && mkdir $mkdirv $mkdirp /usr/share/icons/large && chmod $chmodv 755 /usr/share/icons/large
	test -d /usr/share/icons/large && cp $cpv $share_dir/images/opera_48x48.png /usr/share/icons/large/opera.png

	/usr/bin/update-menus || true
    fi
}

huntdirs()
{
    # $GNOME2PATH and $GNOMEPATH skipped (too few hits google site:gnome.org)
    # http://docs.kde.org/userguide/environment-variables.html
    echo ${GNOME2_PATH}:${KDEDIRS}:${GNOME_PATH}:${KDEDIR}:/usr:/opt/gnome:/opt/kde3 \
    | tr : '\012' | sed -e 's:$:'"/$1:"
}

pathdirs()
{
    # Places to look for executables, one per line:
    echo $PATH | tr : '\012'
    huntdirs bin
}

xdg()
{   # http://standards.freedesktop.org
    UDD=''
    for BIN_DIR in `pathdirs`; do
	test -x ${BIN_DIR}/update-desktop-database || continue
	UDD=${BIN_DIR}/update-desktop-database; break
    done

    # http://standards.freedesktop.org/icon-theme-spec/icon-theme-spec-latest.html
    if test "$UDD"; then
	for ICON_DIR in `echo ${XDG_DATA_DIRS}:/usr/local/share:/usr/share|tr : '\012'|sed -e '/^$/d;s:$:/icons/hicolor:'` /usr/share/pixmaps/hicolor; do
	    test -d ${ICON_DIR} && break
	done

	if   test ! -d ${ICON_DIR}; then echo "Could not find icon installation directory, icons not installed." >&2
	elif test ! -w ${ICON_DIR}; then echo "Directory \"${ICON_DIR}\" not writable by user \"${USER}\", icons not installed." >&2
	else
	    for RESOLUTION in 48x48 32x32 22x22; do
		TO_DIR=${ICON_DIR}/${RESOLUTION}/apps
		test -d ${TO_DIR} && test -w ${TO_DIR} && cp $cpv $share_dir/images/opera_${RESOLUTION}.png ${TO_DIR}/opera.png
	    done
	fi

	for SHORTCUT_DIR in ${XDG_DATA_HOME}/applications /usr/local/share/applications /usr/share/applications; do
	    test -d ${SHORTCUT_DIR} && break;
	done

	if   test ! -d ${SHORTCUT_DIR}; then echo "Could not find shortcut installation directory, desktop entry not installed." >&2; return
	elif test ! -w ${SHORTCUT_DIR}; then echo "Directory \"${SHORTCUT_DIR}\" not writable by user \"${USER}\", desktop entry not installed." >&2; return
	fi
	generate_desktop ${SHORTCUT_DIR} xdg
	${UDD}
    else
	icons
	gnome
	kde 3
	kde 2
	kde1
	mandrake
    fi
}

echo test | sed -n -e 's/test//' || error 'sed'

# AnyBSD systems don't have $USERNAME by default
if test -z "${USERNAME}" && test "${USER}"
then
    USERNAME="${USER}"
    debug_msg 0 "setting USERNAME to ${USER}"
fi

check_prefixes "$@"
select_installation_type
set_os_spesific
run_install
exit 0
