#include <stdio.h>
#include <string.h>
#include <stdlib.h>



#define printk(x...) printf(" * " x)
#define kmalloc(a,b) malloc(a)
#define KERN_WARNING " "

typedef unsigned char u8;
typedef unsigned short u16;
typedef unsigned int u32;
typedef unsigned long long u64;


typedef int     acpi_status;

#define AE_OK	0
#define AE_BAD_PARAMETER 1
#define AE_NO_MEMORY 2


struct acpi_table_header {
	char            signature[4];	/* ACPI signature (4 ASCII
					 * characters) */
	u32             length;	/* Length of table, in bytes, including
				 * header */
	u8              revision;	/* ACPI Specification minor version # */
	u8              checksum;	/* To make sum of entire table == 0 */
	char            oem_id[6];	/* OEM identification */
	char            oem_table_id[8];	/* OEM table identification */
	u32             oem_revision;	/* OEM revision number */
	char            asl_compiler_id[4];	/* ASL compiler vendor ID */
	u32             asl_compiler_revision;	/* ASL compiler revision */
}               __attribute__((packed));

struct acpi_generic_address {
	u8              address_space_id;	/* Address space where struct
						 * or register exists. */
	u8              register_bit_width;	/* Size in bits of given
						 * register */
	u8              register_bit_offset;	/* Bit offset within the
						 * register */
	u8              reserved;	/* Must be 0 */
	u64             address;/* 64-bit address of struct or register */
}               __attribute__((packed));

struct acpi_table_ecdt {
	struct acpi_table_header header;
	struct acpi_generic_address ec_control;
	struct acpi_generic_address ec_data;
	u32             uid;
	u8              gpe_bit;
	char            ec_id[0];
}               __attribute__((packed));




/* dumping, etc. */

static char    *
append_0(unsigned char *s, int len)
{
	static char     buff[10];

	memcpy(buff, s, len);
	buff[len] = 0;
	return buff;
}

static void
dump_header(struct acpi_table_header * h)
{
	printk("\tsignature %s\n", append_0(h->signature, 4));
	printk("\tlength: %d\n", h->length);
	printk("\trevision: %#x\n", h->revision & 0xff);
	printk("\tchecksum: %#x\n", h->checksum & 0xff);
	printk("\toem_id: %s\n", append_0(h->oem_id, 6));
	printk("\toem_table_id: %s\n", append_0(h->oem_table_id, 8));
	printk("\toem_revision: %#x\n", h->oem_revision);
	printk("\tasl_compiler_id: %s\n", append_0(h->asl_compiler_id, 4));
	printk("\tasl_compiler_revision: %#x\n", h->asl_compiler_revision);
}

static void
dump_GAS(struct acpi_generic_address * gas)
{
	printk("\taddress_space_id: %#x\n", gas->address_space_id & 0xff);
	printk("\taddress_space_bit_width: %#x\n",
	       gas->register_bit_width & 0xff);
	printk("\tregister_bit_offset: %#x\n", gas->register_bit_offset & 0xff);
	printk("\treserved (should be 0): %#x\n", gas->reserved);
	printk("\taddress: %#llx\n", gas->address);
}

static u8       compute_checksum(struct acpi_table_header * h);

static void
dump_ecdt_table(struct acpi_table_ecdt * ecdt)
{
	u8              cs;

	printk("dumping %p\n", ecdt);

	printk("header:\n");
	dump_header(&ecdt->header);

	printk("ec_control:\n");
	dump_GAS(&ecdt->ec_control);
	printk("ec_data:\n");
	dump_GAS(&ecdt->ec_data);
	printk("uid: %#x\n", ecdt->uid);
	printk("gpe: %#x\n", ecdt->gpe_bit);
	printk("ec_id: %s\n", ecdt->ec_id);

	cs = compute_checksum((struct acpi_table_header *) ecdt);
	printk("compute checksum: %s\n", cs ? "BAD" : "passed");
}


/* Begin kernel things for driver/acpi/osl.c */

static          u8
compute_checksum(struct acpi_table_header * h)
{
	int             i;
	u8             *p;
	u8              cs;

	p = (u8 *) h;
	cs = 0;
	for (i = 0; i < h->length; i++)
		cs += *p++;
	return cs;
}

static          acpi_status
try_to_replace_ibm_ecdt(struct acpi_table_ecdt * old,
			struct acpi_table_ecdt ** new)
{
	u32             len;
	u8             *p;
	struct acpi_table_ecdt *ecdt;

	*new = NULL;

	len = old->header.length;
	ecdt = kmalloc(len + 1, GFP_KERNEL);
	if (!ecdt)
		return AE_NO_MEMORY;

	memcpy(ecdt, old, len);

	ecdt->header.length = len + 1;

	for (p = (u8 *) ecdt + ecdt->header.length - 1;
	     p != (u8 *) & ecdt->ec_id;
	     --p) {
		*p = *(p - 1);
	}
	ecdt->ec_id[1] = '_';

	ecdt->header.checksum = 0;
	ecdt->header.checksum = -compute_checksum((struct acpi_table_header *) ecdt);

	*new = ecdt;

	/* printk(KERN_WARNING "acpi: IBM's ECDT TABLE sucessfully replaced by a corrected one.\n"); */

	return AE_OK;
}


static          acpi_status
verify_ibm_ecdt(struct acpi_table_ecdt * existing_table,
		struct acpi_table_ecdt ** new_table)
{
	const char     *const ec_id =
	((struct acpi_table_ecdt *) existing_table)->ec_id;

	if (ec_id[0] == '\\' && ec_id[1] == 'S') {
		printk(KERN_WARNING "Found bad ACPI ECDT table from IBM in the firmware\n");
		return (try_to_replace_ibm_ecdt(existing_table, new_table));
	}
	return AE_OK;
}



acpi_status
acpi_os_table_override(struct acpi_table_header * existing_table,
		       struct acpi_table_header ** new_table)
{
	if (!existing_table || !new_table)
		return AE_BAD_PARAMETER;

	if (!memcmp(existing_table->signature, "ECDT", 4)) {
		struct acpi_table_ecdt *old = (struct acpi_table_ecdt *) existing_table;
		struct acpi_table_ecdt **new = (struct acpi_table_ecdt **) new_table;

		if (!memcmp(old->header.oem_id, "IBM", 3))
			return (verify_ibm_ecdt(old, new));
	}
	return AE_OK;
}
/* End kernel things */

void
do_generate_header(struct acpi_table_ecdt *ecdt)
{
	int             len;
	int             i;
	u8             *p;

	i = 0;
	len = ecdt->header.length;
	p = (u8 *) ecdt;

	printf("unsigned char ecdt_table[]= {\n\t");

	do {
		printf("0x%.2x,%s", *p++ & 0xff, ++i % 16 ? " " : "\n\t");
	} while (--len);
	printf("\n};\n");
}

int
main(int argc, char *argv[])
{
	int             retval = 0;
	int             n;
	u8             *ibm_ecdt = NULL;
	struct acpi_table_header *ibm_ecdt_corrected = NULL;

#define at_least (sizeof(struct acpi_table_header) + \
		  2 * sizeof(struct acpi_generic_address) + \
		  1 * sizeof(u32) + 5)

	ibm_ecdt = malloc(1001);
	if (!ibm_ecdt)
		goto out;
	n = fread(ibm_ecdt, 1, 1000, stdin);


	if (n < at_least || memcmp("ECDT", ibm_ecdt, 4)) {
		printf("size seems to be no correct.\n");
		exit(1);
	}

	printf("/*\n");
	printf(" * ECDT table before:\n");
	dump_ecdt_table((struct acpi_table_ecdt *) ibm_ecdt);

	retval = acpi_os_table_override((struct acpi_table_header *) ibm_ecdt, &ibm_ecdt_corrected);
	if (ibm_ecdt_corrected) {
		printf(" * \n");
		printf(" * The ECDT table for this IBM Thinkpad is wrong.\n");
		printf(" * It should contains this info instead.\n");
		printf(" * \n");
		dump_ecdt_table((struct acpi_table_ecdt *) ibm_ecdt_corrected);
		printf(" * \n");
		printf(" */\n\n");
		do_generate_header((struct acpi_table_ecdt *) ibm_ecdt_corrected);
		free(ibm_ecdt_corrected);
	} else {
		fprintf(stderr, "This table is correct.  No need to replace it.\n");
	}
	free(ibm_ecdt);
out:
	return retval;
}
