/* Copyright (c) 2003 SuSE Linux AG, Germany
 * Copyright (c) 1999, 2000, 2001, 2002 SuSE GmbH Nuernberg, Germany
 * Author: Thorsten Kukuk <kukuk@suse.de>
 *
 * This version based on the login program from util-linux 2.9s, but
 * has a lot of enhancements and code cleanups.
 */

/*
 * Copyright (c) 1980, 1987, 1988 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * login [ name ]
 * login -h hostname    (for telnetd, etc.)
 * login -f name        (for pre-authenticated login: datakit, xterm, etc.)
 * login -p             (don't destroy environment)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define _GNU_SOURCE

#include <sys/param.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <getopt.h>
#include <memory.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/file.h>
#include <termios.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <signal.h>
#include <errno.h>
#include <grp.h>
#include <pwd.h>
#include <utmp.h>
#include <setjmp.h>
#include <stdlib.h>
#include <string.h>
#include <sys/syslog.h>
#include <sys/sysmacros.h>
#include <netdb.h>
#include <locale.h>
#include <libintl.h>
#include <sys/sysmacros.h>
#include <linux/major.h>
#include <security/pam_appl.h>
#include <security/pam_misc.h>

#ifdef WITH_SELINUX
#include <selinux/get_context_list.h>
#include <selinux/flask.h>
#include <selinux/selinux.h>
#endif

#include "faillog.h"
#include "failure.h"

#define PAM_FAIL_CHECK if (retcode != PAM_SUCCESS) { \
       fprintf (stderr, "\n%s\n", pam_strerror (pamh, retcode)); \
       syslog (LOG_ERR, "%s", pam_strerror (pamh, retcode)); \
       sleep (getdef_num ("FAIL_DELAY", 1)); \
       pam_end (pamh, retcode); \
       exit (1); \
   }
#define PAM_END { pam_setcred(pamh, PAM_DELETE_CRED); \
                  retcode = pam_close_session(pamh,0); \
		  pam_end(pamh,retcode); }

#include "setproctitle.h"
#include "getdef.h"

#ifndef _
#define _(s) gettext(s)
#endif

#ifdef WITH_SELINUX
int selinux_enabled=0;
security_context_t user_context=NULL;
security_context_t ttyn_context=NULL; /* The current context of ttyn device */
security_context_t vcsn_context=NULL; /* The current context of vcsn device */
security_context_t vcsan_context=NULL;/* The current context of vcsan device */
#endif

/* Print the version information.  */
static inline void
print_version (void)
{
  fprintf (stdout, "login (%s) %s\n", PACKAGE, VERSION);
  fprintf (stdout, gettext ("\
Copyright (C) 1980, 1987, 1988 The Regents of the University of California.\n\
Copyright (C) 1999, 2000, 2001, 2002 SuSE GmbH Nuernberg, Germany.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
"));
  fprintf (stdout, _("Written by Thorsten Kukuk.\n"));
}

static inline void
print_usage (void)
{
  fputs (_("Usage: login [name] [-p] [-h host] [-f name]\n"),
	 stdout);
}

static void
print_help (void)
{
  print_usage ();
  fputs (_("login - sign on\n\n"), stdout);

  fputs (_("  -p             Do not destroy the environment\n"), stdout);
  fputs (_("  -h hostname    Name of the remote host for umtp/wtmp\n"),
	 stdout);
  fputs (_("  -f name        Skip a second login authentication\n"), stdout);
  fputs (_("  --help         Give this help list\n"), stdout);
  fputs (_("  --usage        Give a short usage message\n"), stdout);
  fputs (_("  --version      Print program version\n"), stdout);
}

static inline void
print_error (void)
{
  const char *program = "login";

  fprintf (stderr,
	   _("Try `%s --help' or `%s --usage' for more information.\n"),
	   program, program);
}

jmp_buf motdinterrupt;

static void
sigint (int unused __attribute__ ((unused)))
{
  longjmp (motdinterrupt, 1);
}

/*
 * motd -- output the /etc/motd file
 *
 * motd() determines the name of a login announcement file and outputs
 * it to the user's terminal at login time.  The MOTD_FILE configuration
 * option is a colon-delimited list of filenames.
 */

static void
motd (void)
{
  int fd, nchars;
  void (*oldint) (int);
  char tbuf[8192];
  char motdlist[BUFSIZ], *mb, *motdfile;

  if ((mb = getdef_str ("MOTD_FILE")) == NULL)
    return;

  strncpy (motdlist, mb, sizeof (motdlist));
  motdlist[sizeof (motdlist) - 1] = '\0';

  for (mb = motdlist; (motdfile = strtok (mb, ":")) != NULL; mb = NULL)
    {

      if ((fd = open (motdfile, O_RDONLY, 0)) >= 0)
	{
	  oldint = signal (SIGINT, sigint);
	  if (setjmp (motdinterrupt) == 0)
	    while ((nchars = read (fd, tbuf, sizeof (tbuf))) > 0)
	      write (fileno (stdout), tbuf, nchars);
	  signal (SIGINT, oldint);
	  close (fd);
	}
    }
}

static struct hostent *hostaddress = NULL;
static int timeout;

static void
timedout (int sig __attribute__ ((unused)))
{
  struct termio ti;

  fprintf (stderr, _("Login timed out after %d seconds\n"), timeout);

  /* reset echo */
  ioctl (0, TCGETA, &ti);
  ti.c_lflag |= ECHO;
  ioctl (0, TCSETA, &ti);
  exit (0);
}

#ifdef WITH_SELINUX

static security_context_t
security_label_tty (const char *tty, security_context_t usercon)
{
  security_context_t newdev_context=NULL; /* The new context of a device */
  security_context_t prev_context=NULL; /* The new context of a device */
  if (getfilecon (tty, &prev_context) < 0)
    {
      fprintf (stderr, _("Warning!  Could not get current context for %s, not relabeling.\n"), tty);
      syslog (LOG_NOTICE, "Warning!  Could not get current context for %s, notrelabeling.\n", tty);
      return NULL;
    }
  if (security_compute_relabel (usercon, prev_context, SECCLASS_CHR_FILE,
				&newdev_context) != 0)
    {
      fprintf (stderr, _("Warning!  Could not get new context for %s, not relabeling.\n"), tty);
      syslog (LOG_NOTICE, "Warning!  Could not get new context for %s, not relabeling.\n", tty);
      return NULL;
    }
  if (setfilecon (tty,newdev_context))
    {
      fprintf (stderr, _("Warning!  Could not relabel %s with %s, not relabeling.\n"), tty, newdev_context);
      syslog (LOG_NOTICE, "Warning!  Could not relabel %s with %s, not relabeling.\n", tty, newdev_context);
      return NULL;
    }
  freecon (newdev_context);
  return prev_context;
}

static void
security_restorelabel_tty (const char *tty, security_context_t context)
{
  if (context == NULL)
    return;

  if (setfilecon (tty, context))
    {
      fprintf (stderr, _("Warning!  Could not relabel %s with %s, not relabeling.\n"), tty, context);
      syslog (LOG_NOTICE, "Warning!  Could not relabel %s with %s, not relabeling.\n", tty, context);
    }
  freecon (context);
  return;
}
#endif /* WITH_SELINUX */


/* Nice and simple code provided by Linus Torvalds 16-Feb-93 */
/* Nonblocking stuff by Maciej W. Rozycki, macro@ds2.pg.gda.pl, 1999.
   He writes: "Login performs open() on a tty in a blocking mode.
   In some cases it may make login wait in open() for carrier infinitely,
   for example if the line is a simplistic case of a three-wire serial
   connection. I believe login should open the line in the non-blocking mode
   leaving the decision to make a connection to getty (where it actually
   belongs). */
static void
opentty (const char *tty)
{
  int i, flags;
  int fd = open(tty, O_RDWR | O_NONBLOCK);

  if (fd == -1)
    {
      syslog (LOG_ERR, "FATAL: can't reopen tty: %s",
	      strerror (errno));
      sleep (5);
      exit (1);
    }

  flags = fcntl (fd, F_GETFL);
  flags &= ~O_NONBLOCK;
  fcntl (fd, F_SETFL, flags);

  for (i = 0; i < fd; ++i)
    close (i);
  for (i = 0; i < 3; ++i)
    if (fd != i)
      {
	if (dup2 (fd, i) == -1)
	  {
	    syslog (LOG_ERR, "FATAL: can't reopen stdin/stdout/stderr: %s",
		    strerror (errno));
	    sleep (5);
	    exit (1);
	  }
      }
  if (fd >= 3)
    close (fd);
}

/* In case login is suid it was possible to use a hardlink as stdin
   and exploit races for a local root exploit. The problem is
     ttyn := ttyname(0); ...; chown(ttyn);
   here ttyname() might return "/tmp/x", a hardlink to a pseudotty.
   All of this is a problem only when login is suid, which it isnt. */
static void
check_ttyname (char *ttyn)
{
  struct stat statbuf;

  if (lstat (ttyn, &statbuf)
      || !S_ISCHR (statbuf.st_mode)
      || (statbuf.st_nlink > 1 &&
	  strncmp(ttyn, _PATH_DEV, strlen (_PATH_DEV)) != 0))
    {
      syslog (LOG_ERR, "FATAL: bad tty");
      sleep (5);
      exit (1);
    }
}

/* true if the filedescriptor fd is a console tty, very Linux specific */
/* XXX Does this work with serial console ?? (kukuk@suse.de) */
static int
consoletty (int fd)
{
  struct stat stb;

  if ((fstat (fd, &stb) >= 0)
      && (major (stb.st_rdev) == TTY_MAJOR) && (minor (stb.st_rdev) < 64))
    return 1;
  return 0;
}

/* Create btmp entry if configured for it. */
static void
logbtmp (const char *line, const char *username, const char *hostname)
{
  struct utmp ut;
  struct timeval ut_tv;
  char *path_btmp = getdef_str ("FTMP_FILE");

  /* If no FTMP_FILE is defined in /etc/login.defs, we shouldn't
     add btmp entries */
  if (path_btmp == NULL)
    return;

  memset (&ut, 0, sizeof (ut));

  /* We made sure, that logbtmp is only called with a valid
     username.  */
  strncpy (ut.ut_user, username, sizeof (ut.ut_user));

  strncpy (ut.ut_id, line + 3, sizeof (ut.ut_id));
  strncpy (ut.ut_line, line, sizeof (ut.ut_line));
  ut.ut_line[sizeof (ut.ut_line) - 1] = 0;
  gettimeofday (&ut_tv, NULL);
  ut.ut_tv.tv_sec = ut_tv.tv_sec;
  ut.ut_tv.tv_usec = ut_tv.tv_usec;
  ut.ut_type = LOGIN_PROCESS;
  ut.ut_pid = getpid ();
  if (hostname)
    {
      strncpy (ut.ut_host, hostname, sizeof (ut.ut_host));
      ut.ut_host[sizeof (ut.ut_host) - 1] = 0;
      if (hostaddress && hostaddress->h_addr_list)
	memcpy (&ut.ut_addr, hostaddress->h_addr_list[0], sizeof (ut.ut_addr));
    }

  updwtmp (path_btmp, &ut);
}

/* If we create a lastlog file, show the data from it. */
static void
dolastlog (int quiet, uid_t uid, const char *tty, const char *hostname)
{
  struct lastlog ll;
  time_t ll_time;
  int fd;

  if (!getdef_bool ("LASTLOG_ENAB"))	/* give last login and log this one */
    return;

  if ((fd = open (_PATH_LASTLOG, O_RDWR, 0)) >= 0)
    {
      if (lseek (fd, (off_t) uid * sizeof (ll), SEEK_SET) == (off_t)-1)
	{
	  int errsv = errno;
	  fprintf (stderr, _("login: lseek() failed: %s\n"),
		   strerror (errsv));
	  return;
	}
      if (!quiet)
	{
	  if (read (fd, (char *) &ll, sizeof (ll)) == sizeof (ll) &&
	      ll.ll_time != 0)
	    {
	      ll_time = ll.ll_time;

	      printf (_("Last login: %.*s "),
		      24 - 5, ctime (&ll_time));

	      if (*ll.ll_host != '\0')
		printf (_("from %.*s\n"),
			(int) sizeof (ll.ll_host), ll.ll_host);
	      else
		printf (_("on %.*s\n"),
			(int) sizeof (ll.ll_line), ll.ll_line);
	    }
	  lseek (fd, (off_t) uid * sizeof (ll), SEEK_SET);
	}
      memset (&ll, 0, sizeof (ll));
      time (&ll_time);
      ll.ll_time = ll_time;
      strncpy (ll.ll_line, tty, sizeof (ll.ll_line));
      ll.ll_line[sizeof (ll.ll_line) - 1] = 0;
      if (hostname)
	{
	  strncpy (ll.ll_host, hostname, sizeof (ll.ll_host));
	  ll.ll_host[sizeof (ll.ll_host) - 1] = 0;
	}
      write (fd, &ll, sizeof (ll));
      close (fd);
    }
}

/* Find out which TERM variable we need to set for this terminal. */
static char *
ttytype (const char *line)
{
  FILE *fp;
  char buf[BUFSIZ];
  char *typefile;
  char *cp;
  char type[BUFSIZ];
  char port[BUFSIZ];

  if ((typefile = getdef_str ("TTYTYPE_FILE")) == NULL)
    return strdup ("dump");

  if (access (typefile, F_OK))
    return strdup ("dump");

  if (!(fp = fopen (typefile, "r")))
    {
      perror (typefile);
      return strdup ("dump");
    }

  while (fgets (buf, sizeof buf, fp))
    {
      if (buf[0] == '#')
	continue;

      if ((cp = strchr (buf, '\n')))
	*cp = '\0';

      if (sscanf (buf, "%s %s", type, port) == 2 && strcmp (line, port) == 0)
	break;
    }
  if (feof (fp) || ferror (fp))
    strcpy (type, "dump");

  fclose (fp);

  return strdup (type);
}

/* Create the login prompt string. */
static char *
new_pam_prompt (char *thishost)
{
  char *buffer = alloca (20 + strlen (thishost));
  char *cp = buffer;
  char *ret;

  *cp++ = '\n';
  cp = stpcpy (cp, thishost);
  cp = stpcpy (cp, " login: ");

  ret = strdup (buffer);

  if (ret == NULL)
    return "login:";
  else
    return ret;
}

/* Install some signal handler */
static inline void
init_sighandler (void)
{
  timeout = getdef_num ("LOGIN_TIMEOUT", 60);

  signal (SIGALRM, timedout);
  alarm ((unsigned int) timeout);
  signal (SIGQUIT, SIG_IGN);
  signal (SIGINT, SIG_IGN);
}

int
main (int argc, char **argv)
{
  struct group *gr;
  int ask, fflag, hflag, pflag, cnt, errsv;
  int quietlog, passwd_req;
  char *domain, *ttyn, *tty, *group;
  char thishost[MAXPATHLEN + 1];
  char *termenv;
  char *childArgv[10];
  char *buff;
  int childArgc = 0;
  int retcode;
  pam_handle_t *pamh = NULL;
  struct pam_conv conv = { misc_conv, NULL };
  pid_t childPid;
  char *hushfile;
  gid_t gid;
  struct faillog faillog;
  char *hostname;
  const char *username;
  const void *void_username;
  int retries = getdef_num ("LOGIN_RETRIES", 3);
  int buflen = 256;
  char *buffer = alloca (buflen);
  struct passwd resultbuf;
  struct passwd *pwd;
  char vcsn[20], vcsan[20];


  init_sighandler ();

  setlocale (LC_ALL, "");
  bindtextdomain ("pam_login", LOCALEDIR);
  textdomain ("pam_login");

  setpriority (PRIO_PROCESS, 0, 0);
#ifdef HAVE_QUOTA
  quota (Q_SETUID, 0, 0, 0);
#endif
  initproctitle (argc, argv);

  if (gethostname (thishost, sizeof (thishost)) < 0)
    {
      errsv = errno;
      fprintf (stderr, _("login: gethostname() failed: %s\n"),
	       strerror (errsv));
      strncpy (thishost, "localhost", sizeof (thishost));
    }
  thishost[sizeof (thishost) - 1] = 0;
  domain = strchr (thishost, '.');

  username = tty = hostname = NULL;
  fflag = hflag = pflag = 0;
  passwd_req = 1;


  while (1)
    {
      int c;
      int option_index = 0;
      struct option long_options[] = {
	{"version", no_argument, NULL, '\255'},
	{"usage", no_argument, NULL, '\254'},
	{"help", no_argument, NULL, '\253'},
	{NULL, 0, NULL, '\0'}
      };

      c = getopt_long (argc, argv, "fh:p", long_options, &option_index);
      if (c == (-1))
	break;
      switch (c)
	{
	case 'f':
	  fflag = 1;
	  break;

	case 'h':
	  {
	    char *cp;
	    int hostbuflen = 512;
	    char *hostbuffer = malloc (hostbuflen);
	    struct hostent resulthostbuf;

	    if (getuid ())
	      {
		fprintf (stderr, _("login: -h for super-user only.\n"));
		exit (1);
	      }

	    if (hostbuffer == NULL)
	      {
		fprintf (stderr, _("login: out of memory.\n"));
		exit (1);
	      }

	    hflag = 1;
	    if (domain && (cp = strchr (optarg, '.')) &&
		strcasecmp (cp, domain) == 0)
	      *cp = '\0';

	    hostname = strdup (optarg);
	    if (hostname == NULL)
	      {
		fprintf (stderr, _("login: out of memory.\n"));
		exit (1);
	      }

	    hostaddress = NULL;
	    while (gethostbyname_r (hostname, &resulthostbuf, hostbuffer,
				    hostbuflen, &hostaddress, &h_errno) != 0
		   && h_errno == NETDB_INTERNAL
		   && errno == ERANGE)
	      {
		char *buf;

		errno = 0;
		hostbuflen += 256;
		buf = realloc (hostbuffer, hostbuflen);
		if (buf != NULL)
		  hostbuffer = buf;
		else
		  {
		    fprintf (stderr, _("login: out of memory.\n"));
		    if (hostbuffer)
		      free (hostbuffer);
		    exit (1);
		  }
	      }
	  }
	  break;

	case 'p':
	  pflag = 1;
	  break;
	case '\253':
	  print_help ();
	  return 0;
	case '\255':
	  print_version ();
	  return 0;
	case '\254':
	  print_usage ();
	  return 0;
	default:
	  print_error ();
	  return 1;
	}
    }

  argc -= optind;
  argv += optind;

  if (!isatty(0) || !isatty(1) || !isatty(2))
    return 1;                /* must be a terminal */

  if (argc > 1)
    {
      fputs (_("login: Too many arguments\n"), stderr);
      print_error ();
      return 1;
    }

  if (*argv)
    {
      char *cp = *argv;
      username = strdup (cp);
      if (username == NULL)
	{
	  fputs (_("login: out of memory.\n"), stderr);
	  exit (1);
	}
      ask = 0;
      /* wipe name - some people mistype their password here */
      /* (of course we are too late, but perhaps this helps a little ..) */
      while (*cp)
	*cp++ = ' ';
    }
  else
    ask = 1;

  for (cnt = getdtablesize (); cnt > 2; cnt--)
    close (cnt);

  setpgrp ();

  openlog ("login", LOG_ODELAY | LOG_PID, LOG_AUTHPRIV);

  ttyn = ttyname (0);
  if (ttyn != NULL && *ttyn != '\0')
    {
      struct termios tt, ttt;

      /* Make sure, ttyn is no link to something outside of _PATH_DEV.  */
      check_ttyname (ttyn);

      tcgetattr (0, &tt);
      ttt = tt;
      ttt.c_cflag &= ~HUPCL;

      if ((chown (ttyn, 0, 0) == -1) ||
	  (chmod (ttyn, getdef_num ("TTYPERM", 0600)) == -1))
	{
	  errsv = errno;
	  syslog (LOG_ERR, "FATAL: cannot change permissions of TTY: %s",
		  strerror (errsv));
	  fprintf (stderr, _("FATAL: cannot change permissions of TTY: %s"),
		   strerror (errsv));
	  sleep (5);
	  exit (1);
	}

      tcsetattr (0, TCSAFLUSH, &ttt);
      signal (SIGHUP, SIG_IGN);	/* so vhangup() wont kill us */
      vhangup ();
      signal (SIGHUP, SIG_DFL);

      /* re-open stdin,stdout,stderr after vhangup() closed them */
      /* if it did, after 0.99.5 it doesn't! */
      opentty (ttyn);
      tcsetattr (0, TCSAFLUSH, &tt);

      if ((tty = strrchr (ttyn, '/')))
	++tty;
      else
	tty = ttyn;
    }
  else
    {
      errsv = errno;
      syslog (LOG_ERR, "ttyname(0) failed: %s", strerror (errsv));
      tty = NULL;
    }

  /* username is initialized to NULL
     and if specified on the command line it is set.
     Therefore, we are safe not setting it to anything
  */

  retcode = pam_start ("login", username, &conv, &pamh);
  if (retcode != PAM_SUCCESS)
    {
      fprintf (stderr, _("login: PAM Failure, aborting: %s\n"),
	       pam_strerror (pamh, retcode));
      syslog (LOG_ERR, "Couldn't initialize PAM: %s",
	      pam_strerror (pamh, retcode));
      exit (99);
    }
  /* hostname & tty are either set to NULL or their correct values,
     depending on how much we know */
  retcode = pam_set_item (pamh, PAM_RHOST, hostname);
  PAM_FAIL_CHECK;
  if (ttyn)
    {
      retcode = pam_set_item (pamh, PAM_TTY, ttyn);
      PAM_FAIL_CHECK;
    }
  retcode = pam_set_item (pamh, PAM_USER_PROMPT, new_pam_prompt (thishost));
  PAM_FAIL_CHECK;

  /* if fflag == 1, then the user has already been authenticated */
  if (fflag && (getuid () == 0))
    passwd_req = 0;
  else
    passwd_req = 1;

  if (passwd_req == 1)
    {
      int failcount = 0;

      /* there may be better ways to deal with some of these
         conditions, but at least this way I don't think we'll
         be giving away information... */
      /* Perhaps someday we can trust that all PAM modules will
         pay attention to failure count and get rid of MAX_LOGIN_TRIES? */

      do {
	retcode = pam_authenticate (pamh, 0);

	if (retcode != PAM_SUCCESS)
	  {
	    const char *userptr;

	    pam_get_item (pamh, PAM_USER, &void_username);
	    username = void_username;

	    pwd = NULL; /* only to be really sure */
	    if (username && *username)
	      while (getpwnam_r (username, &resultbuf, buffer, buflen,
				 &pwd) != 0 && errno == ERANGE)
		{
		  errno = 0;
		  buflen += 256;
		  buffer = alloca (buflen);
		}

	    if (pwd)
	      userptr = pwd->pw_name;
	    else if (getdef_bool ("LOG_UNKFAIL_ENAB"))
	      userptr = username;
	    else
	      userptr = "UNKNOWN";

	    if (++failcount >= retries || retcode == PAM_MAXTRIES)
	      syslog (LOG_NOTICE, "FAILED LOGIN SESSION FROM %s FOR %s, %s",
		      hostname ? hostname : (ttyn ? ttyn : "<unknown>"),
		      userptr, pam_strerror (pamh, retcode));
	    else
	      syslog (LOG_NOTICE, "FAILED LOGIN %d FROM %s FOR %s, %s",
		      failcount, hostname ? hostname :
		      (ttyn ? ttyn : "<unknown>"), userptr,
		      pam_strerror (pamh, retcode));
	    if (ttyn && strlen (ttyn) > strlen (_PATH_DEV))
	      logbtmp (ttyn + strlen (_PATH_DEV), userptr, hostname);
	    else
	      syslog (LOG_NOTICE, "btmp logging disabled, no tty found");
	    if (getdef_bool ("FAILLOG_ENAB"))
	      {
		if (pwd)
		  failure (pwd->pw_uid, tty, &faillog);
	      }
	    fprintf (stderr, _("Login incorrect\n\n"));
	    pam_set_item (pamh, PAM_USER, NULL);
	    sleep (getdef_num ("FAIL_DELAY", 1));
	  }
      } while ((failcount < retries) &&
	       ((retcode == PAM_AUTH_ERR) ||
		(retcode == PAM_USER_UNKNOWN) ||
		(retcode == PAM_CRED_INSUFFICIENT) ||
		(retcode == PAM_AUTHINFO_UNAVAIL)));

      if (retcode != PAM_SUCCESS)
	{
	  pam_end (pamh, retcode);
	  exit (0);
	}

      retcode = pam_acct_mgmt (pamh, 0);

      if (retcode == PAM_NEW_AUTHTOK_REQD)
	retcode = pam_chauthtok (pamh, PAM_CHANGE_EXPIRED_AUTHTOK);

      PAM_FAIL_CHECK;
    }

  /* Grab the user information out of the password file for future usage
   * First get the username that we are actually using, though.
   */
  retcode = pam_get_item (pamh, PAM_USER, &void_username);
  PAM_FAIL_CHECK;
  username = void_username;

  if (username == NULL || *username == '\0')
    {
      fprintf (stderr, _("\nSession setup problem, abort.\n"));
      syslog (LOG_ERR, "NULL user name in %s:%d. Abort.", __FUNCTION__, __LINE__);
      pam_end (pamh, PAM_SYSTEM_ERR);
      exit (1);
    }

  pwd = NULL; /* only to be really sure */
  while (getpwnam_r (username, &resultbuf, buffer, buflen, &pwd) != 0
         && errno == ERANGE)
    {
      errno = 0;
      buflen += 256;
      buffer = alloca (buflen);
    }

  if(!pwd)
    {
      syslog (LOG_ERR, "Failed to look up user '%s'.", username);
      pam_end (pamh, PAM_SYSTEM_ERR);
      exit (1);
    }

  if (getdef_bool ("FAILLOG_ENAB") && !failcheck (pwd->pw_uid))
    {
      printf (_("exceeded failure limit for `%s'%s\n"), pwd->pw_name, hostname);
      syslog (LOG_CRIT, "exceeded failure limit for `%s'%s\n",
	      pwd->pw_name, hostname);
      exit (1);
    }

  /* Initialize the supplementary group list. This should be done
     before pam_setcred because the PAM modules might add groups
     during pam_setcred. For root we don't call initgroups, instead
     we call setgroups with group 0. This avoids the need to step
     through the whole group file, which can cause problems if NIS,
     NIS+, LDAP or something similar is used and the machine has
     network problems.  */
  if (pwd->pw_uid == 0)
    {
      gid_t groups[2] = {pwd->pw_gid, 0};
      int ngroups = 1;

      if (setgroups (ngroups, groups) < 0)
	{
	  syslog (LOG_ERR, "setgroups: %m");
	  fprintf (stderr, _("\nSession setup problem, abort.\n"));
	  pam_end (pamh, PAM_SYSTEM_ERR);
	  exit (1);
	}
    }
  else if (initgroups (pwd->pw_name, pwd->pw_gid) < 0)
    {
      syslog (LOG_ERR, "initgroups: %m");
      fprintf (stderr, _("\nSession setup problem, abort.\n"));
      pam_end (pamh, PAM_SYSTEM_ERR);
      exit (1);
    }

  retcode = pam_setcred (pamh, PAM_ESTABLISH_CRED);
  PAM_FAIL_CHECK;

  retcode = pam_open_session (pamh, 0);
  PAM_FAIL_CHECK;

#ifdef WITH_SELINUX
  /* Make sure SELINUX is really running on this system */
  if ((selinux_enabled = is_selinux_enabled ()))
    {
      int ret;
      security_context_t* contextlist;
      int num_contexts = get_ordered_context_list (username, 0, &contextlist);
      if (num_contexts > 0)
	{
#if 0
	  ret = query_user_context (contextlist, &user_context);
	  freeconary (contextlist);

	  if (ret < 0)
	    {
	      fprintf (stderr, _("%s:  query_user_context failed\n"), argv[0]);
	      syslog (LOG_ERR, "%s:  query_user_context failed\n", argv[0]);
	      sleep (5);
	      exit (3);
	    }
#else
	  user_context = (security_context_t) strdup (contextlist[0]);
	  freeconary (contextlist);
#endif
	}
      else
	{
	  fprintf (stderr, _("login: unable to obtain context for %s.\n"),
		   username);
	  ret = manual_user_enter_context (username, &user_context);
	  if (ret < 0)
	    {
	      syslog (LOG_ERR, "UNABLE TO GET VALID CONTEXT FOR %s",
		      username);
	      sleep (5);
	      exit (3);
	    }
	}
    }
#endif

  /* committed to login -- turn off timeout */
  alarm ((unsigned int) 0);

#ifdef HAVE_QUOTA
  if (quota (Q_SETUID, pwd->pw_uid, 0, 0) < 0 && errno != EINVAL)
    {
      switch (errno)
	{
	case EUSERS:
	  fprintf (stderr,
		   _("Too many users logged on already.\nTry again later.\n"));
	  break;
	case EPROCLIM:
	  fprintf (stderr,
		   _("You have too many processes running.\n"));
	  break;
	default:
	  perror ("quota (Q_SETUID)");
	}
      sleep (5);
      exit (0);           /* %% */
    }
#endif

  /* paranoia... */
  endpwent ();

  quietlog = 0;
  if ((hushfile = getdef_str ("HUSHLOGIN_FILE")) != NULL)
    {
      /* If this is not a fully rooted path then see if the
         file exists in the user's home directory. */

      if (hushfile[0] != '/')
	{
	  /* This requires some explanation: As root we may not be able to
	     read the directory of the user if it is on an NFS mounted
	     filesystem. We temporarily set our effective uid to the user-uid
	     making sure that we keep root privs. in the real uid.

	     A portable solution would require a fork(), but we rely on Linux
	     having the BSD setreuid() */

	  char *tmpstr =
	    alloca (strlen (pwd->pw_dir) + strlen (hushfile) + 2);
	  uid_t ruid = getuid ();
	  gid_t egid = getegid ();

	  sprintf (tmpstr, "%s/%s", pwd->pw_dir, hushfile);
	  setregid (-1, pwd->pw_gid);
	  setreuid (0, pwd->pw_uid);
	  quietlog = (access (tmpstr, F_OK) == 0);
	  setuid (0);		/* setreuid doesn't do it alone! */
	  setreuid (ruid, 0);
	  setregid (-1, egid);
	}
      else
	{
	  FILE *fp;
	  char buf[BUFSIZ];

	  if ((fp = fopen (hushfile, "r")) != NULL)
	    {
	      for (quietlog = 0; !quietlog && fgets (buf, sizeof buf, fp);)
		{
		  buf[strlen (buf) - 1] = '\0';
		  quietlog = !strcmp (buf, buf[0] == '/' ?
				      pwd->pw_shell : pwd->pw_name);
		}
	      fclose (fp);
	    }
	}
    }

  /* for linux, write entries in utmp and wtmp */
  {
    struct utmp ut;
    struct utmp *utp;
    struct timeval ut_tv;
    pid_t mypid = getpid ();

    utmpname (_PATH_UTMP);
    setutent ();

    /* Find mypid in utmp.
       login sometimes overwrites the runlevel entry in /var/run/utmp,
       confusing sysvinit. I added a test for the entry type,
       and the problem was gone. (In a runlevel entry, st_pid is not
       really a pid but some number calculated from the previous and
       current runlevel).
       Michael Riepe <michael@stud.uni-hannover.de>
     */
    while ((utp = getutent ()))
      if (utp->ut_pid == mypid
	  && utp->ut_type >= INIT_PROCESS && utp->ut_type <= DEAD_PROCESS)
	break;

    /*
     * If we can't find a pre-existing entry by pid, try by line
     * BSD network daemons may rely on this.
     */
    if (utp == NULL)
      {
	endutent ();
	setutent ();
	memset (&ut, 0, sizeof (ut));
	ut.ut_type = LOGIN_PROCESS;	/* XXX doesn't matter */
	strncpy (ut.ut_id, ttyn + strlen (_PATH_TTY), sizeof (ut.ut_id));
	strncpy (ut.ut_line, ttyn + strlen (_PATH_DEV), sizeof (ut.ut_line));
	utp = getutid (&ut);
      }

    if (utp)
      memcpy (&ut, utp, sizeof (ut));
    else	    /* some gettys/telnetds don't initialize utmp... */
      memset (&ut, 0, sizeof (ut));

    if (ut.ut_id[0] == 0)
      strncpy (ut.ut_id, ttyn + strlen (_PATH_TTY), sizeof (ut.ut_id));

    strncpy (ut.ut_user, username, sizeof (ut.ut_user));
    strncpy (ut.ut_line, ttyn + strlen (_PATH_DEV), sizeof (ut.ut_line));
    ut.ut_line[sizeof (ut.ut_line) - 1] = 0;
    gettimeofday (&ut_tv, NULL);
    ut.ut_tv.tv_sec = ut_tv.tv_sec;
    ut.ut_tv.tv_usec = ut_tv.tv_usec;
    ut.ut_type = USER_PROCESS;
    ut.ut_pid = mypid;
    if (hostname)
      {
	strncpy (ut.ut_host, hostname, sizeof (ut.ut_host));
	ut.ut_host[sizeof (ut.ut_host) - 1] = 0;
	if (hostaddress && hostaddress->h_addr_list)
	  memcpy (&ut.ut_addr, hostaddress->h_addr_list[0],
		  sizeof (ut.ut_addr));
      }

    pututline (&ut);
    endutent ();

    updwtmp (_PATH_WTMP, &ut);
  }

  dolastlog (quietlog, pwd->pw_uid, tty, hostname);  /* Maybe we move this to PAM ? */

  if ((group = getdef_str ("TTYGROUP")) == NULL)
    gid = pwd->pw_gid;
  else if (group[0] >= '0' && group[0] <= '9')
    gid = atoi (group);
  else if ((gr = getgrnam (group)))
    gid = gr->gr_gid;
  else
    gid = pwd->pw_gid;

  chown (ttyn, pwd->pw_uid, gid);
  chmod (ttyn,  getdef_num ("TTYPERM", 0600));

#ifdef WITH_SELINUX
  if (selinux_enabled)
    ttyn_context = security_label_tty (ttyn, user_context);
#endif

  /* if tty is one of the VC's then change owner and mode of the
     special /dev/vcs devices as well */
  if (consoletty (0))
    {
      /* find names of Virtual Console devices */
      char *p = ttyn;
      /* find number of tty */
      while (*p && !isdigit (*p))
	++p;

      strcpy (vcsn, "/dev/vcs");
      strcat (vcsn, p);
      strcpy (vcsan, "/dev/vcsa");
      strcat (vcsan, p);
#if 0
      /*
       * Please don't add code to chown /dev/vcs* to the user logging in -
       * it's a potential security hole.  I wouldn't like the previous user
       * to hold the file descriptor open and watch my screen.  We don't
       * have the *BSD revoke() system call yet, and vhangup() only works
       * for tty devices (which vcs* is not).
       */
      chown (vcsn, pwd->pw_uid, gid);
      chown (vcsan, pwd->pw_uid, gid);
      chmod (vcsn, getdef_num ("TTYPERM", 0600));
      chmod (vcsan, getdef_num ("TTYPERM", 0600));
#endif
#ifdef WITH_SELINUX
      if (selinux_enabled)
        {
	  vcsn_context = security_label_tty (vcsn, user_context);
	  vcsan_context = security_label_tty (vcsan, user_context);
	}
#endif
    }

  setgid (pwd->pw_gid);

#ifdef HAVE_QUOTA
  quota (Q_DOWARN, pwd->pw_uid, (dev_t)-1, 0);
#endif

  if (*pwd->pw_shell == '\0')
    pwd->pw_shell = _PATH_BSHELL;

  /* preserve TERM even without -p flag */
  {
    char *ep;

    if (!((ep = getenv ("TERM")) && (termenv = strdup (ep))))
      termenv = ttytype (tty);
  }

  /* destroy environment unless user has requested preservation */
  if (!pflag)
    {
      environ = (char **) malloc (sizeof (char *));
      memset (environ, 0, sizeof (char *));
    }

  if (setenv ("HOME", pwd->pw_dir, 0) == -1)	/* legal to override */
    fputs (_("login: setenv (\"HOME\") failed: insufficient space."),
	   stderr);
  if (pwd->pw_uid)
    setenv ("PATH", getdef_str ("ENV_PATH"), 1);
  else
    setenv ("PATH", getdef_str ("ENV_ROOTPATH"), 1);

  if (setenv ("SHELL", pwd->pw_shell, 1) == -1)
    fputs (_("login: setenv (\"SHELL\") failed: insufficient space."),
	   stderr);
  if (setenv ("TERM", termenv, 1) == -1)
    fputs (_("login: setenv (\"TERM\") failed: insufficient space."),
	   stderr);

  /* mailx will give a funny error msg if you forget this one */
  /* avoid snprintf */
  if (sizeof (_PATH_MAILDIR) + strlen (pwd->pw_name) + 1 < MAXPATHLEN)
    {
      char tmp[MAXPATHLEN];

      sprintf (tmp, "%s/%s", _PATH_MAILDIR, pwd->pw_name);
      if (setenv ("MAIL", tmp, 0) == -1)
	fputs (_("login: setenv (\"MAIL\") failed: insufficient space."),
	       stderr);
    }

  /* Export the user name.  For BSD derived systems, it's "USER", for
     all others it's "LOGNAME".  We set both of them. */
  if (setenv ("USER", pwd->pw_name, 1) == -1)
    fputs (_("login: setenv (\"USER\") failed: insufficient space."),
	   stderr);
  if (setenv ("LOGNAME", pwd->pw_name, 1) == -1)
    fputs (_("login: setenv (\"LOGNAME\") failed: insufficient space."),
	   stderr);

  {
    int i;
    char **env;

    env = pam_getenvlist (pamh);

    if (env != NULL)
      {
	for (i = 0; env[i++];)
	  {
	    putenv (env[i - 1]);
	    /* D(("env[%d] = %s", i-1,env[i-1])); */
	  }
      }
  }

  setproctitle ("login", username);

#ifdef WITH_SELINUX
  if (selinux_enabled)
    {
      if (pwd->pw_uid == 0)
	{
	  if (hostname)
	    syslog (LOG_NOTICE, "ROOT LOGIN ON %s FROM %s USING %s",
		    ttyn, hostname, user_context);
	  else
	    syslog (LOG_NOTICE, "ROOT LOGIN ON %s USING %s", ttyn,
		    user_context);
	}
      else
	{
	  if (hostname)
	    syslog (LOG_INFO, "LOGIN ON %s BY %s FROM %s USING %s",
		    ttyn, pwd->pw_name, hostname, user_context);
	  else
	    syslog (LOG_INFO, "LOGIN ON %s BY %s USING %s", ttyn,
		    pwd->pw_name, user_context);
	}
    }
#endif

  if (!quietlog)
    motd ();

  signal (SIGALRM, SIG_DFL);
  signal (SIGQUIT, SIG_DFL);
  signal (SIGTSTP, SIG_IGN);

  /*
   * We must fork before setuid() because we need to call
   * pam_close_session() as root.
   *
   * Problem: if the user's shell is a shell like ash that doesnt do
   * setsid() or setpgrp(), then a ctrl-\, sending SIGQUIT to every
   * process in the pgrp, will kill us.
   */
  signal (SIGINT, SIG_IGN);
  closelog();
  signal (SIGHUP, SIG_IGN);
  ioctl(0, TIOCNOTTY, NULL);
  signal (SIGHUP, SIG_DFL);
  childPid = fork ();
  if (childPid < 0)
    {
      errsv = errno;
      /* error in fork() */
      fprintf (stderr, _("login: failure forking: %s"), strerror (errsv));
      PAM_END;
      exit (0);
    }
  else if (childPid)
    {
      /* parent - wait for child to finish, then cleanup session */
      chdir ("/");

      signal(SIGINT, SIG_IGN);
      signal(SIGQUIT, SIG_IGN);
      signal(SIGTERM, SIG_IGN);

      openlog ("login", LOG_ODELAY | LOG_PID, LOG_AUTHPRIV);

      while (waitpid (childPid, NULL, 0) == -1)
	{
	  errsv = errno;
	  if (errsv != EINTR)
	    fprintf (stderr, _("login: waitpid (%d, NULL, 0) failed: %s\n"),
		     childPid, strerror (errsv));
	}
#ifdef WITH_SELINUX
      if (selinux_enabled)
        {
          /* We need to change the contexts of the terminal devices back to
             the system when the user's session ends.  */
          security_restorelabel_tty (ttyn, ttyn_context);
          if (consoletty (0))
	    {
	      security_restorelabel_tty (vcsn, vcsn_context);
	      security_restorelabel_tty (vcsan, vcsan_context);
	    }
        }
#endif

      PAM_END;
      exit (0);
    }
  /* child */
  /* start new session */
  setsid();
  if (ioctl (0, TIOCSCTTY, (char *) 1) == -1)
    {
      errsv = errno;
      fprintf (stderr, _("login: couldn't set controlling tty: %s\n"),
	       strerror (errsv));
      exit (1);
    }

  signal (SIGINT, SIG_DFL);

  openlog ("login", LOG_ODELAY | LOG_PID, LOG_AUTHPRIV);

#ifdef WITH_SELINUX
  if (selinux_enabled)
    {
      if (setexeccon (user_context))
	{
	  fprintf (stderr,
		   _("Error!  Unable to set executable context %s.\n"),
		   user_context);
	  syslog(LOG_ERR, "Error!  Unable to set executable context %s.\n",
		 user_context);
	  exit (3);
	}
      freecon (user_context);
    }
#endif

  /* discard permissions last so can't get killed and drop core */
  if (setuid (pwd->pw_uid) < 0 && pwd->pw_uid)
    {
      syslog (LOG_ALERT, "setuid() failed");
      exit (1);
    }

  /* wait until here to change directory! */
  if (chdir (pwd->pw_dir) < 0)
    {
      printf (_("No directory %s!\n"), pwd->pw_dir);
      if (!getdef_bool ("DEFAULT_HOME"))
	exit (0);
      if (chdir ("/"))
	exit (0);
      pwd->pw_dir = "/";
      printf (_("Logging in with home = \"/\".\n"));
    }

  /* if the shell field has a space: treat it like a shell script */
  if (strchr (pwd->pw_shell, ' '))
    {
      buff = malloc (strlen (pwd->pw_shell) + 6);

      if (!buff)
	{
	  fprintf (stderr, _("login: no memory for shell script.\n"));
	  exit (0);
	}

      strcpy (buff, "exec ");
      strcat (buff, pwd->pw_shell);
      childArgv[childArgc++] = "/bin/sh";
      childArgv[childArgc++] = "-sh";
      childArgv[childArgc++] = "-c";
      childArgv[childArgc++] = buff;
    }
  else
    {
      char tbuf[MAXPATHLEN + 1];
      char *cp;

      tbuf[0] = '-';
      strncpy (tbuf + 1, ((cp = strrchr (pwd->pw_shell, '/')) ?
			  cp + 1 : pwd->pw_shell), sizeof (tbuf) - 1);
      tbuf[sizeof (tbuf) - 1] = 0;

      childArgv[childArgc++] = pwd->pw_shell;
      childArgv[childArgc++] = strdup (tbuf);
    }

  childArgv[childArgc++] = NULL;

  execvp (childArgv[0], childArgv + 1);

  errsv = errno;

  if (strcmp (childArgv[0], "/bin/sh") == 0)
    fprintf (stderr, _("login: couldn't exec shell script: %s.\n"),
	     strerror (errsv));
  else
    fprintf (stderr, _("login: no shell: %s.\n"), strerror (errsv));

  exit (0);
}
