/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _HeaderPane_h_
#define _HeaderPane_h_

#include <gtk/gtk.h>
#include <pan/data/article-cache.h>
#include <pan/data/data.h>
#include <pan/data/filter-info.h>
#include <pan/gui/action-manager.h>
#include <pan/gui/pan-tree.h>
#include <pan/gui/prefs.h>

namespace pan
{
  /**
   * Base class for functors taking a const Article& and returning bool.
   * @ingroup GUI
   */
  struct ArticleTester {
    virtual ~ArticleTester() {}
    virtual bool operator()(const Article&) const = 0;
  };

  /**
   * Base class for actions performed on a header pane row.
   */
  struct RowActionFunctor {
    virtual ~RowActionFunctor () {}
    virtual void operator() (GtkTreeModel* model, GtkTreeIter* iter, const Article& a) = 0;
  };

  /**
   * Header Pane in the main window of Pan's GUI.
   * @ingroup GUI
   */
  class HeaderPane:
    private Data::Listener,
    private Data::ArticleTree::Listener,
    private Prefs::Listener
  {
    public:
      HeaderPane (ActionManager&, Data& data, const ArticleCache&, Prefs&);
      ~HeaderPane ();

    public:
      void refilter ();
      void set_show_type (const Data::ShowType);

    public:
      void set_focus ();
      void unselect_all ();
      void select_all ();
      void select_threads ();
      void select_subthreads ();
      void select_similar ();

    public:
      void read_next_article ();
      void read_next_unread_article ();
      void read_next_thread ();
      void read_next_unread_thread ();
      void read_previous_article ();
      void read_previous_thread ();
      void read_parent_article ();

    private:
      void action_next_if (const ArticleTester& test, RowActionFunctor& action);
      void read_next_if (const ArticleTester&);
      void read_prev_if (const ArticleTester&);

    public:
      GtkWidget* root () { return _root; }
      GtkWidget* create_filter_entry ();
      const Article* get_first_selected_article () const;
      std::set<const Article*> get_full_selection () const;
      std::set<const Article*> get_nested_selection () const;
      bool set_group (const Quark& group);
      const Quark& get_group () { return _group; }

    private:
      void rebuild ();

    private:
      virtual void on_group_read (const Quark& group);

    private:
      virtual void on_tree_change (const Data::ArticleTree::Diffs&);

    private:
      virtual void on_prefs_flag_changed (const StringView& key, bool value);
      virtual void on_prefs_int_changed (const StringView& key, int value) { }
      virtual void on_prefs_string_changed (const StringView& key, const StringView& value);
      virtual void on_prefs_color_changed (const StringView& key, const GdkColor& color) {}

    public: // pretend it's private
      ActionManager& _action_manager;

    private:
      Data& _data;
      Prefs& _prefs;
      Quark _group;
      Data::ArticleTree * _atree;
      GtkWidget * _root;
      GtkWidget * _tree_view;
      PanTreeStore * _tree_store;
      FilterInfo _filter;
      Data::ShowType _show_type;

    private:
      void rebuild_filter (const std::string&, int);
      void refresh_font ();

    public: // public so that anonymous namespace can reach -- don't call
      void filter (const std::string& text, int mode);
      static void do_popup_menu (GtkWidget*, GdkEventButton*, gpointer);
      static void on_row_activated (GtkTreeView*, GtkTreePath*, GtkTreeViewColumn*, gpointer);
      static gboolean on_button_pressed (GtkWidget*, GdkEventButton*, gpointer);
      const ArticleCache& _cache;

    private:
      void get_nested_foreach (GtkTreeModel*, GtkTreePath*, GtkTreeIter*, gpointer) const;
      static void get_nested_foreach_static (GtkTreeModel*, GtkTreePath*, GtkTreeIter*, gpointer);
  };
}

#endif
