/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <iostream>
#include <algorithm>
#include <cstdarg>
#include <gobject/gvaluecollector.h>
#include <pan/general/debug.h>
#include "pan-tree.h"

#define IS_SORTED(tree) \
  (tree->sort_column_id != GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID)

struct PanTreeStore :: Node
{
  /** Pointer to the parent node. */
  Node * parent;

  int n_children () const { return (int) children.size(); }

  /** Pointers to the child nodes. */
  std::vector<Node*> children;

  Node* nth_child (int n) {
    Node * ret (0);
    if (0<=n && n<(int)children.size())
      ret = children[n];
    return ret;
  }

  /** This node's index in it's parent's `children' vector.
      Used for fast building of GtkTreePath. */
  int child_index;

  /** This row's values, created by the client and given to us via insert_n() */
  Row * values;

  Node(): parent(0), child_index(-1), values(0) {}
  ~Node() { }
};

/***
****  Row Helper Functions
***/

void
PanTreeStore :: Row :: set_value_int (GValue * setme, int value)
{
  g_value_init (setme, G_TYPE_INT);
  g_value_set_int (setme, value);
}

void
PanTreeStore :: Row :: set_value_pointer (GValue * setme, gpointer value)
{
  g_value_init (setme, G_TYPE_POINTER);
  g_value_set_pointer (setme, value);
}

void
PanTreeStore :: Row :: set_value_ulong (GValue * setme, unsigned long value)
{
  g_value_init (setme, G_TYPE_ULONG);
  g_value_set_ulong (setme, value);
}

void
PanTreeStore :: Row :: set_value_string (GValue * setme, const char * value)
{
  g_value_init (setme, G_TYPE_STRING);
  g_value_set_string (setme, value);
}

void
PanTreeStore :: Row :: set_value_static_string (GValue * setme, const char * value)
{
  g_value_init (setme, G_TYPE_STRING);
  g_value_set_static_string (setme, value);
}


/***
****  Low-level utilities
***/

PanTreeStore :: Node*
PanTreeStore :: get_node (GtkTreeIter * iter)
{
  g_assert (iter);
  g_assert (iter->stamp == stamp);
  Node * n = static_cast<Node*>(iter->user_data);
  g_assert (n);
  return n;
}

PanTreeStore :: Row*
PanTreeStore :: get_row (GtkTreeIter * iter)
{
  return get_node(iter)->values;
}

const PanTreeStore :: Node*
PanTreeStore :: get_node (const GtkTreeIter * iter) const
{
  g_assert (iter);
  g_assert (iter->stamp == stamp);
  const Node * n = static_cast<const Node*>(iter->user_data);
  g_assert (n);
  return n;
}

const PanTreeStore :: Row*
PanTreeStore :: get_row (GtkTreeIter * iter) const
{
  return get_node(iter)->values;
}

void
PanTreeStore :: set_iter (GtkTreeIter * iter,
                          Node        * node)
{
  iter->stamp = stamp;
  iter->user_data = node;
  iter->user_data2 = 0;
  iter->user_data3 = 0;
}

void
PanTreeStore :: invalidate_iter (GtkTreeIter * iter)
{
  iter->stamp      = 0;
  iter->user_data  = (void*) 0xDEADBEEF;
  iter->user_data2 = (void*) 0xDEADBEEF;
  iter->user_data2 = (void*) 0xDEADBEEF;
}

/*****
******
******  implementing GtkTreeModel's interface
******
*****/

GtkTreeModelFlags
PanTreeStore :: model_get_flags (GtkTreeModel *tree_model)
{
  return GTK_TREE_MODEL_ITERS_PERSIST;
}

gint
PanTreeStore :: model_get_n_columns (GtkTreeModel * model)
{
  const PanTreeStore * store (PAN_TREE_STORE(model));
  g_return_val_if_fail (store, 0);
  return store->n_columns;
}

GType
PanTreeStore :: model_get_column_type (GtkTreeModel * tree,
                                       gint           n)
{
  return (*((PanTreeStore*)(tree))->column_types)[n];
}

gboolean
PanTreeStore :: model_get_iter (GtkTreeModel * model,
                                GtkTreeIter  * iter,
                                GtkTreePath  * path)
{
  PanTreeStore * store = PAN_TREE_STORE(model);
  g_return_val_if_fail (store, false);
  g_return_val_if_fail (path, false);

  // find the Node that corresponds to this path.
  PanTreeStore::Node * node (store->root);
  const int depth (gtk_tree_path_get_depth (path));
  const int * indices = gtk_tree_path_get_indices (path);
  for (int i=0; i<depth; ++i) {
    node = node->nth_child (*indices++);
    if (!node)
      return false;
  }

  // build an iter from that node.
  store->set_iter (iter, node);
  return true;
}

GtkTreePath*
PanTreeStore :: model_get_path (GtkTreeModel * model,
                                GtkTreeIter  * iter)
{
  PanTreeStore * store (PAN_TREE_STORE(model));
  g_return_val_if_fail (store, 0);
  return store->get_path (iter);
}

void
PanTreeStore :: model_get_value (GtkTreeModel * model,
                                 GtkTreeIter  * iter,
                                 gint           column,
                                 GValue       * dest_value)
{
  g_assert (iter);
  g_assert (dest_value);
  PanTreeStore * store = PAN_TREE_STORE(model);
  g_assert (store);
  g_assert (iter->stamp == store->stamp);
  g_assert (0<=column && column<store->n_columns);

  store->get_node(iter)->values->get_value (column, dest_value);
}

gboolean
PanTreeStore :: model_iter_next (GtkTreeModel * model,
                                 GtkTreeIter  * iter)
{
  PanTreeStore * tree = PAN_TREE_STORE(model);
  g_return_val_if_fail (tree, false);
  g_return_val_if_fail (iter, false);
  g_return_val_if_fail (iter->stamp == tree->stamp, false);

  PanTreeStore::Node * node (tree->get_node (iter));
  node = node->parent->nth_child (node->child_index + 1);

  if (node)
    tree->set_iter (iter, node);
  else
    invalidate_iter (iter);

  return node != 0;
}

gboolean
PanTreeStore :: model_iter_children (GtkTreeModel * model,
                                     GtkTreeIter  * iter,
                                     GtkTreeIter  * parent)
{
  return model_iter_nth_child (model, iter, parent, 0);
}

gint
PanTreeStore :: model_iter_n_children (GtkTreeModel * model,
                                       GtkTreeIter  * iter)
{
  PanTreeStore * tree = PAN_TREE_STORE(model);
  g_return_val_if_fail (tree, 0);
  g_return_val_if_fail (!iter || iter->stamp == tree->stamp, 0);

  const PanTreeStore::Node * node (iter ? tree->get_node(iter) : tree->root);
  return node->n_children();
}

gboolean
PanTreeStore :: model_iter_has_child (GtkTreeModel *model,
                                      GtkTreeIter  *iter)
{
  return model_iter_n_children (model, iter) != 0;
}

gboolean
PanTreeStore :: model_iter_nth_child (GtkTreeModel * model,
                                      GtkTreeIter  * iter,
                                      GtkTreeIter  * parent,
                                      gint           n)
{
  PanTreeStore * tree = PAN_TREE_STORE(model);
  g_return_val_if_fail (tree, false);
  g_return_val_if_fail (iter, false);
  g_return_val_if_fail (!parent || parent->stamp == tree->stamp, false);

  PanTreeStore::Node * node (parent ? tree->get_node(parent) : tree->root);
  node = node->nth_child (n);

  if (node)
    tree->set_iter (iter, node);
  else
    invalidate_iter (iter);

  return node != 0;
}

gboolean
PanTreeStore :: model_iter_parent (GtkTreeModel * model,
                                   GtkTreeIter  * iter,
                                   GtkTreeIter  * child)
{
  return PAN_TREE_STORE(model)->get_parent (iter, child);
}

/*****
******
*****/

GtkTreePath*
PanTreeStore :: get_path (GtkTreeIter  * iter)
{
  g_return_val_if_fail (iter, false);
  g_return_val_if_fail (iter->stamp == stamp, false);

  GtkTreePath * path = gtk_tree_path_new ();
  PanTreeStore::Node * node (get_node (iter));
  std::vector<int> indices;
  while (node && node!=root) {
    indices.push_back (node->child_index);
    node = node->parent;
  }
  for (std::vector<int>::const_reverse_iterator it(indices.rbegin()), end(indices.rend()); it!=end; ++it)
    gtk_tree_path_append_index (path, *it);

  return path;
}

bool
PanTreeStore :: get_parent (GtkTreeIter * iter,
                            GtkTreeIter * child)
{
  g_return_val_if_fail (child, false);
  g_return_val_if_fail (iter, false);
  g_return_val_if_fail (child->stamp == stamp, false);

  const PanTreeStore::Node * node = get_node (child);
  const bool has_parent = node->parent != root;
  if (has_parent)
    set_iter (iter, node->parent);
  else
    invalidate_iter (iter);
  return has_parent;
}

bool
PanTreeStore :: is_root (const GtkTreeIter* iter) const
{
  return iter && get_node(iter)->parent==root;
}



/*****
******
******  implementing GtkTreeSortable
******
*****/

gboolean
PanTreeStore :: sortable_get_sort_column_id (GtkTreeSortable  * sortable,
                                             gint             * sort_column_id,
                                             GtkSortType      * order)
{
  PanTreeStore * tree (PAN_TREE_STORE (sortable));
  g_return_val_if_fail (tree, false);

  if (sort_column_id)
     *sort_column_id = tree->sort_column_id;
  if (order)
     *order = tree->order;

  return (tree->sort_column_id != GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID)
      && (tree->sort_column_id != GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID);
}

gboolean
PanTreeStore :: sortable_has_sort_func (GtkTreeSortable *sortable, gint col)
{
  PanTreeStore * tree (PAN_TREE_STORE (sortable));
  g_return_val_if_fail (tree, false);
  std::map<int,PanTreeStore::SortInfo>::const_iterator it =
    tree->sort_info->find (col);
  return (it!=tree->sort_info->end()) && (it->second.sort_func!=0);
}

struct
PanTreeStore :: SortTuple
{
  int offset;
  PanTreeStore::Node * node;
  GtkTreeIter iter;
};

struct PanTreeStore :: SortHelperData
{
  PanTreeStore * store;
  GtkTreeModel * model;
  PanTreeStore::SortInfo& sort_info;
  GtkSortType order;
  SortHelperData (PanTreeStore           * tree_store,
                  PanTreeStore::SortInfo & s,
                  GtkSortType              o):
    store(tree_store), model(GTK_TREE_MODEL(tree_store)), sort_info(s), order(o) {}
};

int
PanTreeStore :: tuple_compare_func (gconstpointer a_gpointer,
                                    gconstpointer b_gpointer,
                                    gpointer user_data)
{
  const SortTuple * a = (const SortTuple*) a_gpointer;
  const SortTuple * b = (const SortTuple*) b_gpointer;
  const SortHelperData * help = (const SortHelperData*) user_data;
  int val = (help->sort_info.sort_func) (help->model,
                                         const_cast<GtkTreeIter*>(&a->iter),
                                         const_cast<GtkTreeIter*>(&b->iter),
                                         help->sort_info.user_data);
  if (help->order == GTK_SORT_DESCENDING)
    val = -val;

  if (!val) // inplace sort
    val = help->store->get_node(const_cast<GtkTreeIter*>(&a->iter))->child_index - 
          help->store->get_node(const_cast<GtkTreeIter*>(&b->iter))->child_index;

  return val;
}

void
PanTreeStore :: renumber_children (Node * parent,
                                   int child_lo,
                                   int child_hi)
{
  const int n_children (parent->n_children());
  child_hi = std::min (child_hi, n_children);

  g_assert (parent);
  g_assert (child_lo >= 0);
  g_assert (child_lo <= child_hi);
  g_assert (child_hi <= n_children);

  Node ** it = &parent->children[child_lo];
  for (int i=child_lo; i!=child_hi; ++i, ++it) {
    (*it)->child_index = i;
    (*it)->parent = parent;
  }

#if 0
  // just for testing.. these lines slow things down
  // and will be commented out
  for (int i=0; i<n_children; ++i) {
    g_assert (parent->children[i]->child_index == i);
    g_assert (parent->children[i]->parent == parent);
  }
#endif
}


void
PanTreeStore :: sort_helper (SortInfo  & sort_info,
                             Node      * parent,
                             bool        recurse)
{
  g_assert (parent);

  const int n (parent->n_children());
  if (n < 2) // no need to sort [0...1] children
    return;

  // build a temporary array to sort
  SortTuple * sort_array = new SortTuple [n];
  for (int i=0; i<n; ++i) {
    SortTuple& tuple = sort_array[i];
    tuple.offset = i;
    tuple.node = parent->children[i];
    set_iter (&tuple.iter, tuple.node);
  }

  // sort 'em...
  SortHelperData help (this, sort_info, order);
  g_qsort_with_data (sort_array, n,
                     sizeof(SortTuple),
                     tuple_compare_func,
                     &help);

  // update the child indices...
  for (int i=0; i<n; ++i) {
    Node * child = sort_array[i].node;
    child->child_index = i;
    parent->children[i] = child;
    g_assert (child->parent == parent);
  }

  // let the world know we've changed
  int * new_order (new int [n]);
  bool differ (false);
  for (int i=0; i<n; ++i) {
    differ |= (i != sort_array[i].offset);
    new_order[i] = sort_array[i].offset;
  }

  // tell the world...
  GtkTreeIter tmp_iter, *parent_iter;
  GtkTreePath * path;
  if (parent == root) {
    parent_iter = 0;
    path = gtk_tree_path_new ();
  } else {
    set_iter (&tmp_iter, parent);
    parent_iter = &tmp_iter;
    path = gtk_tree_model_get_path (help.model, parent_iter);
  }
  gtk_tree_model_rows_reordered (help.model, path, parent_iter, new_order);
  gtk_tree_path_free (path);
  delete [] new_order;

  delete [] sort_array;

  for (int i=0; recurse && i<n; ++i) {
    Node * child = parent->children[i];
    sort_helper (sort_info, child, recurse);
  }
}

void
PanTreeStore :: sort ()
{
  if (!sort_paused && (sort_column_id != GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID))
  {
    g_assert (sortable_has_sort_func (GTK_TREE_SORTABLE(this), sort_column_id));
    SortInfo& info = (*sort_info)[sort_column_id];
    sort_helper (info, root, true);
  }
}

void
PanTreeStore :: sortable_set_sort_column_id (GtkTreeSortable * sortable,
                                             gint              sort_column_id,
                                             GtkSortType       order)
{
  PanTreeStore * tree (PAN_TREE_STORE (sortable));
  g_return_if_fail (tree);

  // if no change, there's nothing to do...
  if (tree->sort_column_id==sort_column_id && tree->order==order)
    return;

  // sanity checks...
  if (sort_column_id != GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID) {
    g_return_if_fail (tree->sort_info->count(sort_column_id) != 0);
    g_return_if_fail (tree->sort_info->find(sort_column_id)->second.sort_func != 0);
  }

  tree->sort_paused = 0;
  tree->sort_column_id = sort_column_id;
  tree->order = order;
  gtk_tree_sortable_sort_column_changed (sortable);
  tree->sort ();
}

gboolean
PanTreeStore :: sortable_has_default_sort_func (GtkTreeSortable *sortable)
{
  return sortable_has_sort_func (sortable, GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID);
}

void
PanTreeStore :: sortable_set_sort_func (GtkTreeSortable        *sortable,
                                        gint                    col,
                                        GtkTreeIterCompareFunc  func,
                                        gpointer                data,
                                        GtkDestroyNotify        destroy)
{
  PanTreeStore * tree (PAN_TREE_STORE (sortable));
  g_return_if_fail (tree);

  (*tree->sort_info)[col].assign (func, data, destroy);

  if (tree->sort_column_id == col)
    tree->sort ();
}

void
PanTreeStore :: sortable_set_default_sort_func (GtkTreeSortable        * s,
                                                GtkTreeIterCompareFunc   f,
                                                gpointer                 p,
                                                GtkDestroyNotify         d)
{
  sortable_set_sort_func (s, GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID, f, p, d);
}

namespace
{
  GObjectClass *parent_class (0);

  typedef std::vector<GValue> values_t;
}

struct
PanTreeStore :: FreeNodeWalker: public PanTreeStore::WalkFunctor
{
  PanTreeStore * store;
  FreeNodeWalker (PanTreeStore *t): store(t) {}
  virtual ~FreeNodeWalker () {}
  virtual bool operator()(PanTreeStore *tree, PanTreeStore::Row *row, GtkTreeIter*it, GtkTreePath *p) {
    delete row;
    delete store->get_node(it);
    return true; // keep marching
  }
};

void
PanTreeStore :: pan_tree_finalize (GObject *object)
{
  // erase the remaining nodes
  PanTreeStore * store = PAN_TREE_STORE(object);
  FreeNodeWalker walk (store);
  store->postfix_walk (walk);
  delete store->root;
   
  delete store->column_types;
  delete store->sort_info;

  (*parent_class->finalize) (object);
}

void
PanTreeStore :: pan_tree_class_init (PanTreeStoreClass *klass)
{
  GObjectClass *object_class;

  parent_class = (GObjectClass*) g_type_class_peek_parent (klass);
  object_class = (GObjectClass*) klass;

  object_class->finalize = pan_tree_finalize;
}

void
PanTreeStore :: pan_tree_model_init (GtkTreeModelIface *iface)
{
  iface->get_flags       = model_get_flags;
  iface->get_n_columns   = model_get_n_columns;
  iface->get_column_type = model_get_column_type;
  iface->get_iter        = model_get_iter;
  iface->get_path        = model_get_path;
  iface->get_value       = model_get_value;
  iface->iter_next       = model_iter_next;
  iface->iter_children   = model_iter_children;
  iface->iter_has_child  = model_iter_has_child;
  iface->iter_n_children = model_iter_n_children;
  iface->iter_nth_child  = model_iter_nth_child;
  iface->iter_parent     = model_iter_parent;
}

void
PanTreeStore :: pan_tree_sortable_init (GtkTreeSortableIface *iface)
{
  iface->get_sort_column_id    = sortable_get_sort_column_id;
  iface->set_sort_column_id    = sortable_set_sort_column_id;
  iface->set_sort_func         = sortable_set_sort_func;
  iface->set_default_sort_func = sortable_set_default_sort_func;
  iface->has_default_sort_func = sortable_has_default_sort_func;
}

void
PanTreeStore :: pan_tree_init (PanTreeStore * tree)
{
  tree->stamp = g_random_int();
  tree->n_columns = 0;
  tree->root = new PanTreeStore::Node ();
  tree->column_types = new std::vector<GType>();
  tree->sort_paused = 0;
  tree->sort_info = new std::map<int,PanTreeStore::SortInfo>();
  tree->sort_column_id = GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID;
  tree->order = GTK_SORT_ASCENDING;
}

void
PanTreeStore :: alloc_nodes (size_t               n_rows,
                             std::vector<Node*> & setme)
{
  setme.resize (n_rows);
  for (std::vector<Node*>::iterator it(setme.begin()), end(setme.end()); it!=end; ++it)
    *it = new Node ();
}

/***
****
***/

GType
PanTreeStore :: get_type ()
{
  static GType pan_tree_type (0);
  if (pan_tree_type)
    return pan_tree_type;

  static const GTypeInfo pan_tree_info = {
    sizeof (PanTreeStoreClass),
    NULL, // base_init
    NULL, // base_finalize
    (GClassInitFunc) pan_tree_class_init,
    NULL, // class finalize
    NULL, // class_data
    sizeof (PanTreeStore),
    0, // n_preallocs
    (GInstanceInitFunc) pan_tree_init
  };

  pan_tree_type = g_type_register_static (
    G_TYPE_OBJECT, "PanTreeStore", &pan_tree_info, (GTypeFlags)0);
  static const GInterfaceInfo tree_model_info =
    { (GInterfaceInitFunc)pan_tree_model_init, 0, 0 };
  g_type_add_interface_static (
    pan_tree_type, GTK_TYPE_TREE_MODEL, &tree_model_info);
  static const GInterfaceInfo sortable_info =
    { (GInterfaceInitFunc)pan_tree_sortable_init, 0, 0 };
  g_type_add_interface_static (
    pan_tree_type, GTK_TYPE_TREE_SORTABLE, &sortable_info);
  return pan_tree_type;
}

PanTreeStore*
PanTreeStore :: new_tree (int n_columns, ...)
{
  g_return_val_if_fail (n_columns>0, 0);
  PanTreeStore* tree = (PanTreeStore*) g_object_new (PAN_TREE_STORE_TYPE, NULL);

  va_list args;
  va_start (args, n_columns);
  for (int i=0; i<n_columns; ++i) {
    const GType type = va_arg (args, GType);
    tree->column_types->push_back (type);
  }
  va_end (args);
  tree->n_columns = tree->column_types->size();

  return tree;
}

/***
****
***/

void
PanTreeStore :: row_changed (GtkTreeIter * iter)
{
  GtkTreeModel * model = GTK_TREE_MODEL(this);
  GtkTreePath * path = gtk_tree_model_get_path (model, iter);
  gtk_tree_model_row_changed (model, path, iter);
  gtk_tree_path_free (path);
}

/***
****
***/

typedef std::vector<PanTreeStore::Row*> rows_t;

void
pan_tree_store_insert     (PanTreeStore                     * tree,
                           GtkTreeIter                      * iter,
                           GtkTreeIter                      * parent,
                           PanTreeStore::Row                * row,
                           int                                pos)
{
  rows_t rows;
  rows.push_back (row);
  pan_tree_store_insert_n (tree, iter, parent, rows, pos);
}

void
pan_tree_store_insert_n  (PanTreeStore                    * tree,
                          GtkTreeIter                     * iter,
                          GtkTreeIter                     * parent_it,
                          rows_t                          & rows,
                          int                               position)
{
  tree->insert_n (iter, parent_it, rows, position);
}

void
PanTreeStore :: insert_n (GtkTreeIter                     * iter,
                          GtkTreeIter                     * parent_it,
                          rows_t                          & rows,
                          int                               position)
{
  g_return_if_fail (!rows.empty());
  g_return_if_fail (position >= 0);

  // allocate the nodes
  std::vector<PanTreeStore::Node*> new_children;
  const size_t n_rows (rows.size());
  alloc_nodes (n_rows, new_children);
  for (size_t i(0), count(n_rows); i!=count; ++i)
    new_children[i]->values = rows[i];

  // insert the nodes
  PanTreeStore::Node * parent_node (parent_it ? get_node (parent_it) : root);
  const int old_size (parent_node->n_children());
  position = std::min (position, old_size);
  parent_node->children.insert (parent_node->children.begin()+position,
                                new_children.begin(),
                                new_children.end());
  renumber_children (parent_node, position);

  // set the return iter
  GtkTreeIter walk;
  set_iter (&walk, parent_node->children[position]);
  if (iter)
    *iter = walk;

  // emit the 'row inserted' signals
  GtkTreePath * path = get_path (&walk);
  GtkTreeModel * model = GTK_TREE_MODEL(this);
  PanTreeStore::Node** it = &parent_node->children[position];
  for (size_t i=0; i<n_rows; ++i) {
    PanTreeStore::Node * node (*it++);
    set_iter (&walk, node);
    gtk_tree_model_row_inserted (model, path, &walk);
    gtk_tree_path_next (path);
  }

  // maybe emit the 'row has childed' signal
  if (!old_size && parent_node!=root) {
    gtk_tree_path_up (path);
    gtk_tree_model_row_has_child_toggled (model, path, parent_it);
  }

  // cleanup
  gtk_tree_path_free (path);
}

void
pan_tree_store_append  (PanTreeStore       * tree,
                        GtkTreeIter        * iter,
                        GtkTreeIter        * parent,
                        PanTreeStore::Row  * row)
{
  rows_t rows;
  rows.push_back (row);
  pan_tree_store_append_n (tree, iter, parent, rows);
}

void
pan_tree_store_append_n  (PanTreeStore  * tree,
                          GtkTreeIter   * iter,
                          GtkTreeIter   * parent,
                          rows_t        & rows)
{
  pan_tree_store_insert_n (tree, iter, parent, rows, INT_MAX);
}

/***
****
***/


struct PanTreeStore::ReparentWalker: public PanTreeStore::WalkFunctor
{
  GtkTreeModel * model;

  ReparentWalker (PanTreeStore *store): model(GTK_TREE_MODEL(store)) {}

  virtual ~ReparentWalker () {}

  virtual bool operator()(PanTreeStore *store, Row* row,
                          GtkTreeIter *iter, GtkTreePath *path)
  {
    Node * node (store->get_node (iter));
    const int n_children (node->n_children());

    if (n_children)
    {
      // fire a "row inserted" for each child
      GtkTreePath * cpath (gtk_tree_path_copy (path));
      gtk_tree_path_down (cpath);
      for (int i=0; i<n_children; ++i) {
        Node * child (node->nth_child (i));
        GtkTreeIter citer;
        store->set_iter (&citer, child);
        gtk_tree_model_row_inserted (model, cpath, &citer);
        gtk_tree_path_next (cpath);
      }
      gtk_tree_path_free (cpath);

      // fire a "has children toggled"
      gtk_tree_model_row_has_child_toggled (model, path, iter);
    }

    return true; // keep marching
  }
};

void
PanTreeStore :: reparent (GtkTreeIter    * parent_iter,
                          GtkTreeIter    * iter,
                          int              position)
{
  g_return_if_fail (iter != 0);

  GtkTreeModel * model (GTK_TREE_MODEL(this));

  Node * node (get_node (iter));
  Node * old_parent (node->parent);
  Node * new_parent (parent_iter ? get_node(parent_iter) : root);
  const int new_parent_old_n_children (new_parent->n_children());

  // remove our subtree's toplevel from its old parent
  remove_siblings (&node, 1, false);

  // if this was the old parent's last child, fire a has-child-toggled event
  if (!old_parent->n_children()) {
    GtkTreeIter tmp;
    set_iter (&tmp, old_parent);
    GtkTreePath * path (get_path (&tmp));
    gtk_tree_model_row_has_child_toggled (model, path, &tmp);
    gtk_tree_path_free (path);
  }

  // add the subtree's toplevel to its new parent node
  position = std::min (position, new_parent_old_n_children);
  new_parent->children.insert (new_parent->children.begin()+position, node);
  renumber_children (new_parent, position);

  // emit a row-inserted for iter
  GtkTreePath * path (get_path (iter));
  gtk_tree_model_row_inserted (model, path, iter);
  gtk_tree_path_free (path);

  // this emits all the row-inserted and has-child-toggled signals EXCEPT
  // for iter's row-inserted (above) and parent-iter's has-child-toggled (below).
  // It's kind of kludgy but gets all the signals emitted in the right sequence.
  ReparentWalker walk (this);
  prefix_walk (walk, iter, true);

  // if this was the new parent's first child, fire a has-child-toggled event
  if (!new_parent_old_n_children) {
    GtkTreePath * path (get_path (parent_iter));
    gtk_tree_model_row_has_child_toggled (model, path, parent_iter);
    gtk_tree_path_free (path);
  }
}

/***
****
***/

struct PanTreeStore :: NodeCompare
{
  Node * root;
  NodeCompare (Node * r): root(r) {}

  int depth (const Node * node) const {
    int depth (0);
    while (node->parent != root) {
      ++depth;
      node = node->parent;
    }
    return depth;
  }

  // this sort function is very particular...
  // we go deepest first, so the first batches don't invalidate later ones.
  // we group by parent so to get groups of siblings for remove_siblings().
  // we sort by child_index so the first siblings removed in
  // remove_siblings() don't invalidate the tree path it uses.
  bool operator() (const Node* a, const Node* b) const {
    // deepest first
    const int a_depth (depth (a));
    const int b_depth (depth (b));
    if (a_depth != b_depth)
      return b_depth < a_depth;
    // group by parent
    if (a->parent != b->parent)
      return a->parent < b->parent;
    // from last child to first
    return b->child_index < a->child_index;
  }
};

void
PanTreeStore :: remove (GtkTreeIter * iters,
                        size_t        n_iters)
{
  g_return_if_fail (iters);
  g_return_if_fail (n_iters);

  Node ** nodes = new Node* [n_iters];
  for (size_t i=0; i<n_iters; ++i)
    nodes[i] = get_node (iters+i);

  NodeCompare compare (root);
  std::sort (nodes, nodes+n_iters, compare);

  Node * parent (0);
  Node ** children (0);
  for (Node **nit=nodes, **nend=nodes+n_iters; nit!=nend; ++nit) {
    if (parent && ((*nit)->parent != parent)) {
      remove_siblings (children, nit-children, true);
      parent = 0;
    }
    if (!parent) {
      parent = (*nit)->parent;
      g_assert (parent);
      children = nit;
    }
  }
  if (parent) {
    size_t n = (nodes + n_iters) - children;
    remove_siblings (children, n, true);
  }

  delete [] nodes;
}

struct PanTreeStore::ClearWalker: public PanTreeStore::WalkFunctor
{
  virtual ~ClearWalker () {}

  void clear_children (PanTreeStore* store, Node * node)
  {
    if (node->n_children()) {
      std::vector<Node*> children = node->children;
      std::reverse (children.begin(), children.end());
      store->remove_siblings (&children.front(), children.size(), true);
    }
  }

  virtual bool operator()(PanTreeStore* store,
                          PanTreeStore::Row* row,
                          GtkTreeIter* iter, GtkTreePath* path)
  {
    clear_children (store, store->get_node(iter));
    return true;
  }
};

void
PanTreeStore :: clear ()
{
  ClearWalker walker;
  postfix_walk (walker, 0, false);
  walker.clear_children (this, root);
}


// note that the siblings passed in here must
// be sorted from highest child index to lowest child index
void
PanTreeStore :: remove_siblings (Node ** siblings,
                                 size_t n_siblings,
                                 bool free_memory)
{
  // entry assertions
  g_assert (siblings);
  g_assert (n_siblings);
  Node * parent = siblings[0]->parent;
  for (Node **nit(siblings), **nend(nit+n_siblings); nit!=nend; ++nit) {
    g_assert ((*nit)->parent == parent); // all are siblings
    g_assert (nit==siblings || ((*nit)->child_index < nit[-1]->child_index)); // sorted from high to low child index
  }

  // get a path to the first sibilng.
  // we'll reuse it for all siblings.
  GtkTreeIter iter;
  set_iter (&iter, siblings[0]);
  GtkTreeModel * model (GTK_TREE_MODEL(this));
  GtkTreePath * path = gtk_tree_model_get_path (model, &iter);
  int * last_path_index = gtk_tree_path_get_indices (path);
  last_path_index += gtk_tree_path_get_depth(path)-1;

  // walk through all the siblings and unthread them.
  for (Node** nit(siblings), **nend(siblings+n_siblings); nit!=nend; ++nit)
  {
    Node * node (*nit);
    g_assert (node->parent);
    g_assert (node->parent == parent);

    // unthread the node
    const int pos = node->child_index;
    parent->children.erase (parent->children.begin() + pos);

    // let the world know the row was removed
    *last_path_index = node->child_index;
    gtk_tree_model_row_deleted (model, path);

    // free `node' and all its descendants
    if (free_memory) {
      FreeNodeWalker walk (this);
      GtkTreeIter iter;
      set_iter (&iter, *nit);
      postfix_walk (walk, &iter);
    }
  }

  // renumber the surviving children
  renumber_children (parent);

  // if the parent's has-children state has changed, let the world know that too
  if (parent!=root && !parent->n_children()) {
    GtkTreeIter pit;
    set_iter (&pit, parent);
    gtk_tree_path_up (path);
    gtk_tree_model_row_has_child_toggled (model, path, &pit);
  }

  gtk_tree_path_free (path);
}

/***
****
***/

bool
PanTreeStore :: walk_helper (int           walk_mode,
                             Node        * node,
                             GtkTreePath * path,
                             WalkFunctor & walker)
{
  g_assert (node);
  g_assert (walk_mode==WALK_PREFIX || walk_mode==WALK_POSTFIX);

  bool more (true);

  GtkTreeIter iter;
  set_iter (&iter, node);

  if (more && node!=root && walk_mode==WALK_PREFIX)
    more = walker (this, node->values, &iter, path);

  const size_t n_children (node->n_children());
  if (more && n_children) {
    if (path)
      gtk_tree_path_append_index (path, 0);
    for (Node ** it(&node->children.front()),
              ** end(it+n_children); more && it!=end; ++it) {
      more = walk_helper (walk_mode, *it, path, walker);
      if (path)
        gtk_tree_path_next (path);
    }
    if (path)
      gtk_tree_path_up (path);
  }

  if (more && node!=root && walk_mode==WALK_POSTFIX)
    more = walker (this, node->values, &iter, path);

  return more;
}

void
PanTreeStore :: walk (int             walk_mode,
                      WalkFunctor   & walker,
                      GtkTreeIter   * top,
                      bool            need_path)
{
  GtkTreePath * path (0);
  if (need_path)
    path = top ? get_path(top) : gtk_tree_path_new();

  PanTreeStore::Node * node (top ? get_node(top) : root);
  walk_helper (walk_mode, node, path, walker);
  gtk_tree_path_free (path);
}

void
PanTreeStore :: prefix_walk (WalkFunctor   & walker,
                             GtkTreeIter   * top,
                             bool            need_path)
{
  walk (WALK_PREFIX, walker, top, need_path);
}

void
PanTreeStore :: postfix_walk (WalkFunctor   & walker,
                              GtkTreeIter   * top,
                              bool            need_path)
{
  walk (WALK_POSTFIX, walker, top, need_path);
}
