/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <cassert>
#include <pan/general/debug.h>
#include <pan/general/foreach.h>
#include <pan/general/quark.h>
#include <pan/data/article.h>
#include <pan/data/filter-info.h>
#include "article-filter.h"
#include "data-impl.h"

using namespace pan;

/****
*****  ArticleTree functions
****/

void
DataImpl :: MyTree :: get_children (const Quark& mid, articles_t & setme) const
{
  if (mid.empty()) // get the roots
  {
    foreach_const (nodes_t, _nodes, it)
    if (it->second->_article && !it->second->_parent)
      setme.push_back (it->second->_article);
  }
  else // get children of a particular article
  {
    nodes_t::const_iterator parent_it (_nodes.find (mid));
    if (parent_it != _nodes.end()) {
      ArticleNode::children_t& kids (parent_it->second->_children);
      foreach_const (ArticleNode::children_t, kids, it)
        setme.push_back ((*it)->_article);
    }
  }
}

const Article*
DataImpl :: MyTree :: get_parent (const Quark& mid) const
{
  const Article * parent (0);
  const ArticleNode * parent_node (0);

  nodes_t::const_iterator child_it (_nodes.find (mid));
  if (child_it != _nodes.end())
    parent_node = child_it->second->_parent;
  if (parent_node)
    parent = parent_node->_article;

  return parent;
}

const Article*
DataImpl :: MyTree :: get_article (const Quark& mid) const
{
  nodes_t::const_iterator it (_nodes.find (mid));
  return it==_nodes.end() ? 0 : it->second->_article;
}

size_t
DataImpl :: MyTree :: size () const 
{
  return _nodes.size();
}

void
DataImpl :: MyTree :: set_filter (const Data::ShowType    show_type,
                                  const FilterInfo      * criteria)
{
  if (criteria)
    _filter = *criteria;
  else 
    _filter.clear ();

  // apply the filter to everything in the group...
  articles_t all, pass, fail;
  const GroupHeaders * h =  _data.get_group_headers (_group);
  assert (h != 0);
  foreach_const (nodes_t, h->_nodes, it)
    if (it->second->_article)
      all.push_back (it->second->_article);
  if (criteria)
    _data._article_filter.test_articles (_data, *criteria, _group, all, pass, fail);
  else
    pass = all;

  //
  //  maybe include threads or subthreads...
  //

  if (show_type == Data::SHOW_THREADS)
  {
    unique_articles_t top;
    foreach_const (articles_t, pass, it)
      top.insert (h->find_top_existing_ancestor ((*it)->message_id));
    pass.clear ();
    pass.insert (pass.begin(), top.begin(), top.end());
  }

  if (show_type == Data::SHOW_THREADS || show_type == Data::SHOW_SUBTHREADS)
  {
    unique_sorted_quarks_t top_mids;
    foreach_const (articles_t, pass, it)
      top_mids.get_container().push_back ((*it)->message_id);
    top_mids.sort ();

    pass.clear ();
    fail.clear ();
    foreach_const (articles_t, all, it) {
      const Article * a (*it);
      const ArticleNode * node (h->find_node (a->message_id));
      bool show (false);
      if (a->score <= -9999) // "hide ignored" outweighs "show relatives"
        show = _data._article_filter.test_article (_data, *criteria, _group, *a);
      else if (top_mids.count (a->message_id))
        show = true;
      else if (_data.find_closest_ancestor (node, top_mids))
        show = true;
      (show ? pass : fail).push_back (*it);
    }
  }

  // passing articles not in the tree should be added...
  quarks_t mids_to_add;
  foreach_const (articles_t, pass, it)
    if (!_nodes.count ((*it)->message_id))
      mids_to_add.insert ((*it)->message_id);
  add_articles (mids_to_add, false); // false means we've already filtered these
  debug ("after adding passing articles, I've got " << _nodes.size() << '.');

  // failing articles in the tree should be removed...
  quarks_t mids_to_remove;
  foreach_const (articles_t, fail, it)
    if (_nodes.count ((*it)->message_id))
      mids_to_remove.insert ((*it)->message_id);
  remove_articles (mids_to_remove);
  debug ("now I'm done, and I have " << _nodes.size() << " articles.");
}

/****
*****  
****/

DataImpl :: MyTree :: MyTree (DataImpl              & data_impl,
                              const Quark           & group,
                              const Data::ShowType    show_type,
                              const FilterInfo      * filter):
  _group (group),
  _data (data_impl)
{
  _filter.set_type_aggregate_or (); // a harmless default that, with no children, filters out nothing
  _data.ref_group (_group);
  _data._trees.insert (this);
  set_filter (show_type, filter);
}

DataImpl :: MyTree :: ~MyTree ()
{
  _nodes.clear ();
  _data._trees.erase (this);
  _data.unref_group (_group);
}

/****
*****  
****/

void
DataImpl :: MyTree :: remove_articles (const quarks_t& mids)
{
  debug ("I have " << size() <<
         " articles and am about to remove " << mids.size() << " of them.");

  ArticleTree::Diffs diffs;

  std::set<ArticleNode*> parents;

  // zero out any corresponding nodes in the tree...
  foreach_const (quarks_t, mids, it)
  {
    const Quark& mid (*it);

    nodes_t::iterator nit (_nodes.find (mid));
    if (nit == _nodes.end())
      continue;

    ArticleNode * node (nit->second);
    if (node->_article) {
      diffs.removed.insert (mid);
      node->_article = 0;
    }
    if (node->_parent)
      parents.insert (node->_parent);
  }

  foreach (std::set<ArticleNode*>, parents, it)
  {
    ArticleNode *parent (*it);
    ArticleNode::children_t kids;
    foreach (ArticleNode::children_t, parent->_children, kit)
      if (!mids.count ((*kit)->_mid))
        kids.push_back (*kit);
    parent->_children.swap (kids);
  }

  // reparent any children who need it...
  foreach_const (nodes_t, _nodes, it)
  {
    ArticleNode * n (it->second);

    // if it's an article with a removed parent, reparent to ancestor
    if (n->_article && n->_parent && !n->_parent->_article)
    {
      // start filling out a diff entry
      ArticleTree::Diffs::Reparent r;
      r.message_id = n->_mid;
      r.old_parent = n->_parent->_mid;

      // look for a suitable parent
      ArticleNode * newparent (n->_parent);
      while (newparent && !newparent->_article)
        newparent = newparent->_parent;

      // reparent the node
      if (newparent) {
        newparent->_children.push_front (n);
        r.new_parent = newparent->_mid;
      }
      n->_parent = newparent;

      debug (r.message_id << " has been reparented");
      diffs.reparented.push_back (r);
    }
  }

  fire_diffs (diffs);

  foreach_const (quarks_t, diffs.removed, it)
    _nodes.erase (*it);

  debug ("I now have " << _nodes.size() << " articles left");
}

namespace
{
  bool contains (std::vector<Quark>& sorted_quarks, const Quark& q)
  {
    return binary_search (sorted_quarks.begin(), sorted_quarks.end(), q);
  }
}

void
DataImpl :: MyTree :: add_articles (const quarks_t& mids, bool need_to_filter)
{
  //std::cerr << LINE_ID << " adding " << mids.size()<< " articles to " << this << "..." << std::endl;

  // make a list of the mids already in the tree...
  typedef std::vector<Quark> quark_v;
  quark_v old_mids (_nodes.size());
  int i = 0;
  foreach_const (nodes_t, _nodes, it)
    old_mids[i++] = it->first; // these quarks are already sorted by virtue of being keys in a map...
  //std::cerr << LINE_ID << " previously there were " << old_mids.size() << " articles..." << std::endl;

  // we don't want to add articles that we already have...
  quark_v new_mids;
  new_mids.reserve (mids.size());
  std::set_difference (mids.begin(), mids.end(),
                       old_mids.begin(), old_mids.end(),
                       inserter (new_mids, new_mids.begin()));
  //std::cerr << LINE_ID << ' ' << new_mids.size() << " of the new articles aren't already in the tree..." << std::endl;

  // only add articles which pass the filter
  GroupHeaders * h (_data.get_group_headers (_group));
  assert (h);
  if (need_to_filter) {
    quark_v tmp;
    tmp.reserve (new_mids.size());
    foreach_const (quark_v, new_mids, it) {
      Article * a (h->find_article (*it));
      if (_data._article_filter.test_article (_data, _filter, _group, *a))
        tmp.push_back (*it);
    }
    new_mids.swap (tmp); // now new_mids only holds those passing the filter
    //std::cerr << LINE_ID << ' ' << new_mids.size() << " survived the remove-duplicates and filter tests." << std::endl;
  }
    
  // construct new Nodes for the new mids...
  int node_index (_node_chunk.size());
  _node_chunk.resize (_node_chunk.size() + new_mids.size());
  foreach_const (quark_v, new_mids, it) {
    Article * a (h->find_article (*it));
    ArticleNode * node (&_node_chunk[node_index++]);
    node->_mid = a->message_id;
    node->_article = a;
    std::pair<nodes_t::iterator,bool> result (_nodes.insert (std::pair<Quark,ArticleNode*>(node->_mid, node)));
    assert (result.second); // freshly added; not a duplicate
  }

  // reparent everybody...
  ArticleTree::Diffs diffs;
  unique_sorted_quarks_t all_mids;
  all_mids.reserve (old_mids.size() + new_mids.size());
  set_union (old_mids.begin(), old_mids.end(),
             new_mids.begin(), new_mids.end(),
             std::back_inserter (all_mids.get_container()));
  foreach_const (unique_sorted_quarks_t, all_mids, it)
  {
    ArticleNode * hchild (h->_nodes.find (*it)->second);
    ArticleNode * hparent (find_closest_ancestor (hchild, all_mids));
    ArticleNode * tchild (_nodes.find (*it)->second);
    ArticleNode * tparent (hparent ? _nodes.find (hparent->_mid)->second : 0);

    if (tchild->_parent != tparent) // if the parent changed:
    {
      ArticleTree::Diffs::Reparent r;
      r.message_id = *it;

      if (tchild->_parent) { // unparent from the old
        r.old_parent = tchild->_parent->_mid;
        tchild->_parent->_children.remove (tchild);
        tchild->_parent = 0;
      }

      if (tparent) { // reparent to the new
        r.new_parent = tparent->_mid;
        tchild->_parent = tparent;
        tparent->_children.push_front (tchild);
      }

      if (!contains (old_mids, r.message_id)) // new article...
      {
        debug ("I think that " << r.message_id << " is a new article");
        ArticleTree::Diffs::Added a;
        a.message_id = r.message_id;
        a.parent = r.new_parent;
        diffs.added.push_back (a);
      }
      else // old article -- reparenting
      {
        debug ("I think " << r.message_id << " needs reparenting");
        assert (r.old_parent != r.new_parent);
        diffs.reparented.push_back (r);
      }
    }
    else if (contains (new_mids, *it)) // this is a new root article
    {
      ArticleTree::Diffs::Added a;
      a.message_id = *it;
      diffs.added.push_back (a);
    }
  }

  if (!diffs.reparented.empty() || !diffs.added.empty())
    fire_diffs (diffs);
}
