/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <cctype>
#include <cmath>
#include <iostream>
extern "C" {
  #include <glib/gi18n.h>
  #include <gtk/gtk.h>
  #include <gdk/gdkkeysyms.h>
  #include <gmime/gmime.h>
}
#include <pan/general/debug.h>
#include <pan/general/foreach.h>
#include <pan/general/log.h>
#include <pan/usenet-utils/gnksa.h>
#include <pan/usenet-utils/mime-utils.h>
#include <pan/usenet-utils/utf8-utils.h>
#include <pan/usenet-utils/url-find.h>
#include <pan/icons/pan-pixbufs.h>
#include "body-pane.h"
#include "pad.h"
#include "xface.h"
#include "url.h"

#define FIRST_PICTURE "first-picture"

using namespace pan;

/**
***  Pixbuf Cache
**/
namespace
{
  class PixbufCache
  {
    private:
      typedef std::map<int,GdkPixbuf*> pixbufs_t;
      pixbufs_t _pixbufs;

    public:
      PixbufCache() {}
      ~PixbufCache() { clear(); }

    public:
      void clear () {
        foreach (pixbufs_t, _pixbufs, it)
          g_object_unref (it->second);
        _pixbufs.clear ();
      }

    public:
      GdkPixbuf* get_pixbuf_at_offset (int offset) {
        GdkPixbuf* ret (0);
        if (_pixbufs.count(offset))
          ret = _pixbufs[offset];
        return ret;
      }
      void set_pixbuf_at_offset (int offset, GdkPixbuf* new_p) {
        GdkPixbuf *& p (_pixbufs[offset]);
        if (p)
          g_object_unref (p);
        p = new_p;
        g_object_ref (p);
      }
  };

  void delete_pixbuf_cache (gpointer pc) {
    delete static_cast<PixbufCache*>(pc);
  }

  PixbufCache& get_pixbuf_cache (gpointer gp) {
    static const char * PIXBUF_CACHE ("pixbuf-cache");
    GObject * o (G_OBJECT (gp)); 
    PixbufCache *pc ((PixbufCache*) g_object_get_data(o, PIXBUF_CACHE));
    if (!pc) {
      pc = new PixbufCache ();
      g_object_set_data_full (o, PIXBUF_CACHE, pc, delete_pixbuf_cache);
    }
    return *pc;
  }

  GdkPixbuf* get_pixbuf_at_offset (gpointer o, int offset) {
    return get_pixbuf_cache(o).get_pixbuf_at_offset (offset);
  }

  void set_pixbuf_at_offset (gpointer o, int offset, GdkPixbuf* p) {
    get_pixbuf_cache(o).set_pixbuf_at_offset (offset, p);
  }

  void clear_pixbuf_cache (gpointer o) {
    get_pixbuf_cache(o).clear ();
  }
}

/**
***  "fullsize" flag
**/
namespace
{
  #define FULLSIZE "fullsize"

  void set_fullsize_flag (gpointer o, bool b) {
    g_object_set_data (G_OBJECT(o), FULLSIZE, GINT_TO_POINTER(b));
  }
  bool get_fullsize_flag (gpointer o) {
    return g_object_get_data (G_OBJECT(o), FULLSIZE) != 0;
  }
  bool toggle_fullsize_flag (gpointer o) {
    const bool b (!get_fullsize_flag (o));
    set_fullsize_flag (o, b);
    return b;
  }
}

/**
***  Cursors
**/
namespace
{
  enum {
    CURSOR_IBEAM,
    CURSOR_HREF,
    CURSOR_ZOOM_IN,
    CURSOR_ZOOM_OUT,
    CURSOR_QTY
  };

  GdkCursor * cursors[CURSOR_QTY];
  GdkCursor * cursor_current (0);

  void free_cursors (void)
  {
    for (int i=0; i<CURSOR_QTY; ++i)
      gdk_cursor_destroy (cursors[i]);
  }

  void ensure_cursors_created (GtkWidget * w)
  {
    static bool created (false);
    if (!created)
    {
      created = true;
      GdkDisplay * display (gtk_widget_get_display (w));
      GdkPixbuf * in (gdk_pixbuf_new_from_inline (-1, icon_zoom_in, false, 0));
      GdkPixbuf * out (gdk_pixbuf_new_from_inline (-1, icon_zoom_out, false, 0));

      cursors[CURSOR_IBEAM] = gdk_cursor_new (GDK_XTERM);
      cursors[CURSOR_HREF] = gdk_cursor_new (GDK_HAND2);
      cursors[CURSOR_ZOOM_IN] = gdk_cursor_new_from_pixbuf (display, in, 20, 18);
      cursors[CURSOR_ZOOM_OUT] = gdk_cursor_new_from_pixbuf (display, out, 20, 18);

      g_object_unref (out);
      g_object_unref (in);

      g_atexit (free_cursors);
    }
  }

  void set_cursor_from_iter (GdkWindow *window, GtkWidget *w, GtkTextIter *it)
  {
    GtkTextView * text_view (GTK_TEXT_VIEW (w));
    GtkTextBuffer * buf (gtk_text_view_get_buffer (text_view));
    GtkTextTagTable * tags (gtk_text_buffer_get_tag_table (buf));
    GtkTextTag * pix_tag (gtk_text_tag_table_lookup (tags, "pixbuf"));
    GtkTextTag * url_tag (gtk_text_tag_table_lookup (tags, "url"));
    const bool in_url (gtk_text_iter_has_tag (it, url_tag));
    const bool in_pix (gtk_text_iter_has_tag (it, pix_tag));
    const bool fullsize (get_fullsize_flag (buf));

    ensure_cursors_created (w);

    GdkCursor * cursor_new;
    if (in_pix && fullsize)
      cursor_new = cursors[CURSOR_ZOOM_OUT];
    else if (in_pix)
      cursor_new = cursors[CURSOR_ZOOM_IN];
    else if (in_url)
      cursor_new = cursors[CURSOR_HREF];
    else
      cursor_new = cursors[CURSOR_IBEAM];

    // change the cursor if needed
    if (cursor_new != cursor_current)
      gdk_window_set_cursor (window, cursor_current=cursor_new);
  }
}


namespace
{
  GtkTextTag* get_named_tag_from_view (GtkWidget * w, const char * key)
  {
    GtkTextView * text_view (GTK_TEXT_VIEW(w));
    GtkTextBuffer * buf = gtk_text_view_get_buffer (text_view);
    GtkTextTagTable * tags = gtk_text_buffer_get_tag_table (buf);
    return gtk_text_tag_table_lookup (tags, key);
  }

  void get_iter_from_event_coords (GtkWidget * w,
                                   int x, int y,
                                   GtkTextIter * setme)
  {
    GtkTextView * text_view (GTK_TEXT_VIEW (w));
    gtk_text_view_window_to_buffer_coords (text_view,
                                           GTK_TEXT_WINDOW_WIDGET,
                                           x, y, &x, &y);
    gtk_text_view_get_iter_at_location (text_view, setme, x, y);
    gtk_text_iter_backward_char (setme);
  }

  gboolean motion_notify_event (GtkWidget       * w,
                                GdkEventMotion  * event,
                                gpointer          unused)
  {
    if (event->window != NULL)
    {
      int x, y;
      if (event->is_hint)
        gdk_window_get_pointer (event->window, &x, &y, NULL);
      else {
        x = (int) event->x;
        y = (int) event->y;
      }
      GtkTextIter iter;
      get_iter_from_event_coords (w, x, y, &iter);
      set_cursor_from_iter (event->window, w, &iter);
    }

    return false;
  }

  char* get_url_from_location (GtkWidget * w, int x, int y)
  {
    GtkTextIter iter;
    get_iter_from_event_coords (w, x, y, &iter);

    GtkTextTag * url_tag (get_named_tag_from_view (w, "url"));

    char * retval (0);
    if (gtk_text_iter_has_tag (&iter, url_tag))
    {
      GtkTextIter begin(iter), end(iter);
      gtk_text_iter_backward_to_tag_toggle (&begin, NULL);
      gtk_text_iter_forward_to_tag_toggle (&end, NULL);
      retval = gtk_text_iter_get_text (&begin, &end);
    }

    return retval;
  }

  /* returns a GdkPixbuf of the scaled image.
     unref it when no longer needed. */
  GdkPixbuf* size_to_fit (GdkPixbuf           * pixbuf,
                          const GtkAllocation * size)
  {
    const int nw (size ? size->width : 0);
    const int nh (size ? size->height : 0);

    GdkPixbuf * out (0);
    if (nw>=100 && nh>=100)
    {
      const int ow (gdk_pixbuf_get_width (pixbuf));
      const int oh (gdk_pixbuf_get_height (pixbuf));
      double scale_factor (std::min (nw/(double)ow, nh/(double)oh));
      scale_factor = std::min (scale_factor, 1.0);
      const int scaled_width ((int) std::floor (ow * scale_factor + 0.5));
      const int scaled_height ((int) std::floor (oh * scale_factor + 0.5));
      out = gdk_pixbuf_scale_simple (pixbuf,
                                     scaled_width, scaled_height,
                                     GDK_INTERP_BILINEAR);
    }

    if (!out)
    {
      g_object_ref (pixbuf);
      out = pixbuf;
    }

    return out;
  }

  void resize_picture_at_iter (GtkTextBuffer        * buf,
                               GtkTextIter          * iter,
                               bool                   fullsize,
                               const GtkAllocation  * size,
                               GtkTextTag           * apply_tag)
  {
    const int offset (gtk_text_iter_get_offset (iter));

    GdkPixbuf * original (get_pixbuf_at_offset (buf, offset));

    // remove the old
    GtkTextIter end (*iter);
    gtk_text_iter_forward_to_tag_toggle (&end, apply_tag);
    gtk_text_buffer_delete (buf, iter, &end);

    // insert the new
    GdkPixbuf * scaled (size_to_fit (original, (fullsize ? 0 : size)));
    gtk_text_buffer_insert_pixbuf (buf, iter, scaled);
    g_object_unref (scaled);

    // and apply the tag.
    end = *iter;
    gtk_text_iter_set_offset (iter, offset);
    gtk_text_buffer_apply_tag (buf, apply_tag, iter, &end);
    *iter = end;
  }

  gboolean mouse_button_pressed (GtkWidget       * w,
                                 GdkEventButton  * event,
                                 gpointer          unused)
  {
    g_return_val_if_fail (GTK_IS_TEXT_VIEW(w), false);

    if (event->button==1 || event->button==2) {
      char * url (get_url_from_location (w, (int)event->x, (int)event->y));
      if (url) {
        /* this is kind of a crude way of making sure that double-click
         * doesn't open two or three browser windows. */
        static time_t last_url_time (0);
        const time_t this_url_time (time (NULL));
        if (this_url_time != last_url_time) {
          last_url_time = this_url_time;
          pan_url_show (url);
          g_free (url);
        }
      } else { // maybe we're zooming in/out on a pic...
        GtkTextIter iter;
        get_iter_from_event_coords (w, (int)event->x, (int)event->y, &iter);
        GtkTextTag * tag (get_named_tag_from_view (w, "pixbuf"));
        if (gtk_text_iter_begins_tag (&iter, tag))
        {
          const int offset (gtk_text_iter_get_offset (&iter));
          GtkTextBuffer * buf (gtk_text_view_get_buffer (GTK_TEXT_VIEW(w)));
          const bool fullsize (toggle_fullsize_flag (buf));
          resize_picture_at_iter (buf, &iter, fullsize, &w->allocation, tag);
          gtk_text_iter_set_offset (&iter, offset);
          set_cursor_from_iter (event->window, w, &iter);
        }
      }
    }
    return false;
  }
}

/***
****  INIT
***/
namespace
{
  GtkTextTag* get_or_create_tag (GtkTextTagTable * table, const char * key)
  {
    g_assert (table);
    g_assert (key && *key);

    GtkTextTag * tag (gtk_text_tag_table_lookup (table, key));
    if (!tag) {
      tag = gtk_text_tag_new (key);
      gtk_text_tag_table_add (table, tag);
      g_object_unref (tag); // table refs it
    }
    return tag;
  }

  void
  set_text_buffer_tags (GtkTextBuffer * buffer, const Prefs& p)
  {
    GtkTextTagTable * table = gtk_text_buffer_get_tag_table (buffer);
    get_or_create_tag (table, "pixbuf");
    get_or_create_tag (table, "quote_0");
    g_object_set (get_or_create_tag(table,"bold"),
      "weight", PANGO_WEIGHT_BOLD,
      NULL);
    g_object_set (get_or_create_tag (table, "italic"),
      "style", PANGO_STYLE_ITALIC,
      NULL);
    g_object_set (get_or_create_tag (table, "underline"),
      "underline", PANGO_UNDERLINE_SINGLE,
      NULL);
    g_object_set (get_or_create_tag (table, "url"),
      "underline", PANGO_UNDERLINE_SINGLE,
      "foreground", p.get_color_str ("body-pane-color-url", "#0000ff").c_str(),
      NULL);
    g_object_set (get_or_create_tag (table, "quote_1"),
      "foreground", p.get_color_str ("body-pane-color-quote-1", "#550077").c_str(),
      NULL);
    g_object_set (get_or_create_tag (table, "quote_2"),
      "foreground", p.get_color_str ("body-pane-color-quote-2", "#006600").c_str(),
      NULL);
    g_object_set (get_or_create_tag (table, "quote_3"),
      "foreground", p.get_color_str ("body-pane-color-quote-3", "#bb5500").c_str(),
      NULL);
    g_object_set (get_or_create_tag (table, "signature"),
      "foreground", p.get_color_str ("body-pane-color-signature", "#00AAAA").c_str(),
      NULL);
  }
}

/***
****
***/
namespace
{
  // handle up, down, pgup, pgdown to scroll
  gboolean text_key_pressed (GtkWidget * w, GdkEventKey * event, gpointer scroll)
  {
    gboolean handled (false);

    g_return_val_if_fail (GTK_IS_TEXT_VIEW(w), false);
    const bool up = event->keyval==GDK_Up || event->keyval==GDK_KP_Up;
    const bool down = event->keyval==GDK_Down || event->keyval==GDK_KP_Down;

    if (up || down)
    {
      handled = true;
      gtk_text_view_place_cursor_onscreen (GTK_TEXT_VIEW(w));
      GtkAdjustment * adj = gtk_scrolled_window_get_vadjustment (
                                                      GTK_SCROLLED_WINDOW(scroll));
      gdouble val = adj->value;
      if (up)
        val -= adj->step_increment;
      else
        val += adj->step_increment;
      val = MAX(val, adj->lower);
      val = MIN(val, adj->upper-adj->page_size);
      gtk_adjustment_set_value (adj, val);
    }

    return handled;
  }
}

/****
*****
*****   SETTING THE TEXT FROM AN ARTICLE
*****
****/
namespace
{
  bool text_get_show_all_headers () { return false; }

  /**
   * Returns the quote tag ("quote_0", "quote_1", etc) appropriate for the line.
   * The right tag is calculated by adding up the number of quote characters
   * at the beginning of the line.
   *
   * @param utf8_line the line whose quote status we're checking
   * @param utf8_byte_len the byte length of utf8_line
   * @return a const string for the line's quote tag.  Never NULL.
   */
  const char * get_quote_tag (const TextMassager     * text_massager,
                              const char             * utf8_line,
                              int                      utf8_byte_len)
  {
    const char * str = utf8_line;
    const char * line_end = utf8_line + utf8_byte_len;
    const char * retval = "quote_0";

    if (0<utf8_byte_len && str && *str)
    {
      int depth = 0;

      // walk past leading spaces
      while (str!=line_end && g_unichar_isspace(g_utf8_get_char(str)))
        str = g_utf8_next_char (str);

      // count the number of spaces or quote characters
      for (;;) {
        if (str == line_end)
          break;
        else if (text_massager->is_quote_character (g_utf8_get_char (str))) 
          ++depth;
        else if (!g_unichar_isspace(g_utf8_get_char(str)))
          break;
        str = g_utf8_next_char (str);
      }

      if (!depth)
        retval = "quote_0";
      else switch (depth % 3) {
        case 1: retval = "quote_1"; break;
        case 2: retval = "quote_2"; break;
        case 0: retval = "quote_3"; break;
      }
    }

    return retval;
  }

  typedef std::map<std::string,GdkPixbuf*> pixbufs_t;

  // don't use this directly -- use get_emoticons()
  pixbufs_t emoticon_pixbufs;

  void clear_emoticon_pixbufs ()
  {
    foreach_const (pixbufs_t, emoticon_pixbufs, it)
      g_object_unref (it->second);
  }

  pixbufs_t& get_emoticons ()
  {
    static bool inited (false);
    if (!inited) {
      inited = true;
      emoticon_pixbufs[":)"] = gdk_pixbuf_new_from_inline (-1, icon_mozilla_smile, false, 0);
      emoticon_pixbufs[":-)"] = gdk_pixbuf_new_from_inline (-1, icon_mozilla_smile, false, 0);
      emoticon_pixbufs[";)"] = gdk_pixbuf_new_from_inline (-1, icon_mozilla_wink, false, 0);
      emoticon_pixbufs[":("] = gdk_pixbuf_new_from_inline (-1, icon_mozilla_frown, false, 0);
      emoticon_pixbufs[":P"] = gdk_pixbuf_new_from_inline (-1, icon_mozilla_tongueout, false, 0);
      emoticon_pixbufs[":O"] = gdk_pixbuf_new_from_inline (-1, icon_mozilla_surprised, false, 0);
      g_atexit (clear_emoticon_pixbufs);
    }
    return emoticon_pixbufs;
  }

  enum TagMode { ADD, REPLACE };
 
  void
  set_section_tag (GtkTextBuffer     * buffer,
                   GtkTextIter       * start,
                   const StringView  & body,
                   const StringView  & area,
                   const char        * tag,
                   TagMode             mode)
  {
    // if no alnums, chances are area is a false positive
    int alnums (0);
    for (const char *pch(area.begin()), *end(area.end()); pch!=end; ++pch)
      if (::isalnum(*pch))
        ++alnums;
    if (!alnums)
      return;

    GtkTextIter mark_start = *start;
    gtk_text_iter_forward_chars (&mark_start,
                                 g_utf8_strlen(body.str, area.str-body.str));
    GtkTextIter mark_end = mark_start;
    gtk_text_iter_forward_chars (&mark_end, g_utf8_strlen(area.str,area.len));
    if (mode == REPLACE)
      gtk_text_buffer_remove_all_tags (buffer, &mark_start, &mark_end);
    gtk_text_buffer_apply_tag_by_name (buffer, tag, &mark_start, &mark_end);
  }

  void
  replace_emoticon_text_with_pixbuf (GtkTextBuffer      * buffer,
                                     GtkTextMark        * mark,
                                     std::string        & body,
                                     const std::string  & text,
                                     GdkPixbuf          * pixbuf)
  {
    g_assert (!text.empty());
    g_assert (pixbuf != 0);

    std::string::size_type pos;
    while (((pos=body.find(text))) != body.npos)
    {
      GtkTextIter begin;
      gtk_text_buffer_get_iter_at_mark (buffer, &begin, mark);
      gtk_text_iter_forward_chars (&begin, g_utf8_strlen(&body[0],pos));
      GtkTextIter end = begin;
      gtk_text_iter_forward_chars (&end, text.size());
      gtk_text_buffer_delete (buffer, &begin, &end);
      body.erase (pos, text.size());
      gtk_text_buffer_insert_pixbuf (buffer, &end, pixbuf);
      body.insert (pos, 1, '?'); // make body.size() match the textbuf's size
    }
  }

  /**
   * Appends the specified body into the text buffer.
   * This function takes care of muting quotes and marking
   * quoted and URL areas in the GtkTextBuffer.
   */
  void
  append_text_buffer_nolock (const TextMassager  * text_massager,
                             GtkTextBuffer       * buffer,
                             const char          * body_pch,
                             bool                  mute_quotes,
                             bool                  show_smilies,
                             bool                  do_markup)
  {
    g_return_if_fail (buffer!=0);
    g_return_if_fail (GTK_IS_TEXT_BUFFER(buffer));


    // mute the quoted text, if desired
    if (!body_pch) body_pch = "";
    std::string body (mute_quotes ? text_massager->mute_quotes (body_pch) : body_pch);
    //char * pan_find_signature_delimiter (body, &sig_point);

    // insert the text
    GtkTextIter end;
    gtk_text_buffer_get_end_iter (buffer, &end);
    GtkTextMark * mark = gtk_text_buffer_create_mark (buffer, "blah", &end, true);
    gtk_text_buffer_insert (buffer, &end, body.c_str(), -1);
    GtkTextIter start;
    gtk_text_buffer_get_iter_at_mark (buffer, &start, mark);

    StringView v(body), line;
    GtkTextIter mark_start (start);

    // find where the signature begins...
    const char * sig_point (0);
    int offset (0);
    if (GNKSA::find_signature_delimiter (v, offset) != GNKSA::SIG_NONE)
      sig_point = v.str + offset;
   
    // colorize the quoted text 
    GtkTextIter mark_end;
    std::string last_quote_tag;
    bool is_sig (false);
    const char * last_quote_begin (v.str);
    while (v.pop_token (line, '\n'))
    {
      if (line.empty())
        continue;

      if (line.str == sig_point)
        is_sig = true;

      const std::string quote_tag = is_sig
        ?  "signature"
        : get_quote_tag (text_massager, line.str, line.len);

      // if we've changed tags, colorize the previous block
      if (!last_quote_tag.empty() && quote_tag!=last_quote_tag) {
        mark_end = mark_start;
        gtk_text_iter_forward_chars (&mark_end, g_utf8_strlen(last_quote_begin,line.str-1-last_quote_begin));
        gtk_text_buffer_apply_tag_by_name (buffer, last_quote_tag.c_str(), &mark_start, &mark_end);
        mark_start = mark_end;
        gtk_text_iter_forward_chars (&mark_start, 1);
        last_quote_begin = line.str;
      }

      last_quote_tag = quote_tag;
    }

    // apply the final tag, if there is one */
    if (!last_quote_tag.empty()) {
      gtk_text_buffer_get_end_iter (buffer, &mark_end);
      gtk_text_buffer_apply_tag_by_name (buffer, last_quote_tag.c_str(), &mark_start, &mark_end);
    }

    const StringView v_all (body);
    if (do_markup) {
      StringView v(body), line;
      while (v.pop_token (line, '\n')) {
        const char * prev (0);
        for (;;) {
          const char * b (line.strpbrk ("_*/"));
          if (!b) break; 
          if (b != prev) {
            const StringView lineleft (line.substr (b+1, 0));
            const char * e = lineleft.strchr (*b);
            if (e) {
              const char * type (0);
              switch (*b) {
                case '*': type = "bold"; break;
                case '_': type = "underline"; break;
                case '/': type = "italic"; break;
              }
              set_section_tag (buffer, &start, v_all, StringView(b,e+1), type, ADD);
            }
            prev = e;
          }
          line.eat_chars (b+1-line.str);
        }
      }
    }
    
    // colorize urls
    StringView area;
    StringView march (v_all);
    while ((url_find (march, area))) {
      set_section_tag (buffer, &start, v_all, area, "url", REPLACE);
      march = march.substr (area.str + area.len, 0);
    }

    // do this last, since it alters the text instead of just marking it up
    if (show_smilies) {
      pixbufs_t& emoticons (get_emoticons());
      foreach_const (pixbufs_t, emoticons, it)
        replace_emoticon_text_with_pixbuf (buffer, mark, body, it->first, it->second);
    }

    gtk_text_buffer_delete_mark (buffer, mark);
  }

  /**
   * Generates a GtkPixmap object from a given GMimePart that contains an image.
   * Used for displaying attached pictures inline.
   */
  GdkPixbuf* get_pixbuf_from_gmime_part (const GMimePart * part)
  {
    GdkPixbufLoader * l (gdk_pixbuf_loader_new ());
    GError * err (0);

    // populate the loader
    GMimeDataWrapper * wrapper (g_mime_part_get_content_object (part));
    if (wrapper)
    {
      GMimeStream * mem_stream (g_mime_stream_mem_new ());
      g_mime_data_wrapper_write_to_stream (wrapper, mem_stream);
      GByteArray * buffer (GMIME_STREAM_MEM(mem_stream)->buffer);
      if (buffer->len)
          gdk_pixbuf_loader_write (l, (guchar*)buffer->data, buffer->len, &err);
      g_object_unref (mem_stream);
      g_object_unref (wrapper);
    }
  
    // create the pixbuf
    GdkPixbuf * pixbuf (0);
    if (!err)
      pixbuf = gdk_pixbuf_loader_get_pixbuf (l);
    else {
      Log::add_err (err->message);
      g_clear_error (&err);
    }

    // cleanup
    gdk_pixbuf_loader_close (l, &err);
    if (err) {
      Log::add_err (err->message);
      g_clear_error (&err);
    }
    if (pixbuf)
      g_object_ref (G_OBJECT(pixbuf));
    g_object_unref (G_OBJECT(l));
    return pixbuf;
  }
}

void
BodyPane :: append_part (GMimeObject * obj, GtkAllocation * widget_size)
{
  bool is_done (false);

  // we only need leaf parts..
  if (!GMIME_IS_PART (obj))
    return;

  GMimePart * part = GMIME_PART (obj);
  const GMimeContentType * type = g_mime_object_get_content_type (GMIME_OBJECT (part));

  // decide whether or not this part is a picture
  bool is_image (g_mime_content_type_is_type (type, "image", "*"));
  if (!is_image && g_mime_content_type_is_type(type, "application", "octet-stream")) {
    const char *type, *subtype;
    mime::guess_part_type_from_filename(g_mime_part_get_filename(part), &type, &subtype);
    is_image = type && !strcmp(type,"image");
  }

  // if it's a picture, draw it
  if (is_image)
  {
    GdkPixbuf * original (get_pixbuf_from_gmime_part (part));
    const bool fullsize (!_prefs.get_flag ("size-pictures-to-fit", true));
    GdkPixbuf * view_pic (size_to_fit (original, fullsize ? 0 : widget_size));

    if (view_pic != 0)
    {
      GtkTextIter iter;

      // if this is the first thing in the buffer, precede it with a linefeed.
      gtk_text_buffer_get_end_iter (_buffer, &iter);
      if (gtk_text_buffer_get_char_count (_buffer) > 0)
        gtk_text_buffer_insert (_buffer, &iter, "\n", -1);

      // rembember the location of the first picture.
      if (gtk_text_buffer_get_mark (_buffer, FIRST_PICTURE) == NULL)
        gtk_text_buffer_create_mark (_buffer, FIRST_PICTURE, &iter, true);

      // add the picture
      const int begin_offset (gtk_text_iter_get_offset (&iter));
      gtk_text_buffer_insert_pixbuf (_buffer, &iter, view_pic);
      const int end_offset (gtk_text_iter_get_offset (&iter));
      gtk_text_buffer_insert (_buffer, &iter, "\n", -1);
      GtkTextIter iter_end = iter;
      gtk_text_iter_set_offset (&iter, begin_offset);
      gtk_text_iter_set_offset (&iter_end, end_offset);

      // hook onto the tag a reference to the original picture
      // so that we can resize it later if user resizes the text pane.
      set_pixbuf_at_offset (_buffer, begin_offset, original);
      set_fullsize_flag (_buffer, fullsize);

      GtkTextTagTable * tags (gtk_text_buffer_get_tag_table (_buffer));
      GtkTextTag * tag (gtk_text_tag_table_lookup (tags, "pixbuf"));
      gtk_text_buffer_apply_tag (_buffer, tag, &iter, &iter_end);

      g_object_unref (view_pic);
      g_object_unref (original);

      is_done = true;
    }
  }

  // or, if it's text, display it
  else if (g_mime_content_type_is_type (type, "text", "*"))
  {
    const char * fallback_charset (_charset.c_str());
    char * str = mime_part_to_utf8 (part, fallback_charset);

    const bool do_wrap (_prefs.get_flag ("wrap-article-body", false));
    if (str && *str && do_wrap) {
      std::string wrapped = _tm.fill (str);
      g_free (str);
      str = g_strdup (wrapped.c_str());
    }

    const bool do_mute (_prefs.get_flag ("mute-quoted-text", false));
    const bool do_smilies (_prefs.get_flag ("show-smilies-as-graphics", true));
    const bool do_markup (_prefs.get_flag ("show-text-markup", true));
    append_text_buffer_nolock (&_tm, _buffer, str, do_mute, do_smilies, do_markup);
    g_free (str);
    is_done = true;
  }

  // otherwise, bitch and moan.
  if (!is_done) {
    const char * filename = g_mime_part_get_filename (part);
    char * pch = (filename && *filename)
      ? g_strdup_printf (_("Attachment not shown: MIME type %s/%s; filename %s\n"), type->type, type->subtype, filename)
      : g_strdup_printf (_("Attachment not shown: MIME type %s/%s\n"), type->type, type->subtype);
    GtkTextIter iter;
    gtk_text_buffer_get_end_iter (_buffer, &iter);
    gtk_text_buffer_insert (_buffer, &iter, pch, -1);
    g_free (pch);
  }
}
void
BodyPane :: foreach_part_cb (GMimeObject* o, gpointer self)
{
  if (GMIME_IS_MULTIPART (o))
    g_mime_multipart_foreach (GMIME_MULTIPART (o), foreach_part_cb, self);
  else {
    BodyPane * pane = static_cast<BodyPane*>(self);
    GtkWidget * w (pane->_text);
    pane->append_part (o, &w->allocation);
  }
}


/***
****  HEADERS
***/
namespace
{
  void add_bold_header_value (std::string   & s,
                              GMimeMessage  * message,
                              const char    * key,
                              const char    * fallback_charset)
  {
    const char * val (message ? g_mime_message_get_header (message, key) : "");
    char * utf8_val (header_to_utf8 (val, fallback_charset));
    char * e (0);
    if (strcmp (key, "From"))
      e = g_markup_printf_escaped ("<span weight=\"bold\">%s</span>", utf8_val);
    else {
      const StringView v = GNKSA :: get_short_author_name (utf8_val);
      e = g_markup_printf_escaped ("<span weight=\"bold\">%s</span>", v.to_string().c_str());
    }
    s += e;
    g_free (utf8_val);
    g_free (e);
  }

  size_t add_header_line (std::string  & s,
                          const char   * key_i18n,
                          const char   * key,
                          const char   * val,
                          const char   * fallback_charset)
  {
    char * e;
    e = g_markup_printf_escaped ("<span weight=\"bold\">%s:</span> ", key_i18n);
    s += e;
    g_free (e);
    char * utf8_val = header_to_utf8 (val, fallback_charset);
    e = g_markup_printf_escaped ("%s\n", utf8_val ? utf8_val : "");
    s += e;
    const size_t retval (g_utf8_strlen(key,-1) + g_utf8_strlen(utf8_val,-1) + 2);
    g_free (utf8_val);
    g_free (e);
    return retval;
  }

  size_t add_header_line (std::string   & s,
                          GMimeMessage  * msg,
                          const char    * key_i18n,
                          const char    * key,
                          const char    * fallback_charset)
  {
    const char * val (msg ? g_mime_message_get_header (msg, key) : "");
    return add_header_line (s, key_i18n, key, val, fallback_charset);
  }
}

void
BodyPane :: set_text_from_message (GMimeMessage * message)
{
  const char * fallback_charset (_charset.c_str());

  // set the headers...
  std::string s;
  size_t w(0), l(0);
  l = add_header_line (s, message, _("Subject"), "Subject", fallback_charset);
  w = std::max (w, l);
  l = add_header_line (s, message, _("From"), "From", fallback_charset);
  w = std::max (w, l);
  l = add_header_line (s, message, _("Date"), "Date", fallback_charset);
  w = std::max (w, l);
  s.resize (s.size()-1); // remove trailing linefeed
  gtk_label_set_markup (GTK_LABEL(_headers), s.c_str());

#if GTK_CHECK_VERSION(2,6,0)
  // ellipsize mode is useless w/o this in expander...
  gtk_label_set_width_chars (GTK_LABEL(_headers), (int)w);
#endif

  // set the x-face...
  GdkPixbuf * pixbuf (0);
  const char * pch = message ? g_mime_message_get_header (message, "X-Face") : 0;
  if (pch && _xface->window)
    pixbuf = pan_gdk_pixbuf_create_from_x_face (gtk_widget_get_colormap(_xface), _xface->window, pch);
  gtk_image_set_from_pixbuf (GTK_IMAGE(_xface), pixbuf);
  if (pixbuf)
    g_object_unref (pixbuf);

  // set the terse headers...
  s.clear ();
  add_bold_header_value (s, message, "Subject", fallback_charset);
  s += _(" from ");
  add_bold_header_value (s, message, "From", fallback_charset);
  s += _(" at ");
  add_bold_header_value (s, message, "Date", fallback_charset);
  gtk_label_set_markup (GTK_LABEL(_terse), s.c_str());
#if GTK_CHECK_VERSION(2,6,0)
  // ellipsize mode is useless w/o this in expander...
  gtk_label_set_width_chars (GTK_LABEL(_terse), (int)s.size());
#endif

  // clear the text buffer...
  GtkTextIter start, end;
  gtk_text_buffer_get_bounds (_buffer, &start, &end);
  gtk_text_buffer_delete (_buffer, &start, &end);
  if (gtk_text_buffer_get_mark (_buffer, FIRST_PICTURE) != NULL)
    gtk_text_buffer_delete_mark_by_name (_buffer, FIRST_PICTURE);
  clear_pixbuf_cache (_buffer);

  // maybe add the headers
  const bool do_show_headers (_prefs.get_flag ("show-all-headers", false));
  if (message && do_show_headers) {
    char * headers (g_mime_message_get_headers (message));
    GtkTextIter end;
    gtk_text_buffer_get_end_iter (_buffer, &end);
    StringView line, v(headers);
    while (v.pop_token (line, '\n')) {
      char * pch = header_to_utf8 (line, fallback_charset);
      gtk_text_buffer_insert (_buffer, &end, pch, -1);
      gtk_text_buffer_insert (_buffer, &end, "\n", 1);
      g_free (pch);
    }
    gtk_text_buffer_insert (_buffer, &end, "\n", 1);
    g_free (headers);
  }

  // FIXME: need to set a mark here so that when user hits follow-up,
  // the all-headers don't get included in the followup

  // set the text buffer...
  if (message)
    g_mime_message_foreach_part (message, foreach_part_cb, this);

  // if there was a picture, scroll to it.
  // otherwise scroll to the top of the body.
  GtkTextMark * mark = gtk_text_buffer_get_mark (_buffer, FIRST_PICTURE);
  if (mark)
    gtk_text_view_scroll_to_mark (GTK_TEXT_VIEW(_text), mark, 0.0, true, 0.0, 0.0);
  else {
    GtkTextIter iter;
    gtk_text_buffer_get_start_iter  (_buffer, &iter);
    gtk_text_view_scroll_to_iter (GTK_TEXT_VIEW(_text), &iter, 0.0, true, 0.0, 0.0);
  }
}

void
BodyPane :: refresh ()
{
  set_text_from_message (_message);
}

void
BodyPane :: set_article (const Article& a)
{
  _article = a;

  if (_message)
    g_object_unref (_message);
  _message = _cache.get_message (_article.parts);
  refresh ();

  _data.mark_read (_article);
}

void
BodyPane :: clear ()
{
  if (_message)
    g_object_unref (_message);
  _message = 0;
  refresh ();
}

void
BodyPane :: rot13_selected_text ()
{
  GtkTextIter start, end;
  if (gtk_text_buffer_get_selection_bounds (_buffer, &start, &end))
  {
    // replace the range with a rot13'ed copy.
    gchar* pch = gtk_text_buffer_get_text (_buffer, &start, &end, false);
    const size_t len = strlen (pch);
    TextMassager :: rot13_inplace (pch);
    gtk_text_buffer_delete (_buffer, &start, &end);
    gtk_text_buffer_insert (_buffer, &end, pch, len);
    g_free (pch);

    // resync selection.
    // since gtk_text_buffer_insert() invalided start, we rebuild it first.
    start = end;
    gtk_text_iter_backward_chars (&start, len);
    gtk_text_buffer_select_range (_buffer, &start, &end);
  }
}

/***
****
***/

gboolean
BodyPane :: expander_activated_idle (gpointer self_gpointer)
{
  BodyPane *  self (static_cast<BodyPane*>(self_gpointer));
  GtkExpander * ex (GTK_EXPANDER(self->_expander));
  const bool expanded = gtk_expander_get_expanded (ex);
  gtk_expander_set_label_widget (ex, expanded ? self->_verbose : self->_terse);
  self->_prefs.set_flag ("body-pane-headers-expanded", expanded);
  return false;
}
void
BodyPane :: expander_activated_cb (GtkExpander*, gpointer self_gpointer)
{
  g_idle_add (expander_activated_idle, self_gpointer);
}

namespace
{
  guint text_size_allocated_idle_tag (0);

  gboolean text_size_allocated_idle (gpointer p)
  {
    GtkWidget * text (GTK_WIDGET (p));

    // get the resize flag...
    GtkTextBuffer * buf (gtk_text_view_get_buffer (GTK_TEXT_VIEW(text)));
    const bool fullsize (get_fullsize_flag (buf));

    // get the start point...
    GtkTextIter iter;
    gtk_text_buffer_get_start_iter (buf, &iter);

    // walk through the buffer looking for pictures to resize
    GtkTextTag * tag (get_named_tag_from_view (text, "pixbuf"));
    for (;;) {
      if (gtk_text_iter_begins_tag (&iter, tag))
        resize_picture_at_iter (buf, &iter, fullsize, &text->allocation, tag);
      if (!gtk_text_iter_forward_to_tag_toggle (&iter, tag))
        break;
    }

    text_size_allocated_idle_tag = 0;
    return false;
  }
}

void
BodyPane :: text_size_allocated (GtkWidget     * text,
                                 GtkAllocation * allocation,
                                 gpointer        unused)
{
  if (!text_size_allocated_idle_tag)
       text_size_allocated_idle_tag = g_idle_add (text_size_allocated_idle, text);
}

/***
****
***/

namespace {
  void clear_pointer_on_destroy_signal (GtkWidget*& w) {
    g_signal_connect (w, "destroy", G_CALLBACK(gtk_widget_destroyed), &w);
  }
}

BodyPane :: BodyPane (Data& data, ArticleCache& cache, Prefs& prefs):
  _prefs (prefs),
  _data (data),
  _cache (cache),
  _message (0)
{
  GtkWidget * vbox = gtk_vbox_new (false, PAD);
  gtk_container_set_resize_mode (GTK_CONTAINER(vbox), GTK_RESIZE_QUEUE);

  // about this expander... getting the ellipsis to work is a strange process.
  // once you turn ellipsize on, the expander tries to make its label as narrow
  // as it can and just have the three "..."s.  So gtk_label_set_width_chars
  // is used to force labels to want to be the right size... but then they
  // never ellipsize.  But, if we start with gtk_widget_set_size_request() to
  // tell the expander that it _wants_ to be very small, then it will still take
  // extra space given to it by its parent without asking for enough size to
  // fit the entire label.
  GtkWidget * w = _expander = gtk_expander_new (NULL);
  gtk_widget_set_size_request (w, 50, -1);
  g_signal_connect (w, "activate", G_CALLBACK(expander_activated_cb), this);
  gtk_box_pack_start (GTK_BOX(vbox), w, false, false, 0);

  _terse = gtk_label_new ("Expander");
  g_object_ref (_terse);
  gtk_object_sink (GTK_OBJECT(_terse));
  gtk_misc_set_alignment (GTK_MISC(_terse), 0.0f, 0.5f);
  gtk_label_set_use_markup (GTK_LABEL(_terse), true);
#if GTK_CHECK_VERSION(2,6,0)
  gtk_label_set_ellipsize (GTK_LABEL(_terse), PANGO_ELLIPSIZE_MIDDLE);
#endif
  gtk_widget_show (_terse);

  GtkWidget * hbox = _verbose = gtk_hbox_new (false, 0);
  g_object_ref (_verbose);
  gtk_object_sink (GTK_OBJECT(_verbose));
  w = _headers = gtk_label_new ("Headers");
  gtk_misc_set_alignment (GTK_MISC(w), 0.0f, 0.5f);
#if GTK_CHECK_VERSION(2,6,0)
  gtk_label_set_ellipsize (GTK_LABEL(w), PANGO_ELLIPSIZE_MIDDLE);
#endif
  gtk_label_set_use_markup (GTK_LABEL(w), true);
  gtk_box_pack_start (GTK_BOX(hbox), w, true, true, PAD_SMALL);
  w = _xface = gtk_image_new ();
  gtk_widget_set_size_request (w, 48, 48);
  gtk_box_pack_start (GTK_BOX(hbox), w, false, false, PAD_SMALL);
  gtk_widget_show_all (_verbose);

  // setup
  _text = gtk_text_view_new ();
  refresh_fonts ();
  gtk_widget_add_events (_text, GDK_POINTER_MOTION_MASK|GDK_POINTER_MOTION_HINT_MASK);
  gtk_container_set_border_width (GTK_CONTAINER(_text), PAD_SMALL);
  gtk_text_view_set_editable (GTK_TEXT_VIEW(_text), false);
  gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(_text), false);
  GtkWidget * scroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scroll), GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                  GTK_POLICY_AUTOMATIC,
                                  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER(scroll), _text);
  gtk_widget_show_all (vbox);
  gtk_box_pack_start (GTK_BOX(vbox), scroll, true, true, 0);

  // set up the buffer tags
  _buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(_text));
  set_text_buffer_tags (_buffer, _prefs);

  // listen for user interaction
  g_signal_connect (_text, "motion_notify_event", G_CALLBACK(motion_notify_event), 0);
  g_signal_connect (_text, "button_press_event", G_CALLBACK(mouse_button_pressed), 0);
  g_signal_connect (_text, "key_press_event", G_CALLBACK(text_key_pressed), scroll);
  g_signal_connect (_text, "size_allocate", G_CALLBACK(text_size_allocated), this);

  set_text_from_message (0);
  const bool expanded (_prefs.get_flag ("body-pane-headers-expanded", true));
  gtk_expander_set_expanded (GTK_EXPANDER(_expander), expanded);
  expander_activated_idle (this);

  _root = vbox;
  _prefs.add_listener (this);

  clear_pointer_on_destroy_signal (_expander);
  clear_pointer_on_destroy_signal (_terse);
  clear_pointer_on_destroy_signal (_verbose);
  clear_pointer_on_destroy_signal (_headers);
  clear_pointer_on_destroy_signal (_xface);
  clear_pointer_on_destroy_signal (_root);
  clear_pointer_on_destroy_signal (_text);

  gtk_widget_show_all (_root);
}

BodyPane :: ~BodyPane ()
{
  _prefs.remove_listener (this);

  g_object_unref (_verbose);
  g_object_unref (_terse);

  if (_message)
    g_object_unref (_message);
}


namespace
{
  const int smooth_scrolling_speed (10);

  void sylpheed_textview_smooth_scroll_do (GtkAdjustment  * vadj,
                                           gfloat           old_value,
                                           gfloat           new_value,
                                           int              step)
  {
    const bool down (old_value < new_value);
    const int change_value = (int)(down ? new_value-old_value : old_value-new_value);
    for (int i=step; i<=change_value; i+=step)
      gtk_adjustment_set_value (vadj, old_value+(down?i:-i));
    gtk_adjustment_set_value (vadj, new_value);
  }
}

bool
BodyPane :: read_more_or_less (bool more)
{
  GtkWidget * parent = gtk_widget_get_parent (_text);
  GtkAdjustment * v = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(parent));

  // figure out how far we scroll
  const int arbitrary_font_height_pixels_hack (18);
  const float inc (v->page_size - arbitrary_font_height_pixels_hack);
  const gfloat val (CLAMP (v->value + (more ? inc : -inc),
                           v->lower,
                           MAX(v->upper,v->page_size)-MIN(v->upper,v->page_size)));

  // if we can scroll, do so.
  bool handled (false);
  if (v->upper>=v->page_size && val!=v->value)
  {
    if (_prefs.get_flag ("smooth-scrolling", true))
      sylpheed_textview_smooth_scroll_do (v, v->value, val, smooth_scrolling_speed);
    else
      gtk_adjustment_set_value (v, val);

    handled = true;
  }

  return handled;
}

GMimeMessage*
BodyPane :: create_followup ()
{
  GMimeMessage * msg (0);

  if (_message)
  {
    msg = g_mime_message_new (false);

    ///
    ///  HEADERS
    ///

    // newsgroups
    const char * header = "Newsgroups";
    StringView v (g_mime_message_get_header (_message, header));
    g_mime_message_add_header (msg, header, v.str);

    // subject
    v = g_mime_message_get_subject (_message);
    std::string val (v.str, v.len);
    if (val.find ("Re:") != 0)
      val.insert (0, "Re: ");
    g_mime_message_set_subject (msg, val.c_str());

    // attribution line
    const char * cpch = g_mime_message_get_sender (_message);
    g_mime_message_add_header (msg, "X-Draft-Attribution-Author", cpch);
    cpch = g_mime_message_get_message_id (_message);
    g_mime_message_add_header (msg, "X-Draft-Attribution-Id", cpch);
    char * pch = g_mime_message_get_date_string (_message);
    g_mime_message_add_header (msg, "X-Draft-Attribution-Date", pch);
    g_free (pch);

    // references
    header = "References";
    v = g_mime_message_get_header (_message, header);
    val.assign (v.str, v.len);
    if (!val.empty())
      val += ' ';
    val += "<";
    val += g_mime_message_get_message_id (_message);
    val += ">";
    val = GNKSA :: trim_references (val);
    g_mime_message_add_header (msg, header, val.c_str());

    ///
    ///  BODY
    ///

    GtkTextIter start, end;
    if (gtk_text_buffer_get_selection_bounds (_buffer, &start, &end))
    {
      // go with the user-selected region w/o modifications
      pch = gtk_text_buffer_get_text (_buffer, &start, &end, false);
    }
    else
    {
      // get the entire body...
      gtk_text_buffer_get_bounds (_buffer, &start, &end);
      pch = gtk_text_buffer_get_text (_buffer, &start, &end, false);

      // remove the sig...
      int sig_index (0);
      if (GNKSA::find_signature_delimiter (pch, sig_index) != GNKSA::SIG_NONE)
        pch[sig_index] = '\0';

      // trim...
      StringView v (pch);
      v.trim ();
      std::memmove (pch, v.str, v.len);
      pch[v.len] = '\0';
    }

    // quote the body
    std::string s;
    for (const char *c(pch); c && *c; ++c) {
      if (c==pch || c[-1]=='\n')
        s += (*c=='>' ? ">" : "> ");
      s += *c;
    }

    // set the clone's content object with our modified body
    GMimeStream * stream = g_mime_stream_mem_new ();
    g_mime_stream_write_string (stream, s.c_str());
    GMimeDataWrapper * wrapper = g_mime_data_wrapper_new_with_stream (stream, GMIME_PART_ENCODING_8BIT);
    GMimePart * part = g_mime_part_new ();
    GMimeContentType * type = g_mime_content_type_new_from_string ("text/plain; charset=UTF-8");
    g_mime_part_set_content_type (part, type);
    g_mime_part_set_content_object (part, wrapper);
    g_mime_part_set_encoding (part, GMIME_PART_ENCODING_8BIT);
    g_mime_message_set_mime_part (msg, GMIME_OBJECT(part));
    g_object_unref (wrapper);
    g_object_unref (part);
    g_object_unref (stream);
    g_free (pch);
//std::cerr << LINE_ID << " here is the modified clone\n" << g_mime_message_to_string(msg) << std::endl;
  }

  return msg;
}

/***
****
***/

void
BodyPane :: refresh_fonts ()
{
  const bool body_pane_font_enabled = _prefs.get_flag ("body-pane-font-enabled", false);
  const bool monospace_font_enabled = _prefs.get_flag ("monospace-font-enabled", false);

  if (!body_pane_font_enabled && !monospace_font_enabled)
    gtk_widget_modify_font (_text, 0);
  else {
    const std::string str (monospace_font_enabled
      ? _prefs.get_string ("monospace-font", "Monospace 10")
      : _prefs.get_string ("body-pane-font", "Sans 10"));
    PangoFontDescription * pfd (pango_font_description_from_string (str.c_str()));
    gtk_widget_modify_font (_text, pfd);
    pango_font_description_free (pfd);
  }
}

void
BodyPane :: on_prefs_flag_changed (const StringView& key, bool value)
{
  if ((key=="body-pane-font-enabled") || (key=="monospace-font-enabled"))
    refresh_fonts ();

  if ((key=="wrap-article-body") || (key=="mute-quoted-text") ||
      (key=="show-smilies-as-graphics") || (key=="show-all-headers") ||
      (key=="size-pictures-to-fit") || (key=="show-text-markup"))
    refresh ();
}

void
BodyPane :: on_prefs_string_changed (const StringView& key, const StringView& value)
{
  if ((key=="body-pane-font") || (key=="monospace-font"))
    refresh_fonts ();

}

void
BodyPane :: on_prefs_color_changed (const StringView& key, const GdkColor& color)
{
  if (key.strstr ("body-pane-color") != 0)
    refresh_colors ();
}

void
BodyPane :: refresh_colors ()
{
  set_text_buffer_tags (_buffer, _prefs);
  set_text_from_message (_message);
}

void
BodyPane :: set_character_encoding (const char * charset)
{
  if (charset && *charset)
    _charset = charset;
  else
    _charset.clear ();

  refresh ();
}
