/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _HeaderPane_h_
#define _HeaderPane_h_

#include <gtk/gtk.h>
#include <pan/general/e-util.h>
#include <pan/data/article-cache.h>
#include <pan/data/data.h>
#include <pan/data/filter-info.h>
#include <pan/tasks/queue.h>
#include <pan/gui/action-manager.h>
#include <pan/gui/pan-tree.h>
#include <pan/gui/prefs.h>

namespace pan
{
  /**
   * Base class for functors taking a const Article& and returning bool.
   * @ingroup GUI
   */
  struct ArticleTester {
    virtual ~ArticleTester() {}
    virtual bool operator()(const Article&) const = 0;
  };

  /**
   * Base class for tree navigation functors
   * @ingroup GUI
   */
  struct TreeIterFunctor {
    virtual ~TreeIterFunctor () {}
    virtual bool operator() (GtkTreeModel*, GtkTreeIter*) const = 0;
    virtual bool front (GtkTreeModel*, GtkTreeIter*) const = 0;
  };

  /**
   * Base class for actions performed on a header pane row.
   */
  struct RowActionFunctor {
    virtual ~RowActionFunctor () {}
    virtual void operator() (GtkTreeModel* model, GtkTreeIter* iter, const Article& a) = 0;
  };

  /**
   * Header Pane in the main window of Pan's GUI.
   * @ingroup GUI
   */
  class HeaderPane:
    private Data::Listener,
    private Data::ArticleTree::Listener,
    private Prefs::Listener,
    private Queue::Listener,
    private ArticleCache::Listener
  {
    public:
      HeaderPane (ActionManager&, Data& data, Queue&, ArticleCache&, Prefs&);
      ~HeaderPane ();

    public:
      void refilter ();
      void set_show_type (const Data::ShowType);

    public:
      void set_focus ();
      void unselect_all ();
      void select_all ();
      void select_threads ();
      void select_subthreads ();
      void select_similar ();

    public:
      void read_next_article ();
      void read_next_unread_article ();
      void read_next_thread ();
      void read_next_unread_thread ();
      void read_previous_article ();
      void read_previous_thread ();
      void read_parent_article ();

    private:
      void action_next_if (const ArticleTester& test, RowActionFunctor& action);
      void read_next_if (const ArticleTester&);
      void read_prev_if (const ArticleTester&);

    public:
      GtkWidget* root () { return _root; }
      GtkWidget* create_filter_entry ();
      const Article* get_first_selected_article () const;
      std::set<const Article*> get_full_selection () const;
      std::set<const Article*> get_nested_selection () const;
      bool set_group (const Quark& group);
      const Quark& get_group () { return _group; }

    private:
      void rebuild ();
      void rebuild_article_action (const Quark& message_id);
      void rebuild_article_state  (const Quark& message_id);
      void rebuild_all_article_states ();

    private:
      virtual void on_group_read (const Quark& group);

    private:
      virtual void on_tree_change (const Data::ArticleTree::Diffs&);

    private:
      virtual void on_prefs_flag_changed   (const StringView&, bool);
      virtual void on_prefs_int_changed    (const StringView&, int) { }
      virtual void on_prefs_string_changed (const StringView&, const StringView&);
      virtual void on_prefs_color_changed  (const StringView&, const GdkColor&) {}

    private:
      virtual void on_queue_task_active_changed (Queue&, Task&, bool active) { }
      virtual void on_queue_tasks_added (Queue&, int index, int count);
      virtual void on_queue_task_removed (Queue&, Task&, int index);
      virtual void on_queue_task_moved (Queue&, Task&, int new_index, int old_index) { }
      virtual void on_queue_connection_count_changed (Queue&, int count) { }
      virtual void on_queue_size_changed (Queue&, int active, int total) { }
      virtual void on_queue_online_changed (Queue&, bool online) { }

    private:
      virtual void on_cache_added (const Quark& mid);
      virtual void on_cache_removed (const quarks_t& mids);

    public: // pretend it's private
      ActionManager& _action_manager;

    private:
      class Row;
      typedef std::map<Quark,Row*> mid_to_row_t;
      mid_to_row_t _mid_to_row;
      static const Article* get_article (GtkTreeModel*, GtkTreeIter*);
      int find_highest_followup_score (GtkTreeModel*, GtkTreeIter*) const;
      Row* create_row (const EvolutionDateMaker&, const Article*);
      void add_children_to_model (PanTreeStore              * store,
                                  GtkTreeIter               * parent_it,
                                  const Quark               & parent_mid,
                                  const Data::ArticleTree   * atree,
                                  const EvolutionDateMaker  & date_maker,
                                  const bool                  do_thread);
      PanTreeStore* build_model (const Quark&, Data::ArticleTree*, const TextMatch*);
      void build_tree_columns ();

    private:
      Data& _data;
      Queue& _queue;
      Prefs& _prefs;
      Quark _group;
      Data::ArticleTree * _atree;
      GtkWidget * _root;
      GtkWidget * _tree_view;
      PanTreeStore * _tree_store;
      FilterInfo _filter;
      Data::ShowType _show_type;

    private:
      void rebuild_filter (const std::string&, int);
      void refresh_font ();

    public: // public so that anonymous namespace can reach -- don't call
      void filter (const std::string& text, int mode);
      static void do_popup_menu (GtkWidget*, GdkEventButton*, gpointer);
      static void on_row_activated (GtkTreeView*, GtkTreePath*, GtkTreeViewColumn*, gpointer);
      static gboolean on_button_pressed (GtkWidget*, GdkEventButton*, gpointer);
      ArticleCache& _cache;

    private:
      void get_nested_foreach (GtkTreeModel*, GtkTreePath*, GtkTreeIter*, gpointer) const;
      static void get_nested_foreach_static (GtkTreeModel*, GtkTreePath*, GtkTreeIter*, gpointer);

    private:
      static int column_compare_func (GtkTreeModel*, GtkTreeIter*, GtkTreeIter*, gpointer);
    private:
      class CountUnread;
      class SelectFirstArticle;
      class RowInserter;
      class SimilarWalk;
      void walk_and_collect (GtkTreeModel*, GtkTreeIter*, std::set<const Article*>&) const;
    private:
      typedef void RenderFunc (GtkTreeViewColumn*, GtkCellRenderer*, GtkTreeModel*, GtkTreeIter*, gpointer);
      static RenderFunc render_action;
      static RenderFunc render_state;
      static RenderFunc render_score;
      static RenderFunc render_bytes;
      static RenderFunc render_subject;

    private:
      Row* get_row (const Quark& message_id);

    private:
      void find_next_iterator_from (GtkTreeModel            * model,
                                    GtkTreeIter             * start_pos,
                                    const TreeIterFunctor   & iterate_func,
                                    const ArticleTester     & test_func,
                                    RowActionFunctor        & success_func,
                                    bool                      test_the_start_pos);
      void next_iterator (GtkTreeView            * view,
                          GtkTreeModel           * model,
                          const TreeIterFunctor  & iterate_func,
                          const ArticleTester    & test_func,
                          RowActionFunctor       & success_func);
  };
}

#endif
