/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <cassert>
extern "C" {
  #include <glib/gi18n.h>
}
#include <pan/general/debug.h>
#include <pan/general/foreach.h>
#include <pan/general/messages.h>
#include <pan/data/data.h>
#include <pan/usenet-utils/utf8-utils.h>
#include "task-groups.h"

using namespace pan;

namespace
{
  std::string get_description (const Data& data, const Quark& server)
  {
    char buf[1024];
    std::string host (data.get_server_address (server));
    snprintf (buf, sizeof(buf), _("Getting group list from \"%s\""), host.c_str());
    return std::string (buf);
  }
}

TaskGroups :: TaskGroups (Data& data, const Quark& servername):
   Task ("GROUPS", get_description(data,servername)),
   _data (data),
   _servername (servername),
   _group_count (0),
   _step (LIST)
{
   _state.set_need_nntp (servername);
}

TaskGroups :: ~TaskGroups ()
{
}

/***
****
***/

void
TaskGroups :: use_nntp (NNTP * nntp)
{
  debug ("groups task got an nntp " << nntp->_server << "; step is " << _step);

  _state.set_working ();

  if (_step == LIST) // "LIST" for a full list of groups...
    nntp->list (this);
  else if (_step == LIST_NEWSGROUPS) // "LIST NEWSGROUPS" for descriptions...
    nntp->list_newsgroups (this);
  else
    assert (0);
}

void
TaskGroups :: on_nntp_line (NNTP               * nntp,
                            const StringView   & line)
{
  char permission ('?');
  char * name (0);
  char * description (0);

  if (_step == LIST)
  {
    StringView myline(line), tmp, post;
    myline.pop_token (tmp);
    name = pan :: content_to_utf8 (tmp.str, tmp.len);
    myline.pop_token (tmp); // skip low number
    myline.pop_token (tmp); // skip high number
    myline.pop_token (post); // ok to post?
    if (!post.empty())
      permission = tolower (*post.str);
  }
  else // LIST_NEWSGROUPS
  {
    const char * pch = line.begin();
    while (pch!=line.end() && !isspace((int)*pch)) ++pch;
    name = pan :: content_to_utf8 (line.str, pch-line.str);
    description = pan :: content_to_utf8 (pch, line.str+line.len-pch);
  }

  if (name)
    g_strstrip (name);
  if (description)
    g_strstrip (description);

  if (name && *name){
    const Quark name_quark (name);
    Data::NewGroup& ng (_new_groups[name_quark]);
    ng.group = name_quark;
    if (description)
      ng.description = description;
    if (permission != '?')
      ng.permission = permission;
  }

  if (!(++_group_count % 100ul)) {
    char buf[1024];
    snprintf (buf, sizeof(buf), _("Fetched %lu Groups"), _group_count);
    set_status (buf);
  }

  increment_step ();
  g_free (description);
  g_free (name);
}

void
TaskGroups :: on_nntp_done (NNTP    * nntp,
                            Health    health)
{
  debug ("groups task got an on_nntp_done() from " << nntp->_server);

  const bool is_ok (health == OK);

  if (health == RETRY)
  {
    _state.set_need_nntp (_servername);
  }
  else // heath is OK or FAIL
  {
    if (_step == LIST_NEWSGROUPS)
    {
      int i (0);
      Data::NewGroup * ng = new Data::NewGroup [_new_groups.size()];
      foreach_const (new_groups_t, _new_groups, it)
        ng[i++] = it->second;
      _data.add_groups (_servername, ng, i);
      delete [] ng;

      debug ("groups task setting state completed");
      _state.set_completed ();
      set_finished (OK);
    }
    else // _step == LIST
    {
      _state.set_need_nntp (_servername);
      _step = LIST_NEWSGROUPS;
    }
  }

  check_in (nntp, is_ok);
}
