/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <cassert>
#include <algorithm>
#include <pan/general/debug.h>
#include <pan/general/foreach.h>
#include <pan/general/string-view.h>
#include "article.h"

using namespace pan;

Article :: PartState
Article :: get_part_state () const
{
  if (lazy_size != parts.size())
  {
    // not a multipart
    if (!is_binary)
      part_state = SINGLE;

    // someone's posted a followup to a multipart
    else if (!is_line_count_ge(250) && has_reply_leader(subject.to_string()))
      part_state = SINGLE;

    // someone's posted a "000/124" info message
    else if (parts.empty())
      part_state = SINGLE;

    // a multipart
    else {
      part_state = COMPLETE;
      for (Article::parts_t::const_iterator it(parts.begin()), end(parts.end()); part_state==COMPLETE && it!=end; ++it)
        if (it->empty())
          part_state = INCOMPLETE;
    }
  }

  return part_state;
}

bool
Article :: is_line_count_ge (unsigned int test) const
{
  unsigned int lines (0);
  foreach_const (parts_t, parts, it)
    if (((lines += it->lines)) >= test)
      return true;
  return false;
}

unsigned int
Article :: get_line_count () const
{
  unsigned int lines (0);
  foreach_const (parts_t, parts, it)
    lines += it->lines;
  return lines;
}

unsigned int
Article :: get_crosspost_count () const
{
  quarks_t groups;
  foreach_const (Xref, xref,  xit)
    groups.insert (xit->group);
  return (int) groups.size ();
}

bool
Article :: is_byte_count_ge (unsigned long test) const
{
  unsigned long bytes (0);
  foreach_const (parts_t, parts, it)
    if (((bytes += it->bytes)) >= test)
      return true;
  return false;
}

unsigned long
Article :: get_byte_count () const
{
  unsigned long bytes (0);
  foreach_const (parts_t, parts, it)
    bytes += it->bytes;
  return bytes;
}

bool
Article :: has_reply_leader (const StringView& s)
{
  return !s.empty()
    && s.len>4 \
    && (s.str[0]=='R' || s.str[0]=='r')
    && (s.str[1]=='E' || s.str[1]=='e')
    && s.str[2]==':'
    && s.str[3]==' ';
}

void
Article :: set_part_count (unsigned int count)
{
  assert (count > 0);
  parts.resize (count);
}

void
Article :: add_part (const Part& part)
{
//std::cerr << LINE_ID << " parts.size() " << parts.size() << " part.index " << part.index << std::endl;
  assert (part.index != 0); // parts are numbered [1..n]
  const unsigned int index (part.index - 1); // array offset
  assert (parts.size() > index);
  parts[index] = part;
}

bool
Article :: erase_part (unsigned int index)
{
  --index;
  assert (parts.size() > index);
  Part& p (parts[index]);
  const bool found (!p.message_id.empty());
  if (found)
    parts[index].clear ();
  return found;
}

Article :: Part&
Article :: get_part (unsigned int index)
{
  //std::cerr << LINE_ID << " parts.size() " << parts.size() << " index " << index << std::endl;
  --index;
  assert (parts.size() > index);
  return parts[index];
}

const Article :: Part&
Article :: get_part (unsigned int index) const
{
  //std::cerr << LINE_ID << " parts.size() " << parts.size() << " index " << index << std::endl;
  --index;
  assert (parts.size() > index);
  return parts[index];
}
