/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <iostream>
#include <algorithm>
#include <cstdarg>
#include <gobject/gvaluecollector.h>
#include <pan/general/debug.h>
#include "pan-tree.h"

#define IS_SORTED(tree) \
  (tree->sort_column_id != GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID)

/***
****  Row Helper Functions
***/

void
PanTreeStore :: Row :: set_value_int (GValue * setme, int value)
{
  g_value_init (setme, G_TYPE_INT);
  g_value_set_int (setme, value);
}

void
PanTreeStore :: Row :: set_value_pointer (GValue * setme, gpointer value)
{
  g_value_init (setme, G_TYPE_POINTER);
  g_value_set_pointer (setme, value);
}

void
PanTreeStore :: Row :: set_value_ulong (GValue * setme, unsigned long value)
{
  g_value_init (setme, G_TYPE_ULONG);
  g_value_set_ulong (setme, value);
}

void
PanTreeStore :: Row :: set_value_string (GValue * setme, const char * value)
{
  g_value_init (setme, G_TYPE_STRING);
  g_value_set_string (setme, value);
}

void
PanTreeStore :: Row :: set_value_static_string (GValue * setme, const char * value)
{
  g_value_init (setme, G_TYPE_STRING);
  g_value_set_static_string (setme, value);
}


/***
****  Low-level utilities
***/

void
PanTreeStore :: get_iter (Row * row, GtkTreeIter * setme)
{
  g_assert (setme);
  g_assert (row);
  set_iter (setme, row);
}


PanTreeStore :: Row*
PanTreeStore :: get_row (GtkTreeIter * iter)
{
  g_assert (iter);
  g_assert (iter->stamp == stamp);
  Row * row (static_cast<Row*>(iter->user_data));
  g_assert (row);
  return row;
}

const PanTreeStore :: Row*
PanTreeStore :: get_row (const GtkTreeIter * iter) const
{
  g_assert (iter);
  g_assert (iter->stamp == stamp);
  const Row * row (static_cast<const Row*>(iter->user_data));
  g_assert (row);
  return row;
}

void
PanTreeStore :: set_iter (GtkTreeIter * iter,
                          Row         * row)
{
  iter->stamp = stamp;
  iter->user_data = row;
  iter->user_data2 = 0;
  iter->user_data3 = 0;
}

void
PanTreeStore :: invalidate_iter (GtkTreeIter * iter)
{
  iter->stamp      = 0;
  iter->user_data  = (void*) 0xDEADBEEF;
  iter->user_data2 = (void*) 0xDEADBEEF;
  iter->user_data2 = (void*) 0xDEADBEEF;
}

/*****
******
******  implementing GtkTreeModel's interface
******
*****/

GtkTreeModelFlags
PanTreeStore :: model_get_flags (GtkTreeModel *tree_model)
{
  return GTK_TREE_MODEL_ITERS_PERSIST;
}

gint
PanTreeStore :: model_get_n_columns (GtkTreeModel * model)
{
  const PanTreeStore * store (PAN_TREE_STORE(model));
  g_return_val_if_fail (store, 0);
  return store->n_columns;
}

GType
PanTreeStore :: model_get_column_type (GtkTreeModel * tree,
                                       gint           n)
{
  return (*((PanTreeStore*)(tree))->column_types)[n];
}

gboolean
PanTreeStore :: model_get_iter (GtkTreeModel * model,
                                GtkTreeIter  * iter,
                                GtkTreePath  * path)
{
  PanTreeStore * store = PAN_TREE_STORE(model);
  g_return_val_if_fail (store, false);
  g_return_val_if_fail (path, false);

  // find the Row that corresponds to this path.
  PanTreeStore::Row * row (store->root);
  const int depth (gtk_tree_path_get_depth (path));
  const int * indices = gtk_tree_path_get_indices (path);
  for (int i=0; i<depth; ++i) {
    row = row->nth_child (*indices++);
    if (!row)
      return false;
  }

  // build an iter from that row.
  store->set_iter (iter, row);
  return true;
}

GtkTreePath*
PanTreeStore :: model_get_path (GtkTreeModel * model,
                                GtkTreeIter  * iter)
{
  PanTreeStore * store (PAN_TREE_STORE(model));
  g_return_val_if_fail (store, 0);
  return store->get_path (iter);
}

void
PanTreeStore :: model_get_value (GtkTreeModel * model,
                                 GtkTreeIter  * iter,
                                 gint           column,
                                 GValue       * dest_value)
{
  g_assert (iter);
  g_assert (dest_value);
  PanTreeStore * store = PAN_TREE_STORE(model);
  g_assert (store);
  g_assert (iter->stamp == store->stamp);
  g_assert (0<=column && column<store->n_columns);

  store->get_row(iter)->get_value (column, dest_value);
}

gboolean
PanTreeStore :: model_iter_next (GtkTreeModel * model,
                                 GtkTreeIter  * iter)
{
  PanTreeStore * tree = PAN_TREE_STORE(model);
  g_return_val_if_fail (tree, false);
  g_return_val_if_fail (iter, false);
  g_return_val_if_fail (iter->stamp == tree->stamp, false);

  Row * row (tree->get_row (iter));
  row = row->parent->nth_child (row->child_index + 1);

  if (row)
    tree->set_iter (iter, row);
  else
    invalidate_iter (iter);

  return row != 0;
}

gboolean
PanTreeStore :: model_iter_children (GtkTreeModel * model,
                                     GtkTreeIter  * iter,
                                     GtkTreeIter  * parent)
{
  return model_iter_nth_child (model, iter, parent, 0);
}

gint
PanTreeStore :: model_iter_n_children (GtkTreeModel * model,
                                       GtkTreeIter  * iter)
{
  PanTreeStore * tree = PAN_TREE_STORE(model);
  g_return_val_if_fail (tree, 0);
  g_return_val_if_fail (!iter || iter->stamp == tree->stamp, 0);

  const Row * row (iter ? tree->get_row(iter) : tree->root);
  return row->n_children();
}

gboolean
PanTreeStore :: model_iter_has_child (GtkTreeModel *model,
                                      GtkTreeIter  *iter)
{
  return model_iter_n_children (model, iter) != 0;
}

gboolean
PanTreeStore :: model_iter_nth_child (GtkTreeModel * model,
                                      GtkTreeIter  * iter,
                                      GtkTreeIter  * parent,
                                      gint           n)
{
  PanTreeStore * tree = PAN_TREE_STORE(model);
  g_return_val_if_fail (tree, false);
  g_return_val_if_fail (iter, false);
  g_return_val_if_fail (!parent || parent->stamp == tree->stamp, false);

  Row * row (parent ? tree->get_row(parent) : tree->root);
  row = row->nth_child (n);

  if (row)
    tree->set_iter (iter, row);
  else
    invalidate_iter (iter);

  return row != 0;
}

gboolean
PanTreeStore :: model_iter_parent (GtkTreeModel * model,
                                   GtkTreeIter  * iter,
                                   GtkTreeIter  * child)
{
  return PAN_TREE_STORE(model)->get_parent (iter, child);
}

/*****
******
*****/

GtkTreePath*
PanTreeStore :: get_path (const Row * row) const
{
  g_return_val_if_fail (row, false);

  std::vector<int> indices;
  while (row && row!=root) {
    indices.push_back (row->child_index);
    row = row->parent;
  }

  GtkTreePath * path = gtk_tree_path_new ();
  for (std::vector<int>::const_reverse_iterator it(indices.rbegin()), end(indices.rend()); it!=end; ++it)
    gtk_tree_path_append_index (path, *it);

  return path;
}

GtkTreePath*
PanTreeStore :: get_path (GtkTreeIter  * iter)
{
  g_return_val_if_fail (iter, false);
  g_return_val_if_fail (iter->stamp == stamp, false);

  return get_path (get_row (iter));
}

bool
PanTreeStore :: get_parent (GtkTreeIter * iter,
                            GtkTreeIter * child)
{
  g_return_val_if_fail (child, false);
  g_return_val_if_fail (iter, false);
  g_return_val_if_fail (child->stamp == stamp, false);

  const Row * row (get_row (child));
  const bool has_parent = row->parent != root;
  if (has_parent)
    set_iter (iter, row->parent);
  else
    invalidate_iter (iter);
  return has_parent;
}

bool
PanTreeStore :: is_root (const GtkTreeIter* iter) const
{
  return iter && get_row(iter)->parent==root;
}



/*****
******
******  implementing GtkTreeSortable
******
*****/

gboolean
PanTreeStore :: sortable_get_sort_column_id (GtkTreeSortable  * sortable,
                                             gint             * sort_column_id,
                                             GtkSortType      * order)
{
  PanTreeStore * tree (PAN_TREE_STORE (sortable));
  g_return_val_if_fail (tree, false);

  if (sort_column_id)
     *sort_column_id = tree->sort_column_id;
  if (order)
     *order = tree->order;

  return (tree->sort_column_id != GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID)
      && (tree->sort_column_id != GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID);
}

gboolean
PanTreeStore :: sortable_has_sort_func (GtkTreeSortable *sortable, gint col)
{
  PanTreeStore * tree (PAN_TREE_STORE (sortable));
  g_return_val_if_fail (tree, false);
  std::map<int,PanTreeStore::SortInfo>::const_iterator it =
    tree->sort_info->find (col);
  return (it!=tree->sort_info->end()) && (it->second.sort_func!=0);
}

// SORTING

struct
PanTreeStore :: SortRowInfo
{
  int oldpos;
  Row * row;
  GtkTreeIter iter;
};

struct PanTreeStore :: SortHelperData
{
  PanTreeStore * store;
  GtkTreeModel * model;
  PanTreeStore::SortInfo& sort_info;
  GtkSortType order;
  SortHelperData (PanTreeStore           * tree_store,
                  PanTreeStore::SortInfo & s,
                  GtkSortType              o):
    store(tree_store), model(GTK_TREE_MODEL(tree_store)), sort_info(s), order(o) {}
};

int
PanTreeStore :: row_compare_func (gconstpointer a_gpointer,
                                  gconstpointer b_gpointer,
                                  gpointer      user_data)
{
  const SortRowInfo * a = (const SortRowInfo*) a_gpointer;
  const SortRowInfo * b = (const SortRowInfo*) b_gpointer;
  const SortHelperData * help = (const SortHelperData*) user_data;
  int val = (help->sort_info.sort_func) (help->model,
                                         const_cast<GtkTreeIter*>(&a->iter),
                                         const_cast<GtkTreeIter*>(&b->iter),
                                         help->sort_info.user_data);
  if (help->order == GTK_SORT_DESCENDING)
    val = -val;

  if (!val) // inplace sort
    val = a->row->child_index - b->row->child_index;

  return val;
}

void
PanTreeStore :: sort_children (SortInfo  & sort_info,
                               Row       * parent,
                               bool        recurse)
{
  g_assert (parent);

  const int n (parent->n_children());
  if (n < 2) // no need to sort [0...1] children
    return;

  // build a temporary array to sort
  SortRowInfo * sort_array = new SortRowInfo [n];
  for (int i=0; i<n; ++i) {
    SortRowInfo& row_info = sort_array[i];
    row_info.oldpos = i;
    row_info.row = parent->children[i];
    set_iter (&row_info.iter, row_info.row);
  }

  // sort 'em...
  SortHelperData help (this, sort_info, order);
  g_qsort_with_data (sort_array, n, sizeof(SortRowInfo),
                     row_compare_func, &help);

  // update the child indices...
  for (int i=0; i<n; ++i) {
    Row * child = sort_array[i].row;
    child->child_index = i;
    parent->children[i] = child;
    g_assert (child->parent == parent);
  }

  // let the world know we've changed
  int * new_order (new int [n]);
  bool reordered (false);
  for (int i=0; i<n; ++i) {
    reordered |= (i != sort_array[i].oldpos);
    new_order[i] = sort_array[i].oldpos;
  }

  // tell the world...
  if (reordered) {
    GtkTreeIter *parent_iter;
    GtkTreePath * path;
    if (parent == root) {
      parent_iter = 0;
      path = gtk_tree_path_new ();
    } else {
      GtkTreeIter tmp_iter;
      set_iter (&tmp_iter, parent);
      parent_iter = &tmp_iter;
      path = gtk_tree_model_get_path (help.model, parent_iter);
    }
    gtk_tree_model_rows_reordered (help.model, path, parent_iter, new_order);
    gtk_tree_path_free (path);
  }

  delete [] new_order;
  delete [] sort_array;

  for (int i=0; recurse && i<n; ++i)
    sort_children (sort_info, parent->children[i], recurse);
}

void
PanTreeStore :: sort ()
{
  if (!sort_paused && (sort_column_id != GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID))
  {
    g_assert (sortable_has_sort_func (GTK_TREE_SORTABLE(this), sort_column_id));
    SortInfo& info = (*sort_info)[sort_column_id];
    sort_children (info, root, true);
  }
}

void
PanTreeStore :: sortable_set_sort_column_id (GtkTreeSortable * sortable,
                                             gint              sort_column_id,
                                             GtkSortType       order)
{
  PanTreeStore * tree (PAN_TREE_STORE (sortable));
  g_return_if_fail (tree);

  // if no change, there's nothing to do...
  if (tree->sort_column_id==sort_column_id && tree->order==order)
    return;

  // sanity checks...
  if (sort_column_id != GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID) {
    g_return_if_fail (tree->sort_info->count(sort_column_id) != 0);
    g_return_if_fail (tree->sort_info->find(sort_column_id)->second.sort_func != 0);
  }

  tree->sort_paused = 0;
  tree->sort_column_id = sort_column_id;
  tree->order = order;
  gtk_tree_sortable_sort_column_changed (sortable);
  tree->sort ();
}

gboolean
PanTreeStore :: sortable_has_default_sort_func (GtkTreeSortable *sortable)
{
  return sortable_has_sort_func (sortable, GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID);
}

void
PanTreeStore :: sortable_set_sort_func (GtkTreeSortable        *sortable,
                                        gint                    col,
                                        GtkTreeIterCompareFunc  func,
                                        gpointer                data,
                                        GtkDestroyNotify        destroy)
{
  PanTreeStore * tree (PAN_TREE_STORE (sortable));
  g_return_if_fail (tree);

  (*tree->sort_info)[col].assign (func, data, destroy);

  if (tree->sort_column_id == col)
    tree->sort ();
}

void
PanTreeStore :: sortable_set_default_sort_func (GtkTreeSortable        * s,
                                                GtkTreeIterCompareFunc   f,
                                                gpointer                 p,
                                                GtkDestroyNotify         d)
{
  sortable_set_sort_func (s, GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID, f, p, d);
}

namespace
{
  GObjectClass *parent_class (0);

  typedef std::vector<GValue> values_t;
}

struct
PanTreeStore :: FreeRowWalker: public PanTreeStore::WalkFunctor
{
  PanTreeStore * store;
  FreeRowWalker (PanTreeStore *t): store(t) {}
  virtual ~FreeRowWalker () {}
  virtual bool operator()(PanTreeStore *s, Row *r, GtkTreeIter*it, GtkTreePath *p) {
    delete r;
    return true; // keep marching
  }
};

void
PanTreeStore :: pan_tree_finalize (GObject *object)
{
  // erase the remaining nodes
  PanTreeStore * store = PAN_TREE_STORE(object);
  FreeRowWalker walk (store);
  store->postfix_walk (walk);
  delete store->root;
   
  delete store->column_types;
  delete store->sort_info;

  (*parent_class->finalize) (object);
}

void
PanTreeStore :: pan_tree_class_init (PanTreeStoreClass *klass)
{
  GObjectClass *object_class;

  parent_class = (GObjectClass*) g_type_class_peek_parent (klass);
  object_class = (GObjectClass*) klass;

  object_class->finalize = pan_tree_finalize;
}

void
PanTreeStore :: pan_tree_model_init (GtkTreeModelIface *iface)
{
  iface->get_flags       = model_get_flags;
  iface->get_n_columns   = model_get_n_columns;
  iface->get_column_type = model_get_column_type;
  iface->get_iter        = model_get_iter;
  iface->get_path        = model_get_path;
  iface->get_value       = model_get_value;
  iface->iter_next       = model_iter_next;
  iface->iter_children   = model_iter_children;
  iface->iter_has_child  = model_iter_has_child;
  iface->iter_n_children = model_iter_n_children;
  iface->iter_nth_child  = model_iter_nth_child;
  iface->iter_parent     = model_iter_parent;
}

void
PanTreeStore :: pan_tree_sortable_init (GtkTreeSortableIface *iface)
{
  iface->get_sort_column_id    = sortable_get_sort_column_id;
  iface->set_sort_column_id    = sortable_set_sort_column_id;
  iface->set_sort_func         = sortable_set_sort_func;
  iface->set_default_sort_func = sortable_set_default_sort_func;
  iface->has_default_sort_func = sortable_has_default_sort_func;
}

namespace
{
  class RootRow: public PanTreeStore::Row {
    virtual void get_value (int column, GValue * setme) { /* unused */ }
  };
}

void
PanTreeStore :: pan_tree_init (PanTreeStore * tree)
{
  tree->stamp = g_random_int();
  tree->n_columns = 0;
  tree->root = new RootRow ();
  tree->column_types = new std::vector<GType>();
  tree->sort_paused = 0;
  tree->sort_info = new std::map<int,PanTreeStore::SortInfo>();
  tree->sort_column_id = GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID;
  tree->order = GTK_SORT_ASCENDING;
}

/***
****
***/

GType
PanTreeStore :: get_type ()
{
  static GType pan_tree_type (0);
  if (pan_tree_type)
    return pan_tree_type;

  static const GTypeInfo pan_tree_info = {
    sizeof (PanTreeStoreClass),
    NULL, // base_init
    NULL, // base_finalize
    (GClassInitFunc) pan_tree_class_init,
    NULL, // class finalize
    NULL, // class_data
    sizeof (PanTreeStore),
    0, // n_preallocs
    (GInstanceInitFunc) pan_tree_init
  };

  pan_tree_type = g_type_register_static (
    G_TYPE_OBJECT, "PanTreeStore", &pan_tree_info, (GTypeFlags)0);
  static const GInterfaceInfo tree_model_info =
    { (GInterfaceInitFunc)pan_tree_model_init, 0, 0 };
  g_type_add_interface_static (
    pan_tree_type, GTK_TYPE_TREE_MODEL, &tree_model_info);
  static const GInterfaceInfo sortable_info =
    { (GInterfaceInitFunc)pan_tree_sortable_init, 0, 0 };
  g_type_add_interface_static (
    pan_tree_type, GTK_TYPE_TREE_SORTABLE, &sortable_info);
  return pan_tree_type;
}

PanTreeStore*
PanTreeStore :: new_tree (int n_columns, ...)
{
  g_return_val_if_fail (n_columns>0, 0);
  PanTreeStore* tree = (PanTreeStore*) g_object_new (PAN_TREE_STORE_TYPE, NULL);

  va_list args;
  va_start (args, n_columns);
  for (int i=0; i<n_columns; ++i) {
    const GType type = va_arg (args, GType);
    tree->column_types->push_back (type);
  }
  va_end (args);
  tree->n_columns = tree->column_types->size();

  return tree;
}

/***
****
***/

void
PanTreeStore :: row_changed (Row * row)
{
  GtkTreeIter iter;
  get_iter (row, &iter);
  row_changed (&iter);
}

void
PanTreeStore :: row_changed (GtkTreeIter * iter)
{
  GtkTreeModel * model = GTK_TREE_MODEL(this);
  GtkTreePath * path = gtk_tree_model_get_path (model, iter);
  gtk_tree_model_row_changed (model, path, iter);
  gtk_tree_path_free (path);
}

/***
****
***/

void
PanTreeStore :: renumber_children (Row * parent,
                                   int child_lo,
                                   int child_hi)
{
  const int n_children (parent->n_children());
  child_hi = std::min (child_hi, n_children);

  g_assert (parent);
  g_assert (child_lo >= 0);
  g_assert (child_lo <= child_hi);
  g_assert (child_hi <= n_children);

  Row ** it = &parent->children[child_lo];
  for (int i=child_lo; i!=child_hi; ++i, ++it) {
    (*it)->child_index = i;
    (*it)->parent = parent;
  }
}

typedef std::vector<PanTreeStore::Row*> rows_t;

void
pan_tree_store_insert     (PanTreeStore                     * tree,
                           GtkTreeIter                      * iter,
                           GtkTreeIter                      * parent,
                           PanTreeStore::Row                * row,
                           int                                pos)
{
  rows_t rows;
  rows.push_back (row);
  pan_tree_store_insert_n (tree, iter, parent, rows, pos);
}

void
pan_tree_store_insert_n  (PanTreeStore                    * tree,
                          GtkTreeIter                     * iter,
                          GtkTreeIter                     * parent_it,
                          rows_t                          & rows,
                          int                               position)
{
  tree->insert_n (iter, parent_it, rows, position);
}

void
PanTreeStore :: insert_n (GtkTreeIter                     * iter,
                          GtkTreeIter                     * parent_it,
                          rows_t                          & new_children,
                          int                               position)
{
  g_return_if_fail (!new_children.empty());
  g_return_if_fail (position >= 0);

  // insert the rows
  const size_t n_rows (new_children.size());
  Row * parent_row (parent_it ? get_row (parent_it) : root);
  const int old_size (parent_row->n_children());
  position = std::min (position, old_size);
  parent_row->children.insert (parent_row->children.begin()+position,
                               new_children.begin(),
                               new_children.end());
  renumber_children (parent_row, position);

  // set the return iter
  GtkTreeIter walk;
  set_iter (&walk, parent_row->children[position]);
  if (iter)
    *iter = walk;

  // emit the 'row inserted' signals
  GtkTreePath * path = get_path (&walk);
  GtkTreeModel * model = GTK_TREE_MODEL(this);
  Row ** it = &parent_row->children[position];
  for (size_t i=0; i<n_rows; ++i) {
    Row * row (*it++);
    set_iter (&walk, row);
    gtk_tree_model_row_inserted (model, path, &walk);
    gtk_tree_path_next (path);
  }

  // maybe emit the 'row has child toggled' signal
  if (!old_size && parent_row!=root) {
    gtk_tree_path_up (path);
    gtk_tree_model_row_has_child_toggled (model, path, parent_it);
  }

  // cleanup
  gtk_tree_path_free (path);
}

void
pan_tree_store_append  (PanTreeStore       * tree,
                        GtkTreeIter        * iter,
                        GtkTreeIter        * parent,
                        PanTreeStore::Row  * row)
{
  rows_t rows;
  rows.push_back (row);
  pan_tree_store_append_n (tree, iter, parent, rows);
}

void
pan_tree_store_append_n  (PanTreeStore  * tree,
                          GtkTreeIter   * iter,
                          GtkTreeIter   * parent,
                          rows_t        & rows)
{
  pan_tree_store_insert_n (tree, iter, parent, rows, INT_MAX);
}

/***
****
***/


struct PanTreeStore::ReparentWalker: public PanTreeStore::WalkFunctor
{
  GtkTreeModel * model;

  ReparentWalker (PanTreeStore *store): model(GTK_TREE_MODEL(store)) {}

  virtual ~ReparentWalker () {}

  virtual bool operator()(PanTreeStore *store, Row* row,
                          GtkTreeIter *iter, GtkTreePath *path)
  {
    const int n_children (row->n_children());

    if (n_children)
    {
      // fire a "row inserted" for each child
      GtkTreePath * cpath (gtk_tree_path_copy (path));
      gtk_tree_path_down (cpath);
      for (int i=0; i<n_children; ++i) {
        Row * child (row->nth_child (i));
        GtkTreeIter citer;
        store->set_iter (&citer, child);
        gtk_tree_model_row_inserted (model, cpath, &citer);
        if (!i)
          gtk_tree_model_row_has_child_toggled (model, path, iter);
        gtk_tree_path_next (cpath);
      }
      gtk_tree_path_free (cpath);
    }

    return true; // keep marching
  }
};

void
PanTreeStore :: reparent (Row  * new_parent,
                          Row  * row,
                          int    position)
{
  g_return_if_fail (row != 0);

  GtkTreeModel * model (GTK_TREE_MODEL(this));

  Row * old_parent (row->parent);
  if (!new_parent)
    new_parent = root;
  const int new_parent_old_n_children (new_parent->n_children());

  // remove our subtree's toplevel from its old parent
  remove_siblings (&row, 1, false);

  // if this was the old parent's last child, fire a has-child-toggled event
  if (!old_parent->n_children()) {
    GtkTreeIter tmp;
    set_iter (&tmp, old_parent);
    GtkTreePath * path (get_path (&tmp));
    gtk_tree_model_row_has_child_toggled (model, path, &tmp);
    gtk_tree_path_free (path);
  }

  // add the subtree's toplevel to its new parent row
  position = std::min (position, new_parent_old_n_children);
  new_parent->children.insert (new_parent->children.begin()+position, row);
  renumber_children (new_parent, position);

  // emit a row-inserted for iter
  GtkTreeIter iter;
  get_iter (row, &iter);
  GtkTreePath * path (get_path (&iter));
  gtk_tree_model_row_inserted (model, path, &iter);
  gtk_tree_path_free (path);

  // this emits all the row-inserted and has-child-toggled signals EXCEPT
  // for iter's row-inserted (above) and parent-iter's has-child-toggled (below).
  // It's kind of kludgy but gets all the signals emitted in the right sequence.
  ReparentWalker walk (this);
  prefix_walk (walk, &iter, true);

  // if this was the new parent's first child, fire a has-child-toggled event
  if (!new_parent_old_n_children) {
    GtkTreePath * path (get_path (new_parent));
    GtkTreeIter new_parent_iter;
    get_iter (new_parent, &new_parent_iter);
    gtk_tree_model_row_has_child_toggled (model, path, &new_parent_iter);
    gtk_tree_path_free (path);
  }
}


/***
****
***/

struct PanTreeStore :: RowCompare
{
  Row * root;
  RowCompare (Row * r): root(r) {}

  int depth (const Row * row) const {
    int depth (0);
    while (row->parent != root) {
      ++depth;
      row = row->parent;
    }
    return depth;
  }

  // this sort function is very particular...
  // we go deepest first, so the first batches don't invalidate later ones.
  // we group by parent so to get groups of siblings for remove_siblings().
  // we sort by child_index so the first siblings removed in
  // remove_siblings() don't invalidate the tree path it uses.
  bool operator() (const Row* a, const Row* b) const {
    // deepest first
    const int a_depth (depth (a));
    const int b_depth (depth (b));
    if (a_depth != b_depth)
      return b_depth < a_depth;
    // group by parent
    if (a->parent != b->parent)
      return a->parent < b->parent;
    // from last child to first
    return b->child_index < a->child_index;
  }
};

void
PanTreeStore :: remove (GtkTreeIter * iters,
                        size_t        n_iters)
{
  g_return_if_fail (iters);
  g_return_if_fail (n_iters);

  Row ** rows = new Row* [n_iters];
  for (size_t i=0; i<n_iters; ++i)
    rows[i] = get_row (iters+i);

  RowCompare compare (root);
  std::sort (rows, rows+n_iters, compare);

  Row * parent (0);
  Row ** children (0);
  for (Row **nit=rows, **nend=rows+n_iters; nit!=nend; ++nit) {
    if (parent && ((*nit)->parent != parent)) {
      remove_siblings (children, nit-children, true);
      parent = 0;
    }
    if (!parent) {
      parent = (*nit)->parent;
      g_assert (parent);
      children = nit;
    }
  }
  if (parent) {
    size_t n = (rows + n_iters) - children;
    remove_siblings (children, n, true);
  }

  delete [] rows;
}

struct PanTreeStore::ClearWalker: public PanTreeStore::WalkFunctor
{
  virtual ~ClearWalker () {}

  void clear_children (PanTreeStore* store, Row * row)
  {
    if (row->n_children()) {
      std::vector<Row*> children (row->children);
      std::reverse (children.begin(), children.end());
      store->remove_siblings (&children.front(), children.size(), true);
    }
  }

  virtual bool operator()(PanTreeStore* store,
                          PanTreeStore::Row* row,
                          GtkTreeIter* iter, GtkTreePath* path)
  {
    clear_children (store, store->get_row(iter));
    return true;
  }
};

void
PanTreeStore :: clear ()
{
  ClearWalker walker;
  postfix_walk (walker, 0, false);
  walker.clear_children (this, root);
}


// note that the siblings passed in here must
// be sorted from highest child index to lowest child index
void
PanTreeStore :: remove_siblings (Row     ** siblings,
                                 size_t     n_siblings,
                                 bool       free_memory)
{
  // entry assertions
  g_assert (siblings);
  g_assert (n_siblings);
  Row * parent = siblings[0]->parent;
  for (Row **nit(siblings), **nend(nit+n_siblings); nit!=nend; ++nit) {
    g_assert ((*nit)->parent == parent); // all are siblings
    g_assert (nit==siblings || ((*nit)->child_index < nit[-1]->child_index)); // sorted from high to low child index
  }

  // walk through all the siblings and unthread them.
  GtkTreeModel * model (GTK_TREE_MODEL(this));
  GtkTreePath * path = get_path (parent);
  for (Row** nit(siblings), **nend(siblings+n_siblings); nit!=nend; ++nit)
  {
    Row * row (*nit);
    g_assert (row->parent);
    g_assert (row->parent == parent);

    // unthread the row
    const int pos = row->child_index;
    parent->children.erase (parent->children.begin() + pos);

    // let the world know the row was removed
    gtk_tree_path_append_index (path, row->child_index);
    gtk_tree_model_row_deleted (model, path);
    gtk_tree_path_up (path);

    // free `row' and all its descendants
    if (free_memory) {
      FreeRowWalker walk (this);
      GtkTreeIter iter;
      set_iter (&iter, *nit);
      postfix_walk (walk, &iter);
    }
  }

  // renumber the surviving children
  renumber_children (parent);

  // if the parent's has-children state has changed, let the world know that too
  if (parent!=root && !parent->n_children()) {
    GtkTreeIter pit;
    set_iter (&pit, parent);
    gtk_tree_model_row_has_child_toggled (model, path, &pit);
  }

  gtk_tree_path_free (path);
}

/***
****
***/

bool
PanTreeStore :: walk_helper (int           walk_mode,
                             Row         * row,
                             GtkTreePath * path,
                             WalkFunctor & walker)
{
  g_assert (row);
  g_assert (walk_mode==WALK_PREFIX || walk_mode==WALK_POSTFIX);

  bool more (true);

  GtkTreeIter iter;
  set_iter (&iter, row);

  if (more && row!=root && walk_mode==WALK_PREFIX)
    more = walker (this, row, &iter, path);

  const size_t n_children (row->n_children());
  if (more && n_children) {
    if (path)
      gtk_tree_path_append_index (path, 0);
    for (Row ** it(&row->children.front()),
              ** end(it+n_children); more && it!=end; ++it) {
      more = walk_helper (walk_mode, *it, path, walker);
      if (path)
        gtk_tree_path_next (path);
    }
    if (path)
      gtk_tree_path_up (path);
  }

  if (more && row!=root && walk_mode==WALK_POSTFIX)
    more = walker (this, row, &iter, path);

  return more;
}

void
PanTreeStore :: walk (int             walk_mode,
                      WalkFunctor   & walker,
                      GtkTreeIter   * top,
                      bool            need_path)
{
  GtkTreePath * path (0);
  if (need_path)
    path = top ? get_path(top) : gtk_tree_path_new();

  Row * row (top ? get_row(top) : root);
  walk_helper (walk_mode, row, path, walker);
  gtk_tree_path_free (path);
}

void
PanTreeStore :: prefix_walk (WalkFunctor   & walker,
                             GtkTreeIter   * top,
                             bool            need_path)
{
  walk (WALK_PREFIX, walker, top, need_path);
}

void
PanTreeStore :: postfix_walk (WalkFunctor   & walker,
                              GtkTreeIter   * top,
                              bool            need_path)
{
  walk (WALK_POSTFIX, walker, top, need_path);
}
