/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002-2006  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <cassert>
#include <glib/gmessages.h> // for g_assert
#include <pan/general/debug.h>
#include <pan/general/foreach.h>
#include <pan/general/quark.h>
#include <pan/data/article.h>
#include <pan/data/filter-info.h>
#include "article-filter.h"
#include "data-impl.h"

using namespace pan;

/****
*****  ArticleTree functions
****/

void
DataImpl :: MyTree :: get_children (const Quark& mid, articles_t & setme) const
{
  if (mid.empty()) // get the roots
  {
    foreach_const (nodes_t, _nodes, it)
    if (it->second->_article && !it->second->_parent)
      setme.push_back (it->second->_article);
  }
  else // get children of a particular article
  {
    nodes_t::const_iterator parent_it (_nodes.find (mid));
    if (parent_it != _nodes.end()) {
      ArticleNode::children_t& kids (parent_it->second->_children);
      foreach_const (ArticleNode::children_t, kids, it)
        setme.push_back ((*it)->_article);
    }
  }
}

const Article*
DataImpl :: MyTree :: get_parent (const Quark& mid) const
{
  const Article * parent (0);
  const ArticleNode * parent_node (0);

  nodes_t::const_iterator child_it (_nodes.find (mid));
  if (child_it != _nodes.end())
    parent_node = child_it->second->_parent;
  if (parent_node)
    parent = parent_node->_article;

  return parent;
}

const Article*
DataImpl :: MyTree :: get_article (const Quark& mid) const
{
  nodes_t::const_iterator it (_nodes.find (mid));
  return it==_nodes.end() ? 0 : it->second->_article;
}

size_t
DataImpl :: MyTree :: size () const 
{
  return _nodes.size();
}

void
DataImpl :: MyTree :: set_filter (const Data::ShowType    show_type,
                                  const FilterInfo      * criteria)
{
  if (criteria)
    _filter = *criteria;
  else 
    _filter.clear ();

  // apply the filter to everything in the group...
  articles_t all, pass, fail;
  const GroupHeaders * h =  _data.get_group_headers (_group);
  g_assert (h != 0);
  foreach_const (nodes_t, h->_nodes, it)
    if (it->second->_article)
      all.push_back (it->second->_article);
  if (criteria)
    _data._article_filter.test_articles (_data, *criteria, _group, all, pass, fail);
  else
    pass = all;

  //
  //  maybe include threads or subthreads...
  //

  if (show_type == Data::SHOW_THREADS)
  {
    unique_articles_t top;
    foreach_const (articles_t, pass, it)
      top.insert (h->find_top_existing_ancestor ((*it)->message_id));
    pass.clear ();
    pass.insert (pass.begin(), top.begin(), top.end());
  }

  if (show_type == Data::SHOW_THREADS || show_type == Data::SHOW_SUBTHREADS)
  {
    unique_sorted_quarks_t top_mids;
    foreach_const (articles_t, pass, it)
      top_mids.get_container().push_back ((*it)->message_id);
    top_mids.sort ();

    pass.clear ();
    fail.clear ();
    foreach_const (articles_t, all, it) {
      const Article * a (*it);
      const ArticleNode * node (h->find_node (a->message_id));
      bool show (false);
      if (a->score <= -9999) // "hide ignored" outweighs "show relatives"
        show = _data._article_filter.test_article (_data, *criteria, _group, *a);
      else if (top_mids.count (a->message_id))
        show = true;
      else if (_data.find_closest_ancestor (node, top_mids))
        show = true;
      (show ? pass : fail).push_back (*it);
    }
  }

  // passing articles not in the tree should be added...
  quarks_t mids_to_add;
  foreach_const (articles_t, pass, it)
    if (!_nodes.count ((*it)->message_id))
      mids_to_add.insert ((*it)->message_id);
  add_articles (mids_to_add, false); // false means we've already filtered these
  debug ("after adding passing articles, I've got " << _nodes.size() << '.');

  // failing articles in the tree should be removed...
  quarks_t mids_to_remove;
  foreach_const (articles_t, fail, it)
    if (_nodes.count ((*it)->message_id))
      mids_to_remove.insert ((*it)->message_id);
  remove_articles (mids_to_remove);
  debug ("now I'm done, and I have " << _nodes.size() << " articles.");
}

/****
*****  
****/

DataImpl :: MyTree :: MyTree (DataImpl              & data_impl,
                              const Quark           & group,
                              const Data::ShowType    show_type,
                              const FilterInfo      * filter):
  _group (group),
  _data (data_impl)
{
  _filter.set_type_aggregate_or (); // a harmless default that, with no children, filters out nothing
  _data.ref_group (_group);
  _data._trees.insert (this);
  set_filter (show_type, filter);
}

DataImpl :: MyTree :: ~MyTree ()
{
  _nodes.clear ();
  _data._trees.erase (this);
  _data.unref_group (_group);
}

/****
*****  
****/

void
DataImpl :: MyTree :: remove_articles (const quarks_t& mids)
{
  debug ("I have " << size() <<
         " articles and am about to remove " << mids.size() << " of them.");

  ArticleTree::Diffs diffs;

  std::set<ArticleNode*> parents;

  // zero out any corresponding nodes in the tree...
  foreach_const (quarks_t, mids, it)
  {
    const Quark& mid (*it);

    nodes_t::iterator nit (_nodes.find (mid));
    if (nit == _nodes.end())
      continue;

    ArticleNode * node (nit->second);
    if (node->_article) {
      diffs.removed.insert (mid);
      node->_article = 0;
    }
    if (node->_parent)
      parents.insert (node->_parent);
  }

  foreach (std::set<ArticleNode*>, parents, it)
  {
    ArticleNode *parent (*it);
    ArticleNode::children_t kids;
    foreach (ArticleNode::children_t, parent->_children, kit)
      if (!mids.count ((*kit)->_mid))
        kids.push_back (*kit);
    parent->_children.swap (kids);
  }

  // reparent any children who need it...
  foreach_const (nodes_t, _nodes, it)
  {
    ArticleNode * n (it->second);

    // if it's an article with a removed parent, reparent to ancestor
    if (n->_article && n->_parent && !n->_parent->_article)
    {
      // start filling out a diff entry
      ArticleTree::Diffs::Reparent r;
      r.message_id = n->_mid;
      r.old_parent = n->_parent->_mid;

      // look for a suitable parent
      ArticleNode * newparent (n->_parent);
      while (newparent && !newparent->_article)
        newparent = newparent->_parent;

      // reparent the node
      if (newparent) {
        newparent->_children.push_front (n);
        r.new_parent = newparent->_mid;
      }
      n->_parent = newparent;

      debug (r.message_id << " has been reparented");
      diffs.reparented.push_back (r);
    }
  }

  fire_diffs (diffs);

  foreach_const (quarks_t, diffs.removed, it)
    _nodes.erase (*it);

  debug ("I now have " << _nodes.size() << " articles left");
}

namespace
{
  bool contains (std::vector<Quark>& sorted_quarks, const Quark& q)
  {
    return binary_search (sorted_quarks.begin(), sorted_quarks.end(), q);
  }
}

struct
DataImpl :: MyTree :: NodeMidCompare
{
  bool operator ()(const ArticleNode* a, const ArticleNode* b) const {
    return a->_mid < b->_mid;
  }
  bool operator ()(const Quark& a, const std::pair<const pan::Quark, pan::DataImpl::ArticleNode*>& b) const {
    return a < b.first;
  }
  bool operator ()(const std::pair<const pan::Quark, pan::DataImpl::ArticleNode*>& a, const Quark& b) const {
    return a.first < b;
  }
  bool operator ()(const std::pair<const pan::Quark, pan::DataImpl::ArticleNode*>& a,
                   const std::pair<const pan::Quark, pan::DataImpl::ArticleNode*>& b) const {
    return a.first < b.first;
  }
};

void
DataImpl :: MyTree :: accumulate_descendants (quarks_t& descendants, const ArticleNode* node) const
{
  descendants.insert (node->_mid);
  foreach_const (ArticleNode::children_t, node->_children, it)
    accumulate_descendants (descendants, *it);
}

void
DataImpl :: MyTree :: add_articles (const quarks_t& mids, bool need_to_filter)
{
  NodeMidCompare compare;

  // prune out any articles already in the tree
  //std::cerr << LINE_ID << " mids.size() " << mids.size() << std::endl;
  quarks_t new_mids;
  std::set_difference (mids.begin(), mids.end(),
                       _nodes.begin(), _nodes.end(),
                       inserter (new_mids, new_mids.begin()),
                       compare);
  //std::cerr << LINE_ID << " after removing duplicates: " << new_mids.size() << std::endl;

  // convert it to articles
  GroupHeaders * h (_data.get_group_headers (_group));
  g_assert (h);
  nodes_v nodes;
  h->find_nodes (new_mids, nodes);
  
  // prune out any articles that don't pass the filter
  if (need_to_filter) {
    nodes_v tmp;
    tmp.reserve (nodes.size());
    foreach_const (nodes_v, nodes, it)
      if (_data._article_filter.test_article (_data, _filter, _group, *((*it)->_article)))
        tmp.push_back (*it);
    nodes.swap (tmp);
  }
  //std::cerr << LINE_ID << " after filtering: " << nodes.size() << std::endl;
      
  // construct new Nodes for the new mids...
  int node_index (_node_chunk.size());
  _node_chunk.resize (_node_chunk.size() + new_mids.size());
  nodes_v my_nodes;
  my_nodes.reserve (nodes.size());
  foreach_const (nodes_v, nodes, it) {
    Article * a ((*it)->_article);
    ArticleNode * node (&_node_chunk[node_index++]);
    node->_mid = a->message_id;
    node->_article = const_cast<Article*>(a);
//std::cerr << LINE_ID << " added " << node->_mid << " to the tree (unthreaded so far)" << std::endl;
    std::pair<nodes_t::iterator,bool> result (_nodes.insert (std::pair<Quark,ArticleNode*>(node->_mid, node)));
    g_assert (result.second); // freshly added; not a duplicate
    my_nodes.push_back (node);
  }

  // find parents for the new articles

  ArticleTree::Diffs diffs;
  for (size_t i(0), n(nodes.size()); i!=n; ++i)
  {
    const ArticleNode * node (nodes[i]);
    ArticleNode * tree_node (my_nodes[i]);
    g_assert (node != tree_node);
    g_assert (node->_mid == tree_node->_mid);
    g_assert (node->_article == tree_node->_article);

    Diffs::Added added;
    added.message_id = tree_node->_mid;

    // of its ancestors,
    // find the first one present in our tree
    ArticleNode * parent (0);
    const nodes_t::const_iterator nend (_nodes.end());
    for (const ArticleNode *it(node->_parent); it && !parent; it=it->_parent) {
      nodes_t::iterator nit (_nodes.find (it->_mid));
      if (nit != nend)
        parent = nit->second;
    }

    if (parent) {
      tree_node->_parent = parent;
      parent->_children.push_back (tree_node);
      added.parent = parent->_mid;
    }

    //std::cerr << LINE_ID << " child " << added.message_id << " has parent " << added.parent << std::endl;
    diffs.added.push_back (added);
  }

  // maybe find new parents for descendants of the new articles...

  // get a list of all articles that are descendants of the new articles
  quarks_t descendants;
  foreach (nodes_v, nodes, it)
    accumulate_descendants (descendants, *it);

  // don't need to reparent the new nodes because we've already
  // parented them.
  if (1) {
    quarks_t tmp;
    std::set_difference (descendants.begin(), descendants.end(),
                         new_mids.begin(), new_mids.end(),
                         inserter (tmp, tmp.begin()));
    descendants.swap (tmp);
  }
    
  // convert that into a list of tree nodes
  // that need to be tested for reparenting
  nodes_v my_descendants;
  nodes_t::iterator nit(_nodes.begin()), nend(_nodes.end());
  quarks_t::iterator dit(descendants.begin()), dend(descendants.end());
  while (nit!=nend && dit!=dend) {
    if (nit->second->_mid < *dit)
      ++nit;
    else if (*dit < nit->second->_mid)
      ++dit;
    else {
      //std::cerr << LINE_ID << " descendant " << nit->first << " may need reparenting" << std::endl;
      my_descendants.push_back (nit->second);
      ++nit;
      ++dit;
    }
  }
  //std::cerr << LINE_ID << " " << my_descendants.size() << " articles may need to be reparented..." << std::endl;

  // reparent each of those nodes...
  foreach (nodes_v, my_descendants, it)
  {
    ArticleNode * tree_node (*it);
    ArticleNode * new_parent (0);

//std::cerr << LINE_ID << " looking for a new parent for " << tree_node->_mid << std::endl;
    
    const ArticleNode * node (h->find_node (tree_node->_mid));
    node = node->_parent;
    while (node && !new_parent) {
      //std::cerr << LINE_ID << " maybe " << node->_mid <<" can be its new parent..." << std::endl;
      nodes_t::iterator nit (_nodes.find (node->_mid));
      if (nit != _nodes.end())
        new_parent = nit->second;
      else {
        node = node->_parent;
        //std::cerr << LINE_ID << " but no, that parent isn't in the tree." << std::endl;
      }
    }
//std::cerr << LINE_ID << " " << tree_node->_mid << "'s best parent is " << new_parent << std::endl;

    if (new_parent != tree_node->_parent) {
      ArticleNode * old_parent (tree_node->_parent);
      ArticleTree::Diffs::Reparent reparent;
      if (old_parent) {
        reparent.old_parent = old_parent->_mid;
        old_parent->_children.remove (tree_node);
      }
      new_parent->_children.push_back (tree_node);
      tree_node->_parent = new_parent;
      reparent.message_id = tree_node->_mid;
      reparent.new_parent = new_parent->_mid;
//std::cerr << LINE_ID << " REPARENTED: " << reparent.message_id << " has a new parent " << reparent.new_parent << std::endl;
      diffs.reparented.push_back (reparent);
    }
  }

  if (!diffs.reparented.empty() || !diffs.added.empty())
    fire_diffs (diffs);
}
