/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>

#include <string.h>

#include <pan/base/acache.h>
#include <pan/base/argset.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/gnksa.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>
#include <pan/base/status-item.h>

#include <pan/filters/filter-aggregate.h>
#include <pan/filters/filter-phrase.h>
#include <pan/filters/filter-score.h>
#include <pan/filters/score.h>

#include <pan/article-actions.h>
#include <pan/articlelist.h>
#include <pan/message-window.h>
#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/save-ui.h>
#include <pan/task.h>
#include <pan/task-bodies.h>
#include <pan/task-cancel.h>
#include <pan/task-save.h>
#include <pan/util.h>

/***
****  Finding a match in pan.sent
***/

typedef struct
{
	GMimeMessage * message;
	char * body;
	GMimeMessage * match;
}
FindMatchingStruct;

static void
find_matching_sent_foreach (GMimeMessage * compare, gpointer user_data)
{
	char * body;
	gboolean is_html;
	FindMatchingStruct * data = (FindMatchingStruct*) user_data;

	/* do we already have a match? */
	if (data->match != NULL)
		return;

	/* comparison #1: subject must match */
	if (pan_strcmp (g_mime_message_get_subject(compare), g_mime_message_get_subject(data->message)))
		return;

	/* comparison #2: author address must match */
	if (pan_strcmp (g_mime_message_get_sender(compare), g_mime_message_get_sender(data->message)))
		return;

	/* comparison #3: body must match, if we have one */
	body = g_mime_message_get_body (compare, TRUE, &is_html);
	if (body!=NULL && data->body!=NULL) {
		g_strstrip (body);
		if (!strcmp (body, data->body)) {
			g_object_ref (compare);
			data->match = compare;
			g_free (body);
		}
	}
}

/**
 * Find a matching article without comparing by message-id.
 * This is because many news servers alter the message-id of
 * an article being posted, so we can't rely on searching by
 * the message-id that Pan generated.
 */
static GMimeMessage*
find_matching_sent (GMimeMessage* message)
{
	gboolean is_html;
	FindMatchingStruct data;
	debug_enter ("find_matching_sent_article");

	/* sanity clause */
	g_return_val_if_fail (GMIME_IS_MESSAGE(message), NULL);

	/* find the match */
	data.message = message;
	data.body = g_strstrip (g_mime_message_get_body (data.message, TRUE, &is_html));
	data.match = NULL;
	acache_path_foreach (PAN_SENT, find_matching_sent_foreach, &data);
	g_free (data.body);
	return data.match;
}

/***
****
***/

void
article_cancel (Server * server, GMimeMessage * message)
{
	GMimeMessage * ours = find_matching_sent (message);

	if (ours != NULL)
		queue_add (TASK(task_cancel_new (server, ours)));
	else
		log_add_va (LOG_ERROR|LOG_URGENT,
		            _("Unable to cancel article: Couldn't find matching article in folder `pan.sent'!"));
}

/****
*****
****/

void
article_supersede (GMimeMessage * message)
{
	GMimeMessage * ours;
	debug_enter ("article_supersede");

	/* sanity clause */
	g_return_if_fail (GMIME_IS_MESSAGE (message));

	/* find the original copy so that the headers will match. */
	ours = find_matching_sent (message);
	if (ours == NULL)
	{
		log_add_va (LOG_ERROR|LOG_URGENT,
		            _("Unable to supersede article: Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		char       * domain;
		char       * pch;
		const char * old_message_id;
		char       * new_message_id;

		/* get the new message-id */
		old_message_id = g_mime_message_get_message_id (ours);
		pch = g_strdup (old_message_id);
		domain = strchr (pch, '@');
		if (domain)
		{
			size_t len = strlen (++domain);
			if (domain[len-1] == '>')
				domain[len-1]='\0';
		}
		new_message_id = gnksa_generate_message_id (domain);

		/* update the headers for superseding */
		g_mime_message_set_header (ours, HEADER_SUPERSEDES, old_message_id);
		g_mime_message_set_header (ours, HEADER_MESSAGE_ID, new_message_id);

		/* edit the message */
		message_edit_window (ours);

		/* cleanup */
		g_free (new_message_id);
		g_free (pch);
	}

	debug_exit ("article_supersede");
}

/****
*****
****/

static void
article_copy_articles_to_folder_cb (gpointer obj, gpointer arg, gpointer data)
{
	int status;
	Task * task;
	Group * folder;
	ArgSet * args;
	char * acache_key;
	debug_enter ("article_copy_articles_to_folder_cb");

	/* sanity clause */
	g_return_if_fail (obj!=NULL);
	args = (ArgSet*) data;
	g_return_if_fail (args!=NULL);
	acache_key = (char*) argset_get (args, 0);
	g_return_if_fail (is_nonempty_string (acache_key));
	folder = (Group*) argset_get (args, 1);
	g_return_if_fail (group_is_valid(folder));
	g_return_if_fail (group_is_folder(folder));

	/* pump argument list */
	status = GPOINTER_TO_INT(arg);
	task = TASK(obj);

	/* copy them over if the task succeeded ... */
	if (status == TASK_SUCCESS)
	{
		guint i;
		for (i=0; i<task->identifiers->len; ++i)
		{
			MessageIdentifier * mid = MESSAGE_IDENTIFIER (g_ptr_array_index (task->identifiers, i));
			GMimeMessage * message = acache_get_message (acache_key, &mid, 1);
			folder_add_message (folder, message);
			g_object_unref (message);
		}
	}

	/* cleanup */
	g_free (acache_key);
	argset_free (args);
	debug_exit ("articlelist_article_to_folder_cb");
}

gboolean
article_copy_articles_to_folder (Group * folder, const Article ** articles, int qty)
{
	int i;
	int missing_qty;
	MessageIdentifier ** missing;
	debug_enter ("article_copy_articles_to_folder");

	/* sanity clause */
	g_return_val_if_fail (folder!=NULL, FALSE);
	g_return_val_if_fail (group_is_folder(folder), FALSE);
	g_return_val_if_fail (articles!=NULL, FALSE);
	g_return_val_if_fail (qty>0, FALSE);
	g_return_val_if_fail (articles_are_valid(articles, qty), FALSE);

	/* copy anything that's already cached, and make a list of those not cached */
	missing = g_newa (MessageIdentifier*, qty);
	missing_qty = 0;
	for (i=0; i<qty; ++i)
	{
		const Article * article = articles[i];
		const char * acache_key = group_get_acache_key (article->group);
		MessageIdentifier * mid = message_identifier_new_from_article (article);
		GMimeMessage * message = acache_get_message (acache_key, &mid, 1);

		if (message == NULL)
			missing[missing_qty++] = mid;
		else
		{
			folder_add_message (folder, message);
			g_object_unref (message);
			g_object_unref (mid);
		}
	}

	/* queue anything that's missing */
	if (missing_qty != 0)
	{
		Task * task = TASK (task_bodies_new (articles[0]->group->server, missing, missing_qty));

		if (task != NULL)
		{
			const char * source_key = group_get_acache_key (articles[0]->group);
			ArgSet * args = argset_new2 (g_strdup(source_key), folder);
			pan_callback_add (task->task_ran_callback, article_copy_articles_to_folder_cb, args);
			queue_add (task);
		}
	}

	/* cleanup */
	for (i=0; i<missing_qty; ++i)
		g_object_unref (missing[i]);
	debug_exit ("article_copy_articles_to_folder");
	return TRUE;
}

/***
****
****
****   SAVING ARTICLES / OPENING ATTACHMENTS
****
****
***/

static void
save_attachments_impl (Article ** articles, int len)
{
	int i;
	GSList * tasks = NULL;
	debug_enter ("save_attachments_impl");

	g_return_if_fail (articles_are_valid ((const Article**)articles, len));

	for (i=0; i<len; ++i)
	{
		GSList * l;
		Article * a = articles[i];
		PanObject * task;
		GPtrArray * tmp = g_ptr_array_sized_new (64);

		/* build an array of this article and all its children. */
		g_ptr_array_add (tmp, a);
		for (l=a->threads; l!=NULL; l=l->next)
			g_ptr_array_add (tmp, l->data);

		/* queue for decode */
		task_save_weed_duplicates (tmp);
		task = task_save_new_from_articles ((const Article**)tmp->pdata, tmp->len);
		task_save_set_attachments (TASK_SAVE(task), NULL, NULL);
		tasks = g_slist_prepend (tasks, task);

		/* cleanup */
		g_ptr_array_free (tmp, TRUE);
	}

	if (tasks != NULL)
	{
		tasks = g_slist_reverse (tasks);
		queue_insert_tasks (tasks, -1);
		tasks = NULL;
	}

	/* cleanup this loop */
	debug_exit ("save_attachments_impl");
}

void
article_action_selected_save_attachments (void)
{
	GPtrArray * sel;
	debug_enter ("article_action_selected_save_attachments");

	/* decode each selected article */
	sel = articlelist_get_selected_articles_nolock ();
	save_attachments_impl ((Article**)sel->pdata, sel->len);
	g_ptr_array_free (sel, TRUE);

	debug_exit ("article_action_selected_save_attachments");
}

void
article_action_selected_save_as (void)
{
	guint i;
	GPtrArray * sel;
	Group * group = articlelist_get_group ();
	GSList * gslists_of_articles = NULL;
	debug_enter ("articlelist_selected_decode_as");

	g_return_if_fail (group != NULL);

	/* loop through all the selected articles */
	sel = articlelist_get_selected_articles_nolock ();
	for (i=0; i!=sel->len; ++i)
	{
		GSList * l = NULL;
		Article * article = ARTICLE(g_ptr_array_index(sel,i));

		/* build an array of this node and all its children. */
		if (article->parts > 1)
			l = g_slist_copy (article->threads);
		l = g_slist_prepend (l, article);

		/* add this to the gslist of articles */
		gslists_of_articles = g_slist_prepend (gslists_of_articles, l);
	}

	/* save_attach_as owns gslists_of_articles now */
	gslists_of_articles = g_slist_reverse (gslists_of_articles);
	save_attachment_as (group, gslists_of_articles);

	g_ptr_array_free (sel, TRUE);
	debug_exit ("articlelist_selected_decode_as");
}

/***
****   DELETING ARTICLES
***/

void
article_action_delete_selected_articles (void)
{
	GPtrArray * articles;
	int i, len;
	debug_enter ("article_action_delete_selected_articles");

	/* if it's a multipart, delete the children too */
       	articles = articlelist_get_active_articles_nolock ();
	for (i=0, len=articles->len; i!=len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		if (a->part==1 && a->parts>1) {
			GSList * l;
			for (l=a->threads; l!=NULL; l=l->next)
				if (ARTICLE(l->data)->part > 1)
					g_ptr_array_add (articles, l->data);
		}
	}

	/* delete the articles */
	if (articles->len)
	{
		Group * group;
		MessageIdentifier ** mids;

		/* remove the headers */
		mids = g_new (MessageIdentifier*, articles->len);
		for (i=0; i<articles->len; ++i)
			mids[i] = message_identifier_new_from_article (ARTICLE(g_ptr_array_index(articles,i)));
		message_identifiers_delete ((const MessageIdentifier**)mids, articles->len, SERVER_GROUPS_SUBSCRIBED);

		/* remove the cached bodies */
	       	group = articlelist_get_group ();
		acache_expire_message_ids (group_get_acache_key(group), (const MessageIdentifier**)mids, articles->len);

		/* cleanup */
		for (i=0; i<articles->len; ++i)
			g_object_unref (mids[i]);
		g_free (mids);
	}

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	debug_exit ("article_action_delete_selected_articles");
}

void
article_action_edit_selected (void)
{
        Article * a = NULL;
	MessageIdentifier * mid = NULL;
	GMimeMessage * message = NULL;
	debug_enter ("article_action_edit_selected");

	a = articlelist_get_selected_article_nolock ();
	if (a != NULL)
		mid = message_identifier_new_from_article (a);
	if (mid != NULL)
		message = acache_get_message (group_get_acache_key(a->group), &mid, 1);
	if (message != NULL)
		message_edit_window (message);

	g_object_unref (mid);
	debug_exit ("article_action_edit_selected");
}

/***
****
***/

enum {
	DO_WATCH,
	DO_IGNORE,
	DO_LIST
};

static void
article_action_something_thread (const Article * article, int action)
{
	ScoreAddItem add_items[2];
	GPtrArray * articles;
	char * score_name;
	char * thread_message_id;
	struct {
		int score;
		const char * verb;
	} do_list[DO_LIST] = {
		{ 9999, "watch" },
		{ -9999, "ignore" }
	};

	/* sanity checking */
	g_return_if_fail (article_is_valid (article));
	g_return_if_fail (action==DO_WATCH || action==DO_IGNORE);

	/* get the new score's name */
	thread_message_id = article_get_thread_message_id (article);
	score_name = g_strdup_printf ("%s thread \"%s\"",
		do_list[action].verb,
		article_get_subject (article_get_root (article)));

	/* if the article's Message-ID is the top article, 
	 * the article's Referencess contains the top article,
	 * set the score to watch/ignore */
	add_items[0].on = TRUE;
	add_items[0].negate = FALSE;
	add_items[0].key = HEADER_MESSAGE_ID;
	add_items[0].value = filter_phrase_create_regex (thread_message_id, PHRASE_MATCH_IS);
	add_items[1].on = TRUE;
	add_items[1].negate = FALSE;
	add_items[1].key = HEADER_REFERENCES;
	add_items[1].value = filter_phrase_create_regex (thread_message_id, PHRASE_MATCH_STARTS_WITH);
	score_add (score_name, NULL, do_list[action].score, TRUE, 31, AGGREGATE_TYPE_OR, add_items, 2);

	/* rescore the thread */
	articles = g_ptr_array_new ();
	article_get_thread (article, articles);
	ensure_articles_scored (article->group, (Article**)articles->pdata, articles->len);

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	g_free (add_items[1].value);
	g_free (add_items[0].value);
	g_free (score_name);
	g_free (thread_message_id);
}

void
article_action_watch_thread (const Article * article)
{
	article_action_something_thread (article, DO_WATCH);
}

void
article_action_watch_selected_thread (void)
{
	Article * article = articlelist_get_selected_article_nolock ();
	if (article != NULL)
		article_action_watch_thread (article);
}

void
article_action_ignore_thread (const Article * article)
{
	article_action_something_thread (article, DO_IGNORE);
}

void
article_action_ignore_selected_thread (void)
{
	Article * article = articlelist_get_selected_article_nolock ();
	if (article != NULL)
		article_action_ignore_thread (article);
}
